//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
//-----------------------------------------------------------------------------
//
// Description:
//      cdevDataEntry Class Which is contained inside cdevData
//
//      Note: This data entry only handle one dimensional array
//
// Author:  
//      Jie Chen
//      Jefferson Lab HPC Group
//
// Revision History:
//   $Log: cdevDataEntry.java,v $
//   Revision 1.1  1999/12/14 15:31:37  chen
//   initial java implementation
//
//
//
//
package cmlog;

import java.io.IOException;
import cmlog.cdevDataTypes;
import cmlog.cdevTimeStamp;
import cmlog.cdevDataOutputStream;
import cmlog.cdevTagTable;

public final class cdevDataEntry 
{
    private int    tag_;   
    private short  type_;
    private short  ndims_;
    private int    nelems_;
    private Number []data_;
    private String []sdata_;

    /**
     * Construct an empty data entry
     */
    public cdevDataEntry ()
    {
	tag_ = -1;
	type_ = cdevDataTypes.CDEV_INVALID;
	ndims_ = -1;
	nelems_ = 0;
    }

    /**
     * Construct a tagged data entry from a double value
     */
    public cdevDataEntry (int tag, double d)
    {
	tag_ = tag;
	ndims_ = 0;
	nelems_ = 1;
	type_ = cdevDataTypes.CDEV_DOUBLE;
	data_ = new Double[1];
	data_[0] = new Double (d);
    }

    /**
     * Construct a tagged data entry from a float value
     */
    public cdevDataEntry (int tag, float f)
    {
	tag_ = tag;
	ndims_ = 0;
	nelems_ = 1;
	type_ = cdevDataTypes.CDEV_FLOAT;
	data_ = new Float[1];
	data_[0] = new Float(f);
    }


    /**
     * Construct a tagged data entry from a long value
     */
    public cdevDataEntry (int tag, long l)
    {
	tag_ = tag;
	ndims_ = 0;
	nelems_ = 1;
	type_ = cdevDataTypes.CDEV_LONG;
	data_ = new Long[1];
	data_[0] = new Long(l);
    }

    /**
     * Construct a tagged data entry from an integer value
     */
    public cdevDataEntry (int tag, int i)
    {
	tag_ = tag;
	ndims_ = 0;
	nelems_ = 1;
	type_ = cdevDataTypes.CDEV_INT32;
	data_ = new Integer[1];
	data_[0] = new Integer(i);
    }    

    /**
     * Construct a tagged data entry from a short value
     */
    public cdevDataEntry (int tag, short s)
    {
	tag_ = tag;
	ndims_ = 0;
	nelems_ = 1;
	type_ = cdevDataTypes.CDEV_INT16;
	data_ = new Short[1];
	data_[0] = new Short(s);
    }

    /**
     * Construct a tagged data entry from a byte value
     */
    public cdevDataEntry (int tag, byte b)
    {
	tag_ = tag;
	ndims_ = 0;
	nelems_ = 1;
	type_ = cdevDataTypes.CDEV_BYTE;
	data_ = new Byte[1];
	data_[0] = new Byte(b);
    }

    /**
     * Construct a tagged data entry from a string
     */
    public cdevDataEntry (int tag, String str)
    {
	tag_ = tag;
	ndims_ = 0;
	nelems_ = 1;
	type_ = cdevDataTypes.CDEV_STRING;
	sdata_ = new String[1];
	sdata_[0] = str;
    }

    /**
     * Construct a tagged data entry from a time stamp
     */
    public cdevDataEntry (int tag, cdevTimeStamp ts)
    {
	tag_ = tag;
	ndims_ = 0;
	nelems_ = 1;
	type_ = cdevDataTypes.CDEV_TIMESTAMP;
	data_ = new cdevTimeStamp[1];
	data_[0] = ts;
    }

    /**
     * Construct a tagged data entry from an array of byte
     */
    public cdevDataEntry (int tag, byte[] barray)
    {
	tag_ = tag;
	ndims_ = 1;
	nelems_ = barray.length;
	type_ = cdevDataTypes.CDEV_BYTE;
	data_ = new Byte[nelems_];
	for (int i = 0; i < nelems_; i++)
	    data_[i] = new Byte(barray[i]);
    }

    /**
     * Construct a tagged data entry from an array of short
     */
    public cdevDataEntry (int tag, short[] sarray)
    {
	tag_ = tag;
	ndims_ = 1;
	nelems_ = sarray.length;
	type_ = cdevDataTypes.CDEV_INT16;
	data_ = new Short[nelems_];
	for (int i = 0; i < nelems_; i++)
	    data_[i] = new Short(sarray[i]);
    }
    
    /**
     * Construct a tagged data entry from an array of integer
     */
    public cdevDataEntry (int tag, int[] iarray)
    {
	tag_ = tag;
	ndims_ = 1;
	nelems_ = iarray.length;
	type_ = cdevDataTypes.CDEV_INT32;
	data_ = new Integer[nelems_];
	for (int i = 0; i < nelems_; i++)
	    data_[i] = new Integer(iarray[i]);
    }

    /**
     * Construct a tagged data entry from an array of long
     */
    public cdevDataEntry (int tag, long[] larray)
    {
	tag_ = tag;
	ndims_ = 1;
	nelems_ = larray.length;
	type_ = cdevDataTypes.CDEV_LONG;
	data_ = new Long[nelems_];
	for (int i = 0; i < nelems_; i++)
	    data_[i] = new Long(larray[i]);
    }

    /**
     * Construct a tagged data entry from an array of float
     */
    public cdevDataEntry (int tag, float[] farray)
    {
	tag_ = tag;
	ndims_ = 1;
	nelems_ = farray.length;
	type_ = cdevDataTypes.CDEV_FLOAT;
	data_ = new Float[nelems_];
	for (int i = 0; i < nelems_; i++)
	    data_[i] = new Float (farray[i]);	
    }

    /**
     * Construct a tagged data entry from an array of double
     */
    public cdevDataEntry (int tag, double[] darray)
    {
	tag_ = tag;
	ndims_ = 1;
	nelems_ = darray.length;
	type_ = cdevDataTypes.CDEV_DOUBLE;
	data_ = new Double[nelems_];
	for (int i = 0; i < nelems_; i++)
	    data_[i] = new Double(darray[i]);	
    }

    /**
     * Construct a tagged data entry from an array of strings
     */
    public cdevDataEntry (int tag, String[] strs)
    {
	tag_ = tag;
	ndims_ = 1;
	nelems_ = strs.length;
	type_ = cdevDataTypes.CDEV_STRING;
	sdata_ = new String[nelems_];
	for (int i = 0; i < nelems_; i++)
	    sdata_[i] = strs[i];
    }

    /**
     * Construct a tagged data entry from an array of time stamps
     */
    public cdevDataEntry (int tag, cdevTimeStamp[] tsarray)
    {
	tag_ = tag;
	ndims_ = 1;
	nelems_ = tsarray.length;
	type_ = cdevDataTypes.CDEV_TIMESTAMP;
	data_ = new cdevTimeStamp[nelems_];
	for (int i = 0; i < nelems_; i++)
	    data_[i] = tsarray[i];
    }

    /**
     * Hash code for this data entry
     */
    public int hashCode()
    {
	Integer itag = new Integer (tag_);
	return itag.hashCode ();
    }

    /**
     * Compare two data entries
     */
    public boolean equals(cdevDataEntry dobj)
    {
	if (dobj.type_ != type_ ||
	    dobj.nelems_ != nelems_ ||
	    dobj.ndims_ != ndims_ ||
	    dobj.tag_ != tag_)
	    return false;

	int i, j;
	if (type_ != cdevDataTypes.CDEV_INVALID) {
	    if (type_ == cdevDataTypes.CDEV_STRING) {
		for (i = 0; i < nelems_; i++) {
		    if (sdata_[i].equals (dobj.sdata_[i]) == false)
			return false;
		}
	    }
	    else {
		for (i = 0; i < nelems_; i++) {
		    if (data_[i].equals (dobj.data_[i]) == false)
			return false;
		}
	    }
	}
	return true;
    }
	

    /**
     * Retrieve internal data as a byte value
     */
    public byte byteValue ()
    {
	if (type_ != cdevDataTypes.CDEV_INVALID &&
	    type_ != cdevDataTypes.CDEV_STRING)
	    return data_[0].byteValue();
	else if (type_ == cdevDataTypes.CDEV_STRING)
	    return Byte.valueOf(sdata_[0]).byteValue();
	else throw new NumberFormatException("Uninitialize cdevDataEntry Object");
    }

    /**
     * Retrieve internal data as a short value
     */
    public short shortValue ()
    {
	if (type_ != cdevDataTypes.CDEV_INVALID &&
	    type_ != cdevDataTypes.CDEV_STRING)
	    return data_[0].shortValue();
	else if (type_ == cdevDataTypes.CDEV_STRING)
	    return Short.valueOf(sdata_[0]).shortValue();
	else throw new NumberFormatException("Uninitialize cdevDataEntry Object");
    }

    /**
     * Retrieve internal data as an integer value
     */
    public int intValue ()
    {
	if (type_ != cdevDataTypes.CDEV_INVALID &&
	    type_ != cdevDataTypes.CDEV_STRING)
	    return data_[0].intValue();
	else if (type_ == cdevDataTypes.CDEV_STRING)
	    return Integer.valueOf(sdata_[0]).intValue();
	else throw new NumberFormatException("Uninitialize cdevDataEntry Object");
    }

    /**
     * Retrieve internal data as a long value
     */
    public long longValue ()
    {
	if (type_ != cdevDataTypes.CDEV_INVALID &&
	    type_ != cdevDataTypes.CDEV_STRING)
	    return data_[0].longValue();
	else if (type_ == cdevDataTypes.CDEV_STRING)
	    return Long.valueOf(sdata_[0]).longValue();
	else throw new NumberFormatException("Uninitialize cdevDataEntry Object");
    }

    /**
     * Retrieve internal data as a float value
     */
    public float floatValue ()
    {
	if (type_ != cdevDataTypes.CDEV_INVALID &&
	    type_ != cdevDataTypes.CDEV_STRING)
	    return data_[0].floatValue();
	else if (type_ == cdevDataTypes.CDEV_STRING)
	    return Float.valueOf(sdata_[0]).floatValue();
	else throw new NumberFormatException("Uninitialize cdevDataEntry Object");
    }

    /**
     * Retrieve internal data as a double value
     */    
    public double doubleValue ()
    {
	if (type_ != cdevDataTypes.CDEV_INVALID &&
	    type_ != cdevDataTypes.CDEV_STRING)
	    return data_[0].doubleValue();
	else if (type_ == cdevDataTypes.CDEV_STRING)
	    return Double.valueOf(sdata_[0]).doubleValue();
	else throw new NumberFormatException("Uninitialize cdevDataEntry Object");
    }

    /**
     * Retrieve internal data as a string
     */
    public String stringValue ()
    {
	if (type_ != cdevDataTypes.CDEV_INVALID &&
	    type_ != cdevDataTypes.CDEV_STRING)
	    return String.valueOf (data_[0]);
	else if (type_ == cdevDataTypes.CDEV_STRING)	
	    return sdata_[0];
	else throw new NumberFormatException("Uninitialize cdevDataEntry Object");
    }

    /**
     * Retrieve internal data as an array of byte
     */
    public byte[] byteArray ()
    {
	int i;
	if (type_ != cdevDataTypes.CDEV_INVALID &&
	    type_ != cdevDataTypes.CDEV_STRING) {
	    byte[] barray = new byte[nelems_];
	    for (i = 0; i < nelems_; i++)
		barray[i] = data_[i].byteValue();
	    return barray;
	}
	else if (type_ == cdevDataTypes.CDEV_STRING) {
	    byte[] barray = new byte[nelems_];
	    for (i = 0; i < nelems_; i++)
		barray[i] = Byte.valueOf(sdata_[i]).byteValue();
	    return barray;
	}
	else throw new NumberFormatException("Uninitialize cdevDataEntry Object");
    }

    /**
     * Retrieve internal data as an array of short
     */
    public short[] shortArray ()
    {
	int i;
	if (type_ != cdevDataTypes.CDEV_INVALID &&
	    type_ != cdevDataTypes.CDEV_STRING) {
	    short[] sarray = new short[nelems_];
	    for (i = 0; i < nelems_; i++)
		sarray[i] = data_[i].shortValue();
	    return sarray;
	}
	else if (type_ == cdevDataTypes.CDEV_STRING) {
	    short[] sarray = new short[nelems_];
	    for (i = 0; i < nelems_; i++)
		sarray[i] = Short.valueOf(sdata_[i]).shortValue();
	    return sarray;
	}
	else throw new NumberFormatException("Uninitialize cdevDataEntry Object");
    }

    /**
     * Retrieve internal data as an array of integer
     */
    public int[] intArray ()
    {
	int i;
	if (type_ != cdevDataTypes.CDEV_INVALID &&
	    type_ != cdevDataTypes.CDEV_STRING) {
	    int[] iarray = new int[nelems_];
	    for (i = 0; i < nelems_; i++)
		iarray[i] = data_[i].intValue();
	    return iarray;
	}
	else if (type_ == cdevDataTypes.CDEV_STRING) {
	    int[] iarray = new int[nelems_];
	    for (i = 0; i < nelems_; i++)
		iarray[i] = Integer.valueOf(sdata_[i]).intValue();
	    return iarray;
	}
	else throw new NumberFormatException("Uninitialize cdevDataEntry Object");
    }

    /**
     * Retrieve internal data as an array of float
     */
    public float[] floatArray ()
    {
	int i;
	if (type_ != cdevDataTypes.CDEV_INVALID &&
	    type_ != cdevDataTypes.CDEV_STRING) {
	    float[] farray = new float[nelems_];
	    for (i = 0; i < nelems_; i++)
		farray[i] = data_[i].floatValue();
	    return farray;
	}
	else if (type_ == cdevDataTypes.CDEV_STRING) {
	    float[] farray = new float[nelems_];
	    for (i = 0; i < nelems_; i++)
		farray[i] = Float.valueOf(sdata_[i]).floatValue();
	    return farray;
	}
	else throw new NumberFormatException("Uninitialize cdevDataEntry Object");
    }

    /**
     * Retrieve internal data as an array of double
     */
    public double[] doubleArray ()
    {
	int i;
	if (type_ != cdevDataTypes.CDEV_INVALID &&
	    type_ != cdevDataTypes.CDEV_STRING) {
	    double[] darray = new double[nelems_];
	    for (i = 0; i < nelems_; i++)
		darray[i] = data_[i].doubleValue();
	    return darray;
	}
	else if (type_ == cdevDataTypes.CDEV_STRING) {
	    double[] darray = new double[nelems_];
	    for (i = 0; i < nelems_; i++)
		darray[i] = Double.valueOf(sdata_[i]).doubleValue();
	    return darray;
	}
	else throw new NumberFormatException("Uninitialize cdevDataEntry Object");
    }

    /**
     * Retrieve internal data as an array of long
     */
    public long[] longArray ()
    {
	int i;
	if (type_ != cdevDataTypes.CDEV_INVALID &&
	    type_ != cdevDataTypes.CDEV_STRING) {
	    long[] larray = new long[nelems_];
	    for (i = 0; i < nelems_; i++)
		larray[i] = data_[i].longValue();
	    return larray;
	}
	else if (type_ == cdevDataTypes.CDEV_STRING) {
	    long[] larray = new long[nelems_];
	    for (i = 0; i < nelems_; i++)
		larray[i] = Long.valueOf(sdata_[i]).longValue();
	    return larray;
	}
	else throw new NumberFormatException("Uninitialize cdevDataEntry Object");
    }

    /**
     * Retrieve internal data as an array of strings
     */
    public String[] stringArray ()
    {
	int i;
	if (type_ != cdevDataTypes.CDEV_INVALID &&
	    type_ != cdevDataTypes.CDEV_STRING) {
	    String[] sarray = new String[nelems_];
	    for (i = 0; i < nelems_; i++) 
		sarray[i] = String.valueOf (data_[i]);
	    return sarray;
	}
	else if (type_ == cdevDataTypes.CDEV_STRING) {
	    String[] sarray = new String[nelems_];	
	    for (i = 0; i < nelems_; i++) 
		sarray[i] = sdata_[i];
	    return sarray;
	}
	else throw new NumberFormatException("Uninitialize cdevDataEntry Object");
    }

    /**
     *convert the data entry into a string representation
     */
    public String toString ()
    {
	cdevTagTable tagtable = cdevTagTable.tagTable();
	String       stag;


	try {
	    stag = tagtable.convertKey (tag_);
	}catch (IllegalArgumentException e) {
	    stag = String.valueOf (tag_);
	}


	StringBuffer result = new StringBuffer();
	int i, j;

	if (type_ == cdevDataTypes.CDEV_INVALID)
	    result.append ("Invalid DataEntry Object");
	else {
	    result.append ("Key:       ").append (stag).append ("\n");
	    result.append ("Type:      ").append (cdevDataTypes.types[type_]).append("\n");
	    result.append ("NElems:     ").append(nelems_).append("\n");
	    result.append ("Values:     \n");
	    
	    j = 0;
	    String[] sarray = stringArray();
	    if (type_ == cdevDataTypes.CDEV_STRING) {
		result.append ("\n");
		for (i = 0; i < nelems_; i++) {
		    result.append ("\t");
		    result.append(sarray[i]).append("\n");
		}
	    }
	    else {
		for (i = 0; i < nelems_; i++) {
		    if (j == 0)
			result.append ("\t");
		    result.append(sarray[i]).append(" ");
		    j++;
		    if (j >= 8) {
			result.append ("\n");
			j = 0;
		    }
		}
	    }
	}
	return result.toString();
    }

    /**
     * Provide an exact deep copy of this data entry 
     */
    public cdevDataEntry copy ()
    {
	int i;
	cdevDataEntry de = new cdevDataEntry();
	
	de.type_ = type_;
	de.nelems_ = nelems_;
	de.ndims_ = ndims_;
	de.tag_ = tag_;

	if (type_ != cdevDataTypes.CDEV_INVALID) {
	    switch (type_) {
	    case cdevDataTypes.CDEV_STRING:
		de.sdata_ = new String[nelems_];
		for (i = 0; i < nelems_; i++)
		    de.sdata_[i] = new String(sdata_[i]);
		break;
	    case cdevDataTypes.CDEV_BYTE:
		de.data_ = new Byte[nelems_];
		for (i = 0; i < nelems_; i++)
		    de.data_[i] = new Byte(data_[i].byteValue());
		break;
	    case cdevDataTypes.CDEV_INT16:
		de.data_ = new Short[nelems_];
		for (i = 0; i < nelems_; i++)
		    de.data_[i] = new Short(data_[i].shortValue());
		break;
	    case cdevDataTypes.CDEV_INT32:
		de.data_ = new Integer[nelems_];
		for (i = 0; i < nelems_; i++)
		    de.data_[i] = new Integer(data_[i].intValue());
		break;
	    case cdevDataTypes.CDEV_FLOAT:
		de.data_ = new Float[nelems_];
		for (i = 0; i < nelems_; i++)
		    de.data_[i] = new Float(data_[i].floatValue());
		break;
	    case cdevDataTypes.CDEV_DOUBLE:
		de.data_ = new Double[nelems_];
		for (i = 0; i < nelems_; i++)
		    de.data_[i] = new Double(data_[i].doubleValue());
		break;
	    case cdevDataTypes.CDEV_LONG:
		de.data_ = new Long[nelems_];
		for (i = 0; i < nelems_; i++)
		    de.data_[i] = new Long(data_[i].longValue());
		break;
	    }
	}
	return de;
    }

    /**
     * Return tag of this data entry
     */
    public int getTag ()
    {
	return tag_;
    }

    /**
     * Set tag to new tag
     */
    public void setTag (int tag)
    {
	tag_ = tag;
    }

    /**
     * Return data type in numeric value
     */
    public short getType ()
    {
	return type_;
    }

    /**
     * Return data type in Class
     */
    public Class getClassType () throws ClassNotFoundException
    {
	if (type_ == cdevDataTypes.CDEV_INVALID)
	    throw new ClassNotFoundException ("Invalid data entry object");
	else {
	    if (type_ == cdevDataTypes.CDEV_STRING)
		return sdata_.getClass();
	    else
		return data_.getClass();
	}
    }

    /**
     * Return dimensionality information
     */
    public short getDimension ()
    {
	return ndims_;
    }

    /**
     * Return number of elements
     */
    public int getNumElements ()
    {
	return nelems_;
    }

    /**
     * Return streamed data size for this data entry
     */
    public int streamSize ()
    {
	int datasize = 0;
	int i;
	
	if (type_ == cdevDataTypes.CDEV_INVALID)
	    return datasize;

	 // size of tag is fixed
	datasize += cdevDataOutputStream.streamSize (tag_);
	// data type, dim, nelems
	datasize += 3*cdevDataOutputStream.streamSize ((int)1);

	// for compatibility with C++ implementation, We have to
	// pass dimension information
	if (ndims_ > 0) {
	    datasize += cdevDataOutputStream.streamSize ((int)0);
	    datasize += cdevDataOutputStream.streamSize ((int)nelems_);
	}
	
	if (type_ == cdevDataTypes.CDEV_STRING) {
	    if (nelems_ == 1) 
		datasize += cdevDataOutputStream.streamSize (sdata_[0]);
	    else {
		for (i = 0; i < nelems_; i++)
		    datasize += cdevDataOutputStream.streamSize (sdata_[i]);
	    }
	}
	else if(type_ == cdevDataTypes.CDEV_BYTE) 
	    datasize += nelems_*cdevDataOutputStream.streamSize ((byte)1);
	else if(type_ == cdevDataTypes.CDEV_INT16)
	    datasize += nelems_*cdevDataOutputStream.streamSize ((short)1);
	else if(type_ == cdevDataTypes.CDEV_INT32)
	    datasize += nelems_*cdevDataOutputStream.streamSize ((int)1);
	else if(type_ == cdevDataTypes.CDEV_FLOAT)
	    datasize += nelems_*cdevDataOutputStream.streamSize ((float)1.0);
	else if(type_ == cdevDataTypes.CDEV_DOUBLE)
	    datasize += nelems_*cdevDataOutputStream.streamSize ((double)1.0);
	else if(type_ == cdevDataTypes.CDEV_TIMESTAMP){
	    cdevTimeStamp ts = new cdevTimeStamp();
	    datasize += nelems_*cdevDataOutputStream.streamSize (ts);
	}
	else if(type_ == cdevDataTypes.CDEV_LONG)
	    datasize += nelems_*cdevDataOutputStream.streamSize ((long)1);
	
	return datasize;
    }

    /**
     * Convert this data entry into a byte stream
     */
    public void streamOut (cdevDataOutputStream output) throws IOException
    {
	int i;
	if (type_ != cdevDataTypes.CDEV_INVALID) {
	    try {
		output.write (tag_);
	    }catch (IOException e) {
		throw e;
	    }

	    try {
		output.write (type_);
	    }catch (IOException e) {
		throw e;
	    }

	    try {
		output.write (ndims_);
	    }catch (IOException e) {
		throw e;
	    }
	    
	    // to make stream data compatible to output generated
	    // by C++ code
	    int numelems;
	    if (type_ == cdevDataTypes.CDEV_STRING) {
		if (ndims_ == 0)
		    numelems = sdata_[0].length() + 1;
		else
		    numelems = nelems_;
	    }
	    else {
		if (ndims_ == 0)
		    numelems = 0;
		else
		    numelems = nelems_;
	    }
	    
	    try {
		output.write (numelems);
	    }catch (IOException e) {
		throw e;
	    }

	    // to make stream data compatible to output generated
	    // by C++ code, we have to pass dimension information
	    // even though we have dim 0 or 1.
	    if (ndims_ > 0) {
		try {
		    output.write ((int)0);
		}catch (IOException e) {
		    throw e;
		}
		try {
		    output.write (numelems);
		}catch (IOException e) {
		    throw e;
		}
	    }
	    
	    
	    if (type_ == cdevDataTypes.CDEV_STRING) {
		for (i = 0; i < nelems_; i++) {
		    try {
			output.write (sdata_[i]);
		    }catch (IOException e) {
			throw e;
		    }
		}
	    }
	    else if (type_ == cdevDataTypes.CDEV_BYTE) {
		for (i = 0; i < nelems_; i++) {
		    try {
			output.write (data_[i].byteValue());
		    }catch (IOException e) {
			throw e;
		    }
		}
	    }
	    else if (type_ == cdevDataTypes.CDEV_INT16) {
		for (i = 0; i < nelems_; i++) {
		    try {
			output.write (data_[i].shortValue());
		    }catch (IOException e) {
			throw e;
		    }
		}
	    }
	    else if (type_ == cdevDataTypes.CDEV_INT32) {
		for (i = 0; i < nelems_; i++) {
		    try {
			output.write (data_[i].intValue());
		    }catch (IOException e) {
			throw e;
		    }
		}
	    }
	    else if (type_ == cdevDataTypes.CDEV_LONG) {
		for (i = 0; i < nelems_; i++) {
		    try {
			output.write (data_[i].longValue());
		    }catch (IOException e) {
			throw e;
		    }
		}
	    }
	    else if (type_ == cdevDataTypes.CDEV_FLOAT) {
		for (i = 0; i < nelems_; i++) {
		    try {
			output.write (data_[i].floatValue());
		    }catch (IOException e) {
			throw e;
		    }
		}
	    }
	    else if (type_ == cdevDataTypes.CDEV_DOUBLE) {
		for (i = 0; i < nelems_; i++) {
		    try {
			output.write (data_[i].doubleValue());
		    }catch (IOException e) {
			throw e;
		    }
		}
	    }
	    else if (type_ == cdevDataTypes.CDEV_TIMESTAMP) {
		for (i = 0; i < nelems_; i++) {
		    try {
			output.write ((cdevTimeStamp)data_[i]);
		    }catch (IOException e) {
			throw e;
		    }
		}
	    }
	}

    }
    
}

    

