//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
//-----------------------------------------------------------------------------
//
// Description:
//      cdev data object containing multiple dataEntry Object
//
// Author:  
//      Jie Chen
//      Jefferson Lab HPC Group
//
// Revision History:
//   $Log: cdevData.java,v $
//   Revision 1.1  1999/12/14 15:31:37  chen
//   initial java implementation
//
//
//
package cmlog;

import java.io.OutputStream;
import java.io.InputStream;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Enumeration;
import java.lang.NumberFormatException;

public class cdevData
{
    // this is a hash table holding all data entry objects
    private Hashtable store_;
    
    // no need to put synchronized for memeber functions since
    // the hashtable is thread safe

    /** 
     * Construct an empty cdevData Object
     */
    public cdevData ()
    {
	// touch tag table
	cdevTagTable t = cdevTagTable.tagTable();

	// create a hash table with initial capacity of 1 and load factor 0.25
	// to improve look up time for a data entry
	store_ = new Hashtable (1, (float)0.25);
    }

    /**
     * Construct a cdevData Object from an existing data object.
     * This is a deep copy.
     */
    public cdevData (cdevData data)
    {
	store_ = new Hashtable (1, (float)0.25);

	Object obj;
	cdevDataEntry temp;
	Enumeration list = data.store_.elements();
	while (list.hasMoreElements()) {
	    obj = list.nextElement ();
	    temp = ((cdevDataEntry)obj).copy();
	    store_.put (new Integer(temp.getTag()), temp);
	}
    }
    
    /**
     * Return a duplicated data object: deep copy
     */
    public cdevData copy ()
    {
	cdevData data = new cdevData ();

	Object obj;
	cdevDataEntry temp;
	Enumeration list = store_.elements();
	while (list.hasMoreElements()) {
	    obj = list.nextElement ();
	    temp = ((cdevDataEntry)obj).copy();
	    data.store_.put (new Integer(temp.getTag()), temp);
	}
	return data;
    }

    /**
     * Clean up this data object
     */
    public void remove ()
    {
	store_.clear ();
	// no free individual elements :-(
    }

    /**
     * Remove a dataEntry with tag
     */
    public int remove (int tag)
    {
	if (store_.remove (new Integer(tag)) != null)
	    return 0;
	return -1;
    }

    /**
     * Remove a dataEntry with a string tag
     */
    public int remove (String tag) throws IllegalArgumentException
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch (IllegalArgumentException e) {
	    throw e;
	}
	return remove(itag);
    }

    /**
     * Insert a single byte with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public int insert (int tag, byte data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    }

    /**
     * Insert a single byte with a string tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public int insert (String tag, byte data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;
	
	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);
    }


    /**
     * Insert a short value with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public int insert (int tag, short data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    }

    /**
     * Insert a short value with a string tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public int insert (String tag, short data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;
	
	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);
    }

    /**
     * Insert an integer with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public int insert (int tag, int data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    }


    /**
     * Insert an integer with a string tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public int insert (String tag, int data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);
    }


    /**
     * Insert a long value with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public int insert (int tag, long data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    }



    /**
     * Insert a long value with a string tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public int insert (String tag, long data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);
    }


    /**
     * Insert a float value with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public int insert (int tag, float data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    }


    /**
     * Insert a float value with a string tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public int insert (String tag, float data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);
    }
	    

    /**
     * Insert a double with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public int insert (int tag, double data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    }

    /**
     * Insert a double with a string tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public int insert (String tag, double data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);
    }
    
    /**
     * Insert a timestamp with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (int tag, cdevTimeStamp data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    }


    /**
     * Insert a timestamp with a string tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (String tag, cdevTimeStamp data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);
    }

    /**
     * Insert a String Value with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (int tag, String data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    }

    /**
     * Insert a String Value with a string tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (String tag, String data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);
    }

    /**
     * Insert an array of byte with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (int tag, byte[] data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    }    

    /**
     * Insert an array of byte with a string tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (String tag, byte[] data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);	
    }    

    /**
     * Insert an array of short with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (int tag, short[] data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    }

    /**
     * Insert an array of short with a string tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (String tag, short[] data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);
    }

    /**
     * Insert an array of integer with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (int tag, int[] data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    }

    /**
     * Insert an array of integer with a string tag. If a data item 
     * with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (String tag, int[] data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);
    }

    /**
     * Insert an array of long with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (int tag, long[] data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    }

    /**
     * Insert an array of long with a string tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (String tag, long[] data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);
    }

    /**
     * Insert an array of float with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (int tag, float[] data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    }    


    /**
     * Insert an array of float with a string tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (String tag, float[] data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);
    }    

    /**
     * Insert an array of double with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (int tag, double[] data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    }    


    /**
     * Insert an array of double with a string tag.If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (String tag, double[] data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);
    }    

    /**
     * Insert an array of timestamp with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (int tag, cdevTimeStamp[] data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    } 


    /**
     * Insert an array of timestamp with a string tag. 
     * If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (String tag, cdevTimeStamp[] data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);
    } 

    /**
     * Insert an array of string with a tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (int tag, String[] data)
    {
	int status = remove (tag);
	cdevDataEntry entry = new cdevDataEntry (tag, data);
	store_.put (new Integer(tag), entry);
	return status;
    }

    /**
     * Insert an array of string with a string tag.If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (String tag, String[] data)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    System.err.println (e);
	    return -1;
	}
	return insert (itag, data);
    }


    /**
     * Insert a data entry item with a given tag. If a data item with the same
     * tag is already there, the previous data is removed and new
     * data is inserted. 
     */
    public  int insert (cdevDataEntry data)
    {
	int status = remove (data.getTag());
	store_.put (new Integer(data.getTag()), data);
	return status;
    }

    /**
     * Get a data entry with a tag
     */
    public cdevDataEntry get (int tag)
    {
	Object obj = store_.get (new Integer(tag));
	if (obj != null)
	    return (cdevDataEntry)obj;
	return null;
    }

    /**
     * Get a data entry with a string tag
     */
    public cdevDataEntry get (String tag) throws IllegalArgumentException
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch(IllegalArgumentException e) {
	    throw e;
	}
	return get (itag);
    }

    /**
     * Find a data entry with a tag
     */
    public  cdevDataEntry find (int tag)
    {
	Object obj = store_.get (new Integer(tag));
	if (obj != null)
	    return (cdevDataEntry)obj;
	return null;
    }    

    /**
     * Find a data entry with a string tag
     */
    public  cdevDataEntry find (String tag) throws IllegalArgumentException
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch (IllegalArgumentException e) {
	    throw e;
	}
	return find (itag);
    }    

    /**
     * Find whether this data contains entry with a given tag
     */
    public boolean contains (int tag)
    {
	return store_.containsKey (new Integer(tag));
    }

    /**
     * Find whether this data contains entry with a given string tag
     */
    public boolean contains (String tag) throws IllegalArgumentException
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag;

	try {
	    itag = t.convertKey (tag);
	}catch (IllegalArgumentException e) {
	    throw e;
	}

	return contains (itag);
    }

    /**
     * Change a data entry tag from old to new.
     * if old tag has no associated data entry, nothing is done
     */
    public void changeTag (int oldtag, int newtag)
    {
	Object obj = store_.remove (new Integer(oldtag));
	if (obj != null) {
	    cdevDataEntry data =  (cdevDataEntry)obj;
	    data.setTag (newtag);
	    store_.put (new Integer(newtag), data);
	}
    }

    /**
     * Change a data entry tag from old to new.
     * if old tag has no associated data entry, nothing is done
     */
    public void changeTag (String oldtag, String newtag) throws IllegalArgumentException
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int ioldtag, inewtag;

	try {
	    ioldtag = t.convertKey (oldtag);
	}catch (IllegalArgumentException e) {
	    throw e;
	}
	try {
	    inewtag = t.convertKey (newtag);
	}catch (IllegalArgumentException e1) {
	    throw e1;
	}
	changeTag (ioldtag, inewtag);
    }
    
    /**
     * Copy a data entry with a oldtag to create a new entry with a newtag.
     */
    public void dup (int oldtag, int newtag) throws IllegalArgumentException
    {
	if (store_.containsKey (new Integer(oldtag)) != true)
	    throw new IllegalArgumentException ("Dup: oldtag <" + String.valueOf(oldtag) + "> has no associated data");
	else if (store_.containsKey (new Integer(newtag)) == true) 
	    throw new IllegalArgumentException ("Dup: newtag <" + String.valueOf(newtag) + "> has associated data");
	else {
	    cdevDataEntry entry = get (oldtag);
	    cdevDataEntry newentry = entry.copy();
	    newentry.setTag (newtag);
	    store_.put (new Integer(newtag), newentry);
	}
    }

    /**
     * Copy a data entry with a oldtag to create a new entry with a newtag.
     * Tags are string tag
     */
    public void dup (String soldtag, String snewtag) throws IllegalArgumentException
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int oldtag;

	try {
	    oldtag = t.convertKey (soldtag);
	}catch (IllegalArgumentException e) {
	    throw e;
	}

	int newtag;
	try {
	    newtag = t.convertKey (snewtag);
	}catch (IllegalArgumentException e1) {
	    throw e1;
	}

	if (store_.containsKey (new Integer(oldtag)) != true)
	    throw new IllegalArgumentException ("Dup: oldtag <"+soldtag+ "> has no associated data");
	else if (store_.containsKey (new Integer(newtag)) == true) 
	    throw new IllegalArgumentException ("Dup: newtag <"+snewtag+" >has associated data");
	else {
	    cdevDataEntry entry = get (oldtag);
	    cdevDataEntry newentry = entry.copy();
	    newentry.setTag (newtag);
	    store_.put (new Integer(newtag), newentry);
	}
    }

    /**
     * Replace all data entries with entries inside the provided data object
     */
    public void replace (cdevData data)
    {
	Object obj;
	cdevDataEntry temp;

	Enumeration list = data.store_.elements();
	while (list.hasMoreElements ()) {
	    obj = list.nextElement ();
	    temp = (cdevDataEntry)obj;
	    if (contains (temp.getTag()))
		insert (temp);
	}
    }

    /**
     * Compare two data objects. Return true if they equal to each other.
     */
    public boolean equals (cdevData data)
    {
	Object obj0;
	cdevDataEntry temp0;

	Object obj1;
	cdevDataEntry temp1;

	Enumeration list0 = data.store_.elements();
	Enumeration list1 = store_.elements();

	int size0 = 0;
	int size1 = 0;
	while (list0.hasMoreElements ()) {
	    obj0 = list0.nextElement ();
	    size0 ++;
	}

	while (list1.hasMoreElements ()) {
	    obj1 = list1.nextElement ();
	    size1 ++;
	}
	if (size0 != size1)
	    return false;
	
	list0 = data.store_.elements ();
	while (list0.hasMoreElements ()) {
	    // for each element, find whether there is an element
	    // in this data
	    obj0 = list0.nextElement ();
	    temp0 = (cdevDataEntry)obj0;
	    temp1 = get (temp0.getTag());

	    if (temp1 == null || temp0.equals (temp1) == false)
		return false;
	}
	return true;
    }

    /**
     * Return number of data entries inside this object
     */
    public int size ()
    {
	return store_.size();
    }

    /**
     * Return number of valid data entries inside this object
     */
    public int numValidEntries ()
    {
	int count = 0;
	Object obj;
	cdevDataEntry temp;

	Enumeration list = store_.elements();
	while (list.hasMoreElements ()) {
	    obj = list.nextElement ();
	    temp = (cdevDataEntry)obj;
	    if (temp.getType() != cdevDataTypes.CDEV_INVALID)
		count ++;
	}
	return count;
    }

    /**
     * Check whether this data is empty
     */
    public boolean isEmpty ()
    {
	return store_.isEmpty();
    }

    /**
     * Return enumeration of elements
     */
    public Enumeration elements()
    {
	return store_.elements ();
    }

    /**
     * Convert data object to a string representation
     */
    public String toString (String title)
    {
	String result = new String ();
	if (title != null)
	    result = title + "\n";

	Object obj;
	Enumeration list = store_.elements();
	while (list.hasMoreElements()) {
	    obj = list.nextElement ();
	    result += obj.toString ();
	    result += "\n";
	}
	return result;
    }

    /**
     * Convert data object to a string representation
     */
    public String toString ()
    {
	return toString ("Contents: ");
    }
	    
    /**
     * Output contents to standard out
     */
    public void asciiDump ()
    {
	System.out.print ("\nBeginning of display----------------------------\n");
	System.out.print (toString());
	System.out.print ("\nEnd of Display----------------------------------\n");
    }

    /**
     * Calculate stream size of this object
     */
    public int streamSize ()
    {
	cdevDataEntry tmp = null;
	int           datasize = 0;
	int           i;

	// number of elements
	datasize += cdevDataOutputStream.streamSize (numValidEntries());

	// add size for each valid item
	Enumeration list = store_.elements();
	while (list.hasMoreElements()) {
	    tmp = (cdevDataEntry)(list.nextElement());
	    // individual data entry size
	    datasize += tmp.streamSize ();
	}
	return datasize;
    }	    

    /**
     * Stream cdevData into a buffered data output stream
     */
    public void streamOut (OutputStream output) throws IOException
    {
	cdevDataEntry tmp;
	
	cdevDataOutputStream writer = new cdevDataOutputStream (output);

	try {
	    streamOut (writer);
	}catch (IOException e){
	    throw e;
	}
    }

    /**
     * Stream cdevData into a buffered data output stream
     */
    public void streamOut (cdevDataOutputStream writer) throws IOException
    {
	cdevDataEntry tmp;
	
	try {
	    writer.write (numValidEntries ());
	}catch (IOException e){
	    throw e;
	}

	Enumeration list = store_.elements ();
	while (list.hasMoreElements()) {
	    tmp = (cdevDataEntry)(list.nextElement());
	    try {
		tmp.streamOut (writer);
	    }catch (IOException e) {
		throw e;
	    }
	}
    }
	
    /**
     * Take a input data stream and populate this data object
     */
    public void streamIn (InputStream stream) throws IOException
    {
	// clean out this object first
	remove ();

	// attach input stream
	cdevDataInputStream input = new cdevDataInputStream (stream);

	try {
	    streamIn (input);
	}catch (IOException e) {
	    throw e;
	}
    }


    /**
     * Take an input data stream and populate this data object
     */
    public void streamIn (cdevDataInputStream input) throws IOException
    {
	int i, j;
	// clean out this object first
	remove ();

	// read number of elements
	int count;
	try {
	    count = input.readInt ();
	}catch (IOException e) {
	    throw e;
	}

	// read each data entry
	for (i = 0; i < count; i++) {
	    short  type;
	    short  ndims;
	    int    elems, nelems;
	    int    tag;
	    cdevDataEntry de = null;

	    // read tag
	    try {
		tag = input.readInt ();
	    }catch (IOException e) {
		throw e;
	    }

	    // read type
	    try {
		type = input.readShort ();
	    }catch (IOException e) {
		throw e;
	    }
	    
	    // read dimension
	    try {
		ndims = input.readShort ();
	    }catch (IOException e) {
		throw e;
	    }	    
	
	    // read number elements
	    try {
		elems = input.readInt ();
	    }catch (IOException e) {
		throw e;
	    }

	    // get bound information: compatible to C++
	    if (ndims > 0) {
		int begin;
		int end;
		try {
		    begin = input.readInt ();
		}catch (IOException e) {
		    throw e;
		}
		try {
		    end = input.readInt ();
		}catch (IOException e) {
		    throw e;
		}
	    }

	    // read data stream compatible to C++ generated stream
	    nelems = elems;
	    if (type != cdevDataTypes.CDEV_INVALID) {
		if (ndims == 0) 
		    nelems = 1;
		else
		    nelems = elems;
	    }
	    
	    if (type == cdevDataTypes.CDEV_INVALID) 
		throw new IOException ("Data Entry Type is invalid");
	    else if (type == cdevDataTypes.CDEV_BYTE) {
		if (nelems == 1) {
		    byte tmp;
		    try {
			tmp = input.readByte ();
		    }catch (IOException e) {
			throw e;
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
		else {
		    byte[] tmp = new byte[nelems];
		    for (j = 0; j < nelems; j++) {
			try {
			    tmp[j] = input.readByte();
			}
			catch (IOException e) {
			    throw e;
			}
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
	    }
	    else if (type == cdevDataTypes.CDEV_INT16) {
		if (nelems == 1) {
		    short tmp;
		    try {
			tmp = input.readShort ();
		    }catch (IOException e) {
			throw e;
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
		else {
		    short[] tmp = new short[nelems];
		    for (j = 0; j < nelems; j++) {
			try {
			    tmp[j] = input.readShort();
			}
			catch (IOException e) {
			    throw e;
			}
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
	    }
	    else if (type == cdevDataTypes.CDEV_INT32) {
		if (nelems == 1) {
		    int tmp;
		    try {
			tmp = input.readInt ();
		    }catch (IOException e) {
			throw e;
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
		else {
		    int[] tmp = new int[nelems];
		    for (j = 0; j < nelems; j++) {
			try {
			    tmp[j] = input.readInt();
			}
			catch (IOException e) {
			    throw e;
			}
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
	    }
	    else if (type == cdevDataTypes.CDEV_LONG) {
		if (nelems == 1) {
		    long tmp;
		    try {
			tmp = input.readLong ();
		    }catch (IOException e) {
			throw e;
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
		else {
		    long[] tmp = new long[nelems];
		    for (j = 0; j < nelems; j++) {
			try {
			    tmp[j] = input.readLong();
			}
			catch (IOException e) {
			    throw e;
			}
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
	    }
	    else if (type == cdevDataTypes.CDEV_FLOAT) {
		if (nelems == 1) {
		    float tmp;
		    try {
			tmp = input.readFloat ();
		    }catch (IOException e) {
			throw e;
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
		else {
		    float[] tmp = new float[nelems];
		    for (j = 0; j < nelems; j++) {
			try {
			    tmp[j] = input.readFloat();
			}
			catch (IOException e) {
			    throw e;
			}
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
	    }
	    else if (type == cdevDataTypes.CDEV_DOUBLE) {
		if (nelems == 1) {
		    double tmp;
		    try {
			tmp = input.readDouble ();
		    }catch (IOException e) {
			throw e;
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
		else {
		    double[] tmp = new double[nelems];
		    for (j = 0; j < nelems; j++) {
			try {
			    tmp[j] = input.readDouble();
			}
			catch (IOException e) {
			    throw e;
			}
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
	    }
	    else if (type == cdevDataTypes.CDEV_TIMESTAMP) {
		if (nelems == 1) {
		    cdevTimeStamp tmp;
		    try {
			tmp = input.readTimeStamp ();
		    }catch (IOException e) {
			throw e;
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
		else {
		    cdevTimeStamp[] tmp = new cdevTimeStamp[nelems];
		    for (j = 0; j < nelems; j++) {
			try {
			    tmp[j] = input.readTimeStamp();
			}
			catch (IOException e) {
			    throw e;
			}
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
	    }
	    else if (type == cdevDataTypes.CDEV_STRING) {
		if (nelems == 1) {
		    String tmp;
		    try {
			tmp = input.readString ();
		    }catch (IOException e) {
			throw e;
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
		else {
		    String[] tmp = new String[nelems];
		    for (j = 0; j < nelems; j++) {
			try {
			    tmp[j] = input.readString();
			}
			catch (IOException e) {
			    throw e;
			}
		    }
		    de = new cdevDataEntry (tag, tmp);
		}
	    }
	    if (de != null) 
		insert (de);	  
	}
	    
    }

    public static void addTag (String stag)  throws IllegalArgumentException 
    {
	cdevTagTable t = cdevTagTable.tagTable();
	
	try {
	    t.add (stag);
	}catch (IllegalArgumentException e) {
	    throw e;
	}
    }

    public static int convertKey(String stag) throws IllegalArgumentException
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int itag = 0;

	try {
	    itag = t.convertKey (stag);
	}catch (IllegalArgumentException e) {
	    throw e;
	}
	return itag;
    }

    public static String convertKey(int tag)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	
	return t.convertKey (tag);
    }

    public static void insertTag (int tag, String ctag) throws IllegalArgumentException
    {
	cdevTagTable t = cdevTagTable.tagTable();

	try {
	    t.insert (tag, ctag);
	}catch (IllegalArgumentException e) {
	    throw e;
	}
    }

    public static boolean tagExists (String ctag)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	return t.tagExists (ctag);
    }

    public static boolean tagExists (int tag)
    {
	cdevTagTable t = cdevTagTable.tagTable();
	return t.tagExists (tag);
    }
}

    
	
