//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
//-----------------------------------------------------------------------------
//
// Description:
//      cmlog protocol buffering mechanism
//
// Author:  
//      Jie Chen
//      Jefferson Lab HPC Group
//
// Revision History:
//   $Log: Packet.java,v $
//   Revision 1.4  2002/04/04 17:17:55  chen
//   Fix a bug for multiple java clients
//
//   Revision 1.3  2000/03/17 19:18:37  chen
//   add java client
//
//   Revision 1.2  2000/02/07 20:06:10  chen
//   add protocol version 2
//
//   Revision 1.1  2000/01/04 14:26:10  chen
//   new implementation
//
//
//
package cmlog;

import java.io.*;

public class Packet
{
    // maximum buffer size for a Packet
    private static int PACKET_MAX_SIZE = 40960;
    // maximum number of messages inside a Packet
    private static int PACKET_LIST_SIZE = 1024;
    // 8 byte boundary aligned
    private static int ALIGN_WORD_LEN = 8;
    // header for each message
    private static int MSG_PREDATA_SIZE = 4;
    // Header before network data is 2*sizeof (int) for now
    private static int PACKET_PREDATA_SIZE = 2*4;

    private static int roundUp (int a, int b)
    {
	return ((a + b - 1)/b)*b;
    }

    public static int getMagicNumber (int header)
    {
	if (Config.PROTOCOL_MAJOR_VERSION >= 2)
	    return header & 0x000000ff;
	else
	    return Config.CMLOG_MAGIC;
    }

    public static int getMsgSize (int header)
    {
	if (Config.PROTOCOL_MAJOR_VERSION >= 2)
	    return ((header >> 8) & 0x00ffffff);
	else
	    return header;
    }

    public static int setMsgHeader (int size)
    {
	if (Config.PROTOCOL_MAJOR_VERSION >= 2) {
	    if (size > 0x0fffff00) {
		System.err.println ("Fatal: message size is too big\n");
		System.exit (1);
	    }
	    return (((size << 8) & 0xffffff00) | Config.CMLOG_MAGIC);
	}
	else
	    return size;
    }

    // total size of network bytes excluding the size and number
    // of messages information in the beginning
    private int size_;

    // number of cmlogMsg to follow
    private int number_;

    // flag to tell whether this buffer is full
    private boolean full_;

    // list of messages that are buffered
    private cdevMessage[] messages_;

    /**
     * Construct an empty Packet
     */
    public Packet ()
    {
	size_ = 0;
	number_ = 0;
	full_ = false;
	messages_ = new cdevMessage[Packet.PACKET_LIST_SIZE];
	for (int i = 0; i < Packet.PACKET_LIST_SIZE; i++)
	    messages_[i] = null;
    }

    /**
     * Construct a new packet from a existing one
     */
    public Packet (Packet p)
    {
	int i;

	size_ = p.size_;
	number_ = p.number_;
	full_ = p.full_;
	messages_ = new cdevMessage[Packet.PACKET_LIST_SIZE];

	for (i = 0; i < Packet.PACKET_LIST_SIZE; i++)
	    messages_[i] = null;	

	for (i = 0; i < number_; i++)
	    messages_[i] = new cdevMessage(p.messages_[i]);
    }


    /**
     * Insert a cmlogMsg into this buffer
     * return false if the buffer is full
     * return true if the insertion is ok
     */
    public synchronized boolean insert (cdevMessage data)
    {
	int  bsize = data.streamSize () + Packet.MSG_PREDATA_SIZE;

	if (full_)
	    return false;
	
	if (size_ + bsize > Packet.PACKET_MAX_SIZE)
	    return false;

	messages_[number_] = data;
	size_ += bsize;
	number_ ++;

	if (number_ == Packet.PACKET_LIST_SIZE)
	    full_ = true;

	return true;
    }

    /**
     * Check whether a message will overflow the Packet 
     * without adding the msg to the Packet
     */
    public synchronized boolean overflow (cdevMessage data)
    {
	if (full_)
	    return true;

	int bsize = data.streamSize () + Packet.MSG_PREDATA_SIZE;

	if (size_ + bsize > Packet.PACKET_MAX_SIZE)
	    return true;

	return false;
    }

    /**
     * reset the Packet
     */
    public synchronized void reset ()
    {
	for (int i = 0; i < number_; i++)
	    messages_[i] = null;

	number_ = 0;
	size_ = 0;
	full_ = false;
    }

    /**
     * return whether this Packet is full
     */
    public synchronized boolean full ()
    {
	return full_;
    }

    /**
     * return whther this Packet is empty
     */
    public synchronized boolean empty ()
    {
	if (number_ == 0)
	    return true;
	return false;
    }

    /**
     * return number of messages 
     */
    public synchronized int numberOfMessages ()
    {
	return number_;
    }
    

    /**
     * Stream out Packet
     */
    public void streamOut (OutputStream stream) throws IOException
    {
	// no need to buffer outbound data since packet only use socket not 
	// using pipe.
	cdevDataOutputStream writer = new cdevDataOutputStream (stream, false);

	try {
	    streamOut (writer);
	}catch (IOException e) {
	    throw e;
	}
    }

    /**
     * Stream out Packet
     */
    public synchronized void streamOut (cdevDataOutputStream writer) 
	throws IOException
    {
	if (Config.PROTOCOL_MAJOR_VERSION >= 2) {
	    // sending magic number first
	    int magic = Config.CMLOG_PMAGIC;
	    try {
		writer.write (magic);
	    }catch (IOException e) {
		throw e;
	    }
	}

	// clip size is 4 bytes bigger than my buffer size
	int clipsize = size_ + 4;

	// output header of clip Packet: size + count
	try {
	    writer.write (clipsize);
	}catch (IOException e){
	    throw e;
	}

	try {
	    writer.write (number_);
	}catch (IOException e) {
	    throw e;
	}
	 
	// send out each message one by one
	cdevMessage data = null;
	int         bsize = 0;
	int         header = 0;
	for (int i = 0; i < number_; i++) {
	    data = messages_[i];
	    bsize = data.streamSize ();
	    // create message header
	    header = Packet.setMsgHeader (bsize);

	    // message data size first
	    try {
		writer.write (header);
	    }catch (IOException e) {
		throw e;
	    }
	    // streamed data
	    try {
		data.streamOut (writer);
	    }catch (IOException e) {
		throw e;
	    }
	}
    }
    
    /**
     * Convert a incoming data stream into a Packet data
     */
    public void streamIn (InputStream stream) throws IOException 
    {
	// attach input stream
	cdevDataInputStream input = new cdevDataInputStream (stream);

	try {
	    streamIn (input);
	}catch (IOException e) {
	    throw e;
	}
    }	


    /**
     * Convert a incoming data stream into a Packet data
     */
    public void streamIn (cdevDataInputStream input) throws IOException
    {
	// clean out this object first
	reset ();
	
	if (Config.PROTOCOL_MAJOR_VERSION >= 2) {
	    // network header
	    int magic;
	    try {
		magic = input.readInt ();
	    }catch (IOException e) {
		throw e;
	    }
	    if (magic != Config.CMLOG_PMAGIC) {
		throw new IOException ("Packet receives wrong magic number");
	    }
	}

	// read clip header which is 8 bytes
	byte[] header = null;
	int    clipsize = 0;
	try {
	    header = input.readBytes (Packet.PACKET_PREDATA_SIZE);
	}catch (IOException e) {
	    throw e;
	}

	cdevDataInputStream headerstream = 
	    new cdevDataInputStream (new ByteArrayInputStream (header));
	
	try {
	    clipsize = headerstream.readInt();
	}catch (IOException e) {
	    reset ();
	    throw e;
	}
	try {
	    number_ = headerstream.readInt ();
	}catch (IOException e) {
	    reset ();
	    throw e;
	}
	size_ = clipsize - 4;

	if (number_ > 0) {
	    // read data of size 'size_' from input stream
	    byte[] payload = null;
	    try {
		payload = input.readBytes (size_);
	    }catch (IOException e) {
		reset ();
		throw e;
	    }
	    // convert binary data stream into messages
	    int         bsize = 0;
	    int         netmagic = 0;
	    int         netheader = 0;
	    cdevDataInputStream pstream = 
		new cdevDataInputStream (new ByteArrayInputStream (payload));
	    for (int i = 0; i < number_; i++) {
		try {
		    netheader = pstream.readInt ();
		}catch (IOException e) {
		    throw e;
		}
		
		// get magic number for each message
		netmagic = Packet.getMagicNumber (netheader);
		if (netmagic != Config.CMLOG_MAGIC) {
		    throw new IOException("Message magic number error");
		}
		bsize = Packet.getMsgSize (netheader);

		try {
		    messages_[i] = new cdevMessage ();
		    messages_[i].streamIn (pstream);
		}catch (IOException e) {
		    throw e;
		}
	    }
	}
    }


    /**
     * return internal messages
     */
    public cdevMessage[] messages ()
    {
	return messages_;
    }
		
}
    
    
	
