#ifndef _CMLOG_CDEV_PACKET_H_
#define _CMLOG_CDEV_PACKET_H_

#include <cdevData.h>
#include <xdrClass.h>

// *****************************************************************************
// * struct cmlog_cdevPacketMap :
// *	This is the first 32 bit integer that is passed through the socket.  It
// *	contains a bitmap that identifies the version of the packet, as well as
// *	a bit field that indicates the contents of the packet.  This bitfield
// *	simplifies the process of decoding the packet and results in a smaller
// *	transmission size.
// *****************************************************************************
typedef union
	{
	unsigned rawData;
	struct	{
		unsigned version : 16;
		unsigned pad     : 16;
		} value;
	} cmlog_cdevPacketMap;
	
// *****************************************************************************
// * class cmlog_cdevPacket :
// *	This is the base class for all cmlog_cdevPacket classes.  It is a pure
// *    virtual class that defines the mechanisms necessary for identifying
// *	the version of a class as well as reading or generating a binary
// *	stream.
// *****************************************************************************
class cmlog_cdevPacket
{
protected:
	// *********************************************************************
	// * These are the data storage variables.
	// *********************************************************************
	char   * binary;
	unsigned   binaryLen;

public:
	cmlog_cdevPacket             ( void );
	cmlog_cdevPacket             ( cmlog_cdevPacket & packet );
	virtual ~cmlog_cdevPacket    ( void );

	virtual int  getVersion ( short & version );	
	virtual int  streamOut  ( char ** stream, unsigned * len );
	virtual int  streamIn   ( char *  stream, unsigned   len );
	virtual void detachData ( void );
	virtual int  attachData ( char * stream, unsigned len );
};
	
// *****************************************************************************
// * cmlog_cdevPacket::getVersion :
// *	This method is used to obtain the version number of the packet.
// *
// *	Returns 0 on Success or -1 on Error
// *****************************************************************************
inline int cmlog_cdevPacket::getVersion ( short & version )
	{
	int           result=-1;
	XDR_Reader    reader;
	cmlog_cdevPacketMap map;

	if(binary!=NULL && binaryLen>0)
		{
		reader.attachData(binary, binaryLen);
		map.rawData = 0;
		if((result = !reader.get(map.rawData))==0)
			{
			version = map.value.version;
			}
		reader.detachData();
		}
	return result;
	}
	
// *****************************************************************************
// * cmlog_cdevPacket::streamIn :
// *	The streamIn function will read a memory buffer provided by the caller 
// *	and then populate the class with these values. This stream remains the 
// *	property of the caller and he is responsible for deleting it.
// *
// *	Returns 0 on Success
// *****************************************************************************
inline int cmlog_cdevPacket::streamIn ( char * stream, unsigned len )
	{
	if(binary!=NULL)
		{
		delete binary;
		binary = NULL;
		}
	binaryLen = 0;
	
	if(len>0 && stream!=NULL)
		{
		binary    = new char[len];
		binaryLen = len;
		memcpy(binary, stream, binaryLen);
		}
	return 0;
	}

// *****************************************************************************
// * cmlog_cdevPacket::streamOut :
// *	The streamOut and streamIn functions are the bread and butter of the 
// *	cmlog_cdevPacket classes.  They allow the class to convert itself into a 
// *	binary stream for transmission and then be reconstructed into a class
// *	when they are received on the other side.
// *
// *	The streamOut function will allocate a memory buffer to the stream
// *	variable that is sufficient to store the binary representation of
// *	the data that is to be transmitted.  This stream remains the 
// *	property of the cmlog_cdevPacket class and should not be deleted by the
// *	caller.
// *
// *	Returns 0 on Success
// *****************************************************************************
inline int cmlog_cdevPacket::streamOut ( char ** stream, unsigned * len )
	{
	*stream = binary;
	*len    = binaryLen;
	return 0;
	}

// *********************************************************************
// * cmlog_cdevPacket::detachData :
// *	This method allows the caller to obtain a copy of the binary
// *	image that is stored in the cmlog_cdevPacket object (suing streamOut), 
// *	and then force the cmlog_cdevPacket object to detach it (preventing it 
// *	from being deleted when the object is destroyed.)
// *
// *	Returns nothing.
// *********************************************************************
inline void cmlog_cdevPacket::detachData ( void )
	{
	binary = NULL;
	binaryLen = 0;
	}
	
// *********************************************************************
// * cmlog_cdevPacket::attachData :
// *	This method allows the caller to assign a preallocated binary
// *	buffer to this object.  This prevents the cmlog_cdevPacket class
// *	from having to allocate the data.
// *
// *	Returns 0 on Success
// *********************************************************************
inline int cmlog_cdevPacket::attachData ( char * stream, unsigned len )
	{
	if(binary!=NULL) 
		{
		delete binary;
		binary = NULL;
		}
	binaryLen = 0;

	if(stream!=NULL && len>0)
		{
		binary    = stream;
		binaryLen = len;
		}
	return 0;
	}

// *****************************************************************************
// * cmlog_cdevPacket::cmlog_cdevPacket :
// * This is the default constructor for the class.
// *****************************************************************************
inline cmlog_cdevPacket::cmlog_cdevPacket ( void ) 
	: binary(NULL), binaryLen(0) 
	{}

// *****************************************************************************
// * cmlog_cdevPacket::cmlog_cdevPacket :
// *	This is the copy constructor for the class.
// *****************************************************************************
inline cmlog_cdevPacket::cmlog_cdevPacket ( cmlog_cdevPacket & packet )
	: binary(NULL), binaryLen(0)
	{
	streamIn(packet.binary, packet.binaryLen);
	}

// *****************************************************************************
// * cmlog_cdevPacket::~cmlog_cdevPacket :
// *	This is the destructor for the object.
// *****************************************************************************
inline cmlog_cdevPacket::~cmlog_cdevPacket ( void )
	{
	if(binary!=NULL)
		{
		delete binary;
		binary = NULL;
		}
	binaryLen = 0;
	}

#endif /* _CDEV_PACKET_H_ */
