//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Data object between cmlog client/server
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogMsg.h,v $
//   Revision 1.4  2001/07/25 14:26:37  chen
//   64 BIT Initial Port
//
//   Revision 1.3  2000/02/07 16:10:42  chen
//   cmlog protocol 2.0
//
//   Revision 1.2  1999/12/13 21:38:56  chen
//   use CMLOG_BAD_PTR to signal error
//
//   Revision 1.1.1.1  1999/09/07 15:29:10  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:27:25  bickley
// Added cmlog to application development system.
//
//
#ifndef _CMLOG_MSG_H
#define _CMLOG_MSG_H

#include <cmlogProtocol.h>

#include <cdevData.h>
#include <cmlog_cdevMessage.h>
#include <cmlog_cdevPacket1.h>

#ifndef _CMLOG_BUILD_CLIENT
#include <ace/INET_Addr.h>
#include <ace/SOCK_Stream.h>
#endif

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
#include <cpSynch.h>
#endif

// A cmlog message contains a header with lower 8 bit (1 byte)
// as magic number, higher 24 bit as size of message (packet) to follow.
// maximum size of a message therefore is 2^24 = 2^4 * 2^20 = 16 MByte

class cmlogPacket;

#ifdef CMLOG_64BIT_LONG
#define CMLOG_BAD_PTR 0xffffffffffffffff
#define size_t unsigned

#ifndef CMLOG_SIZE_T_DEFINED
#define CMLOG_SIZE_T_DEFINED
#define size_t unsigned
#endif

#else
#define CMLOG_BAD_PTR 0xffffffff
#endif

const int CMLOG_PACKET_PREDATA_SIZE = 2*sizeof (int);

// Important note: if the following number is changed, please
// change stack size on cmlogClientD on vxWorks
const int CMLOG_PACKET_MAX_SIZE = 10*4096;

const int CMLOG_PACKET_LIST_SIZE = 1024;

inline int cmlogRoundup (int a, int b)
{
  return ((a + b - 1)/b)*b;
}

class cmlogMsg
{
public:
  // constructor and destructor
  
  // create empty message
  cmlogMsg  (void);
  // create a simple version of cmlogmsg
  cmlogMsg  (unsigned type, cdevData& data, int reqId = 0);
  // create a message contains a packet
  cmlogMsg  (cmlog_cdevMessage& message);

  cmlogMsg  (cmlogMsg& msg);
  ~cmlogMsg (void);

  // binary size of cmlogMsg = cdevMsg + size + magicnumber info
  int  size (void);


  // operation type
  unsigned type     (void);
  void     type     (unsigned type);

  // client request ID
  int  reqId        (void);

  // pointer of incoming IO object
  void*  channelId   (void);
  // set client ID
  void   channelId   (void* id);

  // return 1: if this msg has the same clientId of input
  int  sameChannel   (void* id);

  // get cmlog_cdevPacket reference
  operator cmlog_cdevMessage& (void);

  // friend function to perform I/O operation
  friend int operator << (int out, cmlogMsg& msg);
#ifndef __vxworks
  friend int operator >> (int in,  cmlogMsg& msg);
#endif

  // return -1: failed: return number of bytes received
  // This routine is used on the same machine
  friend int operator >>  (int in,  cmlog_cdevPacket& packet);

  // receive exact n bytes of data from stream
  static int cmlog_recv_n (int in, char* buffer, size_t len);

  // dump the cmlogMsg into a buffer (allocated)
  // this buffer should have large enough size to hold the cmlogMsg
  // return 0: success, return -1: failure
  // only used by the cmlogPacket
  static int msgToBuffer   (cmlogMsg& msg, char* buffer);
  // in addition to the above with initial len points to buffer size
  // and return actual length
  // return 0: success, return -1: failure
  static int msgToBuffer   (cmlogMsg& msg, char* buffer, int* len);
  // retrieve cmlogMsg from a buffer
  // return size of converted bytes: return -1: fatal error
  static int bufferToMsg   (char* buffer, cmlogMsg& msg, int start, int bufsize);
  // retrieve cmlogMsg from a buffer
  // return size of converted bytes: return -1: fatal error
  // this routine is to retrieve messages on shared memory queue (server access
  // only)
#if !defined (CMLOG_USE_THREAD) || !defined (_REENTRANT)
  static int bufferToMsg   (char* buffer, cmlogMsg& msg);
#endif
  // above two routines can be modified to enhance performace

  static unsigned char getMagicNumber (int iheader);
  // get magic number from an input header
  static unsigned int  getMsgSize     (int iheader);
  // get size of this message
  static int setMsgHeader             (unsigned int size);
  // construct a message header in network byte order from a message size

private:
  // real cdev message
  cmlog_cdevMessage data_;

  // deny access to assignment operator
  cmlogMsg& operator = (const cmlogMsg& msg);

  // friend class
  friend class cmlogPacket;
};

// A cmlogPacket header contains a magic number, size of packet and
// number of messages
class cmlogPacket
{
public:
  // constructor and destructor
  cmlogPacket  (void);
  ~cmlogPacket (void);

  // add a new cmlogMsg to the packet
  // return 0: success, -1: packet is full
  int insert (cmlogMsg& msg);

  // check a new msg will overflow the packet or not without really adding
  // this msg to the packet
  int overflow (cmlogMsg& msg);

  // add new cmlog_cdevPacket to the buffer
  // return 0: success, -1: buffer is full
  // size is total size including header size information
  int insert (cmlog_cdevPacket& packet, int size);
  
  // check a new packet will overflow the buffer or not without really adding
  // this packet to the buffer, size is total size of packet including
  // header size information
  int overflow (cmlog_cdevPacket& packet, int size);

  // empty the packet
  int empty  (void);

  // check whether this packet is full
  int fullPacket (void);
  // check whether this packet is empty
  int emptyPacket (void);

  // return size of data to follow without size + count
  int  dataSize   (void);

  // return total packet size
  int  packetSize (void);


  // return number of cmlogMsg s
  unsigned int numberOfData (void);

  // return pointer of the first data, all these cmlogMsg s are 
  // internal created, caller must free them once finished using them
  cmlogMsg**    messages     (void);


  // reset: caution: only used when error occurs
  // not intended for public
  void          reset        (void);
       

#ifndef _CMLOG_BUILD_CLIENT
  // friend function to perform I/O operation
  friend int operator << (ACE_SOCK_Stream& out, cmlogPacket& p);
  friend int operator >> (ACE_SOCK_Stream& in,  cmlogPacket& p);
#endif

  // friend function to perform I/O operation
  friend int operator << (int out, cmlogPacket& p);
  friend int operator >> (int in,  cmlogPacket& p);

protected:
  // byte swapping utilites
  void decode (void);
  void encode (void);

private:
  // total size of network byte excluding the size and count information
  // size_ <= CMLOG_PACKET_MAX_SIZE
  // clip size = size + sizeof (size_)
  unsigned int  size_;
  // number of cmlogMsg to follow
  unsigned int  number_;
  // flag to tell full or not
  int           full_;

  // buffer holds all data with size of CMLOG_PACKET_MAX_SIZE
  char          *buffer_;


#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutex          lock_;
#endif

  // deny access to copy and assignment
  cmlogPacket (const cmlogPacket &);
  cmlogPacket& operator = (const cmlogPacket& p);
};

#ifdef __vxworks
// this data structure for interrupt service routine only
// to replace standard cmlog_cdevPacket1 which is not suitable for ISR
typedef union
{
  unsigned rawData;
  struct  {
    unsigned version             : 16;
    unsigned clientIDSet         : 1;
    unsigned transIndexSet       : 1;
    unsigned cancelTransIndexSet : 1;
    unsigned localDataIndexSet   : 1;
    unsigned foreignDataIndexSet : 1;
    unsigned operationCodeSet    : 1;
    unsigned deviceListSet       : 1;
    unsigned messageSet          : 1;
    unsigned dataSet             : 1;
    unsigned contextSet          : 1;
    unsigned tagMapSet           : 1;
    unsigned pad                 : 5;
  } value;
}cmlogVxMap1;

class cmlogVxPacket1
{
public:
  // constructor
  cmlogVxPacket1  (void);
  // destructor
  ~cmlogVxPacket1 (void);
  
  // message will only points to exising memory
  void set (short clientId, unsigned operatioCode,
	    char* message, 
	    int arg0, int arg1, int arg2, int arg3,
	    int arg4, int arg5, int arg6, int arg7,
	    int arg8, int arg9);

  size_t size (void);

  // convert this object into binary stream, size is total bytes to send
  // return 0: success, -1: overflow
  // since vxworks has the same byte order as the network byte order
  // we do not use any byte swap inside this routine
  int streamOut (char* buffer, size_t& size, size_t maxsize);

private:
  cmlogVxMap1       map_;
  short             clientId_;
  unsigned          operationCode_;
  char*             message_;

  int               arg0_;
  int               arg1_;
  int               arg2_;
  int               arg3_;
  int               arg4_;
  int               arg5_;
  int               arg6_;
  int               arg7_;
  int               arg8_;
  int               arg9_;
};
#endif  /* __vxworks */

#ifdef CDEV_64BIT_LONG
#undef size_t
#endif
  
#endif
