//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      CMLOG Logging Client Interface (for Unix and vxworks)
//
//      On vxworks, cmlogClient is commly used under a single task.
//      If cmlogClient is used in multi tasks, one has to take care
//      locks etc.
//
// Author:  
//      Danjin Wu & Jie Chen
//
// Revision History:
//   $Log: cmlogClient.h,v $
//   Revision 1.1.1.1  1999/09/07 15:29:09  chen
//   CMLOG version 2.0
//
// Revision 1.2  1998/10/20  15:23:13  chen
// use _CMLOG_USE_THREAD to build thread safe client for Unix
//
// Revision 1.1  1997/08/01  15:26:37  bickley
// Added cmlog to application development system.
//
//
//
#ifndef _CMLOG_CLIENT_H
#define _CMLOG_CLIENT_H

#include <cdevData.h>
#include <cmlogUtil.h>
#include <cmlogFilter.h>
#include <cmlogMsg.h>

#ifdef __vxworks
inline void CMLOG_ISR_CHECK (void)
{
  if (intContext ()) {
    ::logMsg ("Not a callable routine from ISR\n", 0, 0, 0, 0, 0, 0);
    return;
  }
}
#else
inline void CMLOG_ISR_CHECK (void)
{
  // empty
}
#endif

#if defined (_CMLOG_USE_THREAD) || defined (__vxworks)
#include <cpSynch.h>
#endif

class cmlogClient;

class cmlogClientTagCallback: public cdevTagTableCallback
{
public:
  cmlogClientTagCallback  (cmlogClient* client);
  ~cmlogClientTagCallback (void);

  virtual void callback (int tag, char* name);

private:
  cmlogClient* client_;

  // deny access to copy and assignment operator
  cmlogClientTagCallback (cmlogClientTagCallback& cbk);
  cmlogClientTagCallback& operator = (cmlogClientTagCallback & cbk);
};

class cmlogClient
{
public:
#ifndef __vxworks
  // constructors and destructor
  cmlogClient  (char* progname = 0);
#else
  static cmlogClient* logClient (void);
#endif
  ~cmlogClient (void);

  // operations

  // connect to logging server with default set of context
  int connect     (int conntionRetries = 3);
  // connect to logging server with user specified context
  int connect     (const cdevData& data, int connectionRetries = 3);
  // disconnect from logging server
  int disconnect  (void);
  // connected or not
  int connected   (void) const;

  // set and get throttling parameters
  int setThrottle (char* tag,  int size, int limit, double delta);
  // callers have control over tag, should free it if no longer need it
  int getThrottle (char** tag, int& size, int& limit, double& delta);

  // set and get severity/verbosity threshold
  int setVerbosityThreshold (int thresh);
  int getVerbosityThreshold (void) const;

  int setSeverityThreshold (int thresh);
  int getSeverityThreshold (void) const;  

  // send logging message to the logging server with value data
  int postData        (cdevData& data);
  // send logging message to the logging server with static data
  // very useful on vxworks
  int postStaticData  (cdevData& data);
  // send logging message to the logging server with more fermaliar
  // way to the ordinary programmer
  int postError   (int verbosity, int severity, 
		   int code, char* facility,
		   char* format, ...);
  // most common way to send message
  int logMsg     (char* format, ...);

  // return pipe fd
  int getFd      (void) const;

#ifdef __vxworks
  // log msg used by ISR
  int logMsgI    (char* format, int arg0, int arg1,
		  int   arg2,   int arg3, int arg4,
		  int   arg5,   int arg6, int arg7,
		  int   arg8,   int arg9);
  int postErrorI (int verbosity, int severity, 
		  int code, char* facility,
		  char* format, int arg0, int arg1,
		  int   arg2,   int arg3, int arg4,
		  int   arg5);
#endif

protected:
#ifdef __vxworks
  // constructor
  cmlogClient  (char* progname = 0);
#endif

  // get all client information
  void  getUserInfo (void);

  // utility for udp open on the same machine
  // return 0 if OK, else return -1
  int udpToClntDOpen (unsigned short port);

  // talk to client daemon via udp message
  int talkToClntD  (int numRetries);

  // send a data to local client daemon and receive response from
  // the server. Establish a timer and resends as necessary
  // return actual size of received datagram, or -1 if error
  // return 0: timedout
  static int udpSendRecv    (int fd, char* outbuff, int outsize,
			     char* inbuf, int insize, 
			     struct sockaddr* destaddr,
			     int    destlen,
			     int    numRetries);

  // open a named pipe, return 0 if ok, else return -1
  int openNamedPipe         (void);

  // send initial connection information such as tag map and context
  int sendConnectionInfo    (cdevData* cxt = 0);

  // send disconnection signal
  int sendDisconnInfo       (void);

  // send tag map information to the server
  int sendTagMapInfo        (int tag, char* name);

  // find svr port number
  unsigned short findServerUdpPort (void)  ;

#ifdef __vxworks
  static int nvramStore(BOOT_PARAMS* parms);
#else
  static void startClientD (void);
#endif

private:
  // program name
  char            *progname_;
  // flag of connection
  int             connected_;
  // client id assignment from client daemon
  short           clientId_;

  // misc client information
  pid_t           taskid_;
  char*           username_;
  char*           hostname_;
  char*           displayname_;

  // udp socket to local client daemon well known port
  struct sockaddr_in udp_addr_;
  int                fd_;

  // named pipe file descriptor
  int                fifo_;

  // cdev tag table callback object
  cmlogClientTagCallback* tagCbk_;

  // filerting mechanism
  cmlogFilter             filter_;

#ifdef _CMLOG_USE_THREAD
  cpMutex                 lock_;
  cpMutex*                cmlog_rlock;
#endif

#ifdef __vxworks
  // static buffer for interrupt service routines
  char                    sbuf_[CMLOG_CLNT_PIPE_MAX_BYTES];
  // char   buffer for format string 
  char                    format_[CMLOG_CLNT_ISR_FORMAT_STR_LEN];
  // packet object for ISR
  cmlogVxPacket1          packet_;

  // friend function
  friend int cmlog_logmsgI (cmlog_client_t client,
			    int verbosity, int severity, 
			    int code, char* facility,
			    char* format, int arg0, int arg1,
			    int   arg2,   int arg3, int arg4,
			    int   arg5);
  friend int cmlog_logtextI (cmlog_client_t client,
			     char* format, 
			     int arg0, int arg1,
			     int   arg2,   int arg3, int arg4,
			     int   arg5,   int arg6, int arg7,
			     int   arg8,   int arg9);

  // single instance of cmlogClient on vxworks
  static  cmlogClient* gclient_;
#endif

  // friend class
  friend class cmlogClientTagCallback;
  
  // deny access to copy and assignment operations
  cmlogClient     (const cmlogClient& cl);
  cmlogClient&    operator = (const cmlogClient& cl);
};

#endif
