//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Simple Token Class (For Thread only)
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cpToken.h,v $
//   Revision 1.1.1.1  1999/09/07 15:29:12  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:30:36  bickley
// Added cmlog to application development system.
//
//
#ifndef _CP_TOKEN_H
#define _CP_TOKEN_H

#ifndef __vxworks
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <errno.h>
#include <time.h>
#include <cpSynch.h>

// = The following structure implements a FIFO of waiter threads
// that are asleep waiting to obtain the token.
struct cpQueueEntry 
{
  cpQueueEntry (cpMutex &m, cp_thread_t t_id);

  cpQueueEntry *next_;
  // Pointer to next waiter.
  cp_thread_t thread_id_;
  // cp_Thread id of this waiter.

  cpConditionMutex cv_;
  // Condition object used to wake up waiter when it can run again.

  int runable_;
  // Ok to run.
};

class cpToken
  // = TITLE
  //    Class that acquires, renews, and releases a process-local
  //    synchronization token.
  //
  // = DESCRIPTION 
  //    This class is a more general-purpose synchronization mechanism
  //    than SunOS 5.x mutexes.  For example, it implements "recursive
  //    mutex" semantics, where a thread that owns the token can
  //    reacquire it without deadlocking.  In addition, threads that are
  //    blocked awaiting the token are serviced in strict FIFO order as
  //    other threads release the token (SunOS 5.x mutexes don't strictly
  //    enforce an acquisition order).
{
public:
  // = Initialization and termination.

  cpToken (int type = CP_THREAD_PROCESS_PRIVATE);
  ~cpToken (void);

  // = Synchronization operations.

  int acquire (void (*sleep_hook)(void *),
	       void *arg = 0, 
	       const struct timespec *timeout = 0);
  // Acquire the token, sleeping until it is obtained or until
  // <timeout> expires.  If some other thread currently holds the
  // token then <sleep_hook> is called before our thread goes to
  // sleep.  This <sleep_hook> can be used by the requesting thread to
  // unblock a token-holder that is sleeping, e.g., by means of
  // writing to a pipe (the ACE ACE_Reactor uses this functionality).
  // Return values: 
  // 0 if acquires without calling <sleep_hook>
  // 1 if <sleep_hook> is called.
  // -1 if failure or timeout occurs (if timeout occurs errno == ETIME) 
  // If <timeout> == <ACE_Time_Value::zerop> then acquire has polling
  // semantics (and does *not* call <sleep_hook>).

  int acquire (const struct timespec *timeout = 0);
  // This behaves just like the previous <acquire> method, except
  // that it invokes the virtual function called <sleep_hook>
  // that can be overridden by a subclass of cpToken.
  
  virtual void sleep_hook (void);
  // This should be overridden by a subclass to define 
  // the appropriate behavior before <acquire> goes to sleep.
  // By default, this is a no-op...

  int renew (int requeue_position = 0, const struct timespec *timeout = 0);
  // An optimized method that efficiently reacquires the token if no
  // other threads are waiting.  This is useful for situations where
  // you don't want to degrad the quality of service if there are
  // other threads waiting to get the token.  If <requeue_position> ==
  // -1 and there are other threads waiting to obtain the token we are
  // queued at the end of the list of waiters.  If <requeue_position>
  // > -1 then it indicates how many entries to skip over before
  // inserting our thread into the list of waiters (e.g.,
  // <requeue_position> == 0 means "insert at front of the queue").
  // Renew has the rather odd semantics such that if there are other
  // waiting threads it will give up the token even if the
  // nesting_level_ > 1.  I'm not sure if this is really the right
  // thing to do (since it makes it possible for shared data to be
  // changed unexpectedly) so use with caution...

  int tryacquire (void);
  // Become interface-compliant with other lock mechanisms (implements
  // a non-blocking <acquire>).

  int remove (void);
  // Shuts down the cpToken instance.

  int release (void);
  // Relinquish the token.  If there are any waiters then the next one
  // in line gets it.

  // = Accessor methods.

  int waiters (void);
  // Return the number of threads that are currently waiting to get
  // the token.

  cp_thread_t current_owner (void);
  // Return the id of the current thread that owns the token.

  virtual const char* className (void) const {return "cpToken";}

private:
  int shared_acquire (void (*sleep_hook_func)(void *), 
		      void *arg,
		      const struct timespec *timeout);
  // Implements the <acquire> and <tryacquire> methods above.

  void remove_entry (cpQueueEntry *);
  // Remove a waiter from the queue (used when a timeout occurs).

  cpQueueEntry *head_;
  // Head of the list of waiting threads.

  cpQueueEntry *tail_;
  // Tail of the list of waiting threads.

  cpMutex lock_;
  // ACE_Mutex used to lock internal data structures.

  cp_thread_t owner_;
  // Current owner of the token.

  int in_use_;
  // Some thread (i.e., <owner_>) is using the token.  We need this
  // extra variable to deal with POSIX pthreads madness...

  int waiters_;
  // Number of waiters.

  int nesting_level_;
  // Current nesting level.
};

class cpTokenGuard
  // = TITLE
  //     This data structure is meant to be used within a method or
  //     function...  It performs automatic aquisition and release of
  //     an cpToken
  //
  // = DESCRIPTION
  //     This should be a specialization of ACE_Guard, but compiler
  //     bugs preclude this.
{
public:
  cpTokenGuard (cpToken& t);
  // implicitly and automatically acquire the lock

  ~cpTokenGuard (void);
  // Implicitly release the lock

  int locked (void);
  // 1 if locked, 0 if could not acquire the lock
  // (errno will contain reason for this)

  int remove (void);
  // Explicitly relesae the lock
  
  int acquire (void);
  // Explicitly acquire the lock
  
  int tryacquire (void);
  // Conditionally acquire the lock (i.e. won't block)

  int release (void);
  // explictlu release the lock

  virtual const char* className (void) const {return "cpTokenGuard";}

protected:
  cpToken& lock_;

  int result_; 
  // keeps track of whether we acquired the lock or failed

  // deny access to assignment and copy
  void operator = (const cpTokenGuard &) {}
  cpTokenGuard (const cpTokenGuard &);
};
  
#undef INLINE
#if defined (__INLINE__)
#define INLINE inline
#include "cpToken.i"
#else
#define INLINE
#endif /* __INLINE__ */

#endif /* vxworks    */

#endif /* cpToken_H */

