//-----------------------------------------------------------------------------
// Copyright (c) 1991,1992 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
// Email: coda@cebaf.gov  Tel: (804) 249-7101  Fax: (804) 249-7363
//-----------------------------------------------------------------------------
// 
// Description:
//	Simple Manager for Posix Threads
//	
// Author:  Jie Chen
//       CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cpThreadManager.cc,v $
//   Revision 1.3  2001/07/25 14:31:09  chen
//   64 BIT Initial Port
//
//   Revision 1.2  2000/06/20 19:36:20  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.1.1.1  1999/09/07 15:29:12  chen
//   CMLOG version 2.0
//
// Revision 1.2  1997/09/16  17:00:02  chen
// port to hpux-10 thread
//
// Revision 1.1  1997/08/01  15:30:31  bickley
// Added cmlog to application development system.
//
//
#include <errno.h>
#include "cpThreadManager.h"

#if !defined (__INLINE__)
#include "cpThreadManager.i"
#endif 

#ifndef __vxworks
static pthread_t CP_NULL_thread;  // just a unsigned int
#else
#include "taskHookLib.h"

// magic number to find out whether a deleted task is from
// cp thread package
#define CP_THREAD_VX_MAGIC_NUMBER 0xdeadbeaf

static cp_thread_t CP_NULL_thread = 0;
#endif

void
cpThreadManager::dump (void) const
{
  // empty
}

cpThreadDescriptor::cpThreadDescriptor (void)
:thr_id_ (CP_NULL_thread),
 grp_id_ (0),
 thr_state_ (cpThreadDescriptor::IDLE)
{
  // empty
}

// Initialize the synchronization variables.
cpThreadManager::cpThreadManager (unsigned size)
: max_table_size_ (0), 
  thr_table_ (0),
  current_count_ (0),
  grp_id_ (1), 
  lock_ (), 
  zero_cond_ (lock_)
{
  if (this->open (size) == -1) 
    fprintf (stderr, "Create cpThreadManager Error\n");

#ifdef __vxworks
  taskDeleteHookAdd ((FUNCPTR)&(cpThreadManager::taskDeleteCallback));
#endif
}

cpThreadManager::~cpThreadManager (void)
{
  this->close ();

#ifdef __vxworks
  taskDeleteHookDelete ((FUNCPTR)&(cpThreadManager::taskDeleteCallback));
#endif
}


// Return the thread descriptor (indexed by cp_thread_t).
int 
cpThreadManager::descriptor (cp_thread_t thr_id,
			     cpThreadDescriptor &descriptor)
{
  cpTMLocker locker (this);

  for (unsigned i = 0; i < this->current_count_; i++)
    if (cpThread::equal (this->thr_table_[i].thr_id_, thr_id))
      {
	descriptor = this->thr_table_[i];
	return 0;
      }

  return -1;
}

int
cpThreadManager::resize (unsigned size)
{
  cpThreadDescriptor *temp = new cpThreadDescriptor[size];
      
  if (temp == 0) {
    errno = ENOMEM;
    return -1;
  }

  for (unsigned i = 0; i < this->max_table_size_; i++)
    temp[i] = this->thr_table_[i]; // Structure assignment.

  this->max_table_size_ = size;

  delete [] this->thr_table_;

  this->thr_table_ = temp;
  return 0;
}

// Create and initialize the table to keep track of the thread pool.

int
cpThreadManager::open (unsigned size)
{
  cpTMLocker locker (this);

  if (this->max_table_size_ < size)
    this->resize (size);
  return 0;
}

// Close up and release all resources.
int
cpThreadManager::close (void)
{
  if (this->thr_table_ != 0) {
    delete [] this->thr_table_;
    this->thr_table_ = 0;
    this->max_table_size_ = 0;
    this->current_count_ = 0;
  }
  return 0;
}


// Call the appropriate OS routine to spawn a thread.  Should *not* be
// called with the lock_ held...
int 
cpThreadManager::spawn_i (CP_THREAD_FUNC func, 
			  void *args, 
			  long flags, 
			  cp_thread_t *t_id, 
			  unsigned int priority,
			  void *stack, 
			  unsigned stack_size)
{
  cp_thread_t thr_id;
  cpThread::init (thr_id);

  if (t_id == 0)
    t_id = &thr_id;

  int result = cpThread::spawn (func, args, flags, 
				t_id, priority,
				stack, stack_size);
  if (result != 0) {
    errno = result;
    return -1;
  }
  else
    return this->append_thr (*t_id, cpThreadDescriptor::SPAWNED);
}

// Create a new thread running FUNC.  *Must* be called with the lock_
// held...
int 
cpThreadManager::spawn (CP_THREAD_FUNC func, 
			void *args, 
			long flags, 
			cp_thread_t *t_id, 
			unsigned int priority,
			void *stack, 
			unsigned stack_size)
{
  cpTMLocker locker (this);

  if (this->spawn_i (func, args, flags, t_id, 
		     priority, stack, stack_size) == -1)
    return -1;
  else
    return this->grp_id_++; // Increment the group id.
}

// Create N new threads running FUNC.

int 
cpThreadManager::spawn_n (int n, 
			  CP_THREAD_FUNC func, 
			  void *args, 
			  long flags,
			  unsigned int priority)
{
  cpTMLocker locker (this);

  int i;
  for (i = 0; i < n; i++) {
    // @@ What should happen if this fails?! e.g., should we try to
    // cancel the other threads that we've already spawned or what?
    if (this->spawn_i (func, args, flags, 0, priority) == -1)
      return -1;
  }
  
  return this->grp_id_++;   // Increment the group id.
}

// Append a thread into the pool (does not check for duplicates).
// Must be called with locks held.

int
cpThreadManager::append_thr (cp_thread_t t_id, 
			     cpThreadDescriptor::Thread_State thr_state)
{
  // Try to resize the array to twice its existing size if we run out
  // of space...
  if (this->current_count_ >= this->max_table_size_ 
      && this->resize (this->max_table_size_ * 2) == -1)
    return -1;
  else
    {
      cpThreadDescriptor &thr_desc = this->thr_table_[this->current_count_];

      thr_desc.thr_id_ = t_id;
      thr_desc.grp_id_ = this->grp_id_;
      thr_desc.thr_state_ = thr_state;

      this->current_count_++;
      return 0;
    }
}

// Insert a thread into the pool (checks for duplicates and doesn't
// allow them to be inserted twice).

int
cpThreadManager::insert_thr (cp_thread_t t_id)
{
  cpTMLocker locker (this);

  // Check for duplicates and bail out if they're already
  // registered...
  if (this->find (t_id) != -1)
    return -1;

  return this->append_thr (t_id, cpThreadDescriptor::SPAWNED);
}

// Remove a thread from the pool.  Must be called with locks held.

void
cpThreadManager::remove_thr (int i)
{
  if (this->current_count_ > 1)
    // Structure assignment.
    this->thr_table_[i] = this->thr_table_[this->current_count_ - 1];

  this->current_count_--;

  // Tell waiters when there are no more threads left in the pool.
  if (this->current_count_ == 0)
    zero_cond_.signal ();
}

int
cpThreadManager::kill_thr (int i, int signum)
{
  int result = cpThread::kill (this->thr_table_[i].thr_id_, signum);

  if (result != 0) {
    this->remove_thr (i);
    errno = result;
    return -1;
  }
  else
    return 0;
}

int
cpThreadManager::cancel_thr (int i, int)
{
  int result = cpThread::cancel (this->thr_table_[i].thr_id_);

  if (result != 0) {
    this->remove_thr (i);
    errno = result;
    return -1;
  }
  else {
    this->remove_thr (i);
    return 0;
  }
}

// Locate the index in the table associated with <t_id>.  Must be
// called with the lock held.

int 
cpThreadManager::find (cp_thread_t t_id)
{
  unsigned i;

  for (i = 0; i < this->current_count_; i++)
#ifndef __vxworks
    if (pthread_equal (t_id, (this->thr_table_[i].thr_id_)) != 0)
#else
    if (t_id == this->thr_table_[i].thr_id_)
#endif
      return i;

  return -1;    
}

// Kill a single thread.
int 
cpThreadManager::kill (cp_thread_t t_id, int signum)
{
  return cpThread::kill (t_id, signum);
}

int
cpThreadManager::cancel (cp_thread_t t_id)
{
  return cpThread::cancel (t_id);
}

// Get group ids for a particular thread id.
int 
cpThreadManager::get_grp (cp_thread_t t_id, int &grp_id)
{
  cpTMLocker locker (this);

  int i = this->find (t_id);
  grp_id = this->thr_table_[i].grp_id_;
  return 0;
}

// Set group ids for a particular thread id.
int 
cpThreadManager::set_grp (cp_thread_t t_id, int grp_id)
{
  cpTMLocker locker (this);  

  int i = this->find (t_id);
  this->thr_table_[i].grp_id_ = grp_id;
  return 0;
}

// Suspend a group of threads.
int
cpThreadManager::apply_grp (int grp_id, 
			    THR_FUNC func,
			    int arg)
{
  cpTMLocker locker (this);

  int result = 0;

  for (unsigned i = 0; i < this->current_count_; i++)
    if (this->thr_table_[i].grp_id_ == grp_id
	&& (this->*func)(i, arg) == -1)
      result = -1;

  return result;
}

int
cpThreadManager::apply_all (THR_FUNC func, int arg)
{
  cpTMLocker locker (this);

  int result = 0;

  for (unsigned i = 0; i < this->current_count_; i++)
    if ((this->*func)(i, arg) == -1)
      result = -1;

  return result;
}

#ifndef _ARCH_PPC
// Kill a group of threads.
int 
cpThreadManager::kill_grp (int grp_id, int signum)
{
  return apply_grp (grp_id, THR_FUNC (&cpThreadManager::kill_thr), signum);
}

int 
cpThreadManager::kill_all (int sig)
{
  return this->apply_all (&cpThreadManager::kill_thr, sig);
}

int
cpThreadManager::cancel_all (void)
{
  return this->apply_all (&cpThreadManager::cancel_thr, 0);
}
#endif



#ifndef __vxworks
// Must be called when thread goes out of scope to clean up its table
// slot.
void *
cpThreadManager::exit (void *status)
{
  cpTMLocker locker (this);

  /* Locate thread id */
  int i = this->find (cpThread::self ());

  if (i == -1)
    {
      errno = ENOENT; 
      return (void *) -1;
    }
  else
    {
      this->remove_thr (i);

      unlock ();  // destructor never get called for locker
      cpThread::exit (status);
      /* NOTREACHED */
    }
  return 0;
}
#else
void *
cpThreadManager::exit (int code)
{
  cpTMLocker locker (this);

  /* Locate thread id */
  int i = this->find (cpThread::self ());

  if (i == -1)
    {
      errno = ENOENT; 
      return (void *) -1;
    }
  else
    {
      this->remove_thr (i);

      unlock ();  // destructor never get called for locker
      cpThread::exit (code);
      /* NOTREACHED */
    }
  return 0;
}

int
cpThreadManager::taskDeleteCallback (WIND_TCB *ptr)
{
  // this routine is get called in the context of the task being deleted
  // Task id is integer representation of WIND_TCB
  int tid = (int)ptr;

  if (ptr->spare4 == CP_THREAD_VX_MAGIC_NUMBER) {
    if (ptr->spare1 != 0) {  // must be a pointer to the manager
      cpThreadManager* manager = (cpThreadManager *)ptr->spare1;

      cpTMLocker locker (manager);

      // Locate thread id 
      int i = manager->find (tid);

      if (i == -1) 
	errno = ENOENT; 
      else
	manager->remove_thr (i);
    }  
    if (ptr->spare2 != 0) {
      CP_THREAD_CLEANUP_FUNC handler = 0;
      handler = (CP_THREAD_CLEANUP_FUNC)ptr->spare2;
      (*handler)((void *)ptr->spare3);
    }
  }
  // disable later clean up routines
  ptr->spare4 = 0;
  return 0;
}
#endif

// Wait for all the threads to exit. 
int
cpThreadManager::wait (const struct timespec *timeout)
{
  cpTMLocker locker (this);

  while (this->current_count_ > 0) {
    if (zero_cond_.wait (timeout) != 0)
      return -1;
  }
  return 0;
}

void
cpThreadControl::dump (void) const
{
  // empty
}

// Initialize the thread controller.
cpThreadControl::cpThreadControl (cpThreadManager *t, 
				  int insert)
:tm_ (t)
{
  if (this->tm_ != 0 && insert) 
    this->tm_->insert_thr (cpThread::self ());

#ifdef __vxworks
  WIND_TCB *tptr = 0;
  tptr = taskTcb (taskIdSelf ());
  if (tptr != 0) {
    tptr->spare1 = (int)t;
    tptr->spare4 = CP_THREAD_VX_MAGIC_NUMBER;  // magic number
  }
#endif
}

// Automatically kill thread on exit.
cpThreadControl::~cpThreadControl (void)
{
#ifndef __vxworks
  this->exit (this->status_);
#else
  this->exit (0);
#endif
}

#ifndef __vxworks
// Exit from thread (but clean up first).
void *
cpThreadControl::exit (void *exit_status)
{
  if (this->tm_ != 0)
    return this->tm_->exit (exit_status);
  else {
    cpThread::exit (exit_status);
    return 0;
  }
}
#else
void *
cpThreadControl::exit (int code)
{
  // set spare1 to zero which disables task delete hook from calling
  // removing this task from manager again
  WIND_TCB* tptr = taskTcb (taskIdSelf());
  tptr->spare1 = 0;

  if (this->tm_ != 0)
    return this->tm_->exit (code);
  else {
    cpThread::exit (code);
    return 0;
  }
}
#endif
