//-----------------------------------------------------------------------------
// Copyright (c) 1991,1992 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
// Email: coda@cebaf.gov  Tel: (804) 249-7101  Fax: (804) 249-7363
//-----------------------------------------------------------------------------
// 
// Description:
//	Simple Interface to Posix Threads and vxWorks Task
//	
// Author:  Jie Chen
//       CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cpThread.i,v $
//   Revision 1.3  2001/07/25 14:31:09  chen
//   64 BIT Initial Port
//
//   Revision 1.2  2000/06/20 19:36:20  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.1.1.1  1999/09/07 15:29:12  chen
//   CMLOG version 2.0
//
// Revision 1.4  1997/09/16  19:43:25  chen
// fix bug for vxworks
//
// Revision 1.3  1997/09/16  17:00:02  chen
// port to hpux-10 thread
//
// Revision 1.2  1997/08/26  16:57:18  chen
// cmlogClientD can be delete/restart on vx
//
// Revision 1.1  1997/08/01  15:30:29  bickley
// Added cmlog to application development system.
//
//

#ifndef __vxworks
// Allocates a <keyp> that is used to identify data that is specific
// to each thread in the process.  The key is global to all threads in
// the process.
INLINE int
cpThread::keycreate (cp_thread_key_t *keyp,
		     void (*destructor)(void *value))
{
  return pthread_key_create (keyp, destructor);
}

// Bind value to the thread-specific data key, <key>, for the calling
// thread.
INLINE int
cpThread::setspecific (cp_thread_key_t key, void *value)
{
  return ::pthread_setspecific (key, value);
}

// Stores the current value bound to <key> for the calling thread
// into the location pointed to by <valuep>.
INLINE int 
cpThread::getspecific (cp_thread_key_t key, void **valuep)
{
#if defined (solaris) || defined (__linux)
  *valuep = pthread_getspecific (key);
#elif defined (__hpux)
  pthread_getspecific (key, valuep);
#endif
  if (*valuep != NULL)
    return 0;
  return -1;
}

INLINE cp_thread_t 
cpThread::self (void)
{
  return pthread_self ();
}

INLINE void 
cpThread::exit (void *status)
{
  pthread_exit (status);
}

INLINE void
cpThread::yield (void)
{
  fprintf (stderr, "Pthread is not supporting pthread_yield....\n");
}

INLINE int 
cpThread::spawn (CP_THREAD_FUNC func, 
		 void *arg, 
		 long flags, 
		 cp_thread_t *thr_id, 
		 unsigned int priority,
		 void *stack, 
		 unsigned stacksize)
{
  int result;
  pthread_attr_t attr;
  pthread_t tmp_thr;
  pthread_t *pthr;
  int spolicy = SCHED_OTHER;

  if (pthread_attr_init (&attr) != 0) 
    return -1;
  
  if (stacksize != 0) {
    unsigned size = stacksize > PTHREAD_STACK_MIN ? 
      stacksize : PTHREAD_STACK_MIN;
    
    if (pthread_attr_setstacksize (&attr, size) != 0) {
      pthread_attr_destroy (&attr);
      return -1;
    }
  }

#if defined (solaris)
  if (stack != 0) {
    if (pthread_attr_setstackaddr (&attr, stack) != 0) {
      pthread_attr_destroy (&attr);
      return -1;
    }
  }
#endif

  // check flags
  if (flags != 0) {
#if defined (solaris) || defined (__linux)
    // hpux threads does not have this function
    if (flags & CP_THREAD_DETACHED) {
      int dstate = PTHREAD_CREATE_DETACHED;
      if (pthread_attr_setdetachstate (&attr, dstate) != 0) {
	pthread_attr_destroy (&attr);
	return -1;
      }
    }
#endif

    if ((flags & CP_THREAD_SCH_FIFO) ||
	(flags & CP_THREAD_SCH_RR)   ||
	(flags & CP_THREAD_SCH_DEFAULT)) {

      if (flags & CP_THREAD_SCH_DEFAULT)
	spolicy = SCHED_OTHER;
      else if (flags & CP_THREAD_SCH_FIFO)
	spolicy = SCHED_FIFO;
      else
	spolicy = SCHED_RR;
      
      if (pthread_attr_setschedpolicy (&attr, spolicy)  != 0) {
	pthread_attr_destroy (&attr);
	return -1;
      }
    }

    
    if ((flags & CP_THREAD_SCH_INHERIT) ||
	(flags & CP_THREAD_SCH_EXPLICIT)) {

#if defined (solaris) || defined (__linux)
      int sched = PTHREAD_EXPLICIT_SCHED;
#elif defined (__hpux)
      int sched = PTHREAD_DEFAULT_SCHED;
#endif

      if (flags & CP_THREAD_SCH_INHERIT)
	sched = PTHREAD_INHERIT_SCHED;
      if (pthread_attr_setinheritsched (&attr, sched) != 0) {
	pthread_attr_destroy (&attr);
	return -1;
      }
    }
  }
  // check priority
  if (priority != 0) {
#if defined (solaris) || defined (__linux)
    struct sched_param sparam;
    int    minp, maxp, newp;

    minp = ::sched_get_priority_min (spolicy);
    maxp = ::sched_get_priority_max (spolicy);
    if (priority > maxp)
      newp = maxp;
    else if (priority < minp)
      newp = minp;
    else
      newp = priority;

    sparam.sched_priority = newp;

    if (pthread_attr_setschedparam (&attr, &sparam) != 0) {
      pthread_attr_destroy (&attr);
      return -1;
    }

#elif defined (__hpux)
    int    minp, maxp, newp;

    if (spolicy == SCHED_FIFO) {
      minp = PRI_FIFO_MIN;
      maxp = PRI_FIFO_MAX;
    }
    else if (spolicy == SCHED_RR) {
      minp = PRI_RR_MIN;
      maxp = PRI_RR_MAX;
    } 
    else if (spolicy == SCHED_FG_NP) {
      minp = PRI_FG_MIN_NP;
      maxp = PRI_FG_MAX_NP;
    } 
    else if (spolicy == SCHED_BG_NP) {
      minp = PRI_BG_MIN_NP;
      maxp = PRI_BG_MAX_NP;
    } 

    if (priority > maxp)
      newp = maxp;
    else if (priority < minp)
      newp = minp;
    else
      newp = priority;

    if (pthread_attr_setprio (&attr, newp) != 0) {
      pthread_attr_destroy (&attr);
      return -1;
    }
#endif
  }
  // get thread id pointer
  pthr = (thr_id == 0 ? &tmp_thr : thr_id);

#if defined (solaris) || defined (__linux)
  result = ::pthread_create (pthr, &attr, func, arg);
#elif defined (__hpux)
  result = ::pthread_create (pthr, attr, func, arg);
#endif

  pthread_attr_destroy (&attr);

  if (result != 0)
    return -1;

  return 0;
}

INLINE int 
cpThread::spawn_n (unsigned n, 
		   CP_THREAD_FUNC func, 
		   void *arg, 
		   long flags, 
		   unsigned int priority,
		   void *stack, 
		   unsigned stack_size)
{
  cp_thread_t t_id;
  unsigned i;

  for (i = 0; i < n; i++)
    // Bail out if error occurs.
    if (cpThread::spawn (func, arg, flags, &t_id,
			 priority, stack, stack_size) != 0)
      break;
  
  return i;
}

INLINE int 
cpThread::spawn_n (cp_thread_t thread_ids[],
		   unsigned n, 
		   CP_THREAD_FUNC func, 
		   void *arg, 
		   long flags, 
		   unsigned int priority,
		   void *stack, 
		   unsigned stack_size)
{
  cp_thread_t t_id;
  unsigned i;

  for (i = 0; i < n; i++)
    // Bail out if error occurs.
    if (cpThread::spawn (func, arg, flags, &t_id,
			 priority, stack, stack_size) == 0)
      thread_ids[i] = t_id;
    else
      break;

  return i;
}

INLINE int 
cpThread::kill (cp_thread_t t_id, int signum)
{
#if defined (solaris) || defined (__linux)
  return pthread_kill (t_id, signum);
#elif defined (__hpux)
  printf ("pthread_kill is not implemented on hpux\n");
  return -1;
#endif
}

INLINE int 
cpThread::join (cp_thread_t wait_for, 
		void **status)
{
  return pthread_join (wait_for, status);
}

INLINE int
cpThread::equal (cp_thread_t id0, cp_thread_t id1)
{
  return pthread_equal (id0, id1);
}

INLINE int 
cpThread::sigsetmask (int how, 
		      const sigset_t *set, 
		      sigset_t *oset)
{
#if defined (solaris) || defined (__linux)
  return pthread_sigmask (how, set, oset);
#elif defined (__hpux)
  // according to hp manual, this is the way to do it
  return sigprocmask (how, set, oset);
#endif
}

INLINE int
cpThread::disablecancel (struct cp_cancel_state *old_state)
{
  int old_cstate;
  int retval;

#if defined (solaris) || defined (__linux)
  if ((retval = pthread_setcancelstate (PTHREAD_CANCEL_DISABLE,
					&old_cstate) == 0))
    if (old_state != 0) {
      memset (old_state, 0, sizeof(old_state));
      old_state->cancelstate = old_cstate;
    }
  return retval;
#elif defined (__hpux)
  old_cstate = pthread_setcancel (CP_THREAD_CANCEL_DISABLE);
  old_state->cancelstate = old_cstate;
  return 0;
#endif
}

INLINE int
cpThread::enablecancel (struct cp_cancel_state *old_state, 
			int flag)
{
  int old_cstate;
  int old_ctype;
  int retval;

#if defined (solaris) || defined (__linux)
  retval = pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, &old_cstate);

  if (retval != 0)
    return retval;

  retval = pthread_setcanceltype (flag, &old_ctype);

  if (retval != 0)
    return retval;

  if (old_state != 0) {
    old_state->cancelstate = old_cstate;
    old_state->canceltype = old_ctype;
  }
  return 0;

#elif defined (__hpux)
  old_cstate = pthread_setcancel (CP_THREAD_CANCEL_ENABLE);
  old_state->cancelstate = old_cstate;
  return 0;  
#endif
}

INLINE int
cpThread::setcancelstate (struct cp_cancel_state &new_state,
			  struct cp_cancel_state *old_state)
{
  int old_cstate;
  int old_ctype;

#if defined (solaris) || defined (__linux)
  if (new_state.cancelstate != 0
      && pthread_setcancelstate (new_state.cancelstate, &old_cstate) == 0)
    return -1;

  if (new_state.canceltype != 0 
      && pthread_setcanceltype (new_state.canceltype, &old_ctype) == 0) {
    int o_cstate;
    pthread_setcancelstate (old_cstate, &o_cstate);
    return -1;
  }

  if (old_state != 0) {
    old_state->cancelstate = old_cstate;
    old_state->canceltype = old_ctype;
  }
  return 0;
#elif defined (__hpux)
  return 0;
#endif
}

INLINE int
cpThread::cancel (cp_thread_t t_id)
{
  return pthread_cancel (t_id);
}

INLINE void
cpThread::testcancel (void)
{
  pthread_testcancel ();
}

INLINE void
cpThread::self (cp_thread_t& t_id)
{
  t_id = pthread_self ();
}

INLINE void
cpThread::init (cp_thread_t& t_id)
{
#if defined (solaris) || defined (__linux)
  t_id = 0;
#elif defined (__hpux)
t_id.field1 = 0;
  t_id.field2 = 0;
  t_id.field3 = 0;
#endif
}  

#else
INLINE int 
cpThread::spawn (CP_THREAD_FUNC func, 
		 void *arg, 
		 long flags, 
		 cp_thread_t *thr_id, 
		 unsigned int priority,
		 void *stack, 
		 unsigned stacksize)
{
  int result;
  cp_thread_t tmp_thr;
  cp_thread_t* pthr;
  char*        pname = 0;
  char         cname[20];
  
  // get thread id pointer
  pthr = (thr_id == 0 ? &tmp_thr : thr_id);

  // get name of parent task
  pname = taskName (taskIdSelf ());
  if (pname) {
    strncpy (cname, pname, 7);
    cname[7] = '\0';
    strcat  (cname, "_c");
  }
  else
    strcpy  (cname, "unknown_c");

  *pthr = ::taskSpawn (cname, priority, flags, stacksize, func,
		       (int)arg, 0, 0, 0, 0, 0, 0, 0, 0, 0);

  if (*pthr == ERROR)
    return -1;
  return 0;
}

INLINE int 
cpThread::spawn_n (unsigned n, 
		   CP_THREAD_FUNC func, 
		   void *arg, 
		   long flags, 
		   unsigned int priority,
		   void *stack, 
		   unsigned stack_size)
{
  cp_thread_t t_id;

  for (unsigned i = 0; i < n; i++)
    // Bail out if error occurs.
    if (cpThread::spawn (func, arg, flags, &t_id,
			 priority, stack, stack_size) != 0)
      break;
  
  return i;
}

INLINE int 
cpThread::spawn_n (cp_thread_t thread_ids[],
		   unsigned n, 
		   CP_THREAD_FUNC func, 
		   void *arg, 
		   long flags, 
		   unsigned int priority,
		   void *stack, 
		   unsigned stack_size)
{
  cp_thread_t t_id;

  for (unsigned i = 0; i < n; i++)
    // Bail out if error occurs.
    if (cpThread::spawn (func, arg, flags, &t_id,
			 priority, stack, stack_size) == 0)
      thread_ids[i] = t_id;
    else
      break;

  return i;
}

INLINE int
cpThread::join (cp_thread_t id, void **arg)
{
  while (taskIdVerify (id) == OK) 
    taskDelay (sysClkRateGet ());
  return 0;
}

INLINE int
cpThread::kill (cp_thread_t id, int signum)
{
  return kill (id, signum);
}

INLINE void
cpThread::yield (void)
{
  // move this task to the end of queue
  taskDelay (NO_WAIT);
}

INLINE cp_thread_t
cpThread::self (void)
{
  return taskIdSelf ();
}

INLINE void 
cpThread::self (cp_thread_t& id)
{
  id = taskIdSelf ();
}

INLINE void
cpThread::init (cp_thread_t& t_id)
{
   t_id = 0;
}  

INLINE int
cpThread::equal (cp_thread_t id0, cp_thread_t id1)
{
  return (id0 == id1);
}

INLINE void 
cpThread::exit (int code)
{
  ::exit (code);
}

INLINE int 
cpThread::sigsetmask (int mask)
{
  return sigmask (mask);
}

INLINE int
cpThread::setspecific (void *ptr)
{
  return ::taskVarAdd (taskIdSelf (), (int *)ptr);
}

INLINE int
cpThread::disablecancel (void)
{
  return taskSafe ();
}

INLINE int
cpThread::enablecancel (void)
{
  return taskUnsafe ();
}

INLINE int
cpThread::cancel (cp_thread_t id)
{
  return taskDelete (id);
}

INLINE void
cpThread::add_cleanup_handler (CP_THREAD_CLEANUP_FUNC handler, void* arg)
{
  WIND_TCB *tptr = 0;
  tptr = taskTcb (taskIdSelf ());
  if (tptr) {
    tptr->spare2 = (int)handler;
    tptr->spare3 = (int)arg;
  }
}

INLINE void
cpThread::remove_cleanup_handler (int exec)
{
  WIND_TCB *tptr = 0;
  tptr = taskTcb (taskIdSelf ());
  CP_THREAD_CLEANUP_FUNC handler = 0;

  if (exec) {
    if (tptr->spare2 != 0) {
      handler = (CP_THREAD_CLEANUP_FUNC) tptr->spare2;
      (*handler)((void *)tptr->spare3);
    }
  }
  if (tptr) {
    tptr->spare2 = 0; 
    tptr->spare3 = 0;
  }
}

INLINE int
cpThread::setpriority (cp_thread_t id, int newpri)
{
  return taskPrioritySet (id, newpri);
}

INLINE int
cpThread::getpriority (cp_thread_t id, int *pri)
{
  return taskPriorityGet (id, pri);
}

#endif


