//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Simple Synchronization Mechanism for Posix Thread
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cpSynch.cc,v $
//   Revision 1.1.1.1  1999/09/07 15:29:12  chen
//   CMLOG version 2.0
//
// Revision 1.2  1997/09/16  17:00:00  chen
// port to hpux-10 thread
//
// Revision 1.1  1997/08/01  15:30:20  bickley
// Added cmlog to application development system.
//
//
#include <errno.h>
#include "cpSynch.h"

//=========================================================================
//   Implementation of cpMutex Class
//=========================================================================
#ifndef __vxworks
cpMutex::cpMutex (int type)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cpMutex Class Object\n");
#endif
  pthread_mutexattr_t attr;
  int res = -1;
  
  if (pthread_mutexattr_init (&attr) == 0 &&
      pthread_mutexattr_setpshared (&attr, type) == 0) {
    if (pthread_mutex_init (&lock_, &attr) == 0)
      res = 0;
  }
  pthread_mutexattr_destroy (&attr);
  
  if (res == -1) 
    fprintf (stderr, "Cannot initialize pthread_mutex_t\n");
}

cpMutex::~cpMutex (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cpMutex Class Object\n");
#endif
  remove ();
}

int
cpMutex::remove (void)
{
  return ::pthread_mutex_destroy (&lock_);
}

int
cpMutex::acquire (void)
{
  return ::pthread_mutex_lock (&lock_);
}

int
cpMutex::tryAcquire (void)
{
  return ::pthread_mutex_trylock (&lock_);
}

int
cpMutex::release (void)
{
  return ::pthread_mutex_unlock (&lock_);
}
#else
cpMutex::cpMutex (int type)
:lock_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cpMutex Class Object\n");
#endif
  lock_ = ::semMCreate (type);
}

cpMutex::~cpMutex (void)
{
#ifdef _TRACE_OBJECTS
  printf ("delete cpMutex Class Object\n");
#endif
  remove ();
}

int
cpMutex::remove (void)
{
  if (lock_ != NULL)
    return semDelete (lock_);
  lock_ = 0;
}

int
cpMutex::acquire (void)
{
  if (lock_ != NULL)
    return semTake (lock_, WAIT_FOREVER);
  return -1;
}

int
cpMutex::tryAcquire (void)
{
  if (lock_ != NULL)
    return semTake (lock_, NO_WAIT);
  return -1;
}

int
cpMutex::release (void)
{
  if (lock_ != NULL)
    return semGive (lock_);
  return -1;
}
#endif

//=========================================================================
//              implementation of cpRWMutex class
//=========================================================================
#ifndef __vxworks
cpRWMutex::cpRWMutex (int type)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cpRWMutex Class Object\n");
#endif
  pthread_mutexattr_t attr;
  int res = -1;
  
  if (pthread_mutexattr_init (&attr) == 0 &&
      pthread_mutexattr_setpshared (&attr, type) == 0) {
    if (pthread_mutex_init (&lock_, &attr) == 0)
      res = 0;
  }
  pthread_mutexattr_destroy (&attr);

  if (res == 0) {
    pthread_condattr_t cv_attr;

#if defined (solaris)
    if (pthread_condattr_init (&cv_attr) == 0 &&
	pthread_condattr_setpshared (&cv_attr, type) == 0) {
#elif defined (__hpux) || defined (__linux)
    if (pthread_condattr_init (&cv_attr) == 0) {
#endif
      pthread_cond_init (&readers_ok_, &cv_attr);
      pthread_cond_init (&writers_ok_, &cv_attr);
      rwlock_ = 0;
      waitingWriters_ = 0;
      pthread_condattr_destroy (&cv_attr);
    }
    else
      fprintf (stderr, "Fatal: Initialize the cpRWMutex Error\n");
  }
  else 
    fprintf (stderr, "Fatal: Initialize the cpRWMutex Error\n");
}

cpRWMutex::~cpRWMutex (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Destroy cpRWMutex Class Object\n");
#endif
  pthread_cond_destroy (&readers_ok_);
  pthread_cond_destroy (&writers_ok_);
  pthread_mutex_destroy (&lock_);
}

int
cpRWMutex::acquireRLock (void)
{
  ::pthread_mutex_lock (&lock_);
  while (rwlock_ < 0 || waitingWriters_)
    ::pthread_cond_wait (&readers_ok_, &lock_);
  rwlock_ ++;
  ::pthread_mutex_unlock (&lock_);
  return 0;
}

int
cpRWMutex::acquireWLock (void)
{
  ::pthread_mutex_lock (&lock_);
  while (rwlock_ != 0) {
    waitingWriters_ ++;
    ::pthread_cond_wait (&writers_ok_, &lock_);
    waitingWriters_ --;
  }
  rwlock_ = -1;
  ::pthread_mutex_unlock (&lock_);
  return 0;
}

int
cpRWMutex::release (void)
{
  int ww, wr;
  
  ::pthread_mutex_lock (&lock_);
  if (rwlock_ < 0)  // locked for writing
    rwlock_ = 0;
  else
    rwlock_ --;

  ww = (waitingWriters_ && rwlock_ == 0);
  wr = (waitingWriters_ == 0);
  ::pthread_mutex_unlock (&lock_);

  // wake up those waiting writers first
  if (ww)
    ::pthread_cond_signal (&writers_ok_);
  else if (wr)
    ::pthread_cond_broadcast (&readers_ok_);

  return 0;
}

#else

cpRWMutex::cpRWMutex (int type)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cpRWMutex Class Object\n");
#endif
  if (type & SEM_Q_PRIORITY)
    lock_ = semMCreate (type | SEM_DELETE_SAFE | SEM_INVERSION_SAFE);
  else
    lock_ = semMCreate (type | SEM_DELETE_SAFE);
  
  if (lock_ != NULL) {
    if ((readers_ok_ = semBCreate (type, SEM_FULL)) != NULL)
      writers_ok_ = semBCreate (type, SEM_FULL);
  }
  rwlock_ = 0;
  waitingWriters_ = 0;
}

cpRWMutex::~cpRWMutex (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Destroy cpRWMutex Class Object\n");
#endif
  semDelete (readers_ok_);
  semDelete (writers_ok_);
  semDelete (lock_);
}

int
cpRWMutex::acquireRLock (void)
{
  semTake (lock_, WAIT_FOREVER);

  while (rwlock_ < 0 || waitingWriters_)
    semTake (readers_ok_, WAIT_FOREVER);
  rwlock_ ++;
  
  semGive (lock_);
  return 0;
}

int
cpRWMutex::acquireWLock (void)
{
  semTake (lock_, WAIT_FOREVER);

  while (rwlock_ != 0) {
    waitingWriters_ ++;
    semTake (writers_ok_, WAIT_FOREVER);
    waitingWriters_ --;
  }
  rwlock_ = -1;

  semGive (lock_);
  return 0;
}

int
cpRWMutex::release (void)
{
  int ww, wr;
  
  semTake (lock_, WAIT_FOREVER);

  if (rwlock_ < 0)  // locked for writing
    rwlock_ = 0;
  else
    rwlock_ --;

  ww = (waitingWriters_ && rwlock_ == 0);
  wr = (waitingWriters_ == 0);
  semGive (lock_);

  // wake up those waiting writers first
  if (ww)
    semGive (writers_ok_);
  else if (wr)
    semGive (readers_ok_);

  return 0;
}
#endif

//=========================================================================
// Implementation of cpRecursiveLock Class
//=========================================================================
#ifndef __vxworks
cpRecursiveMutex::cpRecursiveMutex (int type)
:nestingLevel_ (0), lock_ (type), mutex_ (type)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cpRecursiveMutex Class Object\n");
#endif
  cpThread::init (thrId_);
}

cpRecursiveMutex::~cpRecursiveMutex (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Destroy cpRecursiveMutex Class Object\n");
#endif
}

int
cpRecursiveMutex::acquire (void)
{
  mutex_.acquire ();
  cp_thread_t tid = ::pthread_self ();

  if (pthread_equal (tid, thrId_)) {
    ++ nestingLevel_;
    mutex_.release ();
    return 0;
  }
  else {
    mutex_.release ();
    if (lock_.acquire () == 0) {
      setThreadId (tid);
      setNestingLevel (0);
      return 0;
    }
    return -1;
  }
}

int
cpRecursiveMutex::release (void)
{
  mutex_.acquire ();
  
  if (nestingLevel_ > 0) {
    --nestingLevel_;
    mutex_.release ();
    return 0;
  }
  else {
#if defined (solaris) || defined (__linux)
    setThreadId (0);
#elif defined (__hpux)
    cpThread::init (thrId_);
#endif

    mutex_.release ();
    return lock_.release ();
  }
}

int
cpRecursiveMutex::tryAcquire (void)
{
  mutex_.acquire ();
  cp_thread_t tid = ::pthread_self ();

  if (pthread_equal (tid, thrId_)) {
    ++nestingLevel_;
    mutex_.release ();
    return 0;
  }
  else {
    mutex_.release ();
    if (lock_.tryAcquire () == 0) {
      setThreadId (tid);
      setNestingLevel (0);
      return 0;
    }
    return -1;
  }
}

cpRecursiveMutex::operator cpMutex &()
{
  return lock_;
}

cp_thread_t
cpRecursiveMutex::getThreadId (void)
{
  cpMutexGuard guard (mutex_);
  return thrId_;
}

void
cpRecursiveMutex::setThreadId (cp_thread_t t)
{
#if defined (solaris) || defined (__linux)
  thrId_ = t;
#elif defined (__hpux)
  thrId_.field1 = t.field1;
  thrId_.field2 = t.field2;
  thrId_.field3 = t.field3;
#endif
}

int
cpRecursiveMutex::getNestingLevel (void)
{
  cpMutexGuard guard (mutex_);
  return nestingLevel_;
}

void
cpRecursiveMutex::setNestingLevel (int d)
{
  nestingLevel_ = d;
}
#else
cpRecursiveMutex::cpRecursiveMutex (int type)
:nestingLevel_ (0), thrId_ (0), lock_ (type), mutex_ (type)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cpRecursiveMutex Class Object\n");
#endif
}

cpRecursiveMutex::~cpRecursiveMutex (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Destroy cpRecursiveMutex Class Object\n");
#endif
}

int
cpRecursiveMutex::acquire (void)
{
  mutex_.acquire ();
  cp_thread_t tid = taskIdSelf ();

  if (tid == thrId_) {
    ++ nestingLevel_;
    mutex_.release ();
    return 0;
  }
  else {
    mutex_.release ();
    if (lock_.acquire () == 0) {
      setThreadId (tid);
      setNestingLevel (0);
      return 0;
    }
    return -1;
  }
}

int
cpRecursiveMutex::release (void)
{
  mutex_.acquire ();
  
  if (nestingLevel_ > 0) {
    --nestingLevel_;
    mutex_.release ();
    return 0;
  }
  else {
    setThreadId (0);
    mutex_.release ();
    return lock_.release ();
  }
}

int
cpRecursiveMutex::tryAcquire (void)
{
  mutex_.acquire ();
  cp_thread_t tid = taskIdSelf ();

  if (tid ==  thrId_) {
    ++nestingLevel_;
    mutex_.release ();
    return 0;
  }
  else {
    mutex_.release ();
    if (lock_.tryAcquire () == 0) {
      setThreadId (tid);
      setNestingLevel (0);
      return 0;
    }
    return -1;
  }
}

cpRecursiveMutex::operator cpMutex &()
{
  return lock_;
}

cp_thread_t
cpRecursiveMutex::getThreadId (void)
{
  cpMutexGuard guard (mutex_);
  return thrId_;
}

void
cpRecursiveMutex::setThreadId (cp_thread_t t)
{
  thrId_ = t;
}

int
cpRecursiveMutex::getNestingLevel (void)
{
  cpMutexGuard guard (mutex_);
  return nestingLevel_;
}

void
cpRecursiveMutex::setNestingLevel (int d)
{
  nestingLevel_ = d;
}
#endif


//=========================================================================
// Implementation of cpConditionMutex class
//=========================================================================
#ifndef __vxworks
cpConditionMutex::cpConditionMutex (cpMutex& m, int type)
:mutex_ (m)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cpConditionMutex Class Object\n");
#endif
  pthread_condattr_t cv_attr;

#if defined (solaris)
  if (pthread_condattr_init (&cv_attr) == 0 &&
      pthread_condattr_setpshared (&cv_attr, type) == 0) {
#elif defined (__hpux) || defined (__linux)
  if (pthread_condattr_init (&cv_attr) == 0) {
#endif
    pthread_cond_init (&cond_, &cv_attr);
    pthread_condattr_destroy (&cv_attr);
  }
  else
    fprintf (stderr, "Fatal: Error in initializing cpConditionMutex\n");
}

cpConditionMutex::~cpConditionMutex (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cpConditionMutex Class Object\n");
#endif

  // if condition is busy when one tries to destroy it, 
  // the pthread_cond_destroy () will return -1 and set errno
  // to EBUSY. 
  int res = 0;

  while ((res = pthread_cond_destroy (&cond_)) == -1
	 && errno == EBUSY) {
    pthread_cond_broadcast (&cond_);
  }
}

int
cpConditionMutex::wait (const struct timespec* abs)
{
  if (abs == 0) 
    return ::pthread_cond_wait (&cond_, &mutex_.lock_);
#if defined (solaris) || defined (__linux)
  return ::pthread_cond_timedwait (&cond_, &mutex_.lock_, abs);
#elif defined (__hpux)
  return ::pthread_cond_timedwait (&cond_, &mutex_.lock_, 
				   (struct timespec *)abs);
#endif
}

int
cpConditionMutex::signal (void)
{
  return ::pthread_cond_signal (&cond_);
}

int
cpConditionMutex::broadcast (void)
{
  return ::pthread_cond_broadcast (&cond_);
}

cpMutex&
cpConditionMutex::mutex (void)
{
  return mutex_;
}
#else  /* vxworks */
cpConditionMutex::cpConditionMutex (cpMutex& m, int type)
:mutex_ (m)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cpConditionMutex Class Object\n");
#endif
  cond_ = semBCreate (type, SEM_EMPTY);
}

cpConditionMutex::~cpConditionMutex (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cpConditionMutex Class Object\n");
#endif
  // make sure not to delete the semaphore when one waits on it
  semDelete (cond_);
}

int
cpConditionMutex::wait (const struct timespec* abs)
{
  int status = 0;
  if (abs == 0) {
    mutex_.release ();
    semTake (cond_, WAIT_FOREVER);
  }
  else {
    mutex_.release ();
    int timeout = abs->tv_sec*sysClkRateGet () + 
      (abs->tv_nsec)/(1000000000*60);
    status = semTake (cond_, timeout);
  }
  mutex_.acquire ();
  return status;
}

int
cpConditionMutex::signal (void)
{
  return ::semGive (cond_);
}

int
cpConditionMutex::broadcast (void)
{
  return ::semFlush (cond_);
}

cpMutex&
cpConditionMutex::mutex (void)
{
  return mutex_;
}
#endif

//=========================================================================
// Implementation of cpConditionRWMutex class
//=========================================================================
#ifndef __vxworks
cpConditionRWMutex::cpConditionRWMutex (cpRWMutex& m, int type)
:mutex_ (m)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cpConditionRWMutex Class Object\n");
#endif
  pthread_condattr_t cv_attr;

#if defined (solaris)
  if (pthread_condattr_init (&cv_attr) == 0 &&
      pthread_condattr_setpshared (&cv_attr, type) == 0) {
#elif defined (__hpux) || defined (__linux)
  if (pthread_condattr_init (&cv_attr) == 0) {
#endif

    pthread_cond_init (&cond_, &cv_attr);
    pthread_condattr_destroy (&cv_attr);
  }
  else
    fprintf (stderr, "Fatal: Error in initializing cpConditionRWMutex\n");
}

cpConditionRWMutex::~cpConditionRWMutex (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cpConditionRWMutex Class Object\n");
#endif

  // if condition is busy when one tries to destroy it, 
  // the pthread_cond_destroy () will return -1 and set errno
  // to EBUSY. 
  int res = 0;

  while ((res = pthread_cond_destroy (&cond_)) == -1
	 && errno == EBUSY) {
    pthread_cond_broadcast (&cond_);
  }
}

int
cpConditionRWMutex::wait (const struct timespec* abs)
{
  int writer = 0;
  int timeout = 0;

  ::pthread_mutex_lock (&mutex_.lock_);
  if (mutex_.rwlock_ < 0) { // locked for writing
    mutex_.rwlock_ = 0;
    writer = 1;  // remeber this is a write lock
  }
  else 
    mutex_.rwlock_ --;
  
  // signal waiting writer first, otherwise wake up all readers
  if (mutex_.waitingWriters_ && mutex_.rwlock_ == 0) 
    ::pthread_cond_signal (&mutex_.writers_ok_);
  else
    ::pthread_cond_broadcast (&mutex_.readers_ok_);

  // release lock, wait for wake up. caller must retest condition
  if (abs == 0)
    timeout = ::pthread_cond_wait (&cond_, &mutex_.lock_);
  else 
#if defined (solaris) || defined (__linux)
    timeout = ::pthread_cond_timedwait (&cond_, &mutex_.lock_, abs);
#elif (__hpux)
    timeout = ::pthread_cond_timedwait (&cond_, &mutex_.lock_, 
					(struct timespec *)abs);
#endif


  ::pthread_mutex_unlock (&mutex_.lock_);
  
  // reacquire read or write lock
  if (writer)
    mutex_.acquireWLock ();
  else
    mutex_.acquireRLock ();

  return timeout;
}

int
cpConditionRWMutex::signal (void)
{
  return ::pthread_cond_signal (&cond_);
}

int
cpConditionRWMutex::broadcast (void)
{
  return ::pthread_cond_broadcast (&cond_);
}

cpRWMutex&
cpConditionRWMutex::mutex (void)
{
  return mutex_;
}
#else /* vxworks */
cpConditionRWMutex::cpConditionRWMutex (cpRWMutex& m, int type)
:mutex_ (m)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cpConditionRWMutex Class Object\n");
#endif
  cond_ = semBCreate (type, SEM_EMPTY);
}

cpConditionRWMutex::~cpConditionRWMutex (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cpConditionRWMutex Class Object\n");
#endif

  semDelete (cond_);
}

int
cpConditionRWMutex::wait (const struct timespec* abs)
{
  int writer = 0;
  int timeout = 0;

  // first release this RW lock
  semTake (mutex_.lock_, WAIT_FOREVER);
  if (mutex_.rwlock_ < 0) { // remeber this is locked for writing
    mutex_.rwlock_ = 0;
    writer = 1;  
    // remeber this is a write lock since this  gets
    // called right after one of the mutex.acquirR(W)lock
  }
  else 
    mutex_.rwlock_ --;
  
  semGive (mutex_.lock_);

  // signal waiting writer first, otherwise wake up all readers
  if (mutex_.waitingWriters_ && mutex_.rwlock_ == 0) 
    semGive (mutex_.writers_ok_);
  else
    semFlush (mutex_.readers_ok_);

  /* wait on the condition */
  int status = 0;
  if (abs == 0) 
    semTake (cond_, WAIT_FOREVER);
  else {
    int timeout = abs->tv_sec*sysClkRateGet () + 
      (abs->tv_nsec)/(1000000000*60);
    status = semTake (cond_, timeout);
  }

  // reacquire read or write lock
  if (writer)
    mutex_.acquireWLock ();
  else
    mutex_.acquireRLock ();

  return status;
}

int
cpConditionRWMutex::signal (void)
{
  return semGive (cond_);
}

int
cpConditionRWMutex::broadcast (void)
{
  return semFlush (cond_);
}

cpRWMutex&
cpConditionRWMutex::mutex (void)
{
  return mutex_;
}
#endif


//=========================================================================
// Implementation of class cpMutexGuard
//=========================================================================
cpMutexGuard::cpMutexGuard (cpMutex& m)
:m_ (m)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cpMutexLock Class Object\n");
#endif
  m_.acquire ();
}

cpMutexGuard::~cpMutexGuard (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Destroy cpMutexLock Class Object\n");
#endif
  m_.release ();
}

//=========================================================================
// Implementation of class cpRecursiveMutexLock
//=========================================================================
cpRecursiveMutexGuard::cpRecursiveMutexGuard (cpRecursiveMutex& m)
:m_ (m)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cpRecursiveMutexLock Class Object\n");
#endif
  m_.acquire ();
}

cpRecursiveMutexGuard::~cpRecursiveMutexGuard (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Destroy cpRecursiveMutexLock Class Object\n");
#endif
  m_.release ();
}


