//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      cmlog Server to browser IO handler
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogBrowserIO.cc,v $
//   Revision 1.4  2001/07/25 14:30:18  chen
//   64 BIT Initial Port
//
//   Revision 1.3  1999/12/13 21:24:39  chen
//   Add more error checking using BAD_PTR
//
//   Revision 1.2  1999/12/13 15:17:33  chen
//   Add robust check on all incoming messages
//
//   Revision 1.1.1.1  1999/09/07 15:29:11  chen
//   CMLOG version 2.0
//
// Revision 1.3  1998/10/07  17:03:23  chen
// Fix a compiler (egcs) warning about cmlog_cdevMessage& = (cmlog_cdevMessage)
//
// Revision 1.2  1998/03/27  13:56:35  chen
// add information about connection
//
// Revision 1.1  1997/08/01  15:28:49  bickley
// Added cmlog to application development system.
//
//
#include <cmlogUtil.h>
#include <cmlogData.h>
#include <cmlogDatabase.h>
#include <cmlogDataManager.h>
#include <cmlogQuerySvc.h>
#include "cmlogBrowserIO.h"

extern int serverMsgToPacket (char* msg, cmlogPacket& packet);

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
cmlogBrowserIO::cmlogBrowserIO (ACE_Reactor& r, 
				cmlogDataQueue* q,
				cmlogDatabase* dbase,
				cmlogDatabase* cxtdbase)
:cmlogIO (r, q), map_ (), queryManager_ (), 
 dbase_ (dbase), cxtdbase_ (cxtdbase), username_ (0),
 displayname_ (0), taskid_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogBrowserIO Class Object\n");
#endif
  // empty
}
#else
#include <cpProcessManager.h>

cmlogBrowserIO::cmlogBrowserIO (ACE_Reactor& r, 
				cmlogDataQueue* q,
				cmlogDataQueue* cq,
				cmlogDatabase* dbase,
				cmlogDatabase* cxtdbase)
:cmlogIO (r, q, cq), map_ (), queryManager_ (),
 dbase_ (dbase), cxtdbase_ (cxtdbase), username_ (0), 
 displayname_ (0), taskid_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogBrowserIO Class Object\n");
#endif
  // empty
}
#endif

cmlogBrowserIO::~cmlogBrowserIO (void)
{
#ifdef _TRACE_OBJECTS
  printf ("    Delete cmlogBrowserIO Class Object\n");
#endif
  if (username_)
    delete []username_;
  if (displayname_)
    delete []displayname_;
}

int
cmlogBrowserIO::handle_close (int fd, ACE_Reactor_Mask mask)
{
  // remove all data callbacks assocaited with this IO channel
  dataManager->removeCbksOfCh ((void *)this);
  // call default handle close
  return cmlogIO::handle_close (fd, mask);
}

int
cmlogBrowserIO::handle_input (int fd)
{

  return reader ();
}

int
cmlogBrowserIO::reader (void)
{
  cmlogPacket packet;

  int n = sockStream_ >> packet;
#ifdef _CMLOG_DEBUG
  printf ("Received %d bytes from browser\n", n);
#endif
  if (n <= 0) {
#ifdef _CMLOG_DEBUG
    printf ("browser %s connection closed\n", clientHost_);
#endif
    packet.reset ();

    // log server message
    char tmp[128];
    char username[80];
    char display[256];

    if (username_)
      strncpy (username, username_, sizeof (username) - 1);
    else
      strcpy (username, "unknown");

    if (displayname_)
      strncpy (display, displayname_, sizeof (display) - 1);
    else
      strcpy (display, "unknown");
    
    sprintf (tmp, "User %s started browser from %s at display %s closed\n", 
	     username, clientHost_, display);
    logServerMsg (tmp);
    cmlogUtil::serverPrintf (tmp);

    connectionFlag (0);
    deactivate ();
    cleanup ();
    return 0;
  }
  
  return processData (packet);
}

#define CMLOG_BROWSERIO_SENDERR(err,data,msg) {\
        sendErrorBack (err, idata); \
        delete msg; \
        continue; \
        }

int
cmlogBrowserIO::processData (cmlogPacket& packet)
{
  int         st = 0;
  cdevData*   tagmap = 0;
  char*       message = 0;
  cdevData*   data = 0;
  double      start, end;
  char        verb[80], attribute[80];
  int         deletemsg = 1;

  // remember to free all msgs ansg msgs[] somewhere
  cmlogMsg** msgs = packet.messages ();
  if (msgs == (cmlogMsg **)CMLOG_BAD_PTR) {
    // log server msg
    char tmp[80];
    sprintf (tmp, "Browser %s processing message error. Closing connection\n", 
	     clientHost_);
    logServerMsg (tmp);

    cmlogUtil::serverPrintf (tmp);
    
    return -1;
  }

  for (int i = 0; i < packet.numberOfData (); i++) {
    // set this pointer to the cmlog_cdevMessage
    msgs[i]->channelId ((void *)this);

    // retrieve the cmlog_cdevMessage
    cmlog_cdevMessage& idata = (*msgs[i]);
    // get tagmap first
    tagmap = idata.getTagMap ();
    if (tagmap) 
      setTagMap (tagmap);
    
    // get query message
    message = idata.getMessage ();
    if (message) {
      if ((st = ::sscanf (message, "%s %s", verb, attribute)) == 1) {
	if (::strcasecmp (verb, "query") == 0) {
	  // get start and end time
	  data = idata.getData ();

	  // process data with tag map
	  processDataWithTagMap (data);
    
	  if (!data) CMLOG_BROWSERIO_SENDERR(CMLOG_INVALID,idata, msgs[i]);

	  if (data->get (cmlogUtil::CMLOG_START_TAG, &start) != CDEV_SUCCESS) 
	    CMLOG_BROWSERIO_SENDERR(CMLOG_INVALID,idata,msgs[i]);

	  if (data->get (cmlogUtil::CMLOG_END_TAG, &end) != CDEV_SUCCESS) 
	    CMLOG_BROWSERIO_SENDERR(CMLOG_INVALID,idata,msgs[i]);

	  if (end <= start)  // end time must be bigger than start time
	    CMLOG_BROWSERIO_SENDERR(CMLOG_INVALID,idata,msgs[i]);

	  idata.setOperationCode (CMLOG_FIND_DATA);

	  // get tag map of this browser and attach to this message to send to remote
	  // process if using multiple processes implentation
#if !defined (CMLOG_USE_THREAD) || !defined (_REENTRANT)
	  cdevData* tagdata = map_.tagmap ();
	  if (tagdata)
	    idata.setTagMap (tagdata);
#endif

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
	  // create a detached thread that will delete msgs[i]
	  cmlogQuerySvc* svc = new cmlogQuerySvc (this, msgs[i]);
	  long           thflags = CP_THREAD_DETACHED; 
	  if(cpThread::spawn (CP_THREAD_FUNC(&(cmlogQuerySvc::querySvcThread)),
			      (void *)svc, thflags) != 0) {
#ifdef _CMLOG_DEBUG
	    fprintf (stderr, "Cannot spawn query thread anymore\n");
#endif
	    sendErrorBack (CMLOG_THREAD_ERR, idata);
	  }
	  else
	    deletemsg = 0;
#else   // CMLOG_USE_THREAD
	  // create a detached process
	  cmlogQuerySvc* svc = new cmlogQuerySvc (this, msgs[i]);
	  pid_t          proc;
	  if (cpProcessManager::spawn_detached_process
	      (CP_PROCESS_FUNC(&(cmlogQuerySvc::querySvcThread)), 
	       (void *)svc, &proc) != 0) {
#ifdef _CMLOG_DEBUG
	    fprintf (stderr, "Cannot fork new query process\n");
#endif
	    sendErrorBack (CMLOG_THREAD_ERR, idata);
	  }
	  else {
	    // register this process to query manager
	    svc->queryProcessId (proc);
	    queryManager_.registerSvc (svc);
	    // insert this message to data queue too, so that one can flush
	    // data base before query command
	    dataQueue_->enqueue (*msgs[i]);
	  }
#endif  
	}
	else if (strcasecmp (verb, "stopQuery") == 0) {
#ifdef _CMLOG_DEBUG
	  fprintf (stderr, "Trying to stop a query\n");
#endif
	  queryManager_.abort (&idata);
	}
	else 
	  CMLOG_BROWSERIO_SENDERR(CMLOG_INVALID,idata, msgs[i]);
      }
      else if (st >= 2) {
	if (strcasecmp (verb, "monitorOn") == 0) {
	  // try to find this attribute
	  cmlogData* mdata = 0;
	  if (dataManager->findData (attribute, mdata) != 0)  // not found
	    CMLOG_BROWSERIO_SENDERR(CMLOG_NOTFOUND,idata, msgs[i]);	
	  
	  // add this monitor callback to this data
	  // msgs[i] cannot be deleted since idata is needed
	  mdata->monitorOn (&idata);
	  mdata->update ();
	  deletemsg = 0;
	}
	else if (strcasecmp (verb, "monitorOff") == 0) {
	  // try to find this attribute
	  cmlogData* mdata = 0;
	  if (dataManager->findData (attribute, mdata) != 0) // not found
	    CMLOG_BROWSERIO_SENDERR(CMLOG_NOTFOUND,idata, msgs[i]);	

	  // add this monitor callback to this data
	  mdata->monitorOff (&idata);
	  sendErrorBack (CMLOG_CBK_FINISHED, idata);
	}
	else 
	  sendErrorBack (CMLOG_NOTFOUND, idata);
      }
    }
    else {
      unsigned opcode = msgs[i]->type ();
      switch (opcode) {
      case CMLOG_CONN_INFO:
	{
	  char brconnmsg[256];
	  cdevData* cxt = idata.getData();

	  if (cxt) {
	    char username[80];
	    char display[256];
	    long taskid;
	    
	    if (cxt->get (cmlogUtil::CMLOG_USER_TAG, username, sizeof (username)) 
		== CDEV_SUCCESS) {
	      username_ = new char[::strlen(username) + 1];
	      ::strcpy (username_, username);
	    }
	      
	    if (cxt->get (cmlogUtil::CMLOG_PID_TAG,  &taskid) == CDEV_SUCCESS)
	      taskid_ = taskid;

	    if (cxt->get (cmlogUtil::CMLOG_DISP_TAG, display, sizeof (display))
		== CDEV_SUCCESS) {
	      displayname_ = new char[::strlen (display) + 1];
	      ::strcpy (displayname_, display);
	    }

	    sprintf (brconnmsg, 
		     "Received user %s launched browser from %s with process id %d at display %s",
		     username, clientHost_, taskid, display);
	    logServerMsg (brconnmsg);

	    cmlogUtil::serverPrintf (brconnmsg);
	  }
	}
	break;
      default:
	break;
      }
    }
    
    if (deletemsg)
      delete msgs[i];
  }

  packet.empty (); 
  // free memory for msgs
  delete []msgs;

  return 0;
}

void
cmlogBrowserIO::setTagMap (cdevData* tagmap)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  map_.updateTagMap (*tagmap);

#ifdef _CMLOG_DEBUG
  map_.asciiDump ();
#endif
}

void
cmlogBrowserIO::processDataWithTagMap (cdevData* data)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  map_.remoteToLocal (*data);
}

cmlog_cdevTagMap*
cmlogBrowserIO::tagmap (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif
  return &map_;
}

void
cmlogBrowserIO::convertDataLToR (cdevData& data)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif
  map_.localToRemote (data);
}


