//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
//-----------------------------------------------------------------------------
// 
// Description:
//	 Single Linked List for void *
//
//       Note: remove and clean operations on the list
//             will only remove link nodes without removal of
//             the content inside the nodes. It is callers' 
//             responsiblity to clean up those contents
//
//       This is unsafe C++ practice, use this list at you own risk
//       
//       Reason for this list: It is very difficult to instantiate
//       a template class in a stand alone shared library
//	
// Author:  Jie Chen
//       CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogSlist.cc,v $
//   Revision 1.1.1.1  1999/09/07 15:29:10  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:27:31  bickley
// Added cmlog to application development system.
//
//
#include "cmlogSlist.h"

//======================================================================
//	class cmlogSlist implementation
//======================================================================
cmlogSlist::cmlogSlist (void)
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
:ptrToFirstLink(0), lock_ ()
#else
:ptrToFirstLink(0)
#endif
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogSlist Class Object\n");
#endif
  // no further initialization
}

cmlogSlist::cmlogSlist (const cmlogSlist & source)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogSlist Class Object\n");
#endif

  // duplicate elements from source list
  if (source.isEmpty_i())
    ptrToFirstLink = 0;
  else{
    cmlogSlistLink * firstLink = source.ptrToFirstLink;
    ptrToFirstLink = firstLink->duplicate();
  }
}

cmlogSlist::~cmlogSlist (void)
{
  // empty all elements from the list
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogSlist Class Object\n");
#endif
  deleteAllValues();
}

void
cmlogSlist::add (cmlogSlistItem val)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  // add a new value to the front of a linked list
  ptrToFirstLink = new cmlogSlistLink(val, ptrToFirstLink);
  assert(ptrToFirstLink != 0);
}

void
cmlogSlist::addToEnd (cmlogSlistItem val)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif
  if (ptrToFirstLink == 0) {
    ptrToFirstLink = new cmlogSlistLink(val, ptrToFirstLink); 
    return;
  }

  // loop through until last element
  cmlogSlistLink *p = 0;
  for (p = ptrToFirstLink; p->ptrToNextLink; p = p->ptrToNextLink)
    ;

  cmlogSlistLink* q = new cmlogSlistLink (val, 0);
  p->ptrToNextLink = q;
}

void
cmlogSlist::add_i (cmlogSlistItem val)
{
  // add a new value to the front of a linked list
  ptrToFirstLink = new cmlogSlistLink(val, ptrToFirstLink);
  assert(ptrToFirstLink != 0);
}

int
cmlogSlist::remove (cmlogSlistItem val)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  // remove an element from the list
  // loop to test each element
  cmlogSlistLink *q = ptrToFirstLink;
  for (cmlogSlistLink * p = ptrToFirstLink; p; p = p->ptrToNextLink){
    if (val == p->value){
      if (q == p){
	// remove first element
	ptrToFirstLink = p->ptrToNextLink;
	delete p;
      }
      else{
        q->ptrToNextLink = p->ptrToNextLink;
	delete p;
      }
      return 1;
    }
    q = p;
  }
  // not found
  return 0; 
}

void
cmlogSlist::deleteAllValues (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  // clear all items from the list
  cmlogSlistLink * next;

  for (cmlogSlistLink * p = ptrToFirstLink; p != 0; p = next){
    // delete the element pointed to by p
    next = p->ptrToNextLink;
    p->ptrToNextLink = 0;
    delete p;
  }
  
  // mark that the list contains no elements
  ptrToFirstLink = 0;
}

void
cmlogSlist::deleteAllValues_i (void)
{
  // clear all items from the list
  cmlogSlistLink * next;

  for (cmlogSlistLink * p = ptrToFirstLink; p != 0; p = next){
    // delete the element pointed to by p
    next = p->ptrToNextLink;
    p->ptrToNextLink = 0;
    delete p;
  }
  
  // mark that the list contains no elements
  ptrToFirstLink = 0;
}

int
cmlogSlist::count (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  // count how many items are there 
  cmlogSlistLink * next;
  int      num = 0;

  for (cmlogSlistLink *p = ptrToFirstLink; p != 0; p = next){
    next = p->ptrToNextLink;
    num++;
  }
  return num;
}

cmlogSlist*
cmlogSlist::duplicate (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  cmlogSlist * newlist = new cmlogSlist;
  assert(newlist != 0);

  // copy list
  if (ptrToFirstLink)
    newlist->ptrToFirstLink = ptrToFirstLink->duplicate();
  
  // return the new list
  return newlist;
}

cmlogSlistItem
cmlogSlist::firstElement (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  // return first value in list
  assert(ptrToFirstLink != 0);
  return ptrToFirstLink->value;
}

cmlogSlistItem
cmlogSlist::firstElement_i (void)
{
  // return first value in list
  assert(ptrToFirstLink != 0);
  return ptrToFirstLink->value;
}

cmlogSlistItem
cmlogSlist::lastElement (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  assert(ptrToFirstLink != 0);
  // loop through until last element
  cmlogSlistLink *p = 0;
  for (p = ptrToFirstLink; p->ptrToNextLink; p = p->ptrToNextLink)
    ;
  return p->value;
}

cmlogSlistItem
cmlogSlist::lastElement_i (void)
{
  assert(ptrToFirstLink != 0);
  // loop through until last element
  cmlogSlistLink *p = 0;
  for (p = ptrToFirstLink; p->ptrToNextLink; p = p->ptrToNextLink)
    ;
  return p->value;
}

int
cmlogSlist::includes(cmlogSlistItem v)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  // loop to test each element
  for (cmlogSlistLink * p = ptrToFirstLink; p; p = p->ptrToNextLink)
    if (v == p->value)
      return 1;
  
  // not found
  return 0;
}

int
cmlogSlist::isEmpty (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  // test to see if the list is empty
  // list is empty if the pointer to the first link is null
  return ptrToFirstLink == 0;
}

int
cmlogSlist::isEmpty_i (void) const
{
  // test to see if the list is empty
  // list is empty if the pointer to the first link is null
  return ptrToFirstLink == 0;
}

void
cmlogSlist::removeFirst (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  // make sure there is a first element
  assert(ptrToFirstLink != 0);
  
  // save pointer to the removed node
  cmlogSlistLink * p = ptrToFirstLink;

  // reassign the ptrToFirstLink node
  ptrToFirstLink = p->ptrToNextLink;

  // recover memory used by the first element
  delete p;
}

void
cmlogSlist::removeFirst_i (void)
{
  // make sure there is a first element
  assert(ptrToFirstLink != 0);
  
  // save pointer to the removed node
  cmlogSlistLink * p = ptrToFirstLink;

  // reassign the ptrToFirstLink node
  ptrToFirstLink = p->ptrToNextLink;

  // recover memory used by the first element
  delete p;
}

//======================================================================
//	class slink implementation
//           No need to add any mutex protection since all member
//           functions are called with lock held
//======================================================================
cmlogSlistLink* 
cmlogSlistLink::insert(cmlogSlistItem val)
{
  // insert a new link after current node
  ptrToNextLink = new cmlogSlistLink(val, ptrToNextLink);

  // check that allocation was successful
  assert(ptrToNextLink != 0);
  return ptrToNextLink;
}

cmlogSlistLink::cmlogSlistLink (cmlogSlistItem val, cmlogSlistLink * nxt)
:value(val), ptrToNextLink(nxt)
{
  // create and initialize a new link field
}

cmlogSlistLink* 
cmlogSlistLink::duplicate (void)
{
  cmlogSlistLink * newlink;

  // if there is a next field. copy remainder of list
  if (ptrToNextLink != 0)
    newlink = new cmlogSlistLink(value, ptrToNextLink->duplicate());
  else
    newlink = new cmlogSlistLink (value, 0);

  // check that allocation was successful
  assert(newlink != 0);
  return newlink;
}

cmlogSlistLink*
cmlogSlistLink::next (void)
{
  return ptrToNextLink;
}

cmlogSlistItem
cmlogSlistLink::data (void)
{
  return value;
}

//======================================================================
//	class listIterator implementation
//======================================================================
cmlogSlistIterator::cmlogSlistIterator(cmlogSlist & aList)
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
:theList(aList), lock_ ()
#else
:theList(aList)
#endif
{
  // create and initialize a new list
  init_i();
}

int
cmlogSlistIterator::init_i()
{
  // set the iterator to the first element in the list
  previousLink = 0;
  currentLink = theList.ptrToFirstLink;
  return currentLink != 0;
}

int
cmlogSlistIterator::init()
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  // set the iterator to the first element in the list
  previousLink = 0;
  currentLink = theList.ptrToFirstLink;
  return currentLink != 0;
}

cmlogSlistItem
cmlogSlistIterator::operator () (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif
  // return value of current element
  // check to see if there is a current element
  assert(currentLink != 0);

  // return value associated with current element
  return currentLink->value;
}

int cmlogSlistIterator::operator ! (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif
  // test for termination of the iterator
  // if current link references a removed value,
  // update current to point to next position
  if (currentLink == 0)
    if (previousLink != 0)
      currentLink = previousLink->ptrToNextLink;

  // now see if currentLink is valid
  return currentLink != 0;
}

int cmlogSlistIterator::forward_i (void)
{
  // move current pointer to nect element
  // special processing if current link is deleted
  if (currentLink == 0){
    if (previousLink == 0)
      currentLink = theList.ptrToFirstLink;
    else
      currentLink = previousLink->ptrToNextLink;
  }
  else{
    // advance pointer
    previousLink = currentLink;
    currentLink = currentLink->ptrToNextLink;
  }

  // return true if we have a valid current element
  return currentLink != 0;
}


int cmlogSlistIterator::operator ++()
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  // move current pointer to nect element
  // special processing if current link is deleted
  if (currentLink == 0){
    if (previousLink == 0)
      currentLink = theList.ptrToFirstLink;
    else
      currentLink = previousLink->ptrToNextLink;
  }
  else{
    // advance pointer
    previousLink = currentLink;
    currentLink = currentLink->ptrToNextLink;
  }

  // return true if we have a valid current element
  return currentLink != 0;
}

void cmlogSlistIterator::operator = (cmlogSlistItem val)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  // modify value of current element
  assert(currentLink != 0);

  // modify value of the current link
  currentLink->value = val;
}

void cmlogSlistIterator::removeCurrent_i (void)
{
  // remove the current element from a list
  // make sure there is a current element
  assert(currentLink != 0);

  // case 1, removing first element
  if (previousLink == 0)
    theList.ptrToFirstLink = currentLink->ptrToNextLink;
  
  // case 2, not removing the first element
  else
    previousLink->ptrToNextLink = currentLink->ptrToNextLink;
  
  // delete current node
  delete currentLink;

  // and set current pointer to null
  currentLink = 0;
}

void cmlogSlistIterator::removeCurrent (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(lock_);
#endif

  // remove the current element from a list
  // make sure there is a current element
  assert(currentLink != 0);

  // case 1, removing first element
  if (previousLink == 0)
    theList.ptrToFirstLink = currentLink->ptrToNextLink;
  
  // case 2, not removing the first element
  else
    previousLink->ptrToNextLink = currentLink->ptrToNextLink;
  
  // delete current node
  delete currentLink;

  // and set current pointer to null
  currentLink = 0;
}

void cmlogSlistIterator::addBefore(cmlogSlistItem val)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  // a a new element to list before current value
  // case 1, not at start
  if (previousLink)
    previousLink = previousLink->insert(val);

  // case 2, at start of list
  else{
    theList.cmlogSlist::add(val);
    previousLink = theList.ptrToFirstLink;
    currentLink = previousLink->ptrToNextLink;
  }
}

void cmlogSlistIterator::addAfter(cmlogSlistItem val)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(lock_);
#endif

  // a a new element to list after current value
  // case 1, not at start
  if (currentLink != 0)
    currentLink->insert(val);

  // case 2, at end of list
  else if (previousLink != 0)
    currentLink = previousLink->insert(val);
  
  // case 3, start of list
  else
    theList.cmlogSlist::add(val);
}

int cmlogSlistIterator::searchSame(cmlogSlistItem &val)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(lock_);
#endif

  // search the list to find out whether we have this element
  // if we do, move cursor to this element return 1
  // if we don't return 0
  init_i();

  if (currentLink == 0){
  // empty link 
    return 0;
  }
  while (currentLink != 0){
    // advance pointer
    if (currentLink->value == val)
      break;
    previousLink = currentLink;
    currentLink = currentLink->ptrToNextLink;
  }

  // return true if we have a valid current element
  return currentLink != 0;
}

//========================================================================
//          Implementation of cmlogSlistCursor
//                 Implements of cursor without changing list
//========================================================================
cmlogSlistCursor::cmlogSlistCursor(const cmlogSlist & aList)
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
:theList(aList), lock_ ()
#else
:theList(aList)
#endif
{
  // create and initialize a new list
  init_i();
}

int
cmlogSlistCursor::init_i()
{
  // set the iterator to the first element in the list
  previousLink = 0;
  currentLink = theList.ptrToFirstLink;
  return currentLink != 0;
}

int
cmlogSlistCursor::init()
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(lock_);
#endif

  // set the iterator to the first element in the list
  previousLink = 0;
  currentLink = theList.ptrToFirstLink;
  return currentLink != 0;
}

cmlogSlistItem
cmlogSlistCursor::operator () (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(lock_);
#endif

  // return value of current element
  // check to see if there is a current element
  assert(currentLink != 0);

  // return value associated with current element
  return currentLink->value;
}

int cmlogSlistCursor::operator ! (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(lock_);
#endif

  // test for termination of the iterator
  // if current link references a removed value,
  // update current to point to next position
  if (currentLink == 0)
    if (previousLink != 0)
      currentLink = previousLink->ptrToNextLink;
  
  // now see if currentLink is valid
  return currentLink != 0;
}

int cmlogSlistCursor::operator ++ (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(lock_);
#endif

  // move current pointer to nect element
  // special processing if current link is deleted
  if (currentLink == 0){
    if (previousLink == 0)
      currentLink = theList.ptrToFirstLink;
    else
      currentLink = previousLink->ptrToNextLink;
  }
  else{
    // advance pointer
    previousLink = currentLink;
    currentLink = currentLink->ptrToNextLink;
  }

  // return true if we have a valid current element
  return currentLink != 0;
}

int cmlogSlistCursor::searchSame (cmlogSlistItem &val)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(lock_);
#endif
  // search the list to find out whether we have this element
  // if we do, move cursor to this element return 1
  // if we don't return 0
  init_i();

  if (currentLink == 0){
    // empty link 
    return 0;
  }
  while (currentLink != 0){
    // advance pointer
    if (currentLink->value == val)
      break;
    previousLink = currentLink;
    currentLink = currentLink->ptrToNextLink;
  }

  // return true if we have a valid current element
  return currentLink != 0;
}

void cmlogSlistCursor::operator = (cmlogSlistItem )
{
  // empty
}


//========================================================================
//          Implementation of doubleEndedList
//========================================================================
cmlogDoubleEndedSlist::cmlogDoubleEndedSlist()
:cmlogSlist()
{
  ptrToLastLink = 0;
}

cmlogDoubleEndedSlist::cmlogDoubleEndedSlist(const cmlogDoubleEndedSlist &v)
:cmlogSlist(v)
{
  ptrToLastLink = v.ptrToLastLink;
}

void
cmlogDoubleEndedSlist::add(cmlogSlistItem val)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(lock_);
#endif

  // add method needs to check one special case when the first element
  // is added to the list
  if (isEmpty_i()) {
    // call parent method
    cmlogSlist::add_i (val);
    ptrToLastLink = ptrToFirstLink;
  }
  else{
    // always add to the head
    cmlogSlist::add_i(val);
  }
}

void 
cmlogDoubleEndedSlist::addToEnd (cmlogSlistItem val)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(lock_);
#endif

  //add new value T to the end of the list
  // if there is an end, add to it
  if(ptrToLastLink != 0)
    ptrToLastLink = ptrToLastLink->insert(val);
  else {
    assert (isEmpty_i ());
    // call parent method
    cmlogSlist::add_i (val);
    ptrToLastLink = ptrToFirstLink;
  }
}

void
cmlogDoubleEndedSlist::deleteAllValues()
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(lock_);
#endif
  //delete all values from the list
  cmlogSlist::deleteAllValues_i();
  ptrToLastLink = 0;
}

void
cmlogDoubleEndedSlist::removeFirst()
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(lock_);
#endif
  //remove the first element from the list
  // invoke the parent method
  cmlogSlist::removeFirst_i();
  
  // only do something if we removed last element
  if(isEmpty_i())
    ptrToLastLink = 0;
}

