//-----------------------------------------------------------------------------
// Copyright (c) 1991,1992 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
// Email: coda@cebaf.gov  Tel: (804) 249-7101  Fax: (804) 249-7363
//-----------------------------------------------------------------------------
// 
// Description:
//	 Simple FIFO with multiple readers 
//	
// Author:  Jie Chen
//       CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogDataQueue.cc,v $
//   Revision 1.3  2001/07/25 14:26:36  chen
//   64 BIT Initial Port
//
//   Revision 1.2  2000/02/07 16:10:16  chen
//   add more versbose message
//
//   Revision 1.1.1.1  1999/09/07 15:29:10  chen
//   CMLOG version 2.0
//
// Revision 1.4  1999/01/11  20:42:25  chen
// fix a bug for delete dhs under hpux-10.20
//
// Revision 1.3  1998/10/07  17:02:44  chen
// Fix a compiler (egcs) warning about cmlog_cdevMessage& = (cmlog_cdevMessage)
//
// Revision 1.2  1997/12/11  18:17:26  chen
// minor changes
//
// Revision 1.1  1997/08/01  15:27:19  bickley
// Added cmlog to application development system.
//
//
#include "cmlogDataQueue.h"
#include <cmlog_cdevTagMap.h>
#include <cmlogUtil.h>
#include <cmlogDatabase.h>
#include <cmlogCxtDataCache.h>


#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
cmlog_dq_waiter::cmlog_dq_waiter (cpMutex& lock, cp_thread_t t)
:next_ (0), wakeup_ (0), cv_(lock), thr_ (t)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlog_dq_waiter class object\n");
#endif
}

cmlog_dq_waiter::~cmlog_dq_waiter (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlog_dq_waiter class object\n");
#endif
}
#endif

//=======================================================================
//    implementation of cmlogDataQueue class    
//=======================================================================
cmlogDataQueue::cmlogDataQueue (void)
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
:cmlogDoubleEndedSlist (), 
 waiters_head_ (0), waiters_tail_ (0), num_waiters_ (0)
#else
:cmlogDoubleEndedSlist ()
#endif
{
  // empty
}

cmlogQItem 
cmlogDataQueue::front (void)
{
  // firstElement has lock
  cmlogQItem result = firstElement();
  return result;
}

void
cmlogDataQueue::enqueue (cmlogQItem val)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(lock_);
#endif

  //add new value T to the end of the list
  // if there is an end, add to it
  if(ptrToLastLink != 0)
    ptrToLastLink = ptrToLastLink->insert(val);
  else {
    assert (isEmpty_i ());
    // call parent method
    cmlogSlist::add_i (val);
    ptrToLastLink = ptrToFirstLink;
  }
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  if (num_waiters_ > 0) {
    cmlog_dq_waiter* sleeper = waiters_head_;
    assert (sleeper != 0);

    // set wakeup flag and remove it from the list
    sleeper->wakeup_ = 1;
    // signal this sleeper
    sleeper->cv_.signal ();
  }
#endif
}

cmlogQItem 
cmlogDataQueue::dequeue (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(lock_);
  
  if (num_waiters_ > 0 || isEmpty_i ()) { 
    // there are someone waiting already or
    // there are nothing in the queue, so we wait

    // create a new waiter and insert to the tail of the list
#if 0
    printf ("create new waiter for thread %d\n", cpThread::self ());
#endif
    cmlog_dq_waiter waiter (lock_, cpThread::self ());
    num_waiters_ ++;
    if (waiters_head_ == 0) {
      waiters_head_ = &waiter;
      waiters_tail_ = &waiter;
    }
    else {
      waiters_tail_->next_ = &waiter;
      waiters_tail_ = &waiter;
    }
    
    // we just wait here until someone wake us up
    while (waiter.wakeup_ == 0) 
      waiter.cv_.wait ();

#if 0
    printf ("waiter is awaken for thread %d\n", cpThread::self ());
#endif
    assert (waiter.wakeup_);

    // remove first element from the waiters' queue
    waiters_head_ = waiters_head_->next_;
    if (waiters_head_ == 0)
      waiters_tail_ = 0;

    num_waiters_ --;
  }
#else
  if (isEmpty_i ())
    return 0;
#endif
  // get first element
  cmlogQItem result = firstElement_i ();
  //remove the first element from the list
  // invoke the parent method
  cmlogSlist::removeFirst_i();
  
  // only do something if we removed last element
  if(isEmpty_i())
    ptrToLastLink = 0; 
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  else if (num_waiters_ > 0) { // there are something to read
    cmlog_dq_waiter* sleeper = waiters_head_;
    assert (sleeper);
    sleeper->wakeup_ = 1;
    sleeper->cv_.signal ();
  }
#endif
  return result;
}

unsigned int
cmlogDataQueue::numElements (void)
{
  return count();
}
#endif

#if !defined (CMLOG_USE_THREAD) || !defined (_REENTRANT)

#include <cmlogLogicSup.h>

extern char* cmlog_shmemptr;
//=======================================================================
//    implementation of cmlogDataQueueHandler class    
//=======================================================================
cmlogQueueHandler::cmlogQueueHandler (void)
:dlock_ (), disEmpty_ (), disFull_ (), dshmem_ ()
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogQueueHandler Class Object\n");
#endif
  // FIFO for data item to database
  if (dlock_.open (CMLOG_SEM_ID) != 0) 
    fprintf (stderr, "Cannot create semaphore\n");

  if (disEmpty_.open (CMLOG_SEM_ID2) != 0) 
    fprintf (stderr, "Cannot create semaphore 2\n");

  if (disFull_.open (CMLOG_SEM_ID3,
		     ACE_SV_Semaphore_Simple::CREATE, 
		     1) != 0) 
    fprintf (stderr, "Cannot create semaphore 3\n");

  if (dshmem_.open (CMLOG_SHMEM_ID, 
		    (CMLOG_DSHMEM_SIZE)*1024,
		    ACE_SV_Shared_Memory::CREATE) != 0)
    fprintf (stderr, "Cannot create shared memory segment\n");

  if (dshmem_.attach () != 0) 
    fprintf (stderr, "Qhandler::Cannot attach to shared memory\n");

#if 0
  printf ("shm starts at %u 0x%x\n",
	  dshmem_.get_segment_ptr (),  dshmem_.get_segment_ptr ());
#endif
}

void
cmlogQueueHandler::init (void)
{
  // no lock necessary since no other processes will access here
  // create database FIFO header for database operation
  char* sptr = (char *)dshmem_.get_segment_ptr ();
  cmlogShmHeader dheader (sptr, CMLOG_DSHMEM_SIZE*1024);

  dheader.head_ = dheader.tail_ = dheader.beginningOfQueue ();

#if 0
  printf ("database head and tail start at %u and %u\n", 
	  dheader.head_, dheader.tail_);
#endif
  
  dheader.putHeaderInfo ();

  // detach shared memory since this memory is already attached by
  // the same process or parent process
  // dshmem_.detach ();
}

char*
cmlogQueueHandler::shmemPtr (void)
{
  return (char *)(dshmem_.get_segment_ptr ());
}

cmlogQueueHandler::~cmlogQueueHandler (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogQueueHandler Class Object\n");
#endif
  dlock_.remove ();
  disEmpty_.remove ();
  disFull_.remove ();
  dshmem_.remove ();
}


//=======================================================================
//  Implementation of cmlogShmHeader
//=======================================================================
cmlogShmHeader::cmlogShmHeader (char* shm, unsigned int size)
:numElements_ (0), full_ (0), head_ (0), tail_ (0), 
 shmem_ (shm), size_ (size)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogShmHeader Class Object\n");
#endif
  dbaseName_[0] = '\0';
  cxtdbaseName_[0] = '\0';
}

cmlogShmHeader::~cmlogShmHeader (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogShmHeader Class Object\n");
#endif
  // empty
}

void
cmlogShmHeader::getHeaderInfo (void)
{
  memcpy (dbaseName_, shmem_, sizeof (cmlogShmHeader));
#if 0
  printf ("process %d: queue head %u: tail %u \n", getpid (),
	  head_, tail_);
#endif
}

void
cmlogShmHeader::putHeaderInfo (void)
{
  memcpy (shmem_, dbaseName_, sizeof (cmlogShmHeader));
}

char* 
cmlogShmHeader::beginningOfQueue (void)
{
  return (shmem_ + sizeof (cmlogShmHeader));
}

char*
cmlogShmHeader::endOfQueue (void)
{
  return (shmem_ + size_);
}

int
cmlogShmHeader::overflow (cmlogMsg& msg)
{
  if (head_ <= tail_) {
#if 0
    printf ("Overflow: head %u tail %u size %u endq %u and startq %u\n",
	    head_, tail_, msg.size(), endOfQueue(), beginningOfQueue());
#endif
    if (tail_ + msg.size () < endOfQueue ())
      return 0;
    else if (beginningOfQueue () + msg.size () < head_) {
      
      //===========================================================
      // put empty cmlogMsg at the address of tail_ so that dequeue
      // operation can detect the empty cmlogMsg
      //===========================================================
      
      cmlogMsg empty;
      empty.type (CMLOG_EMPTY);
      cmlogMsg::msgToBuffer (empty, tail_);
#if 0
      printf ("Put empty message at %u-------------------------\n", tail_);
#endif

      // update new tail address information
      tail_ = beginningOfQueue ();
      putHeaderInfo ();
      return 0;
    }
  }
  else {
    if (tail_ + msg.size () < head_)
      return 0;
  }
  // turn on full flag
  full_ = 1;
  putHeaderInfo ();
  return 1;
}

void
cmlogShmHeader::roundUpAddress (char* &address)
{
  if (address >= endOfQueue ()) {
#if 0
    printf ("header is %u reach endof queue %u\n", address, endOfQueue ());
#endif
    address = beginningOfQueue ();
  }
}

//=======================================================================
//  Implementation of cmlogDataQueue
//        This class accesses the shared memory and dequeue/enqueue
//        cmlogMsg into the shared memory
//
//        The first 128 bytes of shared memory for data base name
//        The next sizeof (long) will be number of elements in the queue
//        The next  2*sizeof (char *) will be pointer value to
//        head and tail of this queue
//=======================================================================
cmlogDataQueue::cmlogDataQueue (int type)
:lock_ (), isEmpty_ (), isFull_ (), shmem_ (), type_ (type), shmheader_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogDataQueue Class Object\n");
#endif
  if (type_ == cmlogDataQueue::TODBASE) {
    if (lock_.open (CMLOG_SEM_ID, ACE_SV_Semaphore_Simple::OPEN) != 0)
      fprintf (stderr, "Cannot get semaphore lock\n");
    if (isEmpty_.open (CMLOG_SEM_ID2, ACE_SV_Semaphore_Simple::OPEN) != 0)
      fprintf (stderr, "Cannot get semaphore token\n");
    if (isFull_.open (CMLOG_SEM_ID3, ACE_SV_Semaphore_Simple::OPEN) != 0)
      fprintf (stderr, "Cannot get semaphore token\n");
    shmsize_ = CMLOG_DSHMEM_SIZE*1024;
  }
  if (shmem_.open (CMLOG_SHMEM_ID, 
		   (CMLOG_DSHMEM_SIZE)*1024) != 0)
    fprintf (stderr, "Cannot open shared memory\n");
}

cmlogDataQueue::~cmlogDataQueue (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogDataQueue Class Object\n");
#endif  
  lock_.close ();
  isEmpty_.close ();
  isFull_.close ();
  shmem_.detach ();
}

int
cmlogDataQueue::attach (char* mem)
{
  if (!mem) {
    if (shmem_.attach () != 0) {
      fprintf (stderr, "Cannot attach to shared memory\n");
      return -1;
    }

    if (type_ == cmlogDataQueue::TODBASE) {
      shmheader_ = (char *)shmem_.get_segment_ptr ();
      printf ("Proc %d: Command shmheader 0x%x\n", getpid (), shmheader_);

    }
    else {
      shmheader_ = (char *)shmem_.get_segment_ptr () + CMLOG_DSHMEM_SIZE*1024;
      printf ("Proc %d: Command shmheader 0x%x\n", getpid (), shmheader_);
    }

#if 0
    printf ("proc %d shm header at %u and size %d\n", getpid (), shmheader_,
	    shmsize_);
#endif
  }
  else {
    shmheader_ = mem;
#ifdef _CMLOG_DEBUG
    printf ("proc %d shm header at %u and size %d\n", getpid (), shmheader_,
	    shmsize_);
#endif
  }    

  return 0;
}

char*
cmlogDataQueue::shmemHeader (void)
{
  return shmheader_;
}

unsigned int
cmlogDataQueue::shmemSize (void)
{
  return shmsize_;
}

unsigned int
cmlogDataQueue::numElements_i (void)
{
  cmlogShmHeader header (shmheader_, shmsize_);
  header.getHeaderInfo ();
  return header.numElements_;
}

cmlogMsg *
cmlogDataQueue::front (void)
{
  // firstElement has lock
  lock_.acquire ();

  cmlogShmHeader header (shmheader_, shmsize_);
 
  header.getHeaderInfo ();
  if (header.numElements_ == 0) {
    lock_.release ();
    return 0;
  }
    
  cmlogMsg* result = new cmlogMsg ();

  cmlogMsg::bufferToMsg (header.head_, *result);
  lock_.release ();

  return result;
}

unsigned int
cmlogDataQueue::numElements (void)
{
  lock_.acquire ();
  unsigned int num = numElements_i ();
  lock_.release ();
  return num;
}

void
cmlogDataQueue::enqueue (cmlogMsg& msg)
{
  // empty
}

void
cmlogDataQueue::dequeue (cmlogMsg& msg)
{
  // empty
}



//====================================================================
//   Implementation of cmlogDataQueueWriter
//====================================================================
cmlogDataQueueWriter::cmlogDataQueueWriter (int type)
:cmlogDataQueue (type)
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogDataQueueWriter Class Object\n");
#endif
  isEmpty_.acquire ();
}

void
cmlogDataQueueWriter::writeDbaseName (char* name)
{
  lock_.acquire ();

  cmlogShmHeader header (shmheader_, shmsize_);
  header.getHeaderInfo ();

  strncpy (header.dbaseName_, name, CMLOG_DBASENAME_SIZE);

  header.putHeaderInfo ();

  lock_.release ();
}

void
cmlogDataQueueWriter::writeCxtdbaseName (char* name)
{
  lock_.acquire ();

  cmlogShmHeader header (shmheader_, shmsize_);
  header.getHeaderInfo ();

  strncpy (header.cxtdbaseName_, name, CMLOG_DBASENAME_SIZE);

  header.putHeaderInfo ();

  lock_.release ();
}

void
cmlogDataQueueWriter::enqueue (cmlogMsg& msg)
{
  lock_.acquire ();

  cmlogShmHeader header (shmheader_, shmsize_);
  header.getHeaderInfo ();
  

  while (header.overflow (msg)) {
#if 0
    printf ("Queue is full and number elements is %d\n", header.numElements_);
#endif 
    lock_.release ();
    isFull_.acquire ();  // wait for reader to process

    // reacquire the lock
    lock_.acquire ();
    header.getHeaderInfo ();
  }
#if 0
  printf ("Proc %d: Number of elements is %d\n", getpid (),
	  header.numElements_);
#endif

  cmlogMsg::msgToBuffer (msg, header.tail_);
  header.tail_ += msg.size ();
  header.numElements_ += 1;
  header.putHeaderInfo ();

#if 0
  printf ("Proc %d: after enqueue head is at 0x%x tail is at 0x%x\n", 
	  getpid (), header.head_, header.tail_);

#endif

  // check whether this queue was empty, wake up children
  if (header.numElements_ == 1) 
    isEmpty_.release ();

  lock_.release ();

}

//=========================================================================
//     Implementation of data queue object to database
//=========================================================================
cmlogDatabase* cmlogDataQueueToDbase::dbase = 0;

// all signals to catch
int cmlogDataQueueToDbase::signals[] = 
{SIGINT, SIGQUIT, SIGILL, SIGFPE,
 SIGSYS, SIGPIPE, SIGBUS, SIGSEGV,
 SIGTERM, SIGTSTP, SIGCONT, SIGIO, SIGPOLL};

int cmlogDataQueueToDbase::numSignals = 13;

int
cmlogDataQueueToDbase::registerSignalHandlers (void)
{
  struct sigaction act, oact;

  act.sa_handler = &(cmlogDataQueueToDbase::signalFunc);
  sigemptyset (&act.sa_mask);
  act.sa_flags = 0;
#ifdef SA_RESTART
  act.sa_flags |= SA_RESTART;
#endif

  for (int i = 0; i < cmlogDataQueueToDbase::numSignals; i++) 
    if (sigaction (cmlogDataQueueToDbase::signals[i], &act, &oact) < 0)
      return -1;
  
  return 0;
}

void
cmlogDataQueueToDbase::signalFunc (int signo)
{
#ifdef solaris
  char signame[SIG2STR_MAX];
  if (sig2str (signo, signame) == -1)
    sprintf (signame, "unknown");

  fprintf (stderr, "Interrupted by %s signal\n", signame);

  if (signo == SIGSEGV || signo == SIGBUS) {
    if (cmlogDataQueueToDbase::dbase) {
      cmlogDataQueueToDbase::dbase->flush ();
      cmlogDataQueueToDbase::dbase->close ();
      delete cmlogDataQueueToDbase::dbase;
    }
    abort ();
  }
  else if (signo != SIGPIPE && signo != SIGTSTP &&
	   signo != SIGCONT ) {
    if (cmlogDataQueueToDbase::dbase) {
      cmlogDataQueueToDbase::dbase->flush ();
      cmlogDataQueueToDbase::dbase->close ();
      delete cmlogDataQueueToDbase::dbase;
      ::exit (1);
    }
  }
#else
  fprintf (stderr, "Interrupted by %d signal\n", signo);

  if (signo == SIGSEGV || signo == SIGBUS) {
    fprintf (stderr, "Fatal: Caught signal %d, Quit.......\n", signo);
    if (cmlogDataQueueToDbase::dbase) {
      cmlogDataQueueToDbase::dbase->flush ();
      cmlogDataQueueToDbase::dbase->close ();
      delete cmlogDataQueueToDbase::dbase;
    }
    abort ();
  }
  else if (signo != SIGPIPE && signo != SIGTSTP &&
	   signo != SIGCONT ) {
    if (cmlogDataQueueToDbase::dbase) {
      cmlogDataQueueToDbase::dbase->flush ();
      cmlogDataQueueToDbase::dbase->close ();
      delete cmlogDataQueueToDbase::dbase;
    }
    ::exit (1);
  }
#endif
}


cmlogDataQueueToDbase::cmlogDataQueueToDbase (void)
:cmlogDataQueue (cmlogDataQueue::TODBASE)
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogDataQueueToDbase Class Object\n");
#endif
  isFull_.acquire ();
}

void
cmlogDataQueueToDbase::readDbaseName (char* buffer, int size)
{
  lock_.acquire ();

  cmlogShmHeader header (shmheader_, shmsize_);
  header.getHeaderInfo ();

  strncpy (buffer, header.dbaseName_, size);

  lock_.release ();  
}

void
cmlogDataQueueToDbase::readCxtdbaseName (char* buffer, int size)
{
  lock_.acquire ();

  cmlogShmHeader header (shmheader_, shmsize_);
  header.getHeaderInfo ();

  strncpy (buffer, header.cxtdbaseName_, size);

  lock_.release ();  
}

void
cmlogDataQueueToDbase::dequeue (cmlogMsg& msg)
{
  lock_.acquire ();
  int   size = 0;
  int   fullflag = 0;

  cmlogShmHeader header (shmheader_, shmsize_);
  header.getHeaderInfo ();

  while (header.numElements_ == 0) {
    lock_.release ();
#if 0
    printf ("%d :wait on empty semaphore\n", getpid ());
#endif
    isEmpty_.acquire ();  // wait for writer to add something
#if 0
    printf ("%d : is empty released\n", getpid ());
#endif
    // reacquire the lock
    lock_.acquire ();
    header.getHeaderInfo ();
  }

  while (1) {
    if ((size = cmlogMsg::bufferToMsg (header.head_, msg))
	== CMLOG_EMPTY_SIZE) {
#if 0
      printf ("Empty junk size ++++++++++\n");
#endif
      header.head_ = header.beginningOfQueue ();
    }
    else {
      header.head_ += msg.size ();
#if 0
      printf ("Proc %d :Remove a msg with size %d type %d head at 0x%x tail at 0x%x\n", 
	      getpid (), msg.size (), msg.type (), 
	      header.head_, header.tail_);
#endif
      break;
    }
  }

  header.numElements_ -= 1;
  if (header.full_) {
    fullflag = header.full_;
    header.full_ = 0;
  }
  header.putHeaderInfo ();
  
  // check if full flag is on
  if (fullflag) 
    isFull_.release ();

  lock_.release ();
}

void*
cmlogDataQueueToDbase::readerProcess (void * arg)
{
  int fileopened = 0;

  // get cmlogUtil tag function
  cmlogUtil::setTags ();

  // create a reader for the data queue
  cmlogDataQueueToDbase reader;
  // attach to shared memory
  reader.attach (cmlog_shmemptr);

  cmlogMsg msg;
  double   key;

  // create a database handler
  if (cmlogDataQueueToDbase::dbase == 0)
    cmlogDataQueueToDbase::dbase = new cmlogDatabase ();

  // get database name
  char dbaseName[CMLOG_DBASENAME_SIZE];
  reader.readDbaseName (dbaseName, CMLOG_DBASENAME_SIZE);

  if (::strlen (dbaseName) > 0) {
    if (dbase->open  (dbaseName, O_CREAT | O_RDWR) != CDEV_SUCCESS)
      fprintf (stderr, "Fatal: cmlog server cannot open file %s\n", 
	       dbaseName);
    else
      fileopened = 1;
  }

  // register signal handler
  cmlogDataQueueToDbase::registerSignalHandlers ();

  for (;;) {
    reader.dequeue (msg);

    cmlog_cdevMessage& data = msg;

    switch (msg.type ()) {
    case CMLOG_ADD_DATA:
      {
	cdevData* d = data.getData ();
	if (d->get (cmlogUtil::CMLOG_KEY_TAG, &key) != CDEV_SUCCESS) {
	  key = cmlogUtil::currentTime ();
	  d->insert (cmlogUtil::CMLOG_KEY_TAG, key);
	}

	if (fileopened)
	  dbase->put (*d);
      }
      break;
    case CMLOG_CONN_INFO:
    case CMLOG_RECONN_INFO:
      {
	cdevData* d = data.getData ();
	if (d) {
	  if (d->get (cmlogUtil::CMLOG_KEY_TAG, &key) != CDEV_SUCCESS) {
	    key = cmlogUtil::currentTime ();
	    d->insert (cmlogUtil::CMLOG_KEY_TAG, key);
	  }

	  char dbaseName[CMLOG_DBASENAME_SIZE];
	  reader.readCxtdbaseName (dbaseName, CMLOG_DBASENAME_SIZE);

	  cmlogDatabase cxtdbase;

	  if (cxtdbase.open  (dbaseName, O_CREAT | O_RDWR) != CDEV_SUCCESS)
	    fprintf (stderr, "Fatal: cmlog server cannot open file %s\n", 
		     dbaseName);
	  else {
	    cxtdbase.put (*d); 
	    cxtdbase.flush ();
	    cxtdbase.close ();
	  }
	}
      }
      break;
    case CMLOG_ADD_CXT:
      {
	cdevData* d = data.getData ();
	if (d->get (cmlogUtil::CMLOG_KEY_TAG, &key) != CDEV_SUCCESS) {
	  key = cmlogUtil::currentTime ();
	  d->insert (cmlogUtil::CMLOG_KEY_TAG, key);
	}

	char dbaseName[CMLOG_DBASENAME_SIZE];
	reader.readCxtdbaseName (dbaseName, CMLOG_DBASENAME_SIZE);

	cmlogDatabase cxtdbase;

	if (cxtdbase.open  (dbaseName, O_CREAT | O_RDWR) != CDEV_SUCCESS)
	  fprintf (stderr, "Fatal: cmlog server cannot open file %s\n", 
		   dbaseName);
	else {
	  cxtdbase.put (*d); 
	  cxtdbase.flush ();
	  cxtdbase.close ();
	}
      }
      break;
    case CMLOG_DBASE:
      // database file name is changed
      if (fileopened) {
	dbase->flush ();
	dbase->close ();
	fileopened = 0;
      }
      // get new database file name      
      reader.readDbaseName (dbaseName, CMLOG_DBASENAME_SIZE);
      if (::strlen (dbaseName) > 0) {
	if (dbase->open  (dbaseName, O_CREAT | O_RDWR) != CDEV_SUCCESS) {
	  fprintf (stderr, "Fatal: cmlog server cannot open file %s\n", 
		   dbaseName);
	  fileopened = 0;
	}
	else
	  fileopened = 1;
      }

      break;
    case CMLOG_FIND_DATA:
      // this data item actually signals the database to flush everything
      // to disk
      if (fileopened) 
	dbase->flush ();
      break;
    case CMLOG_TAGMAP:
      break;
    case CMLOG_EXIT:
      if (fileopened) {
	dbase->flush ();
	dbase->close ();
	fileopened = 0;
      }
      printf ("Shutdown writer proc %d by admin task\n", getpid ());
      ::exit (0);
      break;
    default:
      printf ("Data queue msg type %d\n",msg.type ());
      // data.asciiDump ();
      break;
    }

  }
  return 0;
}

#endif
