//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      CMLOG Unix Client Daemon
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogClientD.cc,v $
//   Revision 1.5  2002/04/12 15:33:12  chen
//   Fix a compile error for Tonado 2.0.2
//
//   Revision 1.4  2000/06/20 19:32:08  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.3  2000/02/07 16:15:15  chen
//   add protocol message to stdout
//
//   Revision 1.2  1999/10/29 17:39:34  chen
//   Support -i
//
//   Revision 1.1.1.1  1999/09/07 15:29:09  chen
//   CMLOG version 2.0
//
// Revision 1.3  1998/10/07  19:01:33  chen
// fix a bug for linux
//
// Revision 1.2  1998/03/20  21:08:19  chen
// change named pipe path to /tmp/cmlog
//
// Revision 1.1  1997/08/01  15:26:52  bickley
// Added cmlog to application development system.
//
//
//
#include <cmlogProtocol.h>
#include <cmlogMsg.h>
#include <cmlogServerLocater.h>
#include <cmlogUxToServer.h>
#include <cmlogUpIO.h>
#include <cmlogServerHB.h>
#include <cmlogUtil.h>
#include <cmlogClntDSigHandler.h>

sig_atomic_t    finished = 0;

#ifdef __vxworks
#include "cpThread.h"
#include "cpThreadManager.h"

// Interger flag that notifies whether the client daemon is running or not
int        cmlog_clientd_running = 0;

// Mutex lock and conditional variables associated with this flag
cpMutex            *cmlog_rlock = 0;
cpConditionMutex   *cmlog_rcond = 0;


// to prevent the name to be mangled by the C++ compiler
extern "C" int cmlogClientD (int, int);

/* forward declaration */
static int realCmlogClientD (cpThreadManager* manager);

static void
cmlogClientDCleanup (void* reactor)
{
  ((ACE_Reactor *)reactor)->close ();
}

int 
cmlogClientD (int pri, int stackSize)
{
  cpThreadManager manager;

  // make this taks safe from deletion
  cpThread::disablecancel ();

  // stack size should be roughly the same as the cmlog_packet size inside
  // cmlogMsg.cc
  if (pri == 0) {
    printf ("cmlogClient Daemon uses default priority 190\n");
    pri = 190;
  }
  else 
    printf ("cmlogClient daemon runs at priority %d\n", pri);

  if (stackSize == 0) {
    printf ("cmlogClient daemon uses default stack size 20000\n");
    stackSize = 20000;
  }
  else
    printf ("cmlogClient daemon has stack size %d\n", stackSize);

  manager.spawn ((CP_THREAD_FUNC)realCmlogClientD, (void *)&manager,
		 (long)VX_FP_TASK, (cp_thread_t)0, (unsigned int)pri, (void *)0, (unsigned int)stackSize);
  manager.wait ();

  printf ("cmlog client daemon finished\n");

  return 0;
}
#endif

#ifndef __vxworks
static int
cmlogCreateParentDirs (const char* path, mode_t mode)
{
  const char *p;
  const char *q;
  char  tmp[1024];
  int   i, j, numdirs;
  char** dirs;
  int   status = 0;
  
  // check weather the path is the full path name 
  if (( p = strchr (path, '/')) == 0)
    return -1;

  q = p + 1;
  if (!q || !*q) // syntax error: no path after '/'
    return -1;

  i = 0;
  while ((p = strchr (q, '/')) != 0) {
    q = p + 1;
    i++;
    if (!q || !*q) break;
  }
  
  numdirs = i;
  dirs = new char*[numdirs];
  if (!dirs) {
    fprintf (stderr, "cmlogClientD: malloc error, quit\n");
    exit (1);
  }

  // now find all parent directories
  p = strchr (path, '/');
  q = p + 1;

  i = 0;
  while ((p = strchr (q, '/')) != 0) {
    j = p - path;
    strncpy (tmp, path, j);
    tmp [j] = '\0';

    dirs[i] = new char[strlen (tmp) + 1];
    strcpy (dirs[i], tmp);
    i++;

    q = p + 1;
    if (!q || !*q)break;
  }

  // clear umask
  umask (0);
  for (i = 0; i < numdirs; i++) {
#ifndef __linux
    if (mkdir (dirs[i], mode) != 0 && errno != EEXIST) {
#else
    if (mkdir (dirs[i], mode) != 0 && errno != EEXIST && errno != EACCES) {
#endif
      status = -1;
      break;
    }
  }

  // free memory
  for (i = 0; i < numdirs; i++) 
    delete []dirs[i];
  delete []dirs;

  return status;
}
  

int 
main (int argc, char** argv)
#else
static int
realCmlogClientD (cpThreadManager* manager)
#endif
{
#ifdef __vxworks
  cpThreadControl control(manager);

  cpMutex            rlock;
  cpConditionMutex   rcond (rlock);

  cmlog_rlock = &rlock;
  cmlog_rcond = &rcond;
#else
  int cmlog_interactive = 0;
  if (argc >= 2 && strcmp (argv[1], "-i") == 0) 
    cmlog_interactive = 1;

  // create all parent directory for the named pipe
  if (cmlogCreateParentDirs (CMLOG_CLNT_PIPE, 0777) != 0) {
    fprintf (stderr, "Cannot create named pipe %s\n", CMLOG_CLNT_PIPE);
    exit (1);
  }
#endif
  printf ("cmlogClientD with %s\n", CMLOG_PROTOCOL_VERSION_STRING);
  ACE_Reactor reactor;
  cmlogPacket buffer;

  // setup all tags
  cmlogUtil::setTags ();

  // create tcp connection to server
  cmlogUxToServer toServer (reactor, buffer);
  // create server heart beat monitor
  cmlogServerHB   serverHB (reactor);

  // open heart beat listenning port
  if (serverHB.open () == -1) {
    fprintf (stderr, "Cannot attach to local port for heart beat monitor\n");
    ::exit (1);
  }
  if (reactor.register_handler(&serverHB, ACE_Event_Handler::READ_MASK) == -1){
    fprintf (stderr, "Cannot register heart beat handler\n");
    ::exit (1);
  }
  // put pointer of heart beat handler to server handler
  toServer.serverHeartBeatMonitor (&serverHB);

  // put tcp connection pointer inside heart beat
  serverHB.tcpToServer (&toServer);


  // create a fifo to handle user logging processes
  cmlogUpIO fifo (reactor, toServer);
  if (reactor.register_handler (&fifo,
				ACE_Event_Handler::READ_MASK) == -1) {
    fprintf (stderr, "Cannot register fifo handler\n");
    ::exit (1);
  }

  // finally open toServer local port and connect to server
  if (toServer.connToServer () == -1) 
    fprintf (stderr, "Cannot connect to a cmlog server\n");
  else {
    if (reactor.register_handler (&toServer,
				  ACE_Event_Handler::READ_MASK) == -1)
      fprintf (stderr, "Cannot register tcp to server handler\n");
  }

  // register signal handler
#ifdef __vxworks
  if (cmlogClntDSigHandler::registerSignalHandlers () != 0) {
    fprintf (stderr, "Cannot register signal handler\n");
  }
#else
  if (!cmlog_interactive && 
      cmlogClntDSigHandler::registerSignalHandlers () != 0) {
    fprintf (stderr, "Cannot register signal handler\n");
  }
#endif

  fprintf (stderr, "Starting up cmlog client daemon....\n");

#ifdef __vxworks
  // create dumb cdevData to instantiate static variables
  cdevData dumb;
  dumb.insert ("value", 0);

  // register cleanup handler for vxworks
  cpThread::add_cleanup_handler (cmlogClientDCleanup, (void *)&reactor);

  // wake up the client that are waiting for the client daemon
  cmlog_rlock->acquire ();
  cmlog_clientd_running = 1;
  cmlog_rcond->broadcast ();
  cmlog_rlock->release ();

#endif

  while (!finished)
    reactor.handle_events ();

  fprintf (stderr, "Shutting down cmlog client\n");

  return 0;
}

