#include "ace/Timer_Queue.h"

ACE_ALLOC_HOOK_DEFINE(ACE_Timer_Node)

void
ACE_Timer_Node::dump (void) const
{
  ACE_TRACE ("ACE_Timer_Node::dump");
}

ACE_Timer_Node::ACE_Timer_Node (ACE_Event_Handler *h, 
				const void *a, 
				const ACE_Time_Value &t, 
				const ACE_Time_Value &i, 
				ACE_Timer_Node *n,
				int timer_id)
  : handler_ (h), 
    arg_ (a), 
    timer_value_ (t), 
    interval_ (i), 
    next_ (n),
    timer_id_ (timer_id)
{
  ACE_TRACE ("ACE_Timer_Node::ACE_Timer_Node");
}

ACE_ALLOC_HOOK_DEFINE(ACE_Timer_Queue)

void
ACE_Timer_Queue::dump (void) const
{
  ACE_TRACE ("ACE_Timer_Queue::dump");
}

// Create an empty queue.

ACE_Timer_Queue::ACE_Timer_Queue (void)
  : head_ (0),
    timer_id_ (0)
{
  ACE_TRACE ("ACE_Timer_Queue::ACE_Timer_Queue");
}

// Checks if queue is empty. 

int 
ACE_Timer_Queue::is_empty (void) const
{
  ACE_TRACE ("ACE_Timer_Queue::is_empty");
  return this->head_ == 0;
}

// Returns earliest time in a non-empty queue. 

const ACE_Time_Value &
ACE_Timer_Queue::earliest_time (void) const
{
  ACE_TRACE ("ACE_Timer_Queue::earliest_time");
  return this->head_->timer_value_;
}

// Remove all remaining items in the queue. 

ACE_Timer_Queue::~ACE_Timer_Queue (void)
{
  ACE_TRACE ("ACE_Timer_Queue::~ACE_Timer_Queue");
  ACE_MT (ACE_Recursive_Mutex_Guard m (this->lock_));
  
  ACE_Timer_Node *curr = this->head_;
  
  while (curr != 0)
    {
      ACE_Timer_Node *next = curr->next_;
      delete curr;
      curr = next;
    }
}

// Reschedule a periodic timer.  This function must be called with the
// mutex lock held.

void 
ACE_Timer_Queue::reschedule (ACE_Timer_Node *expired)
{
  ACE_TRACE ("ACE_Timer_Queue::reschedule");
  if (this->is_empty () || expired->timer_value_ < this->earliest_time ())
    {
      expired->next_ = this->head_;
      this->head_    = expired;
    }
  else
    {
      ACE_Timer_Node *prev = this->head_;
      ACE_Timer_Node *after  = this->head_->next_;

      // Locate the proper position in the queue. 

      while (after != 0 
	     && expired->timer_value_ > after->timer_value_)
	{
	  prev = after;
	  after  = after->next_;
	}

      expired->next_ = after;
      prev->next_  = expired;
    }
}

// Insert a new handler that expires at time future_time; if interval
// is > 0, the handler will be reinvoked periodically.

int
ACE_Timer_Queue::schedule (ACE_Event_Handler *handler,
			   const void *arg,
			   const ACE_Time_Value	&future_time, 
			   const ACE_Time_Value	&interval)
{
  ACE_TRACE ("ACE_Timer_Queue::schedule");
  ACE_MT (ACE_Recursive_Mutex_Guard m (this->lock_));

  // Increment the sequence number (it will wrap around).
  this->timer_id_++;

  if (this->is_empty () || future_time < this->earliest_time ())
    {
      // Place at the beginning of the list.
      this->head_ = new ACE_Timer_Node (handler, 
					arg, 
					future_time,
					interval, 
					this->head_,
					this->timer_id_);
      return this->head_ ? this->timer_id_ : -1;
    }
  else // Place in the middle of the list somewhere.
    {
      ACE_Timer_Node *prev = this->head_;
      ACE_Timer_Node *after = this->head_->next_;

      while (after != 0 && future_time > after->timer_value_)
	{
	  prev = after;
	  after = after->next_;
	}

      prev->next_ = new ACE_Timer_Node (handler, 
					arg, 
					future_time,
					interval, 
					after,
					this->timer_id_);
      return prev->next_ ? this->timer_id_ : -1;
    }
}

// Locate and remove the single <ACE_Event_Handler> with a value of
// <timer_id> from the timer queue.

int
ACE_Timer_Queue::cancel (int timer_id)
{
  ACE_TRACE ("ACE_Timer_Queue::cancel");
  ACE_MT (ACE_Recursive_Mutex_Guard m (this->lock_));

  ACE_Timer_Node *prev = 0;
  ACE_Timer_Node *curr = 0;

  // Try to locate the ACE_Timer_Node that matches the timer_id.

  for (curr = this->head_; 
       curr != 0 && curr->timer_id_ != timer_id;
       curr = curr->next_)
    prev = curr;

  if (curr != 0)
    {
      if (prev == 0)
	this->head_ = curr->next_;
      else
	prev->next_ = curr->next_;
      delete curr;
      return 0;
    }
  else
    return -1;
}

// Locate and remove all values of <handler> from the timer queue.

int
ACE_Timer_Queue::cancel (ACE_Event_Handler *handler)
{
  ACE_TRACE ("ACE_Timer_Queue::cancel");
  ACE_MT (ACE_Recursive_Mutex_Guard m (this->lock_));

  ACE_Timer_Node *prev = 0;
  ACE_Timer_Node *curr = this->head_;

  while (curr != 0)
    {
      if (curr->handler_ == handler)
	{
	  if (prev == 0)
	    {
	      this->head_ = curr->next_;
	      delete curr;
	      curr = this->head_;
	    }
	  else
	    {
	      prev->next_ = curr->next_;
	      delete curr;
	      curr = prev->next_;
	    }
	}
      else
	{
	  prev = curr;
	  curr = curr->next_;
	}
    }

  return 0;
}

// Run the <handle_timeout> method for all Timers whose values are <= <cur_time>.

int
ACE_Timer_Queue::expire (const ACE_Time_Value &cur_time)
{
  ACE_TRACE ("ACE_Timer_Queue::expire");
  ACE_MT (ACE_Recursive_Mutex_Guard m (this->lock_));

  for (;;)
    {
      if (this->is_empty () || this->earliest_time () > cur_time)
	break;

      ACE_Timer_Node *expired = this->head_;
      ACE_Event_Handler *handler = (ACE_Event_Handler *) expired->handler_;
      const void *arg = expired->arg_;
      int reclaim  = 1;
      int res;

      this->head_ = this->head_->next_;

      if (expired->interval_ > ACE_Time_Value::zero)
	{
	  do
	    expired->timer_value_ += expired->interval_;
	  while (expired->timer_value_ <= cur_time);

	  this->reschedule (expired);
	  reclaim = 0;
	}

      res = handler->handle_timeout (cur_time, arg);

      if (res == -1)
	this->cancel (handler);

      if (reclaim)
	delete expired;
    }
  return 0;
}


