#if !defined (ACE_SYNCH_C)
#define ACE_SYNCH_C

#include "ace/Assert.h"
#include "ace/Thread.h"
#include "ace/Synch.h"
#include "ace/Log_Msg.h"
#include "ace/Time_Value.h"

#if !defined (__INLINE__)
#include "ace/Synch.i"
#endif /* __INLINE__ */

ACE_ALLOC_HOOK_DEFINE(ACE_Null_Mutex)
ACE_ALLOC_HOOK_DEFINE(ACE_File_Lock)

ACE_File_Lock::ACE_File_Lock (ACE_HANDLE h)
{
  ACE_TRACE ("ACE_File_Lock::ACE_File_Lock");
  if (ACE_OS::flock_init (&this->lock_) == -1)
    ACE_ERROR ((LM_ERROR, "%p\n", "ACE_File_Lock::ACE_File_Lock"));
  this->set_handle (h);
}

ACE_File_Lock::ACE_File_Lock (const char *name, 
			      int flags, 
			      mode_t perms)
{
  ACE_TRACE ("ACE_File_Lock::ACE_File_Lock");

  if (ACE_OS::flock_init (&this->lock_, flags, name, perms) == -1)
    ACE_ERROR ((LM_ERROR, "%p\n", "ACE_File_Lock::ACE_File_Lock"));
}

ACE_File_Lock::~ACE_File_Lock (void)
{
  ACE_TRACE ("ACE_File_Lock::~ACE_File_Lock");
  if (this->remove () == -1)
    ACE_ERROR ((LM_ERROR, "%p\n", "ACE_File_Lock::~ACE_File_Lock"));
}

#if defined (ACE_HAS_THREADS)

ACE_ALLOC_HOOK_DEFINE(ACE_Recursive_Mutex)
ACE_ALLOC_HOOK_DEFINE(ACE_Mutex_Guard)
ACE_ALLOC_HOOK_DEFINE(ACE_Recursive_Mutex_Guard)
ACE_ALLOC_HOOK_DEFINE(ACE_Mutex)
ACE_ALLOC_HOOK_DEFINE(ACE_Semaphore)

void
ACE_Semaphore::dump (void) const
{
  ACE_TRACE ("ACE_Semaphore::dump");
}

ACE_Semaphore::ACE_Semaphore (u_int count, 
			      int type, 
			      const char *name, 
			      void *arg)
{
  ACE_TRACE ("ACE_Semaphore::ACE_Semaphore");
  if (ACE_OS::sema_init (&this->semaphore_, count, type, name, arg) != 0)
    ACE_ERROR ((LM_ERROR, "%p\n", "ACE_Semaphore::~ACE_Semaphore"));
}

ACE_Semaphore::~ACE_Semaphore (void)
{
  ACE_TRACE ("ACE_Semaphore::~ACE_Semaphore");
  this->remove ();
}

void
ACE_Mutex::dump (void) const
{
  ACE_TRACE ("ACE_Mutex::dump");
}

ACE_Mutex::ACE_Mutex (int type, const char *name, void *arg)
{
  ACE_TRACE ("ACE_Mutex::ACE_Mutex");

  if (ACE_OS::mutex_init (&this->lock_, type, name, arg) != 0)
    ACE_ERROR ((LM_ERROR, "%p\n", "ACE_Mutex::ACE_Mutex"));
}

ACE_Mutex::~ACE_Mutex (void)
{
  ACE_TRACE ("ACE_Mutex::~ACE_Mutex");
  if (this->remove () != 0)
    ACE_ERROR ((LM_ERROR, "%p\n", "ACE_Mutex::~ACE_Mutex"));
}

void
ACE_Recursive_Mutex::dump (void) const
{
  ACE_TRACE ("ACE_Recursive_Mutex::dump");
}

ACE_Recursive_Mutex::ACE_Recursive_Mutex (int type, 
					 const char *name,
					 void *arg)
  : nesting_level_ (0), 
    thr_id_ (ACE_OS::NULL_thread), 
    lock_ (type, name, arg),
    mutex_ (type, name, arg)
{
  ACE_TRACE ("ACE_Recursive_Mutex::ACE_Recursive_Mutex");
}

ACE_Recursive_Mutex::~ACE_Recursive_Mutex (void)
{
  ACE_TRACE ("ACE_Recursive_Mutex::~ACE_Recursive_Mutex");
}

int
ACE_Recursive_Mutex::acquire (void)
{
  ACE_TRACE ("ACE_Recursive_Mutex::acquire");
  this->mutex_.acquire ();
  thread_t t_id = ACE_Thread::self ();

  if (ACE_OS::thr_equal (t_id, this->thr_id_))
    {
      ++this->nesting_level_;
      this->mutex_.release ();
      return 0;
    }
  else
    {
      this->mutex_.release ();
      int result = this->lock_.acquire ();

      if (result == 0)
	{
	  this->set_thread_id (t_id);
	  this->set_nesting_level (0);
	}
      return result;
    }
}
 
int
ACE_Recursive_Mutex::release (void)
{
  ACE_TRACE ("ACE_Recursive_Mutex::release");
  this->mutex_.acquire ();

  // ACE_ASSERT (this->get_thread_id () == ACE_Thread::self ());
  if (this->nesting_level_ > 0)
    {
      --this->nesting_level_;
      this->mutex_.release ();
      return 0;
    }
  else
    {
      this->set_thread_id (ACE_OS::NULL_thread);
      this->mutex_.release ();
      return this->lock_.release ();
    }
}

int
ACE_Recursive_Mutex::tryacquire (void)
{
  ACE_TRACE ("ACE_Recursive_Mutex::tryacquire");
  this->mutex_.acquire ();
  thread_t t_id = ACE_Thread::self ();

  if (ACE_OS::thr_equal (t_id, this->thr_id_))
    {
      ++this->nesting_level_;
      this->mutex_.release ();
      return 0;
    }
  else
    {
      this->mutex_.release ();
      int result = this->lock_.tryacquire ();

      if (result == 0)
	{
	  this->set_thread_id (t_id);
	  this->set_nesting_level (0);
	}
      return result;
    }
}

ACE_ALLOC_HOOK_DEFINE(ACE_Condition_Mutex)

void
ACE_Condition_Mutex::dump (void) const
{
  ACE_TRACE ("ACE_Condition_Mutex::dump");
}

ACE_Condition_Mutex::ACE_Condition_Mutex (ACE_Mutex &m,
					  int type, 
					  const char *name,
					  void *arg)
  : mutex_ (m)
{
  ACE_TRACE ("ACE_Condition_Mutex::ACE_Condition_Mutex");
  if (ACE_OS::cond_init (&this->cond_, type, name, arg) != 0)
    ACE_ERROR ((LM_ERROR, "%p\n", "ACE_Condition_Mutex::ACE_Condition_Mutex"));
}

ACE_Condition_Mutex::~ACE_Condition_Mutex (void)
{
  ACE_TRACE ("ACE_Condition_Mutex::~ACE_Condition_Mutex");
  if (this->remove () == -1)
    ACE_ERROR ((LM_ERROR, "%p\n", "ACE_Condition_Mutex::~ACE_Condition_Mutex"));
}

// Peform an "alertable" timed wait.  If the argument ABSTIME == 0
// then we do a regular cond_wait(), else we do a timed wait for up to
// ABSTIME using the Solaris cond_timedwait() function.

int
ACE_Condition_Mutex::wait (ACE_Time_Value *abstime)
{
  ACE_TRACE ("ACE_Condition_Mutex::wait");
  if (abstime == 0)
    return ACE_OS::cond_wait (&this->cond_, &this->mutex_.lock_);
  else
    return ACE_OS::cond_timedwait (&this->cond_, &this->mutex_.lock_, abstime);
}

int
ACE_Condition_Mutex::signal (void)
{
  ACE_TRACE ("ACE_Condition_Mutex::signal");
  return ACE_OS::cond_signal (&this->cond_);
}

int
ACE_Condition_Mutex::broadcast (void)
{
  ACE_TRACE ("ACE_Condition_Mutex::broadcast");
  return ACE_OS::cond_broadcast (&this->cond_);
}

ACE_ALLOC_HOOK_DEFINE(ACE_Sub_Barrier)

void
ACE_Sub_Barrier::dump (void) const
{
  ACE_TRACE ("ACE_Sub_Barrier::dump");
}

ACE_Sub_Barrier::ACE_Sub_Barrier (u_int count,
				  ACE_Mutex &lock,
				  int type, 
				  const char *name,
				  void *arg)
  : barrier_finished_ (lock, type, name, arg),
    running_threads_ (count)
{
  ACE_TRACE ("ACE_Sub_Barrier::ACE_Sub_Barrier");
}

ACE_ALLOC_HOOK_DEFINE(ACE_Barrier)
ACE_ALLOC_HOOK_DEFINE(ACE_Thread_Barrier)
ACE_ALLOC_HOOK_DEFINE(ACE_Process_Barrier)

void
ACE_Barrier::dump (void) const
{
  ACE_TRACE ("ACE_Barrier::dump");
}

ACE_Barrier::ACE_Barrier (u_int count, 
			  int type, 
			  const char *name,
			  void *arg)
  : lock_ (type, name, arg),
    current_generation_ (0),
    count_ (count),
    sub_barrier_1_ (count, lock_, type, name, arg),
    sub_barrier_2_ (count, lock_, type, name, arg)
{
  ACE_TRACE ("ACE_Barrier::ACE_Barrier");
  this->sub_barrier_[0] = &this->sub_barrier_1_;
  this->sub_barrier_[1] = &this->sub_barrier_2_;
}

ACE_Process_Barrier::ACE_Process_Barrier (u_int count, const char *name)
  : ACE_Barrier (count, USYNC_PROCESS, name)
{
  ACE_TRACE ("ACE_Process_Barrier::ACE_Process_Barrier");
}

void
ACE_Process_Barrier::dump (void) const
{
  ACE_TRACE ("ACE_Process_Barrier::dump");
}

ACE_Thread_Barrier::ACE_Thread_Barrier (u_int count, const char *name)
  : ACE_Barrier (count, USYNC_THREAD, name)
{
  ACE_TRACE ("ACE_Thread_Barrier::ACE_Thread_Barrier");
}

void
ACE_Thread_Barrier::dump (void) const
{
  ACE_TRACE ("ACE_Thread_Barrier::dump");
}

int
ACE_Barrier::wait (void)
{
  ACE_TRACE ("ACE_Barrier::wait");
  ACE_Mutex_Guard mon (this->lock_);

  ACE_Sub_Barrier *sbp = this->sub_barrier_[this->current_generation_];

  // Check for shutdown...
  if (sbp == 0)
    return -1;

  if (sbp->running_threads_ == 1)
    {
      // We're the last running thread, so swap generations and tell
      // all the threads waiting on the barrier to continue on their
      // way.

      sbp->running_threads_ = this->count_;
      // Swap generations.
      this->current_generation_ = 1 - this->current_generation_;
      sbp->barrier_finished_.broadcast ();
    }
  else
    {
      --sbp->running_threads_;

      // Block until all the other threads wait().
      while (sbp->running_threads_ != this->count_)
	sbp->barrier_finished_.wait ();
    }

  return 0;
}

ACE_ALLOC_HOOK_DEFINE(ACE_Process_Mutex)

void
ACE_Process_Mutex::dump (void) const
{
  ACE_TRACE ("ACE_Process_Mutex::dump");
}

ACE_Process_Mutex::ACE_Process_Mutex (const char *name, void *arg)
  : ACE_Mutex (USYNC_PROCESS, name, arg)
{
  ACE_TRACE ("ACE_Process_Mutex::ACE_Process_Mutex");
}

ACE_ALLOC_HOOK_DEFINE(ACE_Thread_Mutex)

void
ACE_Thread_Mutex::dump (void) const
{
  ACE_TRACE ("ACE_Thread_Mutex::dump");
}

ACE_Thread_Mutex::ACE_Thread_Mutex (const char *name, void *arg)
  : ACE_Mutex (USYNC_THREAD, name, arg)
{
  ACE_TRACE ("ACE_Thread_Mutex::ACE_Thread_Mutex");
}

ACE_ALLOC_HOOK_DEFINE(ACE_RW_Thread_Mutex)
ACE_ALLOC_HOOK_DEFINE(ACE_RW_Process_Mutex)

ACE_RW_Thread_Mutex::ACE_RW_Thread_Mutex (const char *name, 
					  void *arg)
  : ACE_RW_Mutex (USYNC_THREAD, name, arg)
{
  ACE_TRACE ("ACE_RW_Thread_Mutex::ACE_RW_Thread_Mutex");
}

ACE_RW_Process_Mutex::ACE_RW_Process_Mutex (const char *name, 
					   void *arg)
  : ACE_RW_Mutex (USYNC_PROCESS, name, arg)
{
  ACE_TRACE ("ACE_RW_Process_Mutex::ACE_RW_Process_Mutex");
}

void
ACE_RW_Thread_Mutex::dump (void) const
{
  ACE_TRACE ("ACE_RW_Thread_Mutex::dump");
}

void
ACE_RW_Process_Mutex::dump (void) const
{
  ACE_TRACE ("ACE_RW_Process_Mutex::dump");
}

ACE_RW_Mutex::ACE_RW_Mutex (int type, const char *name, void *arg)
{
  ACE_TRACE ("ACE_RW_Mutex::ACE_RW_Mutex");
  if (ACE_OS::rwlock_init (&this->lock_, type, name, arg) != 0)
    ACE_ERROR ((LM_ERROR, "%p\n", "ACE_RW_Mutex::~ACE_RW_Mutex"));
}

ACE_RW_Mutex::~ACE_RW_Mutex (void)
{
  ACE_TRACE ("ACE_RW_Mutex::~ACE_RW_Mutex");
  if (this->remove () == -1)
    ACE_ERROR ((LM_ERROR, "%p\n", "ACE_RW_Mutex::~ACE_RW_Mutex"));
}

#endif /* ACE_HAS_THREADS */
#endif /* ACE_SYNCH_C */

