/* -*- C++ -*- */

// ============================================================================
//
// = LIBRARY
//    ace
// 
// = FILENAME
//    Event_Handler.h
//
// = AUTHOR
//    Doug Schmidt 
// 
// ============================================================================

#if !defined (ACE_EVENT_HANDLER_H)
#define ACE_EVENT_HANDLER_H

#include "ace/ACE.h"

typedef u_long ACE_Reactor_Mask;

class ACE_Event_Handler 
  // = TITLE
  //     Provides an abstract interface for handling various types of
  //     I/O-based, timer-based, and signal-based events.
  //
  // = DESCRIPTION
  //       Derived classes read/write input/output on an I/O
  //       descriptor, handle an exception raised on an I/O
  //       descriptor, handle a timer's expiration, or handle a
  //       signal.
{
public:
  enum 
  {
    LO_PRIORITY = 0,
    HI_PRIORITY = 10,
    NULL_MASK   = 0,
#if defined (ACE_USE_POLL)
    READ_MASK   = POLLIN,
    WRITE_MASK  = POLLOUT,
    EXCEPT_MASK = POLLPRI,
#else /* USE SELECT */
    READ_MASK   = 0x1,
    WRITE_MASK  = 0x4,
    EXCEPT_MASK = 0x2,
#endif /* ACE_USE_POLL */
    RWE_MASK    = READ_MASK | WRITE_MASK | EXCEPT_MASK,
    DONT_CALL   = 0x100
  };

  virtual ~ACE_Event_Handler (void);
  // Destructor is virtual to enable proper cleanup.

  // = The following methods must be supplied by subclasses in order
  // to specialize the behavior of an Event_Handler.
  virtual ACE_HANDLE  get_handle (void) const;
  // Get the I/O handle.
  virtual void set_handle (ACE_HANDLE);
  // Set the I/O handle.

  // = Priority runs from MIN_PRIORITY (which is the "lowest
  // priority") to MAX_PRIORITY (which is the "highest priority").
  virtual int get_priority (void) const;
  // Get the priority of the Event_Handler.
  virtual void set_priority (int priority);
  // Set the priority of the Event_Handler.

  virtual int handle_input (ACE_HANDLE fd = ACE_INVALID_HANDLE);
  // Called when input events occur (e.g., connection or data).

  virtual int handle_output (ACE_HANDLE fd = ACE_INVALID_HANDLE);
  // Called when output events are possible (e.g., flow control
  // abates).

  virtual int handle_exception (ACE_HANDLE fd = ACE_INVALID_HANDLE);
  // Called when execption events occur (e.g., SIGURG).

  virtual int handle_timeout (const ACE_Time_Value &tv, 
			      const void *arg = 0);
  // Called when timer expires.

  virtual int handle_close (ACE_HANDLE fd, 
			    ACE_Reactor_Mask close_mask);
  // Called when object is removed from the ACE_Reactor

  virtual int handle_signal (int signum
#if defined (ACE_HAS_SIGINFO_T)
, siginfo_t * = 0, ucontext_t * = 0
#endif /* ACE_HAS_SIGINFO_T */
);
  // Called when signal is triggered by OS.

protected:
  ACE_Event_Handler (void);
  // Force ACE_Event_Handler to be an abstract base class.

  int priority_;
  // Priority of this Event_Handler.
};

#if defined (__INLINE__)
#include "ace/Event_Handler.i"
#endif /* __INLINE__ */

#endif /* ACE_EVENT_HANDLER_H */
