#include "ace/Assert.h"
#include "ace/IPC_SAP.h"
#include "ace/Time_Value.h"
#include "ace/Handle_Set.h"
#include "ace/ACE.h"

#if !defined (__INLINE__)
#include "ace/ACE.i"
#endif /* __INLINE__ */

// Miscellaneous static methods used throughout ACE.

ssize_t
ACE::send_n (ACE_HANDLE handle, const void *buf, size_t len)
{
  ACE_TRACE ("ACE::send_n");
  size_t bytes_written;
  int	 n;

  for (bytes_written = 0; bytes_written < len; bytes_written += n)
#if defined (ACE_WIN32)
    if ((n = ACE_OS::send (handle, (const char *) buf + bytes_written, 
			   len - bytes_written)) == -1)

#else
    if ((n = ACE_OS::write (handle, (const char *) buf + bytes_written, 
			    len - bytes_written)) == -1)
     
#endif /* ACE_WIN32 */
      return -1;

  return bytes_written;
}

ssize_t
ACE::send_n (ACE_HANDLE handle, const void *buf, size_t len, int flags)
{
  ACE_TRACE ("ACE::send_n");
  size_t bytes_written;
  int	 n;

  for (bytes_written = 0; bytes_written < len; bytes_written += n)
    if ((n = ACE_OS::send (handle, (const char *) buf + bytes_written, 
			   len - bytes_written, flags)) == -1)
      return -1;

  return bytes_written;
}

ssize_t
ACE::recv_n (ACE_HANDLE handle, void *buf, size_t len)
{
  ACE_TRACE ("ACE::recv_n");
  size_t bytes_read;
  int	 n;

  for (bytes_read = 0; bytes_read < len; bytes_read += n)
#if defined (ACE_WIN32)
    if ((n = ACE_OS::recv (handle, (char *) buf + bytes_read,
			   len - bytes_read)) == -1)

#else
    if ((n = ACE_OS::read (handle, (char *) buf + bytes_read, 
		     len - bytes_read)) == -1)
#endif /* ACE_WIN32 */
      return -1;
    else if (n == 0)
      break;

  return bytes_read;      
}

ssize_t
ACE::recv_n (ACE_HANDLE handle, void *buf, size_t len, int flags)
{
  ACE_TRACE ("ACE::recv_n");
  size_t bytes_read;
  int	 n;

  for (bytes_read = 0; bytes_read < len; bytes_read += n)
    if ((n = ACE_OS::recv (handle, (char *) buf + bytes_read, 
			   len - bytes_read, flags)) == -1)
      return -1;
    else if (n == 0)
      break;

  return bytes_read;      
}

// Format buffer into printable format.  This is useful for debugging.
// Portions taken from mdump by J.P. Knight (J.P.Knight@lut.ac.uk)
// Modifications by Todd Montgomery.

void 
ACE::format_hexdump (char *buffer, char *obuf, int size)
{
  ACE_TRACE ("ACE::format_hexdump");

  int i,j;
  unsigned char c;
  char textver[16];

  for (i=0;i<(size >> 4);i++) 
    {
      for (j=0;j<16;j++) 
	{
	  c = buffer[(i << 4)+j];
	  ::sprintf(obuf,"%02x ",c);
	  obuf += 3;
	  textver[j] = ((c<0x20)||(c>0x7e))?'.':c;
	}
      textver[j] = 0;
      ::sprintf(obuf,"\t%s\n",textver);
      while (*obuf != '\0') obuf++;
    }

  for (i=0;i<size%16;i++) 
    {
      c = buffer[size-size%16+i];
      ::sprintf(obuf,"%02x ",c);
      obuf += 3;
      textver[i] = ((c<0x20)||(c>0x7e))?'.':c;
    }

  for (i=size%16;i<16;i++) 
    {
      ::sprintf(obuf,"   ");
      obuf += 3;
      textver[i] = ' ';
    }
  textver[i] = 0;
  ::sprintf(obuf,"\t%s\n",textver);
  while (*obuf != '\0') obuf++;
}

// Returns the current timestamp in the form
// "hour:minute:second:microsecond."  The month, day, and year are
// also stored in the beginning of the date_and_time array.  Returns 0
// if unsuccessful, else returns pointer to beginning of the "time"
// portion of <day_and_time>.

char *
ACE::timestamp (char date_and_time[], int date_and_timelen)
{
  ACE_TRACE ("ACE::timestamp");

  if (date_and_timelen < 35)
    {
      errno = EINVAL;
      return 0;
    }

#if defined (WIN32)
  // @@ Jesper, I think Win32 supports all the UNIX versions below.
  // Therefore, we can probably remove this WIN32 ifdef altogether.
  SYSTEMTIME local;
  ::GetLocalTime (&local);

  ACE_OS::sprintf (date_and_time, "%02d.%02d.%02d.%06d",
		   (int) local.wHour,
		   (int) local.wMinute,
		   (int) local.wSecond,
		   (int) local.wMilliseconds * 1000);
#else  // UNIX
  char timebuf[26]; // This magic number is based on the ctime(3c) man page.
  ACE_Time_Value cur_time = ACE_OS::gettimeofday ();
  time_t secs = cur_time.sec ();
  ACE_OS::ctime_r (&secs, timebuf, sizeof timebuf);
  ACE_OS::strncpy (date_and_time, timebuf, date_and_timelen);
  ACE_OS::sprintf (&date_and_time[19], ".%06d", cur_time.usec ());
#endif /* WIN32 */
  date_and_time[26] = '\0';        
  return &date_and_time[11];
}

// This function rounds the request to a multiple of the page size.

size_t
ACE::round_to_pagesize (off_t len)
{
  ACE_TRACE ("ACE::round_to_pagesize");
  return (len + (ACE_PAGE_SIZE - 1)) & ~(ACE_PAGE_SIZE - 1);
}

ACE_HANDLE 
ACE::handle_timed_complete (ACE_HANDLE h,
			    ACE_Time_Value *tv)
{
  ACE_TRACE ("ACE::handle_timed_complete");
  ACE_Handle_Set rd_handles;
  ACE_Handle_Set wr_handles;

  rd_handles.set_bit (h);
  wr_handles.set_bit (h);

  int n = ACE_OS::select (int (h) + 1, 
			  rd_handles,
			  wr_handles, 
			  0, tv);
  if (n <= 0)
    {
      if (n == 0)
	errno = ETIMEDOUT;
      return ACE_INVALID_HANDLE;
    }
  else if (wr_handles.is_set (h))
    return h;
  else // if (rd_handles.is_set (h))
    return ACE_INVALID_HANDLE;
}

ACE_HANDLE
ACE::handle_timed_connect (ACE_Time_Value *timeout,
			   const char name[], int flags, int perms)
{
  ACE_TRACE ("ACE::handle_timed_connect");

  if (timeout != 0)
    {
      // Open the named pipe or file using non-blocking mode...
      ACE_HANDLE handle = ACE_OS::open (name, flags | ACE_NONBLOCK, perms);

      if (handle == ACE_INVALID_HANDLE
	  && (errno == EWOULDBLOCK 
	      && (timeout->sec () > 0 || timeout->usec () > 0)))
	// This expression checks if we were polling.
	errno = ETIMEDOUT;

      return handle;
    }
  else
    return ACE_OS::open (name, flags, perms);
}

// Wait up to <timeout> amount of time to accept a connection.

int 
ACE::handle_timed_accept (ACE_HANDLE listener, 
			  ACE_Time_Value *timeout,
			  int restart)
{
  ACE_TRACE ("ACE::handle_timed_accept");
  // Make sure we don't bomb out on erroneous values.
  if (listener == ACE_INVALID_HANDLE)
    return -1;

  // Use the select() implementation rather than poll().
  ACE_Handle_Set rd_handle;
  rd_handle.set_bit (listener);

  // We need a loop here if <restart> is enabled.

  for (;;)
    {
      switch (ACE_OS::select (int (listener) + 1, 
			      rd_handle, 0, 0, 
			      timeout))
	{
	case -1:
	  if (errno == EINTR && restart)
	    continue;
	  else
	    return -1;
	  /* NOTREACHED */
	case 0:
	  if (timeout != 0 && timeout->sec() == 0 && timeout->usec() == 0)
	    errno = EWOULDBLOCK;
	  else
	    errno = ETIMEDOUT;
	  return -1;
	  /* NOTREACHED */
	case 1:
	  return 0;
	  /* NOTREACHED */
	default:
	  ACE_ASSERT (!"select/poll() returned expected value\n");
	  /* NOTREACHED */
	}
    }
  /* NOTREACHED */
  return 0;
}

// Bind socket to an unused port.

int
ACE::bind_port (ACE_HANDLE handle)
{
  ACE_TRACE ("ACE::bind_port");
  sockaddr_in sin;
  const int   MAX_SHORT   = 65535;
  static int  upper_limit = MAX_SHORT;
  int	      len         = sizeof sin;
  int	      lower_limit = IPPORT_RESERVED;
  int	      round_trip  = upper_limit;

  ACE_OS::memset ((void *) &sin, 0, sizeof sin);
  sin.sin_family = AF_INET;
#if defined (ACE_HAS_SIN_LEN)
  sin.sin_family = sizeof sin;
#endif /* ACE_HAS_SIN_LEN */
  sin.sin_addr.s_addr = INADDR_ANY;

  for (;;)
    {
      sin.sin_port = htons (upper_limit);

      if (ACE_OS::bind (handle, (sockaddr *) &sin, sizeof sin) >= 0)
	return 0;
      else if (errno != EADDRINUSE)
	return -1;
      else
	{
	  upper_limit--;
		  
	  /* Wrap back around when we reach the bottom. */
	  if (upper_limit <= lower_limit)
	    upper_limit = MAX_SHORT;
		  
	  /* See if we have already gone around once! */
	  if (upper_limit == round_trip)
	    {
	      errno = EAGAIN;	
	      return -1;
	    }
	}
    }
}

// There must be a better way to do this...
#if defined (Linux) || defined (AIX)
#ifdef RLIMIT_OFILE
#   define RLIMIT_NOFILE RLIMIT_OFILE
#else
#   define RLIMIT_NOFILE 200
#endif
#endif

// Make the current process a UNIX daemon.  This is based on Stevens
// code from APUE.

int
ACE::daemonize (void)
{
  ACE_TRACE ("ACE::daemonize");
#if !defined (ACE_WIN32)
  pid_t pid;

  if ((pid = ACE_OS::fork ()) == -1)
    return -1;
  else if (pid != 0)
    ACE_OS::exit (0);			/* parent exits */

  /* child continues */
  ACE_OS::setsid (); /* become session leader */

  ACE_OS::chdir ("/");		/* change working directory */

  ACE_OS::umask (0);			/* clear our file mode creation mask */
  return 0;
#else
  ACE_NOTSUP_RETURN (-1);
#endif /* ACE_WIN32 */
}

// Set the number of currently open handles in the process.
//
// If NEW_LIMIT == -1 set the limit to the maximum allowable.
// Otherwise, set it to be the value of NEW_LIMIT.

int 
ACE::set_handle_limit (int new_limit)
{
  ACE_TRACE ("ACE::set_handle_limit");
#if !defined (ACE_WIN32) && !defined (__vxworks)
  struct rlimit rl;

#if defined (RLIMIT_NOFILE)
  if (ACE_OS::getrlimit (RLIMIT_NOFILE, &rl) == -1)
    return -1;
#else
  rl.rlim_cur = NOFILES_MAX;
#endif /* RLIMIT_NOFILE */

  if (new_limit < 0 || new_limit > rl.rlim_max)
    rl.rlim_cur = rl.rlim_max;
  else
    rl.rlim_cur = new_limit;

  return ACE_OS::setrlimit (RLIMIT_NOFILE, &rl);
#else
  ACE_NOTSUP_RETURN (-1);
#endif /* ACE_WIN32 && __vxworks*/
}

// Flags are file status flags to turn on.

int
ACE::set_fl (ACE_HANDLE fd, int flags)
{
  ACE_TRACE ("ACE::set_fl");
#if !defined (ACE_WIN32)
  int val;

  if ((val = ACE_OS::fcntl (fd, F_GETFL, 0)) == -1)
    return -1;

  val |= flags; /* turn on flags */

  if (ACE_OS::fcntl (fd, F_SETFL, val) == -1)
    return -1;
  else
    return 0;
#else
  ACE_NOTSUP_RETURN (-1);
#endif /* ACE_WIN32 */
}

// Flags are the file status flags to turn off.

int
ACE::clr_fl (ACE_HANDLE fd, int flags)
{
  ACE_TRACE ("ACE::clr_fl");
#if !defined (ACE_WIN32)
  int val;

  if ((val = ACE_OS::fcntl (fd, F_GETFL, 0)) == -1)
    return -1;

  val &= ~flags; /* turn flags off */

  if (ACE_OS::fcntl (fd, F_SETFL, val) == -1)
    return -1;
  else
    return 0;
#else
  ACE_NOTSUP_RETURN (-1);
#endif /* ACE_WIN32 */
}
