//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Implementation cmlogBrowser Class
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogBrowser.cc,v $
//   Revision 1.5  2001/07/25 14:25:06  chen
//   64 BIT Initial Port
//
//   Revision 1.4  2001/07/03 17:23:35  chen
//   Fix a bug on setTagMap
//
//   Revision 1.3  2000/12/13 16:43:54  chen
//   send tagmap information upon connect
//
//   Revision 1.2  2000/06/20 18:44:13  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.1.1.1  1999/09/07 15:29:02  chen
//   CMLOG version 2.0
//
// Revision 1.3  1998/10/07  17:02:16  chen
// Fix a compiler (egcs) warning about cmlog_cdevMessage& = (cmlog_cdevMessage)
//
// Revision 1.2  1997/08/12  16:48:07  chen
// add one connect function
//
// Revision 1.1  1997/08/01  15:17:07  bickley
// Added cmlog to application development system.
//
//
//
#include <cmlogUtil.h>
#include <cmlogNetUtil.h>
#include "cmlogBrowser.h"

#define CMLOG_QUERYCBK_TABLE_SIZE 1024
#define CMLOG_MONITORCBK_TABLE_SIZE 997

#define CMLOG_EVID_MAX            1093939

//===========================================================================
//    Implementation of cmlogBrowserTagCallback
//===========================================================================
cmlogBrowserTagCallback::cmlogBrowserTagCallback (cmlogBrowser* browser)
:cdevTagTableCallback (), browser_ (browser)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogBrowserTagCallback Class Object\n");
#endif
  // empty
}

cmlogBrowserTagCallback::~cmlogBrowserTagCallback (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogBrowserTagCallback Class Object\n");
#endif
  // empty
}

void
cmlogBrowserTagCallback::callback (int tag, char* name)
{
  if (browser_->tagmapNotify ())
    browser_->sendTagMapInfo (tag, name);
}

//=========================================================================
//       Implementation of cmlogBrowser
//=========================================================================
cmlogBrowser::cmlogBrowser (void)
  :connected_ (0), lock_ (0), udp_fd_ (-1), tcp_fd_ (-1), 
  serverPort_ (0), serverHost_ (0), tagmapNotify_ (1), qid_ (93), evid_(1024),
  discCbkList_ (), queryCbkTable_ (CMLOG_QUERYCBK_TABLE_SIZE),
  monitorCbkTable_ (CMLOG_MONITORCBK_TABLE_SIZE, cmlogStrHashFunc)
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogBrowser Class Object\n");
#endif
  // setup all tags
  cmlogUtil::setTags ();

  // register tag table callback
  tagCbk_ = new cmlogBrowserTagCallback (this);
  cdevData::addTagCallback (tagCbk_);
}

cmlogBrowser::~cmlogBrowser (void)
{
#ifdef _TRACE_OBJECTS
  printf ("    Delete cmlogBrowser Class Object\n");
#endif
  
  // cleanup all callbacks
  cleanupCallbacks ();

  // delete server hostname
  if (serverHost_)
    delete []serverHost_;

  if (connected_) 
    disconnect ();

  // delete tag table callback object
  delete tagCbk_;
}

int
cmlogBrowser::connect (int num)
{
  if (!connected_) {
    unsigned short brd_port = cmlogNetUtil::findServerUdpPort ();

    if (cmlogNetUtil::udpOpenLocal (udp_fd_) != 0)
      return CMLOG_ERROR;
    
    if (cmlogNetUtil::udpTalkToServer (udp_fd_, brd_port, num, 
				       CMLOG_BRSERLKSVC,
				       serverPort_, serverHost_) != 0)
      return CMLOG_ERROR;

    // connect to TCP port
    // if code reach here, udp_fd should be closed, reset it to -1
    udp_fd_ = -1;

    if (cmlogNetUtil::tcpConnect (serverPort_, serverHost_, tcp_fd_, 
				  CMLOG_SOCK_BUFSIZE) != 0)
      return CMLOG_ERROR;

    connected_ = 1;

    // send browser info
    if (sendBrowserInfo () != CMLOG_SUCCESS) {
      disconnect ();
      return CMLOG_ERROR;
    }
  }
  return CMLOG_SUCCESS;
}


int
cmlogBrowser::connect (unsigned short brd_port, int num)
{
  if (!connected_) {
    if (cmlogNetUtil::udpOpenLocal (udp_fd_) != 0)
      return CMLOG_ERROR;
    
    if (cmlogNetUtil::udpTalkToServer (udp_fd_, brd_port, num, 
				       CMLOG_BRSERLKSVC,
				       serverPort_, serverHost_) != 0)
      return CMLOG_ERROR;

    // connect to TCP port
    // if code reach here, udp_fd should be closed, reset it to -1
    udp_fd_ = -1;

    if (cmlogNetUtil::tcpConnect (serverPort_, serverHost_, tcp_fd_,
				  CMLOG_SOCK_BUFSIZE) != 0)
      return CMLOG_ERROR;

    connected_ = 1;

    // send browser info
    if (sendBrowserInfo () != CMLOG_SUCCESS) {
      disconnect ();
      return CMLOG_ERROR;
    }
  }
  return CMLOG_SUCCESS;
}


int
cmlogBrowser::connect (char* host, unsigned short brd_port, int num)
{
  if (!connected_) {
    if (cmlogNetUtil::udpOpenLocal (udp_fd_) != 0)
      return CMLOG_ERROR;
    
    if (cmlogNetUtil::udpTalkToGivenServer (udp_fd_, brd_port, 
					    host, num, 
					    CMLOG_BRSERLKSVC,
					    serverPort_) != 0)
      return CMLOG_ERROR;

    // connect to TCP port
    // if code reach here, udp_fd should be closed, reset it to -1
    udp_fd_ = -1;
    if (serverHost_)delete []serverHost_;
    serverHost_ = new char[::strlen(host) + 1];
    ::strcpy (serverHost_, host);
    
    if (cmlogNetUtil::tcpConnect (serverPort_, serverHost_, tcp_fd_,
				  CMLOG_SOCK_BUFSIZE) != 0)
      return CMLOG_ERROR;

    connected_ = 1;

    // send browser info
    if (sendBrowserInfo () != CMLOG_SUCCESS) {
      disconnect ();
      return CMLOG_ERROR;
    }
  }
  return CMLOG_SUCCESS;
}

int
cmlogBrowser::tcpConnect (char* host, unsigned short tcp_port)
{
  if (cmlogNetUtil::tcpConnect (tcp_port, host, tcp_fd_,
				CMLOG_SOCK_BUFSIZE) != 0)
    return CMLOG_ERROR;

  // copy server information
  serverPort_ = tcp_port;
  if (serverHost_) delete []serverHost_;
  serverHost_ = new char[::strlen (host) + 1];
  ::strcpy (serverHost_, host);

  connected_ = 1;
  
  // send browser info
  if (sendBrowserInfo () != CMLOG_SUCCESS) {
    disconnect ();
    return CMLOG_ERROR;
  }
  return CMLOG_SUCCESS;
}

void
cmlogBrowser::cleanupCallbacks (void)
{
  // delete everything from disconnect callback list
  cmlogSlistIterator ite (discCbkList_);
  cmlogCallback*     cbk = 0;
  for (ite.init (); !ite; ++ite) {
    cbk = (cmlogCallback *) ite ();
    delete cbk;
  }
  discCbkList_.deleteAllValues ();

  // delete everything from command callback list
  cmlogIntHashIterator iite (queryCbkTable_);
  for (iite.init (); !iite; ++iite) {
    cbk = (cmlogCallback *) iite ();
    delete cbk;
  }
  queryCbkTable_.deleteAllValues ();

  // delete everything from monitor callback list
  cmlogStrHashIterator site (monitorCbkTable_);
  for (site.init (); !site; ++site) {
    cbk = (cmlogCallback *) site ();
    delete cbk;
  }
  monitorCbkTable_.deleteAllValues ();  
}

int
cmlogBrowser::connected (void) const
{
  return connected_;
}

int
cmlogBrowser::getFd (void) const
{
  return tcp_fd_;
}

int
cmlogBrowser::disconnect (void)
{
  if (connected_) {
    close (tcp_fd_);
    tcp_fd_ = -1;
    connected_ = 0;
  }
  return CMLOG_SUCCESS;
}

int 
cmlogBrowser::disconnectCallback (cmlogBrCallback callback, void* arg)
{
  cmlogCallback* cbk = new cmlogCallback (callback, arg);
  cmlogSlistIterator ite (discCbkList_);
  cmlogCallback* tcbk = 0;
  int            error = 0;

  for (ite.init (); !ite; ++ite) {
    tcbk = (cmlogCallback *) ite ();
    if (*tcbk == *cbk) {
      error = 1;
      break;
    }
  }
  if (!error) {
    discCbkList_.add ((void *)cbk);
    return CMLOG_SUCCESS;
  }
  else {
    delete cbk;
    return CMLOG_ERROR;
  }
}

int 
cmlogBrowser::queryCallback (char* message, cdevData& data, 
			     cmlogBrCallback callback, void* arg)
{
  int n = 0;
  int st = 0;

  if (!connected_ || tcp_fd_ == -1)
    return CMLOG_BADIO;

  char dupm[160];
  char verb[80], attribute[80];

  // copy incoming message: 
  // some compiler will not like sscanf on temparary char*
  ::strncpy (dupm, message, sizeof (dupm));
  if ((st = ::sscanf (dupm, "%s %s", verb, attribute)) == 1) {
    if (::strcasecmp (verb, "query") == 0) {
      // always need time range for query functions
      double start_time, end_time;

      if (data.get ((char *)("start"), &start_time) != CDEV_SUCCESS)
	return CMLOG_INVALID;
      if (data.get ((char *)("end"), &end_time) != CDEV_SUCCESS)
	return CMLOG_INVALID;
      if (end_time <= start_time)
	return CMLOG_INVALID;
    
      cmlogCallback *cbk = new cmlogCallback (callback, arg);

      if (qid_ > 65535)
	qid_ = 93;

      if (evid_ > CMLOG_EVID_MAX)
	evid_ = 1024;

      // construct a new message to go to the server
      cmlog_cdevMessage msg (0, qid_++, 0, 0, evid_++, 0, 
			     0, 0, verb, &data, 0, 0);
      cmlogMsg    qmsg    (msg);
    
      cmlogPacket qpacket;
      if (qpacket.overflow (qmsg))
	return CMLOG_OVERFLOW;

      qpacket.insert (qmsg);
      
      // send over to the server
      n = tcp_fd_ << qpacket;
      if (n > 0) {
	// insert this cbk into query callback table by evid
	cbk->setEvent (evid_ - 1);
	queryCbkTable_.add (evid_ - 1, (void *)cbk);
      	return CMLOG_SUCCESS;
      }
      // remove error message
      return CMLOG_IOERROR;
    }
    if (::strcasecmp (verb, "stopQuery") == 0) {
      cmlogCallback *cbk = 0;
      // first find out whether arg + callback are in the callback table
      if ((cbk = hasQueryCallback (callback, arg)) == 0) 
	return CMLOG_NOTFOUND;

      if (qid_ > 65535)
	qid_ = 93;

      cmlog_cdevMessage msg (0, qid_++, 0, 0, cbk->getEvent(), 0,
			     0, 0, message, 0, 0, 0);
      cmlogMsg qmsg (msg);

      cmlogPacket qpacket;
      if (qpacket.overflow (qmsg))
	return CMLOG_OVERFLOW;

      qpacket.insert (qmsg);

      // send to the server
      n = tcp_fd_ << qpacket;

      if (n > 0) 
	return CMLOG_SUCCESS;
      
      return CMLOG_IOERROR;
    }
  }
  else if (st >= 2) {
    if (::strcasecmp (verb, "monitorOn") == 0) {

      // create a new cmlogCallback
      cmlogCallback* cbk = new cmlogCallback (callback, arg);

      // set query id 
      if (qid_ > 65535)
	qid_ = 93;

      if (evid_ > CMLOG_EVID_MAX)
	evid_ = 1024;

      cmlog_cdevMessage msg (0, qid_++, 0, 0, evid_++, 0, 
			     0, 0, message, &data, 0, 0);
      cmlogMsg    qmsg    (msg);
    
      cmlogPacket qpacket;
      if (qpacket.overflow (qmsg))
	return CMLOG_OVERFLOW;

      qpacket.insert (qmsg);
      
      // send over to the server
      n = tcp_fd_ << qpacket;
      if (n > 0) {
	// insert this cbk into monitorOn callback table
	cbk->setEvent (evid_ - 1);
	monitorCbkTable_.add (attribute, (void *)cbk);
    	return CMLOG_SUCCESS;
      }
      
      // remove error message
      return CMLOG_IOERROR;
    }
    else if (::strcasecmp (verb, "monitorOff") == 0) {
      cmlogCallback* cbk = 0;
      // first find out whether arg + callback in the callback table
      if ((cbk = monitorCallback (attribute, callback, arg)) == 0) {
	// remove error message
	return CMLOG_NOTFOUND;	
      }
    
      // set query id 
      if (qid_ > 65535)
	qid_ = 93;

      cmlog_cdevMessage msg (0, qid_++, 0, 0, cbk->getEvent(), 0, 
			     0, 0, message, &data, 0, 0);
      cmlogMsg    qmsg    (msg);
    
      cmlogPacket qpacket;
      if (qpacket.overflow (qmsg))
	return CMLOG_OVERFLOW;

      qpacket.insert (qmsg);
      
      // send over to the server
      n = tcp_fd_ << qpacket;
      if (n > 0)
	return CMLOG_SUCCESS;
      
      // remove error message
      return CMLOG_IOERROR;
    }
  }
  return CMLOG_INVALID;
}

int
cmlogBrowser::sendBrowserInfo (void)
{
  cdevData ccxt;
  char     username[80];
  int      taskid;
  char     host[256];
  char     display[256];
  char**   tagnames;
  int*     tagvalues;
  int      ntags = 0;
  cdevData tagmap;

  cdevData::readTagTable (tagvalues, tagnames, ntags);
  if (ntags == 0) 
    fprintf (stderr, "no tags found in the tag table\n");
  else {
    // integer values are inserted with tag value 1
    tagmap.insert (1, tagvalues, ntags);
    // names of tags are inserted with tag value 2
    tagmap.insert (2, tagnames, ntags);
    // free memory
    delete []tagvalues;
    delete []tagnames;
  }

  // get user name and process id number
  struct passwd* pwsd = 0;
  if ((pwsd = getpwuid (getuid()) ) != 0) {
    ::strncpy (username, pwsd->pw_name, sizeof (username) - 1);
  }
  else 
    ::sprintf (username, "userid_%d",getuid ());

  taskid = (int)::getpid ();

  // get hostname and if possible X-window display information
  // get host name
  int  len = sizeof (host);
  if (::gethostname (host, len) != 0)
    strcpy (host, "unknown");

  // get X window display name
  char* tmp = ::getenv ("DISPLAY");
  if (!tmp) {
    strncpy (display, host, sizeof (display) - 5);
    strcat (display, ":0.0");
  }
  else {
    if (::strcmp (tmp, ":0.0") == 0) {
      ::strncpy (display, host, sizeof (display) - 5);
      ::strcat (display, tmp);
    }
    else
      strncpy (display, tmp, sizeof (display) - 1);
  }
  ccxt.insert (cmlogUtil::CMLOG_HOST_TAG, host);
  ccxt.insert (cmlogUtil::CMLOG_PID_TAG,  taskid);
  ccxt.insert (cmlogUtil::CMLOG_USER_TAG, username);
  ccxt.insert (cmlogUtil::CMLOG_DISP_TAG, display);

  cmlog_cdevMessage message (0, 0, 0, 0, 0,
			     CMLOG_CONN_INFO, 0, 0, 0, 
			     &ccxt, 0, 0);
  if (ntags)
    message.setTagMap (&tagmap, 1);

  // create cmlogMsg
  cmlogMsg sendmsg (message);

  cmlogPacket qpacket;
  if (qpacket.overflow (sendmsg))
    return CMLOG_OVERFLOW;

  qpacket.insert (sendmsg);
      
  // send over to the server
  int n = tcp_fd_ << qpacket;
  if (n > 0)
    return CMLOG_SUCCESS;
  
  // remove error message
  return CMLOG_IOERROR;  
}

int
cmlogBrowser::pendIO (void)
{
  if (lock_)
    return CMLOG_INVALID;

  if (tcp_fd_ == -1)
    return CMLOG_BADIO;

  int nfound = 0;
  fd_set             readfd;  

  FD_ZERO (&readfd);
  FD_SET  (tcp_fd_, &readfd);

  do {
#ifndef __hpux
    nfound = select (tcp_fd_ + 1, &readfd, 0, 0, 0);
#else

#ifndef __hpux10
    nfound = select (tcp_fd_ + 1, (int *)&readfd, (int *)0, (int *)0,
		     (struct timeval *)0);
#else
    nfound = select (tcp_fd_ + 1, &readfd, 0, 0, 0);
#endif

#endif
  }while (nfound == -1 && errno == EINTR);

  if (nfound == -1) { // something wrong with tcp_fd_
    fprintf (stderr, "Bad tcp fd %d\n", tcp_fd_);
    tcp_fd_ = -1;
    return CMLOG_BADIO;
  }
  else if (nfound == 0) // timeout
    return CMLOG_TIMEOUT;
  else {
    if (FD_ISSET (tcp_fd_, &readfd))
      if (handle_input (tcp_fd_) < 0) { // bad io
	close (tcp_fd_);
	tcp_fd_ = -1;
	return CMLOG_IOERROR;
      }
  }
  return CMLOG_SUCCESS;

}

int
cmlogBrowser::pendIO (double seconds)
{
  if (lock_)
    return CMLOG_INVALID;

  if (tcp_fd_ == -1)
    return CMLOG_BADIO;

  int nfound = 0;
  fd_set             readfd;  
  struct timeval     tv;

  if (seconds <= 0.000001) {
    tv.tv_sec = 0;
    tv.tv_usec = 0;
  }
  else {
    tv.tv_sec = (int)seconds;
    tv.tv_usec = (int)((seconds - tv.tv_sec)*1000000);
  }
    
  FD_ZERO (&readfd);
  FD_SET  (tcp_fd_, &readfd);

  do {
#ifndef __hpux
    nfound = select (tcp_fd_ + 1, &readfd, 0, 0, &tv);
#else

#ifndef __hpux10
    nfound = select (tcp_fd_ + 1, (int *)&readfd, (int *)0, (int *)0,
		     &tv);
#else
    nfound = select (tcp_fd_ + 1, &readfd, 0, 0, &tv);
#endif

#endif
  }while (nfound == -1 && errno == EINTR);

  if (nfound == -1) { // something wrong with tcp_fd_ 
    fprintf (stderr, "Bad tcp fd %d\n", tcp_fd_);
    tcp_fd_ = -1;
    return CMLOG_BADIO;
  }
  else if (nfound == 0) // timeout
    return CMLOG_TIMEOUT;
  else {
    if (FD_ISSET (tcp_fd_, &readfd))
      if (handle_input (tcp_fd_) < 0) { // bad io
	handle_close (tcp_fd_);
	return CMLOG_IOERROR;
      }
  }
  return CMLOG_SUCCESS;
}

int
cmlogBrowser::read_input (void)
{
  int n, status = 0;
  cmlogPacket packet;
  
  // lock this event processing part to prevent recursive calls
  cmlogBrowserLocker locker (this);

  n = tcp_fd_ >> packet;
  if (n <= 0) {
    fprintf (stderr, "Server is dead\n");
    connected_ = 0;
    status = -1;
  }
  else {
    // printf ("Received %d bytes from server\n", n);
    status = processData (packet);
  }

  return status;
}

int
cmlogBrowser::handle_input (int fd)
{
  int n, status = 0;
  cmlogPacket packet;
  
  // lock this event processing part to prevent recursive calls
  cmlogBrowserLocker locker (this);

  n = tcp_fd_ >> packet;
  if (n <= 0) {
    fprintf (stderr, "Server is dead\n");
    connected_ = 0;
    status = -1;
  }
  else {
    // printf ("Received %d bytes from server\n", n);
    status = processData (packet);
  }

  return status;
}

int
cmlogBrowser::processData (cmlogPacket& packet)
{
  char*       message = 0;
  cdevData*   data = 0;
  int         errcode = CMLOG_ERROR;
  unsigned    evid;
  char        verb[80], attribute[80];
  int         st = 0;

  // remember to free all msgs and msgs[] somewhere
  cmlogMsg** msgs = packet.messages ();
  if (msgs == (cmlogMsg **)CMLOG_BAD_PTR) {
    packet.empty();
    return 0;
  }

  // look at the first message which tells status of the query
  cmlog_cdevMessage& idata = (*msgs[0]);
  message = idata.getMessage ();

  if (message) {
    if ((st = ::sscanf (message, "%s %s", verb, attribute)) == 1) {
      if (::strcasecmp (verb, "query") == 0) {
	data = idata.getData ();
	if (data) {
	  if (data->get ((char *)("resultCode"), &errcode) != CDEV_SUCCESS)
	    errcode = CMLOG_ERROR;
	}
	
	evid = idata.getForeignDataIndex ();
	queryCbkFromServer (errcode, evid, packet);
      }
    }
    else if (st >= 2) {
      if (::strcasecmp (verb, "monitorOn") == 0) {
	data = idata.getData ();
	if (data) {
	  if (data->get ((char *)("resultCode"), &errcode) != CDEV_SUCCESS)
	    errcode = CMLOG_ERROR;
	}
 
	evid = idata.getForeignDataIndex ();
	monitorCbkFromServer (errcode, attribute, evid, packet);
      }
      else if (::strcasecmp (verb, "monitorOff") == 0) {
	data = idata.getData ();
	if (data) {
	  if (data->get ((char *)("resultCode"), &errcode) != CDEV_SUCCESS)
	    errcode = CMLOG_ERROR;
	}
 
	evid = idata.getForeignDataIndex ();
	monitorCbkFromServer (errcode, attribute, evid, packet);
      }
    }
  }
  // free memory for msgs
  for (int i = 0; i < packet.numberOfData (); i++)
    delete msgs[i];
  delete []msgs;

  packet.empty (); 
  return 0;
}

int
cmlogBrowser::handle_close (int)
{
  close (tcp_fd_);
  tcp_fd_ = -1;

  // call all disconnect callback
  callAllDiscCbks ();

  // clean up callback tables
  cleanupCallbacks ();
  return 0;
}

void
cmlogBrowser::callAllDiscCbks (void)
{
  cmlogSlistIterator ite (discCbkList_);
  cmlogCallback* cbk = 0;
  cmlogBrCallback callback = 0;
  void *arg = 0;

  for (ite.init (); !ite; ++ite) {
    cbk = (cmlogCallback *)ite ();
    callback = cbk->callbackFunction ();
    arg = cbk->userarg ();
    (*callback)(CMLOG_DISCONNECTED, arg, (cmlogPacket *)0);
  }
}  

int
cmlogBrowser::monitorCbkFromServer (int errcode, char* attribute,
				    unsigned evid,
				    cmlogPacket& packet)
{
  cmlogCallback* cbk = 0;
  cmlogSlist& list = monitorCbkTable_.bucketRef(attribute);
  cmlogSlistIterator ite (list);
  int                found = 0;

  for (ite.init (); !ite; ++ite) {
    cbk = (cmlogCallback *)ite ();
    if (cbk->getEvent() == evid) {
      found = 1;
      break;
    }
  }

  if (found) {
    cmlogBrCallback func = cbk->callbackFunction ();
    void *arg = cbk->userarg ();

    (*func)(errcode, arg, &packet);
    
    if (errcode == CMLOG_NOTFOUND || errcode == CMLOG_CBK_FINISHED) { 
      // not found in the server or callback is done
      // remove this callback
      monitorCbkTable_.remove (attribute, (void *)cbk);
      delete cbk;
    }
  }
  else {
#ifdef _CMLOG_DEBUG
    printf ("Cannot find this monitor callback: %s\n", attribute);
#endif
  }
  return 0;
}

int
cmlogBrowser::queryCbkFromServer (int errcode, unsigned evid,
				  cmlogPacket& packet)
{
  cmlogCallback* cbk = 0;
  cmlogSlist& list = queryCbkTable_.bucketRef(evid);
  cmlogSlistIterator ite (list);
  int                found = 0;

  for (ite.init(); !ite; ++ite) {
    cbk = (cmlogCallback *) ite ();
    if (cbk->getEvent() == evid) {
      found = 1;
      break;
    }
  }

  if (found) {
    cmlogBrCallback func = cbk->callbackFunction ();
    void *arg = cbk->userarg ();

    (*func)(errcode, arg, &packet);

    if (errcode != CMLOG_INCOMPLETE) {
      queryCbkTable_.remove (evid, (void *)cbk);
      delete cbk;
    }
  }
  else {
#ifdef _CMLOG_DEBUG
    fprintf (stderr, "Reply from query has evid %d matches no event\n",evid);
#endif
  }
  return 0;
}

int
cmlogBrowser::sendTagMapInfo (int tagvalue, char* tagname)
{
  cdevData   tagmap;
  
  // integer values are inserted with tag value 1
  tagmap.insert (1, tagvalue);
  // names of tags are inserted with tag value 2
  tagmap.insert (2, tagname);

  cmlog_cdevMessage message (0, 0, 0, 0, 0,
			     CMLOG_TAGMAP, 0, 0, 0, 
			     0, 0, &tagmap);

  // create cmlogMsg
  cmlogMsg sendmsg (message);

  // create cmlogPacket
  cmlogPacket packet;
  
  // insert tagmap message into the packet
  if (packet.insert (sendmsg) == 0)
    return tcp_fd_ << packet;
  return -1;
}

cmlogCallback*
cmlogBrowser::monitorCallback (char* attr, 
			       cmlogBrCallback callback, void* arg)
{
  cmlogSlist& list = monitorCbkTable_.bucketRef (attr);
  cmlogSlistIterator ite (list);
  cmlogCallback* cbk = 0;

  for (ite.init (); !ite; ++ite) {
    cbk = (cmlogCallback *) ite ();
    if (cbk->callbackFunction () == callback && cbk->userarg () == arg)
      return cbk;
  }
  return 0;
}


cmlogCallback*
cmlogBrowser::hasQueryCallback (cmlogBrCallback callback, void* arg)
{
  cmlogIntHashIterator ite (queryCbkTable_);
  cmlogCallback* cbk = 0;
  for (ite.init (); !ite; ++ite) {
    cbk = (cmlogCallback *) ite ();
    if (cbk->callbackFunction () ==callback && cbk->userarg () == arg)
      return cbk;
  }
  return 0;
}

int
cmlogBrowser::removeMonitorCallback (char* attr, 
				     cmlogBrCallback callback, void* arg)
{
  cmlogSlist& list = monitorCbkTable_.bucketRef (attr);
  cmlogSlistIterator ite (list);
  cmlogCallback* cbk = 0;
  int found = 0;

  for (ite.init (); !ite; ++ite) {
    cbk = (cmlogCallback *) ite ();
    if (cbk->callbackFunction () == callback && cbk->userarg () == arg) {
      found = 1;
      break;
    }
  }
  if (found) {
    list.remove ((void *)cbk);
    return CMLOG_SUCCESS;
  }
  return CMLOG_ERROR;
}



int
cmlogBrowser::hasMonitorCallback (char* attr)
{
  cmlogSlist& list = monitorCbkTable_.bucketRef (attr);
  if (list.isEmpty ())
    return 0;
  return 1;
}  

void
cmlogBrowser::enableTagmapNotify (void)
{
  tagmapNotify_ = 1;
}

void
cmlogBrowser::disableTagmapNotify (void)
{
  tagmapNotify_ = 0;
}

int
cmlogBrowser::tagmapNotify (void)
{
  return tagmapNotify_;
}

unsigned short
cmlogBrowser::serverUdpPort (void) const
{
  return cmlogNetUtil::findServerUdpPort ();
}


void
cmlogBrowser::lock (void)
{
  lock_ = 1;
}

void
cmlogBrowser::unlock (void)
{
  lock_ = 0;
}

//=======================================================================
//       Implementation of cmlogBrowserLocker class
//=======================================================================
cmlogBrowserLocker::cmlogBrowserLocker (cmlogBrowser* b)
:b_ (b)
{
  b_->lock ();
}

cmlogBrowserLocker::~cmlogBrowserLocker (void)
{
  b_->unlock ();
}


  
      
				    
  


  
  
