//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      cmlogConverter converts existing flat logging files into
//      cmlog data format
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogConverter.cc,v $
//   Revision 1.1.1.1  1999/09/07 15:29:02  chen
//   CMLOG version 2.0
//
// Revision 1.2  1998/10/29  18:50:29  chen
// port to RedHat 5.1
//
// Revision 1.1  1997/08/01  15:17:01  bickley
// Added cmlog to application development system.
//
//
//
#include <cmlogConfig.h>
#include <cmlogDatabase.h>
#include <cmlogUtil.h>

#ifdef __linux
extern "C"  char *strptime (const char *s, const char *fmt,
			    struct tm *tp);
#endif

// convert time represented by unix date into a double value
// unix date "Wed Mar 19 14:02:18 1997"

static char* cmlog_time_format = "%a %b %d %H:%M:%S %Y";
static char* cmlog_tmp_file = "cmlog_tmpXXXXXX";

static int
cmlog_time_stamp (char* date, double& time)
{
  int dst = -2;
  time_t t;
  struct tm tmv;
  int status = 0;

  if (strptime (date, cmlog_time_format, &tmv) != NULL) {
    tmv.tm_isdst = dst;
    t = mktime (&tmv);
    time = (double)t;
  }
  else 
    status = -1;

  return status;
}

// split whole string into host, date, and text
// The host, date, text are allocated characater buffers
// str looks like the following
// datest Mon Jul 23 09:32:32 1995 Error Happened at IOC12
static int
parse_string (char* str,
	      char* host, size_t hlen,
	      char* date, size_t dlen,
	      char* text, size_t tlen)
{
  char* ptr = str;
  char* dptr = 0;
  char* p = 0;
  char* q = 0;
  int   i = 0;

  // skip white spaces
  while (*ptr == ' ' || *ptr == '\t')
    ptr++;  

  // copy first segment to host buffer
  q = host;
  while (*ptr != ' ' && *ptr != '\t' && i < hlen - 1) {
    *q = *ptr;
    q++; ptr++; i++;
  }
  *q = '\0';


  // skip all white spaces
  while (*ptr == ' ' || *ptr == '\t')
    ptr++;
  
  // this is the beginning of date string
  dptr = ptr;
  
  // move ptr to pass 5 white space segments
  i = 0;
  while (i < 5 && *ptr != '\0' && *ptr != '\n') {
    if (*ptr == ' ' || *ptr == '\t') {
      i++;
      // skip all white spaces
      while (*ptr == ' ' || *ptr == '\t')
	ptr++;
    }
    else
      ptr++;
  }
  if (i < 5)  
    return -1;
  
  if (*ptr == '\0' || *ptr == '\n') {
    *text = '\0';
    return 0;
  }

  // copy date string to date buffer
  q = date;
  p = dptr;
  i = 0;
  while (p != ptr && i < dlen - 1) {
    *q = *p;
    q++; p++; i++;
  }
  *q = '\0';

  // skip all white spaces
  while (*ptr == ' ' || *ptr == '\t')
    ptr++;  

  // now copy remaining text into text buffer
  strncpy (text, ptr, tlen);
  
  return 0;
}

main (int argc, char** argv)
{
  if (argc < 2) {
    fprintf (stderr, "Usage: %s datalogfilename\n", argv[0]);
    exit (1);
  }

  FILE* rfd = fopen (argv[1], "r");
  if (rfd == NULL) {
    fprintf (stderr, "Cannot open file %s\n", argv[1]);
    exit (1);
  }

  cdevData data;
  double   key = 0.0;
  char     host[80], date[80], text[256];
  char     line[416];
  char     tmpfile[40];
  
  // setup all tags
  cmlogUtil::setTags ();
  
  // copy tmpfile template to tmpfile
  strcpy (tmpfile, cmlog_tmp_file);
  // create a unique file name
  mktemp (tmpfile);

  cmlogDatabase dbase;
  if (dbase.open (tmpfile, O_CREAT | O_RDWR) != CDEV_SUCCESS) {
    fprintf (stderr, "Cannot open database %s\n", tmpfile);
    exit (1);
  }

  int i = 0;
  while (!feof (rfd) ) {
    memset (line, 0, sizeof (line));
    fgets (line, sizeof (line), rfd);
    if (parse_string (line, 
		      host, sizeof (host),
		      date, sizeof (date),
		      text, sizeof (text)) == 0) {
      if (cmlog_time_stamp (date, key) == 0 ) {
	data.remove ();
	data.insert (cmlogUtil::CMLOG_KEY_TAG, key);
	data.insert (cmlogUtil::CMLOG_HOST_TAG, host);
	data.insert (cmlogUtil::CMLOG_TEXT_TAG, text);
	data.insert (cmlogUtil::CMLOG_CLASS_TAG, "UNKNOWN");
	data.insert (cmlogUtil::CMLOG_FACILITY_TAG, "EPICS");
	data.insert (cmlogUtil::CMLOG_SEVERITY_TAG, -1);

	dbase.put (data);
      }
    }
    else
      printf ("%s\n", line);
  }
  // close file and database
  fclose (rfd);
  dbase.close ();

  // move database file to the right directory with right name
  // but first get the first element of the database
  if (dbase.open (tmpfile, O_RDONLY) != CDEV_SUCCESS) {
    fprintf (stderr, "Fatal: Cannot read database %s\n", tmpfile);
    exit (1);
  }

  data.remove ();
  key = 0.0;
  if (dbase.cursorInit (data, &key, 1) != CDEV_SUCCESS) {
    fprintf (stderr, "Fatal: Cannot find anything in %s\n", tmpfile);
    dbase.close ();
    exit (1);
  }

  printf ("Name the database with postfix %d\n", (int)key);
  printf ("Enter database prefix name (fullpath)\n");
  
  char prefix[80];
  char fullname[128];
  scanf ("%s", prefix);
  sprintf (fullname, "%s%d", prefix, (int)key);
  
  printf ("Moving current database %s to %s ", tmpfile, fullname);
  if (rename (tmpfile, fullname) == 0) 
    printf ("is completed\n");
  else
    printf ("failed with errno %d\n", errno);

}


  
      
  
  

  

