/**************************************************************************
 * Copyright (c)   2001    Southeastern Universities Research Association,
 *                         Thomas Jefferson National Accelerator Facility
 *
 * This software was developed under a United States Government license
 * described in the NOTICE file included as part of this distribution.
 *
 * Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
 **************************************************************************
 *
 * Description:
 *      Simple Window for Date Input
 *
 * Author:  
 *      Jie Chen
 *      Jefferson Lab HPC Group
 *
 * Revision History:
 *   $Log: DateInput.java,v $
 *   Revision 1.1  2001/10/18 18:45:41  chen
 *   First version of Jcmlog
 *
 *
 *
 */
package cmlog.gui;

import java.io.*;
import java.util.*;
import java.text.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*; 
import javax.swing.text.*; 

public class DateInput extends JPanel implements KeyListener, ActionListener, FocusListener
{
    /**
     * 12 months of a year.
     */
    private static final String[] months_ = {"Jan", "Feb", "Mar",
					     "Apr", "May", "Jun",
					     "Jul", "Aug", "Sep",
					     "Oct", "Nov", "Dec"};

    /**
     * 12 months represented using Calendar format.
     */
    private static final int[]    calmonths_ = {Calendar.JANUARY,
						Calendar.FEBRUARY,
						Calendar.MARCH,
						Calendar.APRIL,
						Calendar.MAY,
						Calendar.JUNE,
						Calendar.JULY,
						Calendar.AUGUST,
						Calendar.SEPTEMBER,
						Calendar.OCTOBER,
						Calendar.NOVEMBER,
						Calendar.DECEMBER};
    /**
     * Hour input field
     */
    private JTextField hour_ = null;

    /**
     * Minute input field.
     */
    private JTextField min_ = null;

    /**
     * Seconds input field.
     */
    private JTextField sec_ = null;

    /**
     * Month field.
     */
    private JComboBox  month_ = null;

    /**
     * Day field.
     */
    private JTextField day_ = null;

    /**
     * Year field.
     */
    private JTextField year_ = null;

    /**
     * Current key code
     */
    private int        keycode_ = KeyEvent.VK_UNDEFINED;

    /**
     * Up arrow button.
     */
    private JButton    up_  = null;

    /**
     * Down arrow button.
     */
    private JButton    down_ = null;

    /**
     * Focused text input field
     */
    private JTextField focusedField_ = null;

    /**
     * Default background and foreground.
     */
    private Color fg_ = null;
    private Color bg_ = null;
    
    public DateInput (Date date)
    {
	// create top level panel
	super ();

	Box hbox = Box.createHorizontalBox ();

	// get a instance of calendar
	Calendar cal = Calendar.getInstance ();
	
	// set the cal to date
	cal.setTime (date);

	// create Combox for month
	int sel = cal.get (Calendar.MONTH) - Calendar.JANUARY;
	month_ = new JComboBox (months_);
	month_.setEditable (false);
	month_.setSelectedIndex (sel);
	hbox.add (month_);

	// create day input field
	sel = cal.get (Calendar.DAY_OF_MONTH);
	day_ = new JTextField (String.valueOf(sel), 2);
	day_.addKeyListener (this);
	day_.addFocusListener (this);
	hbox.add (day_);

	// create year field
	sel = cal.get (Calendar.YEAR);
	year_ = new JTextField (String.valueOf(sel), 4);
	year_.addKeyListener (this);
	year_.addFocusListener (this);
	hbox.add (year_);

	// create a white space of 20 pixel
	Component tmp = hbox.createHorizontalStrut (10);
	hbox.add (tmp);
	

	sel = cal.get (Calendar.HOUR_OF_DAY);
	hour_ = new JTextField (String.valueOf (sel), 2);
	hour_.addKeyListener (this);
	hour_.addFocusListener (this);
	hbox.add (hour_);

	JTextField comm = new JTextField (": ");
	comm.setEditable (false);
	comm.setBorder (null);
	hbox.add (comm);

	sel = cal.get (Calendar.MINUTE);	
	min_ = new JTextField (String.valueOf(sel), 2);
	min_.addKeyListener (this);
	min_.addFocusListener (this);
	hbox.add (min_);

	comm = new JTextField (": ");
	comm.setEditable (false);
	comm.setBorder (null);
	hbox.add (comm);

	sel = cal.get (Calendar.SECOND);	
	sec_ = new JTextField (String.valueOf(sel), 2);
	sec_.addKeyListener (this);
	sec_.addFocusListener (this);
	hbox.add (sec_);

	// create arrow panel
	JPanel apanel = createArrowPanel ();
	hbox.add (apanel);

	// add hbox
	add (hbox);

    }

    /**
     * Create arrow panel.
     */
    private JPanel createArrowPanel ()
    {
	JPanel panel = new JPanel (new GridLayout(2, 1));
	
	up_ = new JButton (new ArrowIcon(16, 8, SwingConstants.TOP));
	up_.setBorder (null);

	down_ = new JButton (new ArrowIcon(14, 7, SwingConstants.BOTTOM));
	down_.setBorder (null);
	
	panel.add (up_);
	panel.add (down_);

	up_.addActionListener (this);
	down_.addActionListener (this);
	up_.addFocusListener (this);
	down_.addFocusListener (this);
	return panel;
    }
	

    /**
     * Set current display to a date.
     */
    public void setDate (Date date)
    {
	// get a instance of calendar
	Calendar cal = Calendar.getInstance ();
	
	// set the cal to date
	cal.setTime (date);

	int sel = cal.get (Calendar.MONTH) - Calendar.JANUARY;
	month_.setSelectedIndex (sel);

	// day input field
	sel = cal.get (Calendar.DAY_OF_MONTH);
	day_.setText(String.valueOf(sel));

	// year field
	sel = cal.get (Calendar.YEAR);
	year_.setText (String.valueOf(sel));

	// hour
	sel = cal.get (Calendar.HOUR_OF_DAY);
	hour_.setText (String.valueOf (sel));

	// minute
	sel = cal.get (Calendar.MINUTE);	
	min_.setText (String.valueOf(sel));
	
	// seconds
	sel = cal.get (Calendar.SECOND);	
	sec_.setText (String.valueOf(sel));
    }

    /**
     * Get current displayed date.
     */
    public Date getDate ()
    {
	// get a instance of calendar
	Calendar cal = Calendar.getInstance ();
	
	// month
	int sel = month_.getSelectedIndex ();
	cal.set (Calendar.MONTH, calmonths_[sel]);

	// year
	cal.set (Calendar.YEAR, Integer.valueOf(year_.getText()).intValue());

	// day
	cal.set (Calendar.DAY_OF_MONTH, 
		 Integer.valueOf(day_.getText()).intValue());

	// hour
	cal.set (Calendar.HOUR_OF_DAY,
		 Integer.valueOf(hour_.getText()).intValue());	

	// minute
	cal.set (Calendar.MINUTE,
		 Integer.valueOf(min_.getText()).intValue());

	// seconds
	cal.set (Calendar.SECOND,
		 Integer.valueOf(sec_.getText()).intValue());

	return cal.getTime ();
    }

    /**
     * Check whether a key is a navigation key
     * left, right arrow
     */
    private static boolean navigationKey (int keycode)
    {
	if (keycode == KeyEvent.VK_LEFT ||
	    keycode == KeyEvent.VK_RIGHT ||
	    keycode == KeyEvent.VK_KP_LEFT ||
	    keycode == KeyEvent.VK_KP_RIGHT)
	    return true;
	return false;
    }

    /**
     * Check whether a key is a delete key
     */
    private static boolean deleteKey (int keycode)
    {
	if (keycode == KeyEvent.VK_BACK_SPACE ||
	    keycode == KeyEvent.VK_DELETE)
	    return true;
	return false;
    }


    /** 
     * check whether type key is ok for a particular field.
     */
    private boolean checkYearValue (JTextField input, int keycode, 
				    char keychar)
    {
	// all navigation key is ok
	if (navigationKey (keycode) == true)
	    return true;
	
	// all delete key
	if (deleteKey (keycode) == true)
	    return true;

	// key must be interger 0 to 9
	if (keycode < KeyEvent.VK_0 || keycode > KeyEvent.VK_9)
	    return false;

	return true;
    }

    /** 
     * check whether type key is ok for a Minute or Second field.
     */
    private boolean checkMinOrSecondValue (JTextField input, int keycode,
					   char keychar)
    {
	// whether we forward a key event
	boolean  forwardevent = true;
	Document doc = input.getDocument ();
	Caret    caret = input.getCaret ();
	int      pos   = caret.getDot ();
	String   extext = null;

	try {
	    extext = doc.getText (0, doc.getLength());
	}catch (Exception e) {
	    return false;
	}
	
	StringBuffer buf = new StringBuffer (extext);


	// this is value of day field a user wants to input
	buf = buf.insert (pos, keychar);

	int maxval = 60;
	int minval = 0;
	
	// user typed day
	String strval = new String(buf);
	int    typeval = Integer.valueOf(strval).intValue();

	if (typeval > maxval) {
	    strval = String.valueOf (maxval);
	    forwardevent = false;
	}
	else if (typeval < minval) {
	    strval = String.valueOf (minval);
	    forwardevent = false;
	}
	
	if (forwardevent == false)
	    input.setText (strval);

	return forwardevent;
    }


    /** 
     * check whether type key is ok for a hour field.
     */
    private boolean checkHourValue (JTextField input, int keycode,
				    char keychar)
    {
	// whether we forward a key event
	boolean  forwardevent = true;
	Document doc = input.getDocument ();
	Caret    caret = input.getCaret ();
	int      pos   = caret.getDot ();
	String   extext = null;

	try {
	    extext = doc.getText (0, doc.getLength());
	}catch (Exception e) {
	    return false;
	}
	
	StringBuffer buf = new StringBuffer (extext);


	// this is value of day field a user wants to input
	buf = buf.insert (pos, keychar);

	int maxhr = 24;
	int minhr = 0;
	
	// user typed day
	String hrstr = new String(buf);
	int typehr = Integer.valueOf(hrstr).intValue();

	if (typehr > maxhr) {
	    hrstr = String.valueOf (maxhr);
	    forwardevent = false;
	}
	else if (typehr < minhr) {
	    hrstr = String.valueOf (minhr);
	    forwardevent = false;
	}
	
	if (forwardevent == false)
	    input.setText (hrstr);

	return forwardevent;
    }


    /** 
     * check whether type key is ok for a day field.
     */
    private boolean checkDayValue (JTextField input, int keycode,
				   char keychar)
    {
	// whether we forward a key event
	boolean  forwardevent = true;
	Document doc = input.getDocument ();
	Caret    caret = input.getCaret ();
	int      pos   = caret.getDot ();
	String   extext = null;

	try {
	    extext = doc.getText (0, doc.getLength());
	}catch (Exception e) {
	    return false;
	}
	
	StringBuffer buf = new StringBuffer (extext);


	// this is value of day field a user wants to input
	buf = buf.insert (pos, keychar);

	// get a instance of calendar
	Calendar cal = Calendar.getInstance ();
	
	// month
	int sel = month_.getSelectedIndex ();
	cal.set (Calendar.MONTH, calmonths_[sel]);

	// year
	cal.set (Calendar.YEAR, Integer.valueOf(year_.getText()).intValue());

	int maxday = cal.getActualMaximum (Calendar.DAY_OF_MONTH);
	int minday = cal.getActualMinimum (Calendar.DAY_OF_MONTH);
	
	// user typed day
	String daystr = new String(buf);
	int typeday = Integer.valueOf(daystr).intValue();

	

	if (typeday > maxday) {
	    daystr = String.valueOf (maxday);
	    forwardevent = false;
	}
	else if (typeday < minday) {
	    daystr = String.valueOf (minday);
	    forwardevent = false;
	}
	
	if (forwardevent == false)
	    input.setText (daystr);

	return forwardevent;
    }

    /** 
     * check whether type key is ok for a particular field.
     */
    private boolean checkValue (JTextField input, int keycode, char keychar)
    {
	// all navigation key is ok
	if (navigationKey (keycode) == true)
	    return true;
	
	// all delete key
	if (deleteKey (keycode) == true)
	    return true;

	// key must be interger 0 to 9
	if (keycode < KeyEvent.VK_0 || keycode > KeyEvent.VK_9)
	    return false;

	if (input == day_)
	    return checkDayValue (input, keycode, keychar);
	else if (input == hour_)
	    return checkHourValue (input, keycode, keychar);
	else
	    return checkMinOrSecondValue (input, keycode, keychar);
    }
	

    public void keyTyped(KeyEvent e)
    {
	JTextField input;
	Document   doc;
	Caret      caret;

	input = (JTextField)e.getSource ();
	doc   = input.getDocument ();
	caret = input.getCaret ();

	if (input == year_) {
	    if (caret.getDot () == 4 && 
		keycode_ != KeyEvent.VK_LEFT &&
		keycode_ != KeyEvent.VK_BACK_SPACE && 
		keycode_ != KeyEvent.VK_DELETE)
		e.consume ();
	    else if (doc.getLength () >= 4 && 
		     keycode_ != KeyEvent.VK_BACK_SPACE &&
		     keycode_ != KeyEvent.VK_DELETE)
		e.consume();
	    else {
		if (checkYearValue (year_, keycode_, e.getKeyChar()) == false)
		    e.consume();
	    }
	}
	else {
	    if (caret.getDot () == 2 && 
		keycode_ != KeyEvent.VK_LEFT &&
		keycode_ != KeyEvent.VK_BACK_SPACE && 
		keycode_ != KeyEvent.VK_DELETE)
		e.consume ();
	    else if (doc.getLength () >= 2 && 
		     keycode_ != KeyEvent.VK_BACK_SPACE &&
		     keycode_ != KeyEvent.VK_DELETE)
		e.consume();
	    else {
		if (checkValue (input, keycode_, e.getKeyChar()) == false)
		    e.consume();
	    }
	}
    }

    /**
     * Focus listener
     */
    public void focusGained(FocusEvent e)
    {
	Object source = e.getSource ();
	
	if (source != up_ && source != down_) {
	    JTextField input = (JTextField)source;

	    if (bg_ == null && fg_ == null) {
		bg_ = input.getBackground ();
		fg_ = input.getForeground ();
	    }
	
	    // reset previous field
	    if (focusedField_ != null) {
		focusedField_.setBackground (bg_);
		focusedField_.setForeground (fg_);
	    }
	    focusedField_ = input;
	}
    }
    
    public void focusLost(FocusEvent e)
    {
	Object source = e.getSource ();
	if (source == up_ || source == down_) {
	    if (focusedField_ != null) {
		focusedField_.setBackground (bg_);
		focusedField_.setForeground (fg_);
	    }
	}
    }
    

    /**
     * Return a text field that has the previous key board focus.
     */
    private JTextField focusedField ()
    {
	// reverse this field
	if (focusedField_ != null) {
	    focusedField_.setBackground (fg_);
	    focusedField_.setForeground (bg_);
	}
	return focusedField_;
    }

    /**
     * Increase a value inside a value field
     */
    private void increaseValueField (JTextField input)
    {
	if (input == day_) 
	    changeDayField (input, 1);
	else if (input == year_)
	    changeField (input, 1, 2100, 1900);
	else if (input == hour_)
	    changeField (input, 1, 24, 0);
	else
	    changeField (input, 1, 60, 0);
    }


    /**
     * Decrease a value inside a value field
     */
    private void decreaseValueField (JTextField input)
    {
	if (input == day_) 
	    changeDayField (input, -1);
	else if (input == year_)
	    changeField (input, -1, 2100, 1900);
	else if (input == hour_)
	    changeField (input, -1, 24, 0);
	else
	    changeField (input, -1, 60, 0);
    }

    /**
     * Change day value filed.
     */
    private void changeDayField (JTextField input, int type)
    {
	// whether we forward a key event
	Document doc =input.getDocument ();
	String   text = null;
	
	try {
	    text = doc.getText (0, doc.getLength());
	}catch (Exception e) {
	    return;
	}

	text = text.trim ();
	
	// get a instance of calendar
	Calendar cal = Calendar.getInstance ();
	
	// month
	int sel = month_.getSelectedIndex ();
	cal.set (Calendar.MONTH, calmonths_[sel]);

	// year
	cal.set (Calendar.YEAR, Integer.valueOf(year_.getText()).intValue());

	int maxday = cal.getActualMaximum (Calendar.DAY_OF_MONTH);
	int minday = cal.getActualMinimum (Calendar.DAY_OF_MONTH);

	if (text == null || text.length() == 0)
	    text = String.valueOf (minday);
	
	// user typed day
	int typeday = Integer.valueOf(text).intValue();

	if (type > 0)
	    typeday ++;
	else
	    typeday --;
	

	if (typeday > maxday) 
	    typeday = minday;
	else if (typeday < minday) 
	    typeday = maxday;
	
	
	input.setText (String.valueOf (typeday));
    }	

    /**
     * Change year value filed.
     */
    private void changeField (JTextField input, int type,
			      int maxval, int minval)
    {
	// whether we forward a key event
	Document doc =input.getDocument ();
	String   text = null;
	
	try {
	    text = doc.getText (0, doc.getLength());
	}catch (Exception e) {
	    ;
	}

	text = text.trim ();

	if (text == null || text.length() == 0)
	    text = String.valueOf (minval);
	
	// user typed value
	int val = Integer.valueOf(text).intValue();

	if (type > 0)
	    val++;
	else
	    val--;
	

	if (val > maxval) 
	    val = minval;
	else if (val < minval) 
	    val = maxval;
	
	input.setText (String.valueOf (val));
    }	
	
    

    public void keyPressed(KeyEvent e)
    {
	keycode_ = e.getKeyCode();
    }

    public void keyReleased(KeyEvent e)
    {
	keycode_ = e.getKeyCode();
    }

    public void actionPerformed (ActionEvent e)
    {
	JTextField input = focusedField ();
	
	if (input == null)
	    return;
	
	if (e.getSource() == up_) 
	    increaseValueField (input);
	else
	    decreaseValueField (input);
    }

    public static void main (String[] args)
    {
	JFrame frame = new JFrame ();

	DateInput win = new DateInput (new Date());

	frame.setContentPane (win);
	
	
	frame.setSize (400, 100);
	frame.setVisible (true);
    }
}

	
