//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
//-----------------------------------------------------------------------------
//
// Description:
//      cdevDataInputStream that converts a byte stream into a cdevData
//
// Author:  
//      Jie Chen
//      Jefferson Lab HPC Group
//
// Revision History:
//   $Log: cdevDataInputStream.java,v $
//   Revision 1.1  1999/12/14 15:31:37  chen
//   initial java implementation
//
//
//
package cmlog;

import java.io.*;
import java.math.*;

public class cdevDataInputStream
{
    // for 32 bit machine
    public static final int _CDEV_STREAM_BYTE_UNIT = 4;
    
    /**
     * Return a long word (32 bit) aligned size from original size
     */
    public static final int _CDEV_RNDUP (int x)
    {
	return ((x + _CDEV_STREAM_BYTE_UNIT - 1)/_CDEV_STREAM_BYTE_UNIT)
	    * _CDEV_STREAM_BYTE_UNIT;
    }

    // underlying data output stream
    private DataInputStream input = null;

    /**
     * Construct a data output stream
     */
    public cdevDataInputStream (InputStream stream)
    {
	try {
	    input = new DataInputStream (stream);
	}catch (Exception e) {
	    e.printStackTrace();
	}
    }

    /**
     * Read a single byte char from stream
     */
    public byte readByte () throws IOException
    {
	int tmp;
	try {
	    tmp = input.readInt ();
	}catch (IOException e) {
	    throw e;
	}
	return (byte)tmp;
    }

    /**
     * Read a value of short from the stream
     */
    public short readShort () throws IOException
    {
	int tmp;
	try {
	    tmp = input.readInt ();
	}catch (IOException e) {
	    throw e;
	}
	return (short)tmp;
    }

    /**
     * Read a value of int from stream
     */
    public int readInt () throws IOException
    {
	int tmp;
	try {
	    tmp = input.readInt ();
	}catch (IOException e) {
	    throw e;
	}
	return tmp;
    }

    /**
     * Read a value of long from stream
     */
    public long readLong () throws IOException
    {
	long tmp;
	try {
	    tmp = input.readLong ();
	}catch (IOException e) {
	    throw e;
	}
	return tmp;
    }


    /**
     * Read a value of float from stream
     */
    public float readFloat () throws IOException
    {
	float tmp;
	try {
	    tmp = input.readFloat ();
	}catch (IOException e) {
	    throw e;
	}
	return tmp;
    }


    /**
     * Read a value of double to stream
     */
    public double readDouble () throws IOException
    {
	double tmp;
	try {
	    tmp = input.readDouble ();
	}catch (IOException e) {
	    throw e;
	}
	return tmp;
    }

    /**
     * Read a string as series of byte using default encoding
     */
    public String readString () throws IOException
    {
	int len, rlen, i;
	byte[] tmp = null;
	
	// first find out str size
	try {
	    len = input.readInt ();
	}catch (IOException e) {
	    throw e;
	}
	rlen = cdevDataInputStream._CDEV_RNDUP(len);
	
	if (rlen == len)
	    tmp = new byte[rlen + 1];
	else
	    tmp = new byte[rlen];

	try {
	    input.readFully (tmp, 0, rlen);
	}catch (IOException e) {
	    throw e;
	}

	int slen;
	if (rlen == len) {
	    tmp[rlen] = (byte)0;
	    slen = rlen;
	}
	else {
	    // find out read length of this string by looking for the first 0
	    // in this byte array
	    for (i = 0; i < len; i++) {
		if (tmp[i] == 0)
		    break;
	    }
	    slen = i;
	}
	return new String (tmp, 0, slen);
    }

    /**
     * Read a series of byte
     */
    public byte[] readBytes () throws IOException
    {
	int len;

	try {
	    len = input.readInt();
	}catch (IOException e){
	    throw e;
	}

	byte[] result = new byte[len];
	try {
	    input.readFully (result, 0, len);
	}catch (IOException e){
	    throw e;
	}

	return result;
    }


    /**
     * Read a number of bytes: blocked until all bytes are read
     */
    public byte[] readBytes (int len) throws IOException
    {
	byte[] result = new byte[len];
	try {
	    input.readFully (result, 0, len);
	}catch (IOException e){
	    throw e;
	}

	return result;
    }
    
    /**
     * Read a time stamp from the stream
     */
    public cdevTimeStamp readTimeStamp () throws IOException
    {
	cdevTimeStamp ts = new cdevTimeStamp ();

	try {
	    ts.secPastEpoch = input.readInt();
	}catch (IOException e){
	    throw e;
	}

	try{
	    ts.nsec = input.readInt();
	}catch (IOException e){
	    throw e;
	}
	return ts;
    }

}
