//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
//-----------------------------------------------------------------------------
//
// Description:
//      CMLOG Client Interface to allow applications to log messages
//
// Author:  
//      Jie Chen
//      Jefferson Lab HPC Group
//
// Revision History:
//   $Log: Client.java,v $
//   Revision 1.2  2002/04/04 15:23:45  chen
//   Fix a bug of multiple java clients writing at the same time
//
//   Revision 1.1  2000/03/17 19:18:37  chen
//   add java client
//
//
//
package cmlog;

import java.io.*;
import java.net.*;
import java.util.*;

public final class Client implements cdevTagChangedEventListener
{
    // flag of connection
    private boolean connected_ = false;

    // client daemon assigned from client daemon
    private short   clientId_  = 0;

    // misc client information
    private String  username_ = null;
    private String  hostname_ = null;
    private String  progname_ = null;
    private int     pid_ = 0;

    // to notify tag map changes to the server
    private boolean tagmapNotify_ = true;

    // file output stream
    private FileOutputStream output_ = null;

    // udp sending back off array
    private static int udpBackOff[] = {1, 2, 4, 8, 16, 32};
    
    // maximum number of udp retries
    private static int maxUdpRetries = 3;

    // flag to tell whether a daemon is started or not
    private boolean    daemonStarted_ = false;

    private void getUserInfo () 
    {
	Properties sysp = System.getProperties ();

	// get user name
	username_ = sysp.getProperty ("user.name");

	// get host name
	boolean foundhost = true;
	InetAddress loc = null;

	try {
	    loc = InetAddress.getLocalHost();
	}catch (UnknownHostException e) {
	    foundhost = false;
	}
	if (foundhost == true)
	    hostname_ = loc.getHostName ();
	else {
	    System.err.println ("Cannot find local hostname, Quit");
	    System.exit (1);
	}

	// get process id for java vm
	try {
	    pid_ = Pid.getpid ();
	}catch (IOException e) {
	    pid_ = 0;
	}
	if (Config.debug == true) 
	    System.out.println ("Pid for this java vm is " + String.valueOf(pid_));
    }

    private String getPipeName ()
    {
	boolean found = false;
	String pipename = new String (Config.CMLOG_CLNT_PIPE);
	File   pipedir  = new File (Config.CMLOG_CLNT_PIPE);
	String[] filenames = null;

	try {
	    filenames = pipedir.list();
	}catch (SecurityException se) {
	    System.err.println ("Security violation to read pipe directory");
	    return null;
	}
	
	if (filenames.length == 0) {
	    System.err.println ("Cannot find client daemon pipe");
	    return null;
	}

	for (int i = 0; i < filenames.length; i++) {
	    if (Config.debug == true)
		System.out.println ("Found filename " + filenames[i]);
	    if (filenames[i].startsWith(Config.CMLOG_CLNT_PIPE_PREFIX)==true){
		pipename = pipename.concat (filenames[i]);
		found = true;
		break;
	    }
	}
	if (found == true)
	    return pipename;
	return null;
    }

	

    /**
     * Constructor
     * @param name  Name of this program
     */
    public Client (String name)
    {
	// set up all tags
	Config.setupTags ();

	// register this Browser to tag table to listen 
	// on the tag changed event
	cdevTagTable tbl = cdevTagTable.tagTable();
	
	tbl.addTagChangedEventListener (this);

	// copy program name
	progname_ = new String (name);

	// get various user information
	getUserInfo ();
    }

    public void tagChanged (cdevTagChangedEvent event)
    {
	if (connected_ && tagmapNotify_ == true) {
	    try {
		sendTagmapInfo (event.tag(), event.name());
	    }catch (IOException e) {
		System.err.println ("Cannot send tagmap info for tag: " + event.tag() + " tag name: " + event.name());
	    }
	}
    }

    private int findServerUdpPort ()
    {
	String env = null;
	int    serverPort = 0;
	try {
	    env = System.getProperty("CMLOG_PORT");
	}catch (SecurityException e) {
	    env = null;
	}
	if (env != null) 
	    serverPort = Integer.valueOf (env).intValue();
	else
	    serverPort = Config.CMLOG_PORT;

	if (Config.debug == true)
	    System.out.println ("Client: cmlogServer UDP port is " + String.valueOf (serverPort));

	return serverPort;
    }

    private int streamMsg (cdevMessage msg, cdevDataOutputStream stream) 
	throws IOException
    {
	int len = msg.streamSize ();
	int header = Packet.setMsgHeader (len);

	try {
	    stream.write (header);
	}catch (IOException e) {
	    throw e;
	}

	try {
	    msg.streamOut (stream);
	}catch (IOException e) {
	    throw e;
	}
	return len + cdevDataOutputStream.streamSize (header);
    }

    private void startClientD ()
    {
	Process p = null;
	String command = new String ("cmlog_activate");

	if (Config.debug == true) 
	    System.out.println ("Start ClientDaemon using command: " + command);
	Runtime r = Runtime.getRuntime();
	try {
	    p = r.exec (command);
	}catch (IOException e) {
	    System.err.println (e);
	}catch (SecurityException se) {
	    System.err.println (se);
	}
    }

    private short swapShort (short s)
    {
	short byte0 = (short)(s & 0xff);
	short byte1 = (short)((s >> 8) & (short)0xff);

	return (short)((byte0 << 8) | byte1);
    }


    private void udpContact (int port) 
	throws UnknownHostException, IOException, ConnectException
    {
	// Talk to UDP port on a given host
	DatagramSocket udpsocket = null;
	InetAddress    dest = null;
	int            isize = 2;    // receiving data buffer size
	int            osize = 1024; // output data buffsize
	int            olen;
	boolean        found = false;

	try {
	    udpsocket = new DatagramSocket ();
	}catch (IOException e) {
	    System.err.println("Cannot create data socket");
	    throw e;
	}
	try {
	    dest = InetAddress.getByName (hostname_);
	}catch (UnknownHostException ue) {
	    throw ue;
	}

	// convert a udp message into a byte array
	ByteArrayOutputStream pout = new ByteArrayOutputStream (osize);
	cdevDataOutputStream iout =  new cdevDataOutputStream (pout, false);

	// create a cdevMessage and send it out to client daemon
	cdevData pdata = new cdevData();
	short    serverport = (short)findServerUdpPort ();
	pdata.insert (Config.VALUE_TAG, serverport);
	
	cdevMessage msg = new cdevMessage ((short)0, 0, 0, 0, 0,
					   Config.CMLOG_USERLKCLNT,
					   null, null, pdata, null, null);
	// stream out this message
	try {
	    olen = streamMsg (msg, iout);
	}catch (IOException e) {
	    throw e;
	}

	// convert this byte array
	byte[] outputbuffer = pout.toByteArray ();
	if (outputbuffer.length != olen) {
	    throw new IOException ("Stream UDP protocol error");
	}

	// construct a output dataPacket
	DatagramPacket outpacket = new DatagramPacket (outputbuffer,
						       olen, dest, port);
	// construct a input dataPacket
	byte[] input = new byte[isize];
	DatagramPacket inputpacket = new DatagramPacket (input, isize);

	int i = 0;
	while (i < Client.maxUdpRetries) {
	    try {
		udpsocket.setSoTimeout (Client.udpBackOff[i] * 1000);
	    }catch (SocketException e) {
		System.err.println (e);
		throw new IOException ("Cannot set udp socket timeout");
	    }

	    // send data
	    try {
		udpsocket.send (outpacket);
	    }catch (IOException e) {
		throw e;
	    }

	    // receive data 
	    try {
		udpsocket.receive (inputpacket);
	    }catch (SocketException se) {
		if (Config.debug == true)
		    System.err.println (se);
		i++;
		continue;
	    }catch (IOException e) {
		if (Config.debug == true)
		    System.err.println ("Timeout on udp read");
		i++;
		continue;
	    }
	    // now we have something to read on udp socket
	    if (Config.debug == true)
		System.err.println ("UDP read getting something");

	    if (inputpacket.getLength () == isize)
		found = true;
	    break;
	}
	if (Config.debug == true) 
	    System.out.println ("UDP received " + String.valueOf (inputpacket.getLength()) + " bytes");

	if (found == false || inputpacket.getLength() < isize) {
	    System.out.println ("cmlogClientD is not found, please start one");
	    throw new IOException ("No client daemon");
	}

	// convert stream into a tcp port number
	ByteArrayInputStream istream = new ByteArrayInputStream (input);
	DataInputStream  distream = new DataInputStream (istream);

	try {
	    clientId_ = distream.readShort ();
	}catch (IOException e) {
	    System.err.println (e);
	    clientId_ = 0;
	    throw e;
	}
	
	if (clientId_ < 0) {
	    // since I know my clientd will never return a negative number, this
	    // negative number is from byte swap done by stupid java on small-endien
	    // machines like intel or alpha
	    clientId_ = swapShort (clientId_);
	}
	    
	if (Config.debug == true)
	    System.out.println ("Get client id " + String.valueOf (clientId_));
	if (clientId_ < 1234) 
	    throw new ConnectException ("Client Daemon rejected connection");
    }

    private void sendMessage (cdevMessage msg) throws IOException
    {
	int header;
	int msgsize;
	int totalsize;
	
	cdevDataOutputStream stream = new cdevDataOutputStream(output_);
	msgsize = msg.streamSize ();
	totalsize = msgsize +  cdevDataOutputStream.streamSize (msgsize);

	header = Packet.setMsgHeader (msgsize);
	
	try {
	    stream.write (header);
	}catch (IOException e) {
	    throw e;
	}

	try {
	    msg.streamOut (stream);
	}catch (IOException e) {
	    throw e;
	}
	try {
	    stream.flush();
	}catch (IOException e) {
	    throw e;
	}
    }

	    

    private void sendConnectionInfo () throws IOException
    {
	cdevTagTable t = cdevTagTable.tagTable();
	int[]    tagval = null;
	String[] tagstr = null;
	int      numtags = 0;

	try {
	    numtags = t.numTags ();
	}catch (IllegalStateException e) {
	    System.err.println ("Internal tag table error");
	    throw new IOException ("Tag table error");
	}
	tagval = new int[numtags];
	tagstr = new String[numtags];

	numtags = t.readTagTable (tagval, tagstr, numtags);
	
	// tagmap
	cdevData tagmap = new cdevData ();
	tagmap.insert (1, tagval);
	tagmap.insert (2, tagstr);
	
	// context
	cdevData cxt = new cdevData();

	cxt.insert (Config.HOST_TAG, hostname_);
	if (pid_ != 0)
	    cxt.insert (Config.PID_TAG, pid_);
	cxt.insert (Config.NAME_TAG, progname_);
	cxt.insert (Config.USER_TAG, username_);

	cdevMessage msg = new cdevMessage (clientId_, 0, 0, 0, 0,
					   Config.CMLOG_CONN_INFO, null, null,
					   cxt, null, tagmap);
	try {
	    sendMessage (msg);
	}catch (IOException e) {
	    throw e;
	}
    }
    
    private int findLocalUdpPort ()
    {
	String env = null;
	int    udpPort = 0;
	try {
	    env = System.getProperty("CMLOG_CLNT_PORT");
	}catch (SecurityException e) {
	    env = null;
	}
	if (env != null) 
	    udpPort = Integer.valueOf (env).intValue();
	else
	    udpPort = Config.CMLOG_CLNT_PORT;

	if (Config.debug == true)
	    System.out.println ("Client: local daemon UDP port is " + String.valueOf (udpPort));

	return udpPort;

    }

    /**
     * Connect to a cmlog system
     */
    public synchronized void connect () throws IOException, ConnectException
    {
	if (connected_ != true) {
	    // contact udp port of cmlogClientD
	    try {
		udpContact (findLocalUdpPort());
	    }catch (UnknownHostException e) {
		System.err.println (e);
		throw new ConnectException ("Unknown local host name");
	    }catch (ConnectException ce) {
		throw ce;
	    }catch (IOException ue) {
		throw ue;
	    }
	    
	    // open a name pipe to the client daemon
	    String pipename = getPipeName ();
	    if (pipename == null) 
		throw new ConnectException ("No client daemon pipe found");

	    try {
		output_ = new FileOutputStream (pipename);
	    }catch (FileNotFoundException e) {
		throw new ConnectException ("Client: cannot find pipe "+pipename);
	    }

	    try {
		sendConnectionInfo ();
	    }catch (IOException e) {
		throw e;
	    }
	    connected_ = true;
	}
    }

    /**
     * Check whether client is connected
     */
    public synchronized boolean connected ()
    {
	return connected_;
    }

    /**
     * Send data to cmlog system
     */
    public void postData (cdevData data) throws IOException
    {
	if (connected_) {
	    boolean dotimestamp = false;
	    cdevDataEntry entry = data.get (Config.KEY_TAG);

	    if (entry != null) {
		short type = entry.getType ();
		if (type >= cdevDataTypes.CDEV_INVALID ||
		    type <= cdevDataTypes.CDEV_UINT16 ||
		    type == cdevDataTypes.CDEV_STRING)
		    dotimestamp = true;
	    }
	    else
		dotimestamp = true;

	    if (dotimestamp == true) {
		Date currtime = new Date();
		Long csec = new Long(currtime.getTime ()/1000);
		double timestamp = csec.doubleValue();
		data.insert (Config.KEY_TAG, timestamp);
	    }

	    cdevMessage msg = new cdevMessage (clientId_, 0, 0, 0, 0,
					       Config.CMLOG_ADD_DATA, 
					       null, null,
					       data, null, null);

	    try {
		sendMessage (msg);
	    }catch (IOException e) {
		throw e;
	    }
	}
	else {
	    System.out.println (data);
	}
    }

    private void sendDisconnInfo () throws IOException
    {
	cdevMessage msg = new cdevMessage (clientId_, 0, 0, 0, 0,
					   Config.CMLOG_CLOSE_CONN, null, null,
					   null, null, null);
	try {
	    sendMessage (msg);
	}catch (IOException e) {
	    throw e;
	}
    }

    /**
     * Close connection to the system
     */
    public synchronized void disconnect () throws IOException
    {
	if (connected_) {
	    try {
		sendDisconnInfo ();
	    }catch (IOException e) {
		throw e;
	    }
	    
	    connected_ = false;
	    
	    output_.flush ();
	    output_.close ();
	    output_ = null;
	}
    }

  
    // send tag map information
    private void sendTagmapInfo (int tag, String name)
	throws IOException
    {
	cdevData tagmap = new cdevData ();

	tagmap.insert (1, tag);
	tagmap.insert (2, name);
	cdevMessage msg = new cdevMessage ((short)0, 0, 0, 0, 0,
					   Config.CMLOG_TAGMAP,
					   null, null, null, null, tagmap);

	try {
	    sendMessage (msg);
	}catch (IOException e) {
	    throw e;
	}
    }
    
    /**
     * Enable tag map notification to the server
     */
    public void enableTagmapNotify ()
    {
	tagmapNotify_ = true;
    }

    /**
     * Disable tag map notification to the server
     */
    public void disableTagmapNotify ()
    {
	tagmapNotify_ = false;
    }

    /**
     * Check whether tagmap notification is enabled
     */
    public boolean tagmapNotifyEnabled ()
    {
	return tagmapNotify_;
    }


}
