#ifndef _CMLOG_CDEV_PACKET_VERSION_1_H_
#define _CMLOG_CDEV_PACKET_VERSION_1_H_

#include <cmlog_cdevPacket.h>

// *****************************************************************************
// * struct cmlog_cdevPacketMap1 :
// *	This is the first 32 bit integer that is passed through the socket.  It
// *	contains a bitmap that identifies the version of the packet, as well as
// *	a bit field that indicates the contents of the packet.  This bitfield
// *	simplifies the process of decoding the packet and results in a smaller
// *	transmission size.
// *****************************************************************************
typedef union
	{
	unsigned rawData;
	struct  {
		unsigned version             : 16;
		unsigned clientIDSet         : 1;
		unsigned transIndexSet       : 1;
		unsigned cancelTransIndexSet : 1;
		unsigned localDataIndexSet   : 1;
		unsigned foreignDataIndexSet : 1;
		unsigned operationCodeSet    : 1;
		unsigned deviceListSet       : 1;
		unsigned messageSet          : 1;
		unsigned dataSet             : 1;
		unsigned contextSet          : 1;
		unsigned tagMapSet           : 1;
		unsigned pad                 : 5;
		} value;
	} cmlog_cdevPacketMap1;

#define _CLIP_VERSION                        0xffff0000
#define _CLIP_IDSET                          0x00008000
#define _CLIP_TRINDEXSET                     0x00004000
#define _CLIP_CTRINDEXSET                    0x00002000
#define _CLIP_LOCALDINDEXSET                 0x00001000
#define _CLIP_FOREIGNDINDEXSET               0x00000800
#define _CLIP_OPCODESET                      0x00000400
#define _CLIP_DEVLISTSET                     0x00000200
#define _CLIP_MSGSET                         0x00000100
#define _CLIP_DATASET                        0x00000080
#define _CLIP_CONTEXTSET                     0x00000040
#define _CLIP_TAGMAPSET                      0x00000020

#define CLIP_CURRENT_VERSION                 0x00010000


// *****************************************************************************
// * The following elements are contained within a cmlog_cdevPacket1 binary object.
// * 
// * short     version          : This is the version of the cmlog_cdevPacket that is
// *                              being constructed.  The format and structure
// *                              of the remainder of the data is based on 
// *                              version 1.
// *
// * short     clientID         : This is the clientID that the server will use
// *                              to uniquely identify this client.
// *
// * unsigned  transIndex       : This is the index that will be used on the
// *                              client side to locate the transaction object
// *                              associated with this request.
// *
// *
// * unsigned  cancelTransIndex : This is the transaction index of a transaction
// *                              that should be canceled.  This field is
// *                              typically used to terminate a monitor.
// *
// * unsigned  localDataIndex   : This is the index that will be used on the
// *                              ~packet senders~ side to identify a list
// *                              of devices.
// *
// * unsigned  foreignDataIndex : This is the index that will be used on the
// *                              ~packet receivers~ side to identify a list
// *                              of devices.
// *
// *
// * unsigned operationCode     : This is a user defined operation code that
// *                              may be used to reduce the size of the data
// *                              that is transmitted between the client and 
// *                              the server by using integer operation codes
// *                              rather than character strings.
// *
// * unsigned  deviceCount      : This is the number of device names that will
// *                              be contained in the following list.
// *
// * char   ** deviceList       : This is a list of character strings that 
// *                              identify cdevDevices.
// *
// * char   *  message          : This is the message text.
// *
// * cdevData  data             : This is a cdevData object containing the
// *                              data associated with the request or
// *                              the result of the request if this packet
// *                              is a response from a server.
// *
// * cdevData  context          : This is a cdevData object specifying the
// *                              context of the request.
// *
// * cdevData  tagMap           : This is a cdevData object that contains the
// *                              mapping of the cdevData tag table for the 
// *                              client.  By virtue of the fact that this table
// *                              cannot rely on any of the known tags to exist
// *                              of be the same... the following integer values
// *                              will be used.
// *
// *                              1)  An array of integers that identify the
// *                                  tag numbers.
// *
// *                              2)  An array of strings that identify the
// *                                  tag names.
// *
// *****************************************************************************
class cmlog_cdevPacket1 : public cmlog_cdevPacket
{
private:
	unsigned int   map;
	XDR_Reader     reader;
	
		
protected:
	typedef enum 
	{	GOTO_CLIENTID=1,
		GOTO_TRANSINDEX,
		GOTO_CANCELTRANSINDEX,
		GOTO_LOCALDATAINDEX,
		GOTO_FOREIGNDATAINDEX,
		GOTO_OPERATIONCODE,
		GOTO_DEVICELIST,
		GOTO_MESSAGE,
		GOTO_DATA,
		GOTO_CONTEXT,
		GOTO_TAGMAP} POSITION_FLAG;

	// *********************************************************************
	// * cmlog_cdevPacket1::setBinaryPosition :
	// *	This method will set the position of the binary stream within an 
	// *	XDR_Reader to the specified data object.
	// *	This method returns 0 on success, or -1 if the data object could 
	// *	not be selected.
	// *********************************************************************
	int setBinaryPosition (XDR_Reader & reader, POSITION_FLAG flag );

public:
	cmlog_cdevPacket1 ( cmlog_cdevPacket1 &packet );
	cmlog_cdevPacket1 ( short            clientID         = -1, 
		      unsigned         transIndex       = 0, 
                      unsigned         cancelTransIndex = 0,
		      unsigned         localDataIndex   = 0, 
		      unsigned         foreignDataIndex = 0,
		      unsigned         operationCode    = 0,
		      unsigned         deviceCount      = 0,
		      char **          deviceList       = NULL,
		      char *           message          = NULL,
		      cdevData *       data             = NULL,
		      cdevData *       context          = NULL,
		      cdevData *       tagMap           = NULL );	
	
	virtual ~cmlog_cdevPacket1 ( void );
	
	// *********************************************************************
	// * The streamIn function will populate the cmlog_cdevPacket using the data
	// * that is specified in the stream.  The stream remains the property
	// * of the caller, who must delete it when it is no longer needed.
	// *********************************************************************
	virtual int  streamIn  ( char * stream, unsigned len );
	
	// *********************************************************************
	// * attachData :
	// *	This method allows the caller to assign a preallocated binary
	// *	buffer to this object.  This prevents the cmlog_cdevPacket1 class
	// *	from having to allocate the data.
	// *********************************************************************
	int attachData ( char * stream, unsigned len );

	// *********************************************************************
	// * Performs a diagnostic dump of the object.
	// *********************************************************************
	virtual void asciiDump ( FILE * fp = stdout );

	// *********************************************************************
	// * The set mechanism: 
	// *	Allows the caller to dynamically change the contents of the
	// *	binary structure.
	// *********************************************************************
	void set ( short            clientID         = -1, 
		   unsigned         transIndex       = 0,
		   unsigned         cancelTransIndex = 0, 
		   unsigned         localDataIndex   = 0, 
		   unsigned         foreignDataIndex = 0,
		   unsigned         operationCode    = 0,
		   unsigned         deviceCount      = 0,
		   char **          deviceList       = NULL,
		   char *           message          = NULL,
		   cdevData *       data             = NULL,
		   cdevData *       context          = NULL,
		   cdevData *       tagMap           = NULL );	

	// *********************************************************************
	// * The set mechanism: 
	// *	Allows the caller to dynamically change the contents of the
	// *	binary structure to a preallocated buffer
	// *********************************************************************
	int   set ( char*            buffer,
	           unsigned           size,
                   short            clientID         = -1, 
		   unsigned         transIndex       = 0,
		   unsigned         cancelTransIndex = 0, 
		   unsigned         localDataIndex   = 0, 
		   unsigned         foreignDataIndex = 0,
		   unsigned         operationCode    = 0,
		   unsigned         deviceCount      = 0,
		   char **          deviceList       = NULL,
		   char *           message          = NULL,
		   cdevData *       data             = NULL,
		   cdevData *       context          = NULL,
		   cdevData *       tagMap           = NULL );	

	// *********************************************************************
	// * The check length mechanism: 
	// *	Allows the caller to dynamically change the contents of the
	// *	binary structure without allocating buffer
	// *********************************************************************
	static unsigned checkLength (short            clientID         = -1, 
	  unsigned         transIndex       = 0,
	  unsigned         cancelTransIndex = 0, 
	  unsigned         localDataIndex   = 0, 
	  unsigned         foreignDataIndex = 0,
	  unsigned         operationCode    = 0,
	  unsigned         deviceCount      = 0,
	  char **          deviceList       = NULL,
	  char *           message          = NULL,
	  cdevData *       data             = NULL,
	  cdevData *       context          = NULL,
	  cdevData *       tagMap           = NULL );	


	// *********************************************************************
	// * The get mechanisms:
	// *	The pointers that are retrieved by these functions are the
	// *	actual pointers that are used within the class.  They should not
	// *	be altered by the caller.  The caller may alter the contents of
	// *	these objects by using the set mechanisms.
	// *********************************************************************
	virtual int getVersion  ( short & version );
	int getClientID         ( short & clientID );
	int getTransIndex       ( unsigned & transIndex );
	int getCancelTransIndex ( unsigned & cancelTransIndex );
	int getLocalDataIndex   ( unsigned & localDataIndex );
	int getForeignDataIndex ( unsigned & foreignDataIndex );
	int getOperationCode    ( unsigned & operationCode );
	int getDeviceList       ( char ** & deviceList, unsigned & deviceCount );
	int getMessage          ( char * & message );
	int getData             ( cdevData & data );
	int getContext          ( cdevData & context );
	int getTagMap           ( cdevData & tagMap );
	int get                 ( short    & clientID, 
				  unsigned & transIndex,
				  unsigned & cancelTransIndex,
				  unsigned & localDataIndex,
				  unsigned & foreignDataIndex,
				  unsigned & operationCode,
				  unsigned & deviceCount,
				  char **  & deviceList,
				  char *   & message,
				  cdevData & data,
				  cdevData & context,
				  cdevData & tagMap);

	// *********************************************************************
	// * The has mechanisms:
	// *	This collection of functions allows the caller to determine if
	// *	certain components of the packet are present.
	// *********************************************************************
	int hasClientID         ( void ) { return map & _CLIP_IDSET; }
	int hasTransIndex       ( void ) { return map & _CLIP_TRINDEXSET;}
	int hasCancelTransIndex ( void ) { return map & _CLIP_CTRINDEXSET; }
	int hasLocalDataIndex   ( void ) { return map & _CLIP_LOCALDINDEXSET; }
	int hasForeignDataIndex ( void ) { return map & _CLIP_FOREIGNDINDEXSET; }
	int hasOperationCode    ( void ) { return map & _CLIP_OPCODESET; }
	int hasDeviceList       ( void ) { return map & _CLIP_DEVLISTSET; }
	int hasMessage          ( void ) { return map & _CLIP_MSGSET; }
	int hasData             ( void ) { return map & _CLIP_DATASET; }
	int hasContext          ( void ) { return map & _CLIP_CONTEXTSET; }
	int hasTagMap           ( void ) { return map & _CLIP_TAGMAPSET;}

        int clipVersionNum (unsigned int a) {return ( ((a & _CLIP_VERSION) >> 16) & 0x0000ffff);}
};

#endif /* CDEV_PACKET_VERSION_1_H_ */
