//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
//-----------------------------------------------------------------------------
// 
// Description:
//	 Single Linked List for pointers void *
//
//       Note: remove and clean operations on the list
//             will only remove link nodes without removal of
//             the content inside the nodes. It is callers' 
//             responsiblity to clean up those contents
//
//       This is unsafe C++ practice, use this list at you own risk
//       
//       Reason for this list: It is very difficult to instantiate
//       a template class in a stand alone shared library
//	
// Author:  Jie Chen
//       CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogSlist.h,v $
//   Revision 1.1.1.1  1999/09/07 15:29:10  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:27:32  bickley
// Added cmlog to application development system.
//
//
//
#ifndef _CMLOG_SLIST_H
#define _CMLOG_SLIST_H

#include <cmlogConfig.h>

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
#include <cpSynch.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>


typedef void*  cmlogSlistItem;

//======================================================================
//	class cmlogSlist
//        Single Linked List for void* pointer
//======================================================================
class cmlogSlistLink;
class cmlogSlistIterator;
class cmlogSlistCursor;
class cmlogStrHash;

class cmlogSlist
{
public:
  // constructors
  cmlogSlist          (void);
  cmlogSlist          (const cmlogSlist & source);
  virtual ~cmlogSlist (void);

  // operations

  // add list item to the beginning of the list
  virtual void        add               (cmlogSlistItem value);
  // add item to the end of list
  virtual void        addToEnd          (cmlogSlistItem value);

  // remove a list item from the list
  // return 0: nothing to remove
  // return 1: remove success
  virtual int         remove            (cmlogSlistItem value);

  // clean up the list. 
  virtual void        deleteAllValues   (void);

  // return first element of the list
  virtual cmlogSlistItem  firstElement  (void);

  // return last element of the list
  virtual cmlogSlistItem    lastElement (void);

  // duplicate ths whole list
  virtual cmlogSlist*       duplicate   (void);
  
  // check whether this list contains a particular item
  // return 1: yes. return 0: no
  virtual int              includes     (cmlogSlistItem value);

  // Is list empty
  // return 1: yes. return 0: no
  virtual int              isEmpty      (void);

  // remove first element of the list
  virtual void             removeFirst  (void);

  // return number of elements inside the list
  virtual int              count        (void);

protected:
  // data field
  cmlogSlistLink*           ptrToFirstLink;

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutex                   lock_;
#endif

  // some internal functions which must be called with lock held
  virtual void        add_i              (cmlogSlistItem value); 
  virtual int         isEmpty_i          (void) const;
  virtual void        deleteAllValues_i  (void);
  virtual void        removeFirst_i      (void);
  virtual cmlogSlistItem  firstElement_i (void);
  virtual cmlogSlistItem  lastElement_i (void);

  // friends
  friend class  cmlogSlistIterator;
  // cannot modify list in anyways
  friend class  cmlogSlistCursor;
  friend class  cmlogStrHash;

private:
  // deny access to assignment operation
  cmlogSlist&   operator = (const cmlogSlist& list);
};

//======================================================================
//	class cmlogSlistLink
//        Single linked list link node
//======================================================================
class cmlogSlistLink
{
public:
  // constructor
  cmlogSlistLink         (cmlogSlistItem linkValue, cmlogSlistLink * nextPtr);

  // insert a new element following the current value
  cmlogSlistLink* insert (cmlogSlistItem val);
  // return data value
  cmlogSlistItem  data   (void);

  // return next data item
  cmlogSlistLink* next   (void);


private:

  // duplicate
  cmlogSlistLink* duplicate (void);

  // data areas
  cmlogSlistItem      value;
  cmlogSlistLink*     ptrToNextLink;

  // friends
  friend class       cmlogSlist;
  friend class       cmlogSlistIterator;
  friend class       cmlogSlistCursor;
};

//===================================================================
//	class cmlogSlistIterator
//		implements iterator protocol for linked lists
//		also permits removal and addition of elements
//===================================================================

class cmlogStrHashIterator;

class cmlogSlistIterator
{
public:
  // constructor
  cmlogSlistIterator        (cmlogSlist& aList);

  // iterator protocol
  virtual int              init (void);
  virtual cmlogSlistItem   operator () (void);
  virtual int              operator !  (void);
  virtual int              operator ++ (void);
  virtual void             operator =  (cmlogSlistItem value);

  // new methods specific to list iterators

  // remove current item pointed by the iterator from the list
  void                     removeCurrent(void);
  
  // add an item to the list before the position pointed by the iterator
  void                     addBefore(cmlogSlistItem  newValue);

  // add an item to the list after the position pointed by the iterator
  void                     addAfter(cmlogSlistItem   newValue);

  // search an item and move the iterator to that position
  int                      searchSame(cmlogSlistItem &value);

protected:
  // some internal functions which must be called with lock held
  virtual int              init_i (void);
  // move cursor forward
  virtual int              forward_i (void);
  // internal remove current
  void                     removeCurrent_i (void);
  // data areas
  cmlogSlistLink *    currentLink;
  cmlogSlistLink *    previousLink;
  cmlogSlist&         theList;
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutex             lock_;
#endif

private:
  // friend class
  friend class cmlogStrHashIterator;

  // deny access of copy and assignment
  cmlogSlistIterator (const cmlogSlistIterator& ir);
  cmlogSlistIterator& operator = (const cmlogSlistIterator& ir);
};

//===================================================================
//	class cmlogSlistCursor
//		implements cursor protocol for linked lists
//===================================================================
class cmlogSlistCursor
{
public:
  // constructor
  cmlogSlistCursor           (const cmlogSlist& aList);

  // iterator protocol
  virtual int               init        (void);
  virtual cmlogSlistItem    operator () (void);
  virtual int               operator !  (void);
  virtual int               operator ++ (void);
  virtual void              operator =  (cmlogSlistItem value);

  int                       searchSame  (cmlogSlistItem &value);

protected:
  // internal functions which must be called with lock 
  virtual int               init_i      (void);  
  // data areas
  cmlogSlistLink *          currentLink;
  cmlogSlistLink *          previousLink;
  const cmlogSlist&         theList;
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutex             lock_;
#endif

private:
  // deny access to copy and assignment operations
  cmlogSlistCursor (const cmlogSlistCursor &);
  cmlogSlistCursor& operator = (const cmlogSlistCursor &);
};

//======================================================================
//      class doubleEndedList
//           Not only keeps a pointer to first node
//           but also keeps a pointer to last node
//======================================================================
class cmlogDoubleEndedSlist: public cmlogSlist{
public:
  //constructor
  cmlogDoubleEndedSlist   (void);
  cmlogDoubleEndedSlist   (const cmlogDoubleEndedSlist &v);
  
  // override the following methods from the cmlogSlist
  virtual void      add             (cmlogSlistItem value);
  virtual void      deleteAllValues (void);
  virtual void      removeFirst     (void);

  // add new element to the end of the list
  virtual void      addToEnd        (cmlogSlistItem value);

protected:
  cmlogSlistLink    *ptrToLastLink;

private:
  // deny access to assignment operation
  cmlogDoubleEndedSlist& operator = (const cmlogDoubleEndedSlist);
};
#endif
