//-----------------------------------------------------------------------------
// Copyright (c) 1991,1992 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
//-----------------------------------------------------------------------------
// 
// Description:
//	 cmlogIntHash: cmlog hash table keyed by an integer
//                    Open Hash with buckets implemented by single linked lists
//    
//              Note: void *s are stored inside the table. This class
//                    will not manage these pointers. Callers have to
//                    manage these pointers
//
// Author: Jie Chen
//
// Revision History:
//   $Log: cmlogIntHash.h,v $
//   Revision 1.2  2001/07/25 14:26:36  chen
//   64 BIT Initial Port
//
//   Revision 1.1.1.1  1999/09/07 15:29:10  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:27:23  bickley
// Added cmlog to application development system.
//
//
//
//     
#ifndef _CMLOG_INT_HASH_H
#define _CMLOG_INT_HASH_H

#include <cmlogSlist.h>

//======================================================================
//  One simple integer hash function
//           This hash function is used in callback hashing
//           It returns integer value between 0 to table size
//======================================================================

typedef int   cmlogIntKeyItem;
typedef void* cmlogHashItem;
//======================================================================
//	class cmlogIntHash
//		collection of buckets indexed by hashed values
//======================================================================
class cmlogIntHashIterator;

class cmlogIntHash
{
public:
  // constructor, construct a table with entry max 
  cmlogIntHash (unsigned int max);
  // destructor
  virtual ~cmlogIntHash (void);

  // operations

  // is the table empty: return 1: yes. return 0: no
  virtual int   isEmpty();

  // clear the elements of the set
  virtual void  deleteAllValues();

  // add an element to the collection
  virtual void  add (cmlogIntKeyItem key, cmlogHashItem ele);

  // test to see whether this hash table includes one particular element
  virtual int   find (cmlogIntKeyItem key, cmlogHashItem ele) const;

  // remove an element. return 0: ele is not inside table. return 1: success
  virtual int   remove (cmlogIntKeyItem key, cmlogHashItem ele);

  // return a reference to a particular bucket according to the key
  cmlogSlist&  bucketRef (cmlogIntKeyItem key);

 protected:
  friend class cmlogIntHashIterator;

  // the actual table itself is a vector of buckets
  const unsigned int    tablesize;
  cmlogSlist*            buckets;

  // convert key into unsigned integer value in range
  unsigned int          hash(const cmlogIntKeyItem key) const;

};

//======================================================================
//	class cmlogIntHashIterator
//		iterator protocol for hash tables
//======================================================================
class cmlogIntHashIterator
{
public:
  // constructor and destructor
  cmlogIntHashIterator  (cmlogIntHash& v);
  ~cmlogIntHashIterator (void);

  // iterator protocol 
  int             init       (void);
  cmlogHashItem    operator ()(void);
  int             operator ! (void);
  int             operator ++(void);
  void            operator = (cmlogHashItem value);

protected:
  cmlogIntHash&    base;
  unsigned int    currentIndex;
  // Single iterator within a bucket
  cmlogSlistIterator*         itr;
  // getNextIterator used to set internal iterator pointer
  // return 1: got it. return 0: no more iterator
  int             getNextIterator (void);
};
#endif
