//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      CMLOG Browser Class
//
//      Unix environment variable CMLOG_PORT changes
//      broadcasting destination port number
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogBrowser.h,v $
//   Revision 1.3  2001/07/25 14:25:06  chen
//   64 BIT Initial Port
//
//   Revision 1.2  2000/06/20 18:44:14  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.1.1.1  1999/09/07 15:29:02  chen
//   CMLOG version 2.0
//
// Revision 1.2  1997/08/12  16:48:07  chen
// add one connect function
//
// Revision 1.1  1997/08/01  15:17:08  bickley
// Added cmlog to application development system.
//
//
//
#ifndef _CMLOG_BROWSER_H
#define _CMLOG_BROWSER_H

#include <cdevData.h>
#include <cmlogMsg.h>
#include <cmlogCallback.h>
#include <cmlogSlist.h>
#include <cmlogStrHash.h>
#include <cmlogIntHash.h>

class cmlogBrowser;

class cmlogBrowserTagCallback: public cdevTagTableCallback
{
public:
  cmlogBrowserTagCallback  (cmlogBrowser* browser);
  ~cmlogBrowserTagCallback (void);

  virtual void callback (int tag, char* name);

private:
  cmlogBrowser* browser_;

  // deny access to copy and assignment operator
  cmlogBrowserTagCallback (cmlogBrowserTagCallback& cbk);
  cmlogBrowserTagCallback& operator = (cmlogBrowserTagCallback& cbk);
};

class cmlogBrowserLocker;

class cmlogBrowser
{
public:
  // constructor
  cmlogBrowser             (void);
  // destructor
  ~cmlogBrowser            (void);

  // operations
  // return CMLOG_SUCCESS for success, CMLOG_ERROR for failure

  // connect to logging server
  // return CMLOG_SUCCESS for successful connection
  // return CMLOG_ERROR   for failure
  int connect              (char* host, unsigned short port,
			    int connectionRetries = 3); 
  int connect              (unsigned short port, int connectionRetries = 3);
  int connect              (int connectionRetries = 3);
  
  // tcp connection
  int tcpConnect           (char* host, unsigned short tcp_port);
  int connected            (void) const;
  // disconnect
  int disconnect           (void);
  // register disconnect callback
  int disconnectCallback   (cmlogBrCallback cbk, void* arg);

  // send query information
  int queryCallback        (char* message, cdevData& data, 
			    cmlogBrCallback cbk, void* arg);

  // get file descriptor
  int getFd                (void) const;

  // pend io on this connection
  // Wait until outstanding events occur
  // seconds = 0.0 polling
  int pendIO               (double seconds);
  
  // wait on this connection forever
  int pendIO               (void);

  // handle low-level io: return 0: success, -1: bad io
  int            read_input      (void);

  // find out whether a callback is registered here
  int    hasMonitorCallback (char* attribute);

  int    removeMonitorCallback (char* attr, 
				cmlogBrCallback callback, void* arg);

  // clean up everything
  void   cleanupCallbacks (void);

  // handle low-level io: return 0: success, -1: bad io
  int            handle_input      (int fd);
  // handle low-level close when server is dead
  int            handle_close      (int fd);


  // disable/enable tagmap notify
  void   enableTagmapNotify (void);
  void   disableTagmapNotify (void);
  int    tagmapNotify (void);

  // return server information
  char*  serverHost (void) const {return serverHost_;}
  unsigned short serverUdpPort (void) const;
  unsigned short serverTcpPort (void) const {return serverPort_;}

protected:
  // call all disconnection notification callbacks
  void callAllDiscCbks (void);

  // send browser information to server
  int  sendBrowserInfo (void);

  // all these callback processing functions
  int    queryCbkFromServer        (int status, unsigned cbkid,
				    cmlogPacket& packet);
  int    monitorCbkFromServer      (int status, char* attribute,
				    unsigned cbkid, cmlogPacket& packet);
  // process data come back from server
  int    processData               (cmlogPacket& packet);

  // send tag map information to the server
  int    sendTagMapInfo (int tagvalue, char* tagname);

  // check whether a callback + arg associated with an attribute is
  // stored in the monitor callback table
  cmlogCallback* monitorCallback (char* attr, 
				  cmlogBrCallback callback, void* arg);

  // check whether a calback + arg is stored in the query callback table
  cmlogCallback* hasQueryCallback (cmlogBrCallback cbk, void* arg);

private:
  // flag of connection
  int connected_;

  // udp socket to server port to find real tcp port
  int                udp_fd_;
  // tcp socket to server
  int                tcp_fd_;

  // server information (tcp)
  unsigned short     serverPort_;          
  char*              serverHost_;

  // call tagmap callback or not flag
  int                tagmapNotify_;

  // lock for browser object to prevent from this object being called 
  // recursively
  int                lock_;
  void               lock   (void);
  void               unlock (void);

  // query id
  unsigned           qid_;
  // event id
  unsigned           evid_;
  
  // disconnected callback list
  cmlogSlist discCbkList_;
  // all query callback
  cmlogIntHash queryCbkTable_;
  // all monitorOn callback
  cmlogStrHash monitorCbkTable_;

  // cdev tag table callback object
  cmlogBrowserTagCallback* tagCbk_;

  // friend class
  friend class cmlogBrowserLocker;
  friend class cmlogBrowserTagCallback;

  // deny access to copy and assignment operations
  cmlogBrowser (const cmlogBrowser& b);
  cmlogBrowser& operator = (const cmlogBrowser& b);
};

  // format of message send to the server in the cdevMessage
  // short      clientID;             -> client id from logging side (0)
  // unsigned   transIndex;           -> browser request index       
  // unsigned   cancelTransIndex;     -> not used
  // unsigned   localDataIndex;       -> socket id (pointer)  (server assign)
  // unsigned   foreignDataIndex;     -> callback function + arg (pointer)
  //                                  -> only in the first one in the packet
  // unsigned   operationCode;        -> operation code
  // unsigned   deviceCount;          -> not used
  // char **    deviceList;           -> not used
  // char *     message;              -> "query", "stopQuery",
  //                                     "get", "monitorOn", 
  //                                     "monitorOff"
  // cdevData * data;                 -> extra for query (outgoing)
  //                                  -> incoming result with status tag denote
  //                                  -> error condition (only first message)
  // cdevData * context;              -> not used
  // cdevData * tagMap;               -> not used

class cmlogBrowserLocker
{
public:
  cmlogBrowserLocker  (cmlogBrowser* b);
  ~cmlogBrowserLocker (void);

private:
  cmlogBrowser* b_;

  // denay access to copy and assignment operations
  cmlogBrowserLocker (const cmlogBrowserLocker& b);
  cmlogBrowserLocker& operator = (const cmlogBrowserLocker& b);
};
#endif
