#include <cmlog_cdevPacket1.h>

// *****************************************************************************
// * cmlog_cdevPacket1::cmlog_cdevPacket1 :
// *	Copy constructaor for the class.  Sets all of the variables to the
// *	values stored in the referenced objects.
// *****************************************************************************
cmlog_cdevPacket1::cmlog_cdevPacket1 ( cmlog_cdevPacket1 & packet )
	: cmlog_cdevPacket()
	{
	streamIn(packet.binary, packet.binaryLen);
	}

		
// *****************************************************************************
// * cmlog_cdevPacket1::cmlog_cdevPacket1 :
// *	This is the default and parameterized constructor. Allows the caller to 
// *    directly specify each of the components.
// *****************************************************************************
cmlog_cdevPacket1::cmlog_cdevPacket1 (short            clientID, 
			  unsigned         transIndex, 
			  unsigned         cancelTransIndex,
			  unsigned         localDataIndex, 
			  unsigned         foreignDataIndex,
			  unsigned         operationCode,
			  unsigned         deviceCount,
			  char **          deviceList,
			  char *           message,
			  cdevData *       data,
			  cdevData *       context,
			  cdevData *       tagMap)
	: cmlog_cdevPacket()
	{
	set(clientID, transIndex, cancelTransIndex, localDataIndex, 
	    foreignDataIndex, operationCode, deviceCount, deviceList, 
	    message, data, context, tagMap);
	}


// *****************************************************************************
// * cmlog_cdevPacket1::~cmlog_cdevPacket1 :
// *	This is the destructor for the class.  
// *****************************************************************************
cmlog_cdevPacket1::~cmlog_cdevPacket1 ( void )
	{
	// *********************************************************************
	// * Execute a detach on the embedded data to ensure that it is not
	// * deleted locally.
	// *********************************************************************
	reader.detachData();
	}
	

// *****************************************************************************
// * cmlog_cdevPacket1::streamIn :
// *	The streamIn function will read a memory buffer provided by the caller 
// *	and then populate the class with these values. This stream remains the 
// *	property of the caller and he is responsible for deleting it.
// *****************************************************************************
int cmlog_cdevPacket1::streamIn ( char *stream, unsigned len )
	{
	int    result = -1;
			
	// *********************************************************************
	// * Delete the binary data buffer and set its length to 0.
	// *********************************************************************
	if(binary!=NULL) 
		{
		delete binary;
		binary = NULL;
		}
	binaryLen = 0;

	// *********************************************************************
	// * Attach the data to the XDR_Reader class so that its components
	// * can be extracted.
	// *********************************************************************
	reader.attachData(stream, len);
	
	// *********************************************************************
	// * Clear the packet map prior to reading data from the stream.
	// *********************************************************************
	map = 0;
	
	// *********************************************************************
	// * Read the packet map from the stream.  This object contains a bit
	// * field that identifies the version of the packet and tells what 
	// * other components are embedded in the binary stream.
	// *********************************************************************
	result = !reader.get(map);
	
	// *********************************************************************
	// * Before processing anything else, the method must ensure that this
	// * packet is of the correct version.  Version 1 for this object.
	// *
	// * Note that reading each field is contingent on the prior fields 
	// * being read correctly.
	// *********************************************************************
	if(!result && clipVersionNum (map) == 1)
		{		
		binaryLen = len;
		binary = new char[binaryLen];
		memcpy(binary, stream, binaryLen);
		}
	else result = -1;
	
	// *********************************************************************
	// * Detach the binary buffer from the XDR_Reader to prevent it from 
	// * being deleted when the XDR_Reader class is destroyed.
	// *********************************************************************
	reader.detachData();
	
	return result;
	}


// *********************************************************************
// * cmlog_cdevPacket1::attachData :
// *	This method allows the caller to assign a preallocated binary
// *	buffer to this object.  This prevents the cmlog_cdevPacket1 class
// *	from having to allocate the data.
// *********************************************************************
int cmlog_cdevPacket1::attachData ( char * stream, unsigned len )
	{
	int    result = -1;
			
	// *********************************************************************
	// * Delete the binary data buffer and set its length to 0.
	// *********************************************************************
	if(binary!=NULL) 
		{
		delete binary;
		binary = NULL;
		}
	binaryLen = 0;

	// *********************************************************************
	// * Attach the data to the XDR_Reader class so that its components
	// * can be extracted.
	// *********************************************************************
	reader.attachData(stream, len);
	
	// *********************************************************************
	// * Clear the packet map prior to reading data from the stream.
	// *********************************************************************
	map  = 0;
	
	// *********************************************************************
	// * Read the packet map from the stream.  This object contains a bit
	// * field that identifies the version of the packet and tells what 
	// * other components are embedded in the binary stream.
	// *********************************************************************
	result = !reader.get(map);
	
	// *********************************************************************
	// * Before processing anything else, the method must ensure that this
	// * packet is of the correct version.  Version 1 for this object.
	// *
	// * Note that reading each field is contingent on the prior fields 
	// * being read correctly.
	// *********************************************************************
	if(!result && clipVersionNum (map) == 1)
		{		
		binaryLen = len;
		binary = stream;
		}
	else result = -1;
	
	// *********************************************************************
	// * Detach the binary buffer from the XDR_Reader to prevent it from 
	// * being deleted when the XDR_Reader class is destroyed.
	// *********************************************************************
	reader.detachData();
	
	return result;
	}


// *****************************************************************************
// * cmlog_cdevPacket1::asciiDump :
// *	This method performs a diagnostic dump of the text representation of the
// *	packets contents.
// *****************************************************************************
void cmlog_cdevPacket1::asciiDump ( FILE * fp )
	{
	short clientID;
	unsigned transIndex, cancelTransIndex, localDataIndex;
	unsigned foreignDataIndex, operationCode, deviceCount;
	char **  deviceList;
	char *   message;
	cdevData temp;
	int i;

	getClientID(clientID);
	getTransIndex(transIndex);
	getCancelTransIndex(cancelTransIndex);
	getLocalDataIndex(localDataIndex);
	getForeignDataIndex(foreignDataIndex);
	getOperationCode (operationCode);
	getDeviceList(deviceList, deviceCount);
	getMessage(message);
	
	fprintf(fp, "----------------------------------------------------------\n");
	fprintf(fp, "             Diagnostic Dump of CDEV Packet\n");
	fprintf(fp, "----------------------------------------------------------\n\n");
	fprintf(fp, "Packet Version     : 1\n");
	fprintf(fp, "Binary Stream Size : %i\n", binaryLen);
	fprintf(fp, "Client Identifier  : %s\n", clientID==-1?"(not specified)":ltoa(clientID));
	fprintf(fp, "Transaction Index  : %s\n", transIndex==0?"(not specified)":ultoa(transIndex));
	fprintf(fp, "Cancel Transaction : %s\n", cancelTransIndex==0?"(not specified)":ultoa(cancelTransIndex));
	fprintf(fp, "Local Data Index   : %s\n", localDataIndex==0?"(not specified)":ultoa(localDataIndex));
	fprintf(fp, "Foreign Data Index : %s\n", foreignDataIndex==0?"(not specified)":ultoa(foreignDataIndex));
	fprintf(fp, "Operation Code     : %s\n", operationCode==0?"(not specified)":ultoa(operationCode));
	fprintf(fp, "Number of Devices  : %s\n", deviceCount==0?"(not specified)":ultoa(deviceCount));
	fprintf(fp, "List of Devices    : ");
	
	if(deviceCount==0) fprintf(fp, "(not specified)\n");
	else fprintf(fp, "%s\n", deviceList[0]);
	
	for(i=1; i<deviceCount; i++) 
		{
		fprintf(fp, "                     %s\n", deviceList[i]);
		}

	fprintf(fp, "Message String     : %s\n", message==NULL?"(not specified)":message);

	// *********************************************************************
	// * The deviceList and message string must be deleted.
	// *********************************************************************
	if(deviceList!=NULL)
		{
		for(i=0; i<deviceCount; i++) delete deviceList[i];
		delete deviceList;
		}		
	if(message != NULL) delete message;
	
	int empty;
	
	fprintf(fp, "Data Object        : %s", (empty=getData(temp))==0?"\n":"(not specified)\n");
	if(!empty) temp.asciiDump(fp);

	fprintf(fp, "Context Object     : %s", (empty=getContext(temp))==0?"\n":"(not specified)\n");
	if(!empty) temp.asciiDump(fp);

	fprintf(fp, "Tag Map            : %s", (empty=getTagMap(temp))==0?"\n":"(not specified)\n");
	if(!empty)
		{
		unsigned  tagNumCnt  = 0;
		int  *  tagNums    = NULL;
		unsigned  tagNameCnt = 0;
		
		if(temp.getElems(1, &tagNumCnt)==CDEV_SUCCESS  && 
		   temp.getElems(2, &tagNameCnt)==CDEV_SUCCESS &&
		   temp.getType (2)==CDEV_STRING               &&
		   tagNumCnt==tagNameCnt)
			{
			char ** tagNames   = NULL;
			char *  singleName = NULL;
	
			tagNums = new int[tagNumCnt];
			temp.get(1, tagNums);

			if(tagNameCnt==1)
				{
				tagNames = &singleName;
				temp.find(2, (void *&)singleName);
				}
			else temp.find(2, (void *&)tagNames);

			for(int i=0; i<tagNumCnt; i++)
				{
				fprintf(fp, "\t%-26.26s= %i\n", tagNames[i], tagNums[i]);
				} 
			delete tagNums;
			}
		}
	fprintf(fp, "----------------------------------------------------------\n");
	fprintf(fp, "            End of Diagnostic Dump of CDEV Packet\n");
	fprintf(fp, "----------------------------------------------------------\n\n");
	fflush(fp);
	}


// *****************************************************************************
// * cmlog_cdevPacket1::setBinaryPosition :
// *	This method will set the position of the binary stream within an 
// *	XDR_Reader to the specified data object.
// *	This method returns 0 on success, or -1 if the data object could not be
// *	selected.
// *****************************************************************************
int cmlog_cdevPacket1::setBinaryPosition (XDR_Reader & localrdr, POSITION_FLAG flag )
	{
	int slen;
	int result = -1;

	// *********************************************************************
	// * Set the position to the beginning of the data buffer.
	// *********************************************************************
	xdr_setpos(localrdr.xdr(), 0);
			
	// *********************************************************************
	// * Clear the packet map prior to reading data from the stream.
	// *********************************************************************
	map  = 0;
	
	// *********************************************************************
	// * Read the packet map from the stream.  This object contains a bit
	// * field that identifies the version of the packet and tells what 
	// * other components are embedded in the binary stream.
	// *********************************************************************
	result = !localrdr.get(map);
	
	// *********************************************************************
	// * Before processing anything else, the method must ensure that this
	// * packet is of the correct version.  Version 1 for this object.
	// *
	// * Note that reading each field is contingent on the prior fields 
	// * being read correctly.
	// *********************************************************************
	if(!result && clipVersionNum (map) ==1)
		{		
		// *************************************************************
		// * If the selected element is beyond the clientID, then 
		// * advance the buffer to the next item.
		// *************************************************************
		if(flag > GOTO_CLIENTID)
			{
			if(map & _CLIP_IDSET)
			    result=!xdr_setpos(
				localrdr.xdr(), 
				xdr_getpos(localrdr.xdr())+XDR_Sizeof((short)1));
			}
		// *************************************************************
		// * Otherwise... set the result to indicate the existence (or
		// * non-existence) of the clientID.
		// *************************************************************
		else if(flag==GOTO_CLIENTID && !(map & _CLIP_IDSET)) result=-1;



		// *************************************************************
		// * If the selected element is beyond the transIndex, then 
		// * advance the buffer to the next item.
		// *************************************************************
		if(flag > GOTO_TRANSINDEX)
			{
			if(!result && (map & _CLIP_TRINDEXSET))
			    result=!xdr_setpos(
				localrdr.xdr(), 
				xdr_getpos(localrdr.xdr())+XDR_Sizeof((unsigned)1));
			}
		// *************************************************************
		// * Otherwise... set the result to indicate the existence (or
		// * non-existence) of the transIndex.
		// *************************************************************
		else if(flag==GOTO_TRANSINDEX && ! (map & _CLIP_TRINDEXSET)) result=-1;
			
			

		// *************************************************************
		// * If the selected element is beyond the cancelTransIndex, 
		// * then advance the buffer to the next item.
		// *************************************************************
		if(flag > GOTO_CANCELTRANSINDEX)
			{
			if(!result && (map & _CLIP_CTRINDEXSET))
			    result=!xdr_setpos(
				localrdr.xdr(), 
				xdr_getpos(localrdr.xdr())+XDR_Sizeof((unsigned)1));
			}
		// *************************************************************
		// * Otherwise... set the result to indicate the existence (or
		// * non-existence) of the cancelTransIndex.
		// *************************************************************
		else if(flag==GOTO_CANCELTRANSINDEX && !(map & _CLIP_CTRINDEXSET)) result=-1;
			
			

		// *************************************************************
		// * If the selected element is beyond the localDataIndex, then 
		// * advance the buffer to the next item.
		// *************************************************************
		if(flag > GOTO_LOCALDATAINDEX)
			{
			if(!result && (map & _CLIP_LOCALDINDEXSET)) 
			    result=!xdr_setpos(
				localrdr.xdr(), 
				xdr_getpos(localrdr.xdr())+XDR_Sizeof((unsigned)1));
			}
		// *************************************************************
		// * Otherwise... set the result to indicate the existence (or
		// * non-existence) of the localDataIndex.
		// *************************************************************
		else if(flag==GOTO_LOCALDATAINDEX && !(map & _CLIP_LOCALDINDEXSET)) result=-1;
			
			

		// *************************************************************
		// * If the selected element is beyond the foreignDataIndex, then 
		// * advance the buffer to the next item.
		// *************************************************************
		if(flag > GOTO_FOREIGNDATAINDEX)
			{
			if(!result && (map & _CLIP_FOREIGNDINDEXSET)) 
			    result=!xdr_setpos(
				localrdr.xdr(), 
				xdr_getpos(localrdr.xdr())+XDR_Sizeof((unsigned)1));
			}
		// *************************************************************
		// * Otherwise... set the result to indicate the existence (or
		// * non-existence) of the foreignDataIndex.
		// *************************************************************
		else if(flag==GOTO_FOREIGNDATAINDEX && !(map & _CLIP_FOREIGNDINDEXSET)) result=-1;
			
		

		// *************************************************************
		// * If the selected element is beyond the operationCode, 
		// * then advance the buffer to the next item.
		// *************************************************************
		if(flag > GOTO_OPERATIONCODE)
			{
			if(!result && (map & _CLIP_OPCODESET))
				result=!xdr_setpos(
				localrdr.xdr(), 
				xdr_getpos(localrdr.xdr())+XDR_Sizeof((unsigned)1));
			}
		// *************************************************************
		// * Otherwise... set the result to indicate the existence (or
		// * non-existence) of the operationCode.
		// *************************************************************
		else if(flag==GOTO_OPERATIONCODE && !(map & _CLIP_OPCODESET)) result=-1;
			
			

		// *************************************************************
		// * If the selected element is beyond the deviceList, then 
		// * advance the buffer to the next item.
		// *************************************************************
		if(flag > GOTO_DEVICELIST)
			{
			if(!result && (map & _CLIP_DEVLISTSET))
				{
				int deviceCount;
				if((result = !localrdr.get(deviceCount))==0)
					{
					// *************************************
					// * Step over each string in the array.
					// *************************************
					for(int i=0; i<deviceCount && !result; i++)
						{
						slen   = localrdr.get_string_len();
						slen   = XDR_Sizeof((char *)NULL, slen);
						result = !xdr_setpos(
								localrdr.xdr(), 
								xdr_getpos(localrdr.xdr())+slen); 
						}
					}
				}
			}			
		// *************************************************************
		// * Otherwise... set the result to indicate the existence (or
		// * non-existence) of the deviceList.
		// *************************************************************
		else if(flag==GOTO_DEVICELIST && !(map & _CLIP_DEVLISTSET)) result=-1;
	


		// *************************************************************
		// * If the selected element is beyond the message, then 
		// * advance the buffer to the next item.
		// *************************************************************
		if(flag > GOTO_MESSAGE)
			{
			if(!result && (map & _CLIP_MSGSET))
				{
				slen   = localrdr.get_string_len();
				slen   = XDR_Sizeof((char *)NULL, slen);
				result = !xdr_setpos(
						localrdr.xdr(), 
						xdr_getpos(localrdr.xdr())+slen); 
				}
			}
		// *************************************************************
		// * Otherwise... set the result to indicate the existence (or
		// * non-existence) of the message.
		// *************************************************************
		else if(flag==GOTO_MESSAGE && !(map & _CLIP_MSGSET)) result=-1;



		// *************************************************************
		// * If the selected element is beyond the data, then 
		// * advance the buffer to the next item.
		// *************************************************************
		if(flag > GOTO_DATA)
			{
			if(!result && (map & _CLIP_DATASET))
				{
				slen   = localrdr.get_string_len();
				slen   = XDR_Sizeof((void *)NULL, slen);
				result = !xdr_setpos(
						localrdr.xdr(), 
						xdr_getpos(localrdr.xdr())+slen); 
				}
			}
		// *************************************************************
		// * Otherwise... set the result to indicate the existence (or
		// * non-existence) of the data.
		// *************************************************************
		else if(flag==GOTO_DATA && !(map & _CLIP_DATASET)) result=-1;
		
		
		
		// *************************************************************
		// * If the selected element is beyond the context, then it
		// * contains an invalid value and should be cleared.
		// *************************************************************
		if(flag > GOTO_CONTEXT)
			{
			if(!result && (map & _CLIP_CONTEXTSET))
				{
				slen   = localrdr.get_string_len();
				slen   = XDR_Sizeof((void *)NULL, slen);
				result = !xdr_setpos(
						localrdr.xdr(), 
						xdr_getpos(localrdr.xdr())+slen); 
				}			
			}
		// *************************************************************
		// * Otherwise... set the result to indicate the existence (or
		// * non-existence) of the context.
		// *************************************************************
		else if(flag==GOTO_CONTEXT && !(map & _CLIP_CONTEXTSET)) result=-1;


		// *************************************************************
		// * If the selected element is beyond the context, then it
		// * contains an invalid value and should be cleared.
		// *************************************************************
		if(flag > GOTO_TAGMAP) result = -1;
		// *************************************************************
		// * Otherwise... set the result to indicate the existence (or
		// * non-existence) of the context.
		// *************************************************************
		else if(flag==GOTO_TAGMAP && !(map & _CLIP_TAGMAPSET)) result=-1;
		}

	return result;
	}


// *****************************************************************************
// * cmlog_cdevPacket1::set :
// *	This method will convert the user provided data into the binary stream.
// *****************************************************************************
void cmlog_cdevPacket1::set ( short            clientID, 
			unsigned         transIndex, 
			unsigned         cancelTransIndex,
			unsigned         localDataIndex, 
			unsigned         foreignDataIndex,
			unsigned         operationCode,
			unsigned         deviceCount,
			char **          deviceList,
			char *           message,
			cdevData *       data,
			cdevData *       context,
			cdevData *       tagMap)
	{
	unsigned dataLen    = 0, dataCnt    = 0;
	unsigned contextLen = 0, contextCnt = 0;
	unsigned tagMapLen  = 0, tagMapCnt  = 0;
	
	if(binary!=NULL) 
		{
		delete binary;
		binary = NULL;
		}
	binaryLen = 0;
	
	// *********************************************************************
	// * Construct the packet map to indicate which components will be 
	// * transmitted.
	// *********************************************************************
	map = 0;
	map |= CLIP_CURRENT_VERSION;
	
	if(clientID   > 0)                    map |= _CLIP_IDSET;
	if(transIndex > 0)                    map |= _CLIP_TRINDEXSET;
	if(cancelTransIndex > 0)              map |= _CLIP_CTRINDEXSET;
	if(localDataIndex > 0)                map |= _CLIP_LOCALDINDEXSET;
	if(foreignDataIndex > 0)              map |= _CLIP_FOREIGNDINDEXSET;
	if(operationCode > 0)                 map |= _CLIP_OPCODESET;
	if(deviceCount>0 && deviceList!=NULL) map |= _CLIP_DEVLISTSET;
	if(message!=NULL)                     map |= _CLIP_MSGSET;
	
	if(data!=NULL)    data->xdrSize   (&dataLen, &dataCnt);
	if(context!=NULL) context->xdrSize(&contextLen, &contextCnt);
	if(tagMap!=NULL)  tagMap->xdrSize (&tagMapLen, &tagMapCnt);
	
	if(dataCnt > 0)                       map |= _CLIP_DATASET;
	if(contextCnt > 0)                    map |= _CLIP_CONTEXTSET;
	if(tagMapCnt > 0)                     map |= _CLIP_TAGMAPSET;
	
	// *********************************************************************
	// * Calculate the total size of the packet.
	// *********************************************************************
	binaryLen = XDR_Sizeof(map);
	if(map & _CLIP_IDSET)               binaryLen += XDR_Sizeof(clientID);
	if(map & _CLIP_TRINDEXSET)           binaryLen += XDR_Sizeof(transIndex);
	if(map & _CLIP_CTRINDEXSET)         binaryLen += XDR_Sizeof(cancelTransIndex);
	if(map & _CLIP_LOCALDINDEXSET)      binaryLen += XDR_Sizeof(localDataIndex);
	if(map & _CLIP_FOREIGNDINDEXSET)    binaryLen += XDR_Sizeof(foreignDataIndex);
	if(map & _CLIP_OPCODESET)           binaryLen += XDR_Sizeof(operationCode);
	if(map & _CLIP_DEVLISTSET)
		{
		binaryLen += XDR_Sizeof(deviceCount);
		for(int i=0; i<deviceCount; i++) binaryLen += XDR_Sizeof(deviceList[i]);
		}
	if(map & _CLIP_MSGSET)              binaryLen += XDR_Sizeof(message);
	if(map & _CLIP_DATASET)
		{
		binaryLen += XDR_Sizeof((void *)NULL, dataLen);
		}
	if(map & _CLIP_CONTEXTSET)
		{
		binaryLen += XDR_Sizeof((void *)NULL, contextLen);
		}
	if(map & _CLIP_TAGMAPSET)
		{
		binaryLen += XDR_Sizeof((void *)NULL, tagMapLen);
		}
		
	// *********************************************************************
	// * Allocate the packet and then write the data items into it.
	// *********************************************************************
	XDR_Writer writer;
	unsigned   position = 0;
	
	binary = new char[binaryLen];
	writer.attachData(binary, binaryLen);
	writer.put(map);
	if(map & _CLIP_IDSET)                    writer.put(clientID);
	if(map & _CLIP_TRINDEXSET)               writer.put(transIndex);
	if(map & _CLIP_CTRINDEXSET)              writer.put(cancelTransIndex);
	if(map & _CLIP_LOCALDINDEXSET)           writer.put(localDataIndex);
	if(map & _CLIP_FOREIGNDINDEXSET)         writer.put(foreignDataIndex);
	if(map & _CLIP_OPCODESET)                writer.put(operationCode);
	if(map & _CLIP_DEVLISTSET)
		{
		writer.put(deviceCount);
		for(int i=0; i<deviceCount; i++) writer.put((char *)deviceList[i]);
		}
	if(map & _CLIP_MSGSET)                   writer.put((char *)message);

	// *********************************************************************
	// * This is an experimental insertion that allows me to place the data
	// * into the xdr buffer directly without having to preallocate a buffer.
	// *********************************************************************
	char * tbuf;
	int    tpos;

	if(map & _CLIP_DATASET) 
		{
		tpos = xdr_getpos(writer.xdr());
		writer.put(dataLen);
		tbuf = writer.buffer()+xdr_getpos(writer.xdr());
		data->xdrExport(tbuf, dataLen, dataCnt);
		xdr_setpos(writer.xdr(), tpos+XDR_Sizeof((void *)NULL, dataLen));
		}
	if(map & _CLIP_CONTEXTSET)
		{
		tpos = xdr_getpos(writer.xdr());
		writer.put(contextLen);
		tbuf = writer.buffer()+xdr_getpos(writer.xdr());
		context->xdrExport(tbuf, contextLen, contextCnt);
		xdr_setpos(writer.xdr(), tpos+XDR_Sizeof((void *)NULL, contextLen));
		}
	if(map & _CLIP_TAGMAPSET)
		{
		tpos = xdr_getpos(writer.xdr());
		writer.put(tagMapLen);
		tbuf = writer.buffer()+xdr_getpos(writer.xdr());
		tagMap->xdrExport(tbuf, tagMapLen, tagMapCnt);
		xdr_setpos(writer.xdr(), tpos+XDR_Sizeof((void *)NULL, tagMapLen));
		}

	writer.detachData();
	}

int cmlog_cdevPacket1::set (char* buffer, unsigned size,
		      short            clientID,
		      unsigned         transIndex, 
		      unsigned         cancelTransIndex,
		      unsigned         localDataIndex, 
		      unsigned         foreignDataIndex,
		      unsigned         operationCode,
		      unsigned         deviceCount,
		      char **          deviceList,
		      char *           message,
		      cdevData *       data,
		      cdevData *       context,
		      cdevData *       tagMap)
{
	unsigned dataLen    = 0, dataCnt    = 0;
	unsigned contextLen = 0, contextCnt = 0;
	unsigned tagMapLen  = 0, tagMapCnt  = 0;
	
	if(binary!=NULL) 
		{
		delete binary;
		binary = NULL;
		}
	binaryLen = 0;
	
	// *********************************************************************
	// * Construct the packet map to indicate which components will be 
	// * transmitted.
	// *********************************************************************
	map = 0;
	map |= CLIP_CURRENT_VERSION;
	
	if(clientID   > 0)                    map |= _CLIP_IDSET;
	if(transIndex > 0)                    map |= _CLIP_TRINDEXSET;
	if(cancelTransIndex > 0)              map |= _CLIP_CTRINDEXSET;
	if(localDataIndex > 0)                map |= _CLIP_LOCALDINDEXSET;
	if(foreignDataIndex > 0)              map |= _CLIP_FOREIGNDINDEXSET;
	if(operationCode > 0)                 map |= _CLIP_OPCODESET;
	if(deviceCount>0 && deviceList!=NULL) map |= _CLIP_DEVLISTSET;
	if(message!=NULL)                     map |= _CLIP_MSGSET;
	
	if(data!=NULL)    data->xdrSize   (&dataLen, &dataCnt);
	if(context!=NULL) context->xdrSize(&contextLen, &contextCnt);
	if(tagMap!=NULL)  tagMap->xdrSize (&tagMapLen, &tagMapCnt);
	
	if(dataCnt > 0)                       map |= _CLIP_DATASET;
	if(contextCnt > 0)                    map |= _CLIP_CONTEXTSET;
	if(tagMapCnt > 0)                     map |= _CLIP_TAGMAPSET;
	
	// *********************************************************************
	// * Calculate the total size of the packet.
	// *********************************************************************
	binaryLen = XDR_Sizeof(map);
	if(map & _CLIP_IDSET)                    binaryLen += XDR_Sizeof(clientID);
	if(map & _CLIP_TRINDEXSET)               binaryLen += XDR_Sizeof(transIndex);
	if(map & _CLIP_CTRINDEXSET)              binaryLen += XDR_Sizeof(cancelTransIndex);
	if(map & _CLIP_LOCALDINDEXSET)           binaryLen += XDR_Sizeof(localDataIndex);
	if(map & _CLIP_FOREIGNDINDEXSET)         binaryLen += XDR_Sizeof(foreignDataIndex);
	if(map & _CLIP_OPCODESET)                binaryLen += XDR_Sizeof(operationCode);
	if(map & _CLIP_DEVLISTSET)
		{
		binaryLen += XDR_Sizeof(deviceCount);
		for(int i=0; i<deviceCount; i++) binaryLen += XDR_Sizeof(deviceList[i]);
		}
	if(map & _CLIP_MSGSET)                   binaryLen += XDR_Sizeof(message);
	if(map & _CLIP_DATASET)
		{
		binaryLen += XDR_Sizeof((void *)NULL, dataLen);
		}
	if(map & _CLIP_CONTEXTSET)
		{
		binaryLen += XDR_Sizeof((void *)NULL, contextLen);
		}
	if(map & _CLIP_TAGMAPSET)
		{
		binaryLen += XDR_Sizeof((void *)NULL, tagMapLen);
		}

	if (binaryLen > size)
	  return -1;
		
	// *********************************************************************
	// * Allocate the packet and then write the data items into it.
	// *********************************************************************
	XDR_Writer writer;
	unsigned   position = 0;
	
	binary = buffer;

	writer.attachData(binary, binaryLen);
	writer.put(map);
	if(map & _CLIP_IDSET)                    writer.put(clientID);
	if(map & _CLIP_TRINDEXSET)               writer.put(transIndex);
	if(map & _CLIP_CTRINDEXSET)              writer.put(cancelTransIndex);
	if(map & _CLIP_LOCALDINDEXSET)           writer.put(localDataIndex);
	if(map & _CLIP_FOREIGNDINDEXSET)         writer.put(foreignDataIndex);
	if(map & _CLIP_OPCODESET)                writer.put(operationCode);
	if(map & _CLIP_DEVLISTSET)
		{
		writer.put(deviceCount);
		for(int i=0; i<deviceCount; i++) writer.put((char *)deviceList[i]);
		}
	if(map & _CLIP_MSGSET)                   writer.put((char *)message);

	// *********************************************************************
	// * This is an experimental insertion that allows me to place the data
	// * into the xdr buffer directly without having to preallocate a buffer.
	// *********************************************************************
	char * tbuf;
	int    tpos;

	if(map &  _CLIP_DATASET) 
		{
		tpos = xdr_getpos(writer.xdr());
		writer.put(dataLen);
		tbuf = writer.buffer()+xdr_getpos(writer.xdr());
		data->xdrExport(tbuf, dataLen, dataCnt);
		xdr_setpos(writer.xdr(), tpos+XDR_Sizeof((void *)NULL, dataLen));
		}
	if(map & _CLIP_CONTEXTSET)
		{
		tpos = xdr_getpos(writer.xdr());
		writer.put(contextLen);
		tbuf = writer.buffer()+xdr_getpos(writer.xdr());
		context->xdrExport(tbuf, contextLen, contextCnt);
		xdr_setpos(writer.xdr(), tpos+XDR_Sizeof((void *)NULL, contextLen));
		}
	if(map & _CLIP_TAGMAPSET)
		{
		tpos = xdr_getpos(writer.xdr());
		writer.put(tagMapLen);
		tbuf = writer.buffer()+xdr_getpos(writer.xdr());
		tagMap->xdrExport(tbuf, tagMapLen, tagMapCnt);
		xdr_setpos(writer.xdr(), tpos+XDR_Sizeof((void *)NULL, tagMapLen));
		}

	writer.detachData();
	}


unsigned cmlog_cdevPacket1::checkLength ( short            clientID, 
				  unsigned         transIndex, 
				  unsigned         cancelTransIndex,
				  unsigned         localDataIndex, 
				  unsigned         foreignDataIndex,
				  unsigned         operationCode,
				  unsigned         deviceCount,
				  char **          deviceList,
				  char *           message,
				  cdevData *       data,
				  cdevData *       context,
				  cdevData *       tagMap)
	{
	unsigned dataLen    = 0, dataCnt    = 0;
	unsigned contextLen = 0, contextCnt = 0;
	unsigned tagMapLen  = 0, tagMapCnt  = 0;
	unsigned int map;
	unsigned binaryLen = 0;
	
	// *********************************************************************
	// * Construct the packet map to indicate which components will be 
	// * transmitted.
	// *********************************************************************
	map = 0;
	map |= CLIP_CURRENT_VERSION;

	if(clientID   > 0)                    map |= _CLIP_IDSET;
	if(transIndex > 0)                    map |= _CLIP_TRINDEXSET;
	if(cancelTransIndex > 0)              map |= _CLIP_CTRINDEXSET;
	if(localDataIndex > 0)                map |= _CLIP_LOCALDINDEXSET;
	if(foreignDataIndex > 0)              map |= _CLIP_FOREIGNDINDEXSET;
	if(operationCode > 0)                 map |= _CLIP_OPCODESET;
	if(deviceCount>0 && deviceList!=NULL) map |= _CLIP_DEVLISTSET;
	if(message!=NULL)                     map |= _CLIP_MSGSET;
	
	if(data!=NULL)    data->xdrSize   (&dataLen, &dataCnt);
	if(context!=NULL) context->xdrSize(&contextLen, &contextCnt);
	if(tagMap!=NULL)  tagMap->xdrSize (&tagMapLen, &tagMapCnt);
	
	if(dataCnt > 0)                       map |= _CLIP_DATASET;
	if(contextCnt > 0)                    map |= _CLIP_CONTEXTSET;
	if(tagMapCnt > 0)                     map |= _CLIP_TAGMAPSET;
	
	// *********************************************************************
	// * Calculate the total size of the packet.
	// *********************************************************************
	binaryLen = XDR_Sizeof(map);
	if(map & _CLIP_IDSET)                    binaryLen += XDR_Sizeof(clientID);
	if(map & _CLIP_TRINDEXSET)               binaryLen += XDR_Sizeof(transIndex);
	if(map & _CLIP_CTRINDEXSET)              binaryLen += XDR_Sizeof(cancelTransIndex);
	if(map & _CLIP_LOCALDINDEXSET)           binaryLen += XDR_Sizeof(localDataIndex);
	if(map & _CLIP_FOREIGNDINDEXSET)         binaryLen += XDR_Sizeof(foreignDataIndex);
	if(map & _CLIP_OPCODESET)                binaryLen += XDR_Sizeof(operationCode);
	if(map & _CLIP_DEVLISTSET)
		{
		binaryLen += XDR_Sizeof(deviceCount);
		for(int i=0; i<deviceCount; i++) binaryLen += XDR_Sizeof(deviceList[i]);
		}
	if(map & _CLIP_MSGSET)                   binaryLen += XDR_Sizeof(message);
	if(map & _CLIP_DATASET)
		{
		binaryLen += XDR_Sizeof((void *)NULL, dataLen);
		}
	if(map & _CLIP_CONTEXTSET)
		{
		binaryLen += XDR_Sizeof((void *)NULL, contextLen);
		}
	if(map & _CLIP_TAGMAPSET)
		{
		binaryLen += XDR_Sizeof((void *)NULL, tagMapLen);
		}

	return binaryLen;
	}

// *****************************************************************************
// * cmlog_cdevPacket1::getVersion :
// *	This method is used to obtain the version number of the packet.
// *****************************************************************************
int cmlog_cdevPacket1::getVersion ( short & version )
        {
	version = 1;
	return 0;
	}


// *****************************************************************************
// * cmlog_cdevPacket1::getClientID :
// *	This method is used to obtain the client identifier for the packet.
// *	This method returns 0 on success, or -1 if the clientID was not
// *	specified.
// *****************************************************************************
int cmlog_cdevPacket1::getClientID ( short & clientID )
	{
	int  result = -1;
	clientID    = -1;
			
	// *********************************************************************
	// * Attach the data to the XDR_Reader class so that its components
	// * can be extracted.
	// *********************************************************************
	reader.attachData(binary, binaryLen);
	
	// *********************************************************************
	// * Reposition the binary structure to point to the correct location
	// * and read the data item.
	// *********************************************************************
	if((result = setBinaryPosition(reader, GOTO_CLIENTID))==0)
		{
		if(map & _CLIP_IDSET)  result=!reader.get(clientID);
		}

	// *********************************************************************
	// * Detach the binary buffer from the XDR_Reader to prevent it from 
	// * being deleted when the XDR_Reader class is destroyed.
	// *********************************************************************
	reader.detachData();
	
	return result;
	}


	
// *****************************************************************************
// * cmlog_cdevPacket1::getTransIndex :
// *	This method is used to obtain the transaction object index.  This
// *	object is used on the client side to obtain a pointer to the transaction
// *	object.
// *	This method returns 0 on success, or -1 if the transIndex could not
// *	be read.
// *****************************************************************************
int cmlog_cdevPacket1::getTransIndex ( unsigned & transIndex )
	{
	int    result = -1;
	transIndex    = 0;
			
	// *********************************************************************
	// * Attach the data to the XDR_Reader class so that its components
	// * can be extracted.
	// *********************************************************************
	reader.attachData(binary, binaryLen);
	
	// *********************************************************************
	// * Reposition the binary structure to point to the correct location
	// * and read the data item.
	// *********************************************************************
	if((result = setBinaryPosition(reader, GOTO_TRANSINDEX))==0)
		{
		result = !reader.get(transIndex);
		}

	// *********************************************************************
	// * Detach the binary buffer from the XDR_Reader to prevent it from 
	// * being deleted when the XDR_Reader class is destroyed.
	// *********************************************************************
	reader.detachData();
	
	return result;
	}


// *****************************************************************************
// * cmlog_cdevPacket1::getCancelTransIndex :
// *	This method is used to obtain the identifier of a transaction object
// *	that is to be canceled.
// *
// *	This method returns 0 on success, or -1 if the cancelCancelTransIndex could not
// *	be read.
// *****************************************************************************
int cmlog_cdevPacket1::getCancelTransIndex ( unsigned & cancelTransIndex )
	{
	int result       = -1;
	cancelTransIndex = 0;
			
	// *********************************************************************
	// * Attach the data to the XDR_Reader class so that its components
	// * can be extracted.
	// *********************************************************************
	reader.attachData(binary, binaryLen);
	
	// *********************************************************************
	// * Reposition the binary structure to point to the correct location
	// * and read the data item.
	// *********************************************************************
	if((result = setBinaryPosition(reader, GOTO_CANCELTRANSINDEX))==0)
		{
		result = !reader.get(cancelTransIndex);
		}

	// *********************************************************************
	// * Detach the binary buffer from the XDR_Reader to prevent it from 
	// * being deleted when the XDR_Reader class is destroyed.
	// *********************************************************************
	reader.detachData();
	
	return result;
	}


// *****************************************************************************
// * cmlog_cdevPacket1::getLocalDataIndex :
// *	This method is used to obtain an index to any data that may be 
// *	maintained on the senders side of the connection.
// *	This method returns 0 on success, or -1 if the localDataIndex could 
// *	not be read.
// *****************************************************************************
int cmlog_cdevPacket1::getLocalDataIndex ( unsigned & localDataIndex )
	{
	int    result  = -1;
	localDataIndex =  0;

	// *********************************************************************
	// * Attach the data to the XDR_Reader class so that its components
	// * can be extracted.
	// *********************************************************************
	reader.attachData(binary, binaryLen);
	
	// *********************************************************************
	// * Reposition the binary structure to point to the correct location
	// * and read the data item.
	// *********************************************************************
	if((result = setBinaryPosition(reader, GOTO_LOCALDATAINDEX))==0)
		{
		result = !reader.get(localDataIndex);
		}
		
	// *********************************************************************
	// * Detach the binary buffer from the XDR_Reader to prevent it from 
	// * being deleted when the XDR_Reader class is destroyed.
	// *********************************************************************
	reader.detachData();
	
	return result;
	}


// *****************************************************************************
// * cmlog_cdevPacket1::getForeignDataIndex :
// *	This method is used to obtain an index to any data that may be 
// *	maintained on the senders side of the connection.
// *	This method returns 0 on success, or -1 if the foreignDataIndex could 
// *	not be read.
// *****************************************************************************
int cmlog_cdevPacket1::getForeignDataIndex ( unsigned & foreignDataIndex )
	{
	int    result    = -1;
	foreignDataIndex = 0;

	// *********************************************************************
	// * Attach the data to the XDR_Reader class so that its components
	// * can be extracted.
	// *********************************************************************
	reader.attachData(binary, binaryLen);
	
	// *********************************************************************
	// * Reposition the binary structure to point to the correct location
	// * and read the data item.
	// *********************************************************************
	if((result = setBinaryPosition(reader, GOTO_FOREIGNDATAINDEX))==0)
		{
		result = !reader.get(foreignDataIndex);
		}
		
	// *********************************************************************
	// * Detach the binary buffer from the XDR_Reader to prevent it from 
	// * being deleted when the XDR_Reader class is destroyed.
	// *********************************************************************
	reader.detachData();
	
	return result;
	}


// *****************************************************************************
// * cmlog_cdevPacket1::getOperationCode :
// *	This method is used to obtain the operation code.  An operation code may
// *	be used by the developer to define a particular message.  This is faster
// *	to manipulate than the supported string messages.
// *****************************************************************************
int cmlog_cdevPacket1::getOperationCode ( unsigned & operationCode )
	{
	int result         = -1;
	operationCode = 0;

	// *********************************************************************
	// * Attach the data to the XDR_Reader class so that its components
	// * can be extracted.
	// *********************************************************************
	reader.attachData(binary, binaryLen);
	
	// *********************************************************************
	// * Reposition the binary structure to point to the correct location
	// * and read the data item.
	// *********************************************************************
	if((result = setBinaryPosition(reader, GOTO_OPERATIONCODE))==0)
		{
		result = !reader.get(operationCode);
		}
		
	// *********************************************************************
	// * Detach the binary buffer from the XDR_Reader to prevent it from 
	// * being deleted when the XDR_Reader class is destroyed.
	// *********************************************************************
	reader.detachData();
	
	return result;
	}


// *****************************************************************************
// * cmlog_cdevPacket1::getDeviceList :
// *	This method is used to obtains an array of device names that are 
// *	associated with the request.  It is the responsibility of the caller
// *	to delete each element of the array, and the array itself when they
// *	are no longer needed.
// *	This method returns 0 on success, or -1 if the deviceList could 
// *	not be read.
// *****************************************************************************
int cmlog_cdevPacket1::getDeviceList ( char ** & deviceList, unsigned & deviceCount)
	{
	int    result = -1;

	deviceList  = NULL;
	deviceCount = 0;

	// *********************************************************************
	// * Attach the data to the XDR_Reader class so that its components
	// * can be extracted.
	// *********************************************************************
	reader.attachData(binary, binaryLen);
	
	// *********************************************************************
	// * Reposition the binary structure to point to the correct location
	// * and read the data item.
	// *********************************************************************
	if((result = setBinaryPosition(reader, GOTO_DEVICELIST))==0)
		{
		if((result = !reader.get(deviceCount))==0)
			{
			// *********************************************
			// * Allocate and initialize an array sufficient
			// * to hold the contents of the stream.
			// *********************************************
			deviceList = new char * [deviceCount];
			memset(deviceList, 0, sizeof(char *)*deviceCount);

			// *********************************************
			// * Read the individual strings into the array.
			// * Note that the XDR_Reader will automatically
			// * allocate data if the pointers are set to 
			// * NULL.
			// *********************************************
			for(int i=0; i<deviceCount && !result; i++)
				result = !reader.get(&deviceList[i]);
			}

		// *****************************************************
		// * In event that a failure occurred after the buffer
		// * was allocated.  Walk through the array and free
		// * each data item... then free the array.
		// *****************************************************
		if(result!=0 && deviceList!=NULL)
			{
			for(int i=0; i<deviceCount; i++)
				{
				if(deviceList[i]!=NULL) delete deviceList[i];
				}
			delete deviceList;
			}
		}
	// *********************************************************************
	// * Detach the binary buffer from the XDR_Reader to prevent it from 
	// * being deleted when the XDR_Reader class is destroyed.
	// *********************************************************************
	reader.detachData();
	
	return result;
	}


// *****************************************************************************
// * cmlog_cdevPacket1::getMessage :
// *	This method is used to obtains an array of device names that are 
// *	associated with the request.  It is the responsibility of the caller
// *	to delete each element of the array, and the array itself when they
// *	are no longer needed.
// *	This method returns 0 on success, or -1 if the deviceList could 
// *	not be read.
// *****************************************************************************
int cmlog_cdevPacket1::getMessage ( char * & message)
	{
	int result = -1;
	message    = NULL;
	
	// *********************************************************************
	// * Attach the data to the XDR_Reader class so that its components
	// * can be extracted.
	// *********************************************************************
	reader.attachData(binary, binaryLen);
	
	// *********************************************************************
	// * Reposition the binary structure to point to the correct location
	// * and read the data item.
	// *********************************************************************
	if((result = setBinaryPosition(reader, GOTO_MESSAGE))==0)
		{
		if((result = !reader.get(&message))!=0)
			{
			delete message;
			}		
		}
	// *********************************************************************
	// * Detach the binary buffer from the XDR_Reader to prevent it from 
	// * being deleted when the XDR_Reader class is destroyed.
	// *********************************************************************
	reader.detachData();
	
	return result;
	}


// *****************************************************************************
// * cmlog_cdevPacket1::getData :
// *	This method is used to read the data cdevData object from the 
// *	binary stream.
// *	This method returns 0 on success, or -1 if the data item could 
// *	not be read.
// *****************************************************************************
int cmlog_cdevPacket1::getData ( cdevData & data)
	{
	int    result = -1;

	// *********************************************************************
	// * Clear the contents of the context data object.
	// *********************************************************************
	data.remove();
		
	// *********************************************************************
	// * Attach the data to the XDR_Reader class so that its components
	// * can be extracted.
	// *********************************************************************
	reader.attachData(binary, binaryLen);
	
	// *********************************************************************
	// * Reposition the binary structure to point to the correct location
	// * and read the data item.
	// *********************************************************************
	if((result = setBinaryPosition(reader, GOTO_DATA))==0)
		{
		int   tlen;
		int   tpos;
		char *tbuf;

		tpos = xdr_getpos(reader.xdr());
		reader.get(tlen);
		tbuf = reader.buffer()+xdr_getpos(reader.xdr());
		if(tlen!=0 && tbuf!=NULL) data.xdrImport(tbuf, tlen);
		else result = -1;
		xdr_setpos(reader.xdr(), tpos+XDR_Sizeof((void *)NULL, tlen));	
		}
	// *********************************************************************
	// * Detach the binary buffer from the XDR_Reader to prevent it from 
	// * being deleted when the XDR_Reader class is destroyed.
	// *********************************************************************
	reader.detachData();
	
	return result;
	}


// *****************************************************************************
// * cmlog_cdevPacket1::getContext :
// *	This method is used to read the context cdevData object from the 
// *	binary stream.
// *	This method returns 0 on success, or -1 if the context could 
// *	not be read.
// *****************************************************************************
int cmlog_cdevPacket1::getContext ( cdevData & context)
	{
	int    result = -1;

	// *********************************************************************
	// * Clear the contents of the context data object.
	// *********************************************************************
	context.remove();
		
	// *********************************************************************
	// * Attach the data to the XDR_Reader class so that its components
	// * can be extracted.
	// *********************************************************************
	reader.attachData(binary, binaryLen);
	
	// *********************************************************************
	// * Reposition the binary structure to point to the correct location
	// * and read the data item.
	// *********************************************************************
	if((result = setBinaryPosition(reader, GOTO_CONTEXT))==0)
		{
		int   tlen;
		int   tpos;
		char *tbuf;

		tpos = xdr_getpos(reader.xdr());
		reader.get(tlen);
		tbuf = reader.buffer()+xdr_getpos(reader.xdr());
		if(tlen!=0 && tbuf!=NULL) context.xdrImport(tbuf, tlen);
		else result = -1;
		xdr_setpos(reader.xdr(), tpos+XDR_Sizeof((void *)NULL, tlen));
		}
	// *********************************************************************
	// * Detach the binary buffer from the XDR_Reader to prevent it from 
	// * being deleted when the XDR_Reader class is destroyed.
	// *********************************************************************
	reader.detachData();
	
	return result;
	}


// *****************************************************************************
// * cmlog_cdevPacket1::getTagMap :
// *	This method is used to read the tagMap cdevData object from the 
// *	binary stream.
// *	This method returns 0 on success, or -1 if the tagMap could not be read.
// *****************************************************************************
int cmlog_cdevPacket1::getTagMap ( cdevData & tagMap)
	{
	int    result = -1;

	// *********************************************************************
	// * Clear the contents of the tagMap data object.
	// *********************************************************************
	tagMap.remove();
		
	// *********************************************************************
	// * Attach the data to the XDR_Reader class so that its components
	// * can be extracted.
	// *********************************************************************
	reader.attachData(binary, binaryLen);
	
	// *********************************************************************
	// * Reposition the binary structure to point to the correct location
	// * and read the data item.
	// *********************************************************************
	if((result = setBinaryPosition(reader, GOTO_TAGMAP))==0)
		{
		int   tlen;
		int   tpos;
		char *tbuf;

		tpos = xdr_getpos(reader.xdr());
		reader.get(tlen);
		tbuf = reader.buffer()+xdr_getpos(reader.xdr());
		if(tlen!=0 && tbuf!=NULL) tagMap.xdrImport(tbuf, tlen);
		else result = -1;
		xdr_setpos(reader.xdr(), tpos+XDR_Sizeof((void *)NULL, tlen));
		}
	// *********************************************************************
	// * Detach the binary buffer from the XDR_Reader to prevent it from 
	// * being deleted when the XDR_Reader class is destroyed.
	// *********************************************************************
	reader.detachData();
	
	return result;
	}


// *****************************************************************************
// * cmlog_cdevPacket1::get :
// *	This method is used to read the complete contents of the packet.
// *	This method returns 0 on success, or -1 if the deviceList could 
// *	not be read.
// *
// *	Note that all data items allocated by this call become the property of
// *	the caller and must be deleted.  Specifically, the deviceList and
// *	message parameters must be deleted.
// *****************************************************************************
int cmlog_cdevPacket1::get (	short    & clientID, 
			unsigned & transIndex,
			unsigned & cancelTransIndex,
			unsigned & localDataIndex,
			unsigned & foreignDataIndex,
			unsigned & operationCode,
			unsigned & deviceCount,
			char **  & deviceList,
			char *   & message,
			cdevData & data,
			cdevData & context,
			cdevData & tagMap)
	{
	int result = -1;

	// *********************************************************************
	// * Set all data to a default value.
	// *********************************************************************
	clientID         = -1;
	transIndex       = 0;
	cancelTransIndex = 0;
	localDataIndex   = 0;
	foreignDataIndex = 0;
	operationCode    = 0;
	deviceCount      = 0;
	deviceList       = NULL;
	message          = NULL;
	data.remove();
	context.remove();
	tagMap.remove();
		
	// *********************************************************************
	// * Attach the data to the XDR_Reader class so that its components
	// * can be extracted.
	// *********************************************************************
	reader.attachData(binary, binaryLen);

	// *********************************************************************
	// * Set the position to the beginning of the data buffer.
	// *********************************************************************
	xdr_setpos(reader.xdr(), 0);
			
	// *********************************************************************
	// * Clear the packet map prior to reading data from the stream.
	// *********************************************************************
	map     = 0;
	
	// *********************************************************************
	// * Read the packet map from the stream.  This object contains a bit
	// * field that identifies the version of the packet and tells what 
	// * other components are embedded in the binary stream.
	// *********************************************************************
	result = !reader.get(map);
	
	// *********************************************************************
	// * Before processing anything else, the method must ensure that this
	// * packet is of the correct version.  Version 1 for this object.
	// *
	// * Note that reading each field is contingent on the prior fields 
	// * being read correctly.
	// *********************************************************************
	if(!result && clipVersionNum (map) != 1)      result=-1;
	if(!result && (map & _CLIP_IDSET))  result=!reader.get(clientID);
	if(!result && (map & _CLIP_TRINDEXSET)) result = !reader.get(transIndex);
	if(!result && (map & _CLIP_CTRINDEXSET)) result = !reader.get(cancelTransIndex);
	if(!result && (map & _CLIP_LOCALDINDEXSET)) result = !reader.get(localDataIndex);
	if(!result && (map & _CLIP_FOREIGNDINDEXSET)) result = !reader.get(foreignDataIndex);
	if(!result && (map & _CLIP_OPCODESET)) result = !reader.get(operationCode);
	if(!result && (map & _CLIP_DEVLISTSET))
		{
		if((result = !reader.get(deviceCount))==0)
			{
			// *********************************************
			// * Allocate and initialize an array sufficient
			// * to hold the contents of the stream.
			// *********************************************
			deviceList = new char * [deviceCount];
			memset(deviceList, 0, sizeof(char *)*deviceCount);

			// *********************************************
			// * Read the individual strings into the array.
			// * Note that the XDR_Reader will automatically
			// * allocate data if the pointers are set to 
			// * NULL.
			// *********************************************
			for(int i=0; i<deviceCount && !result; i++)
				result = !reader.get(&deviceList[i]);
			}
		}
	if(!result && (map & _CLIP_MSGSET)) result = !reader.get(&message);

	// *********************************************************************
	// * This mechanism will bypass the standard XDR extractor and allow
	// * the caller to directly read the data from the stream without
	// * performing an additional memory allocation.
	// *********************************************************************
	int   tlen;
	int   tpos;
	char *tbuf;

	if(!result && (map & _CLIP_DATASET))
		{
		tpos = xdr_getpos(reader.xdr());
		reader.get(tlen);
		tbuf = reader.buffer()+xdr_getpos(reader.xdr());
		if(tlen!=0 && tbuf!=NULL) data.xdrImport(tbuf, tlen);
		else result = -1;
		xdr_setpos(reader.xdr(), tpos+XDR_Sizeof((void *)NULL, tlen));
		}
	if(!result && (map & _CLIP_CONTEXTSET)) 
		{
		tpos = xdr_getpos(reader.xdr());
		reader.get(tlen);
		tbuf = reader.buffer()+xdr_getpos(reader.xdr());
		if(tlen!=0 && tbuf!=NULL) context.xdrImport(tbuf, tlen);
		else result = -1;
		xdr_setpos(reader.xdr(), tpos+XDR_Sizeof((void *)NULL, tlen));
		}
	if(!result && (map & _CLIP_TAGMAPSET)) 
		{
		tpos = xdr_getpos(reader.xdr());
		reader.get(tlen);
		tbuf = reader.buffer()+xdr_getpos(reader.xdr());
		if(tlen!=0 && tbuf!=NULL) tagMap.xdrImport(tbuf, tlen);
		else result = -1;
		xdr_setpos(reader.xdr(), tpos+XDR_Sizeof((void *)NULL, tlen));
		}

	// *****************************************************
	// * In event that a failure occurred after the buffers
	// * was allocated.  Walk through the array and free
	// * each data item... then free the array.
	// *****************************************************
	if(result!=0)
		{
		if(deviceList!=NULL)
			{
			for(int i=0; i<deviceCount; i++)
				{
				if(deviceList[i]!=NULL) delete deviceList[i];
				}
			delete deviceList;
			deviceList = NULL;
			}
		if(message!=NULL) 
			{
			delete message;
			message = NULL;
			}
		clientID         = -1;
		transIndex       = 0;
		localDataIndex   = 0;
		foreignDataIndex = 0;
		deviceCount      = 0;
		data.remove();
		context.remove();
		tagMap.remove();
		}
	
	// *********************************************************************
	// * Detach the binary buffer from the XDR_Reader to prevent it from 
	// * being deleted when the XDR_Reader class is destroyed.
	// *********************************************************************
	reader.detachData();
	
	return result;
	}

