//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Implementation of cmlog xui config
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogXuiConfig.cc,v $
//   Revision 1.9  2001/06/21 18:19:50  chen
//   Port to RedHat 7.1
//
//   Revision 1.8  2000/06/20 19:36:36  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.7  1999/12/13 21:24:06  chen
//   Fix a saving width problem from title bar area
//
//   Revision 1.5  1999/10/29 17:40:06  chen
//   support spaces and quotes in cmlogrc file
//
//   Revision 1.3  1999/09/21 20:27:55  chen
//   Add dialog box for save settings
//
//   Revision 1.1.1.1  1999/09/07 15:29:12  chen
//   CMLOG version 2.0
//
// Revision 1.3  1999/03/16  14:32:39  chen
// Fix a buffer overflow
//
// Revision 1.2  1999/02/02  16:05:51  chen
// add query and update selection message to configuration file
//
// Revision 1.1  1997/08/01  15:31:05  bickley
// Added cmlog to application development system.
//
//
//
#include <stdlib.h>
#include <ctype.h>
#include <cmlogConfig.h>
#include <cmlogUtil.h>
#include <cmlogXuiLogicSup.h>
#include <cmlogXuiCodeConverter.h>
#include "cmlogXuiConfig.h"

const char* cmlogXuiConfig::localConfig_ = ".cmlogrc";
const char* cmlogXuiConfig::configEnv_ = "CMLOG_CONFIG";
const char* cmlogXuiConfig::configDateFormat_ = "%H:%M@%m/%d/%Y";
const char* cmlogXuiConfig::configDateFormat1_ = "-%d:%d";

cmlogXuiConfig* cmlogXuiConfig::config_ = 0;

Dimension cmlogXuiConfig::defwidths_[] = {
  50, 50, 300
};

const char* cmlogXuiConfig::deftitles_[] = {
  "Host", "Time", "Message"
};

const char* cmlogXuiConfig::deftags_[] = {
  "host", "cmlogTime", "text"
};

int cmlogXuiConfig::updateBufsize_ = 2000;

typedef void (*cmlog_opt_func) (char *);

typedef struct _cmlog_option_
{
  const char *opname;
  const char *opfullname;
  int  hasarg;
  cmlog_opt_func func;
  
}cmlog_option;

static cmlog_option _cmlog_xui_options_[] = 
{
  {"-c","-connect", 0, &(cmlogXuiConfig::doConnection)},
  {"-u","-update", 0, &(cmlogXuiConfig::doUpdate)},
  {"-f","-cfg", 1, &(cmlogXuiConfig::useConfigFile)},
  {"-hu",0, 0, &(cmlogXuiConfig::doSearchUpdate)},
  {"-q",0, 0, &(cmlogXuiConfig::doQuery)},
  {"-h", 0, 0, &(cmlogXuiConfig::printUsage)},
};

#define CMLOG_NUM_OPTIONS 5

/* The followings are key words inside a configuration file */
#define CMLOG_COL "Col"
#define CMLOG_QUERY_MSG "queryMessage"
#define CMLOG_CODE_CONV "codeConversion"
#define CMLOG_UPDATE_MSG "updateMessage"
#define CMLOG_UBUFSIZE "ubufsize"
#define CMLOG_WIN_WIDTH "windowWidth"
#define CMLOG_WIN_HEIGHT "windowHeight"
#define CMLOG_NAME "name"

#ifdef __linux
extern "C"  char *strptime (const char *s, const char *fmt,
			    struct tm *tp);
#endif

cmlogXuiConfig::cmlogXuiConfig (void)
:numcols_ (0), colwidths_ (0), coltitles_ (0), coltags_ (0),
 ubufsize_ (2000), qmsg_ (0), qstart_ (-1.0), qend_ (-1.0), 
 qstartType_ (NO_DATE), qendType_ (NO_DATE), qstartStr_ (0), qendStr_ (0),
 umsg_ (0), windowWidth_ (600), windowHeight_ (400), doConnection_ (0),
 doUpdate_ (0), doQuery_ (0), configFile_ (0), histUpdate_ (0), 
 numConverters_ (0), convBufSize_ (5), name_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogXuiConfig Class Objects\n");
#endif
  converters_ = new cmlogXuiCodeConverter*[convBufSize_];
}


cmlogXuiConfig::~cmlogXuiConfig (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogXuiConfig Class Objects\n");
#endif
  int i;
  delete []colwidths_;
  
  for (i = 0; i < numcols_; i++) {
    delete []coltitles_[i];
    delete []coltags_[i];
  }
  delete []colwidths_;
  delete []coltitles_;
  delete []coltags_;

  if (qmsg_)
    delete []qmsg_;
  if (umsg_)
    delete []umsg_;

  if (qstartStr_)
    delete []qstartStr_;

  if (qendStr_)
    delete []qendStr_;

  if (configFile_)
    delete []configFile_;

  if (numConverters_)
    for (i = 0; i < numConverters_; i++)
      delete converters_[i];
  delete []converters_;

  if (name_)
    delete []name_;
}

int
cmlogXuiConfig::init (int argc, char** argv)
{
  int  status = 0;
  int  i = 0;
  // check local file first
  char fullname[128];

  parseCommandLineOption (argc, argv);

  if (!configFile_) {
    // check .cmlogrc file
    char* env = getenv ("HOME");
    if (!env)
      status = -1;
    else {
      strncpy (fullname, env, sizeof (fullname));
      strcat  (fullname, "/");
      strcat  (fullname, cmlogXuiConfig::localConfig_);
    }

    // if unsuccessful 
    if (status == -1 || parseFile (fullname) != 0) {
      // check CMLOG_CONFIG
      char* cenv = getenv (cmlogXuiConfig::configEnv_);
      if (!cenv)
	status = -1;
      else {
	strncpy (fullname, cenv, sizeof (fullname));
	status = parseFile (fullname);
      }
    }
  }
  else {
    if (parseFile (configFile_) != 0) 
      status = -1;
  }

  if (status == -1) { // file does not exist
    fprintf (stderr, "Cannot find any configuration files, using default parameters\n");
    numcols_ = 3;

    colwidths_ = new Dimension[3];
    for (i = 0; i < 3; i++)
      colwidths_[i] = cmlogXuiConfig::defwidths_[i];

    coltitles_ = new char*[3];
    for (i = 0; i < 3; i++) {
      coltitles_[i] = new char[strlen (cmlogXuiConfig::deftitles_[i]) + 1];
      strcpy (coltitles_[i], cmlogXuiConfig::deftitles_[i]);
    }

    coltags_ = new char*[3];
    for (i = 0; i < 3; i++) {
      coltags_[i] = new char[strlen (cmlogXuiConfig::deftags_[i]) + 1];
      strcpy (coltags_[i], cmlogXuiConfig::deftags_[i]);
    }

    ubufsize_ = cmlogXuiConfig::updateBufsize_;
  }
  else if (!configFile_) {
    // remember current config file
    configFile_ = new char[strlen(fullname) + 1];
    strcpy (configFile_, fullname);
  }
  return status;
}
  

int
cmlogXuiConfig::parseCommandLineOption (int argc, char** argv)
{
  int match = 0;
  int i = 1;

  while (i < argc) {
    for (int j = 0; j < CMLOG_NUM_OPTIONS; j++) {
      if (strcmp (argv[i], _cmlog_xui_options_[j].opname) == 0 || 
	  (_cmlog_xui_options_[j].opfullname 
	   && strcmp (argv[i], _cmlog_xui_options_[j].opfullname) == 0)) {
	if (_cmlog_xui_options_[j].hasarg) {
	  if (i + 1 >= argc || argv[i + 1][0] == '-')
	    printUsage (0);
	  (*_cmlog_xui_options_[j].func)(argv[++i]);
	}
	else
	  (*_cmlog_xui_options_[j].func)(0);
	match = 1;
      }
    }
    i++;
  }
  if (!match && argc > 1)
    printUsage (0);

  return 0;
}


int
cmlogXuiConfig::parseFile (char* filename)
{
  int i;
  char* ptr;
  FILE *fd = fopen (filename, "r");
  if (!fd)
    return -1;

  char token0[40], token1[128], token2[80], token3[80];
  int  width;

  // find out how many column definitions are there
  int num = 0;
  char line[256];
  while (!feof (fd)) {
    line[0] = '\0';
    fgets (line, sizeof (line), fd);
    if (line[0] != '#' && line[0] != '\0') {
      if (sscanf(line, "%s",token0) >= 1 && strcasecmp (token0, CMLOG_COL)==0) 
	num++;
    }
  }
  if (num == 0)
    return -1;

  numcols_ = num;
  colwidths_ = new Dimension[numcols_];
  coltitles_ = new char*[numcols_];
  coltags_ = new char*[numcols_];

  // rewind the file descriptor
  num = 0;
  fseek (fd, 0L, SEEK_SET);
  while (!feof (fd)) {
    line[0] = '\0';

    // get a null terminated string
    fgets (line, sizeof (line), fd);
    if ((ptr = strchr(line, '\n')) != 0)
	*ptr = '\0';
    else
      line[sizeof(line) - 1] = '\0';

    if (line[0] != '#') {
      if (sscanf (line, "%s", token0) >= 1) {
	if (strcasecmp (token0, CMLOG_COL) == 0) {
	  // Column definistion
	  if (sscanf (line, "%s %s %s %d", token0, token1, token2, &width) >= 4){
	    colwidths_[num] = (Dimension)width;
	    // title
	    coltitles_[num] = new char[strlen (token1) + 1];
	    strcpy (coltitles_[num], token1);
	    // tags
	    coltags_[num] = new char[strlen (token2) + 1];
	    strcpy (coltags_[num], token2);
	    num++;
	  }
	}
	else if (strcasecmp (token0, CMLOG_QUERY_MSG) == 0) {
	  int qstat = 0;
	  qstat = parseQueryMessageLine (line, 
					 token1, sizeof (token1),
					 token2, sizeof (token2),
					 token3, sizeof (token3));
	  if (qstat < 1) {
	    fprintf (stderr, "Parsing queryMessage error at line <%s>\n", line);
	    exit (1);
	  }
	  if (strcasecmp (token1, "none") != 0) {
	    qmsg_ = new char[strlen (token1) + 1];
	    strcpy (qmsg_, token1);
	  }
	    
	  if (qstat >= 2) {
	    if (queryStartTime(token2, &qstart_, &qstartType_) != 0) {
	      fprintf (stderr, "Parsing query start time string: %s \n", token2);
	      exit (1);
	    }
	    else {
	      qstartStr_ = new char[strlen(token2) + 1];
	      strcpy (qstartStr_, token2);
	    }
	  }
	    
	  if (qstat >= 3) {
	    // get query end time
	    if (queryEndTime(token3, &qend_, &qendType_) != 0) {
	      fprintf (stderr, "Parsing query end time string: %s \n", token3);
	      exit (1);
	    }
	    else {
	      qendStr_ = new char[strlen(token3) + 1];
	      strcpy (qendStr_, token3);
	    }
	  }
	}
	else if (strcasecmp (token0, CMLOG_CODE_CONV) == 0 &&
		 parseCodeConvLine(line, token1, sizeof(token1),
				   token2, sizeof(token2)) == 2) {
	  char seltag[40], seltagVal[40];
	  int  selt, convt;
	  int  pst = 0;
	  // scan selection tag in the format of tag=='value' or none
	  if ((pst = parseSelectionTag(token1, seltag, sizeof(seltag),
				       seltagVal, sizeof(seltagVal))) == -1) {
	    fprintf (stderr, "Selection Tag format error < %s > \n", token1);
	    exit (1);
	  }
	  
	  // convert selection tag string to tag
	  if (pst == 0) {
	    if (!tagOfTitle(seltag) ||
		cdevData::tagC2I (tagOfTitle(seltag), &selt) != CDEV_SUCCESS) {
	      fprintf (stderr, "Illegal selection title name %s\n", seltag);
	      exit (1);
	    }
	  }

	  // convert convertion tag string to tag
	  if (!tagOfTitle(token2) || 
	      cdevData::tagC2I (tagOfTitle(token2), &convt) != CDEV_SUCCESS) {
	    fprintf (stderr, "Illegal conversion titlename %s\n", token2);
	    exit (1);
	  }

	  // if we have selection part 'none', pass selt -1 and seltagVal 0
	  if (pst == 1) {
	    selt = -1;
	    seltagVal[0] = '\0';
	  }

	  // create a new conveter
	  cmlogXuiCodeConverter* converter =
	    new cmlogXuiCodeConverter (selt, convt, seltagVal);
	  // register this code converter to conveters list
	  if (numConverters_ == convBufSize_) {
	    cmlogXuiCodeConverter** nconverters = 
	      new cmlogXuiCodeConverter*[2*convBufSize_];
	    for (i = 0; i < convBufSize_; i++)
	      nconverters[i] = converters_[i];
	    convBufSize_ = 2*convBufSize_;
	    converters_ = nconverters;
	  }
	  converters_[numConverters_ ++] = converter;
	  // pass this file to converter
	  converter->input (fd);
	}
	else if (strcasecmp (token0, CMLOG_UPDATE_MSG) == 0) {
	  int ustat = parseUpdateMessageLine (line, token1, sizeof (token1));
	  if (ustat < 1) {
	    fprintf (stderr, "Parsing updateMessage error at line <%s>\n", line);
	    exit (1);
	  }
	  if (::strcasecmp (token1, "none") != 0) {
	    umsg_ = new char[strlen (token1) + 1];
	    strcpy (umsg_, token1);
	  }
	}
	else if (strcasecmp (token0, CMLOG_UBUFSIZE) == 0 &&
		 sscanf (line, "%s %s", token0, token1) >= 2) 
	  ubufsize_ = atoi (token1);
	else if (strcasecmp (token0, CMLOG_WIN_WIDTH) == 0 &&
		 sscanf (line, "%s %s", token0, token1) >= 2) 		 
	  windowWidth_ = (unsigned short)atoi (token1);
	else if (strcasecmp (token0, CMLOG_WIN_HEIGHT) == 0 &&
		 sscanf (line, "%s %s", token0, token1) >= 2)
	  windowHeight_ = (unsigned short)atoi (token1);
	else if (strcasecmp (token0, CMLOG_NAME) == 0 && 
		 parseNameLine (line, token1, sizeof (token1)) >= 1) {
	  name_ = new char[strlen (token1) + 1];
	  strcpy (name_, token1);
	}
      }
    }	// not comment line
    
  } // while loop

  fclose (fd);

  // now check syntax error on query message and update message
  if (qmsg_ && !cmlogXuiQueryString(qmsg_)) {
    fprintf (stderr, "Query message< %s > syntax error\n", qmsg_);
    exit (1);
  }
  if (umsg_ && !cmlogXuiQueryString(umsg_)) {
    fprintf (stderr, "Updating selection message< %s > syntax error\n", umsg_);
    exit (1);  
  }
  
  if (!numcols_)
    return -1;
  return 0;
}

int
cmlogXuiConfig::saveConfiguration (char fname[], int size)
{
  char fullname[256];
  int status = 0;

  if (configFile_) {
    status = saveConfiguration (configFile_);
    strncpy (fullname, configFile_, sizeof(fullname) - 1);
  }
  else {
    // check .cmlogrc file first
    char* env = getenv ("HOME");
    if (!env)
      status = -1;
    else {
      strncpy (fullname, env, sizeof (fullname));
      strcat  (fullname, "/");
      strcat  (fullname, cmlogXuiConfig::localConfig_);
    }

    if (status == -1 || saveConfiguration (fullname) != 0) 
      status = -1;
  }
  strncpy (fname, fullname, size);
  return status;
}

int
cmlogXuiConfig::saveConfiguration (char* fname)
{
  int i;
  FILE* fd = fopen (fname, "w");
  if (fd == NULL) 
    return -1;

  fprintf (fd, "# Configuration file for cmlog\n");

  fprintf (fd, "# Name of this configuration\n");
  if (name_)
    fprintf (fd, "name %s\n", name_);

  fprintf (fd, "# Type          Title          Tag          Width\n");
  for (i = 0; i < numcols_; i++) {
    fprintf (fd, "Col          %-16s          %-13s          %-13d\n", 
	     coltitles_[i], coltags_[i], colwidths_[i]);
  }
  fprintf (fd, "# Buffer size for updating the message\n");
  fprintf (fd, "ubufsize     %d\n", ubufsize_);

  char* qstr = (qmsg_ != 0) ? qmsg_ : (char *)"none";
  fprintf (fd, "# Default query message\n");
  fprintf (fd, "queryMessage    \"%s\"    ", qstr);
  if (qstartType_ == NO_DATE) 
    fprintf (fd, "\n");
  else {
    if (qstartStr_)
      fprintf (fd, "%s         ", qstartStr_);
    if (qendStr_)
    fprintf (fd, "%s", qendStr_);
    fprintf (fd, "\n");
  }

  char* ustr = (umsg_ != 0) ? umsg_ : (char *)"none";
  fprintf (fd, "# Default update selection message\n");
  fprintf (fd, "updateMessage    \"%s\"\n", ustr);

  fprintf (fd, "# Window Width\n");
  fprintf (fd, "windowWidth  %d\n", windowWidth_);
  fprintf (fd, "# Window Height\n");
  fprintf (fd, "windowHeight  %d\n", windowHeight_);

  for (i = 0; i < numConverters_; i++)
    converters_[i]->output (fd);
  
  fclose (fd);

  return 0;
}

int
cmlogXuiConfig::numColumns (void)
{
  return numcols_;
}

Dimension *
cmlogXuiConfig::columnWidths (void)
{
  return colwidths_;
}

char **
cmlogXuiConfig::columnTitles (void)
{
  return coltitles_;
}

char **
cmlogXuiConfig::columnTags (void)
{
  return coltags_;
}

int
cmlogXuiConfig::updateBufsize (void)
{
  return ubufsize_;
}

void
cmlogXuiConfig::updateBufsize (int size)
{
  ubufsize_ = size;
}


cmlogXuiConfig *
cmlogXuiConfig::config (void)
{
  if (cmlogXuiConfig::config_ == 0) 
    cmlogXuiConfig::config_ = new cmlogXuiConfig ();
  return cmlogXuiConfig::config_;
}

char*
cmlogXuiConfig::tagOfTitle (char* title)
{
  int i;
  for (i = 0; i < numcols_; i++) {
    if (strcmp (title, coltags_[i]) == 0)
      return title;
    else if (strcmp (title, coltitles_[i]) == 0)
      return coltags_[i];
  }
  return 0;
}

char*
cmlogXuiConfig::titleOfTag (char* tag)
{
  int i;
  for (i = 0; i < numcols_; i++) {
    if (strcmp (tag, coltags_[i]) == 0)
      return coltitles_[i];
  }
  return 0;
}

void
cmlogXuiConfig::changeConfiguration (int nc, Dimension* wds,
				     char** titles, char** tags,
				     int ubufsize)
{
  int i = 0;

  /* free memory */
  if (numcols_ > 0) {
    for (i = 0; i < numcols_; i++) {
      delete []coltitles_[i];
      delete []coltags_[i];
    }
    delete []coltitles_;
    delete []coltags_;
    delete []colwidths_;
  }

  /* assign new values */
  numcols_ = nc;
  colwidths_ = new Dimension[numcols_];
  coltitles_ = new char*[numcols_];
  coltags_ = new char*[numcols_];
  for (i = 0; i < numcols_; i++) {
    colwidths_[i] = wds[i];

    coltitles_[i] = new char[strlen (titles[i]) + 1];
    strcpy (coltitles_[i], titles[i]);

    coltags_[i] = new char[strlen (tags[i]) + 1];
    strcpy (coltags_[i], tags[i]);
  }
  ubufsize_ = ubufsize;
}

char *
cmlogXuiConfig::queryMessage (void) const
{
  return qmsg_;
}

void
cmlogXuiConfig::queryMessage (char* msg)
{
  if (qmsg_)
    delete []qmsg_;
  qmsg_ = new char[strlen (msg) + 1];
  strcpy (qmsg_, msg);
}

char *
cmlogXuiConfig::updateSelMessage (void) const
{
  return umsg_;
}

void
cmlogXuiConfig::updateSelMessage (char* msg)
{
  if (umsg_)
    delete []umsg_;
  umsg_ = new char[strlen (msg) + 1];
  strcpy (umsg_, msg);
}

void
cmlogXuiConfig::doUpdate (char *)
{
  cmlogXuiConfig *conf = cmlogXuiConfig::config ();

  conf->doUpdate_ = 1;
}

void
cmlogXuiConfig::doConnection (char *)
{
  cmlogXuiConfig *conf = cmlogXuiConfig::config ();

  conf->doConnection_ = 1;
}

void
cmlogXuiConfig::useConfigFile (char *file)
{
  char fullname[1024];
  cmlogXuiConfig *conf = cmlogXuiConfig::config ();
  
  if (strchr (file, '/') == 0) {  // this file is in current directory
    char path[1024];
    getcwd (path, sizeof (path));
    sprintf (fullname, "%s/%s",path, file);
  }
  else
    strcpy (fullname, file);
    

  conf->configFile_ = new char[strlen (fullname) + 1];
  strcpy (conf->configFile_, fullname);
}

void
cmlogXuiConfig::doSearchUpdate (char *)
{
  cmlogXuiConfig *conf = cmlogXuiConfig::config ();
  conf->histUpdate_ = 1;
}

void
cmlogXuiConfig::doQuery (char* )
{
  cmlogXuiConfig *conf = cmlogXuiConfig::config ();
  conf->doQuery_ = 1;
}

int
cmlogXuiConfig::queryStartTime (char* timestr, double* start,
				cmlogXuiDateType* type)
{
  char temp[256];
  int dst = -2;
  time_t t;
  struct tm tmv;
  int status = -1;


  if (strptime (timestr, cmlogXuiConfig::configDateFormat_, &tmv) != NULL) {
    tmv.tm_isdst = dst;
    tmv.tm_sec = 0;
    t = mktime (&tmv); 
    *type = POSITIVE;
    *start = (double)t;
    status = 0;
  }
  else {
    int hour = 0;
    int min = 0;
    int match = 0;
    double ct = cmlogUtil::currentTime();
    strncpy (temp, timestr, sizeof(temp) - 1);

    if ((match = sscanf (temp, cmlogXuiConfig::configDateFormat1_, &hour, &min))>=2){
      *start = ct - hour*3600 - min*60;
      *type = NEGATIVE;
      status = 0;
    }
    else if (match == 1) {
      *start = ct - hour*3600;      
      *type = NEGATIVE;
      status = 0;
    }
  }
  return status;
}

int
cmlogXuiConfig::queryEndTime (char* timestr, double* end,
			      cmlogXuiDateType* type)
{
  char temp[256];
  int dst = -2;
  time_t t;
  struct tm tmv;
  int status = -1;

  if (::strcasecmp (timestr, "now") == 0) {
    *end = cmlogUtil::currentTime();
    *type = NOW;
    status = 0;
  }

  if (strptime (timestr, cmlogXuiConfig::configDateFormat_, &tmv) != NULL) {
    tmv.tm_isdst = dst;
    tmv.tm_sec = 0;
    t = mktime (&tmv); 
    *end = (double)t;
    *type = POSITIVE;
    status = 0;
  }
  else {
    int hour = 0;
    int min = 0;
    int match = 0;
    double ct = cmlogUtil::currentTime();
    strncpy (temp, timestr, sizeof(temp) - 1);

    if ((match = sscanf (temp, cmlogXuiConfig::configDateFormat1_, &hour, &min))>=2){
      *end = ct - hour*3600 - min*60;
      *type = NEGATIVE;
      status = 0;
    }
    else if (match == 1) {
      *end = ct - hour*3600;      
      *type = NEGATIVE;
      status = 0;
    }
  }    
  return status;
}


int
cmlogXuiConfig::codeConverter (cdevData& data, cmlogXuiCodeConverter* cs[],
			       int* size)
{
  int  nummatch = 0;
  char temp[80];
  int  i, j;

  for (i = 0; i < numConverters_; i++) {
    // if no selection tag we match every code code converter
    if (!converters_[i]->selectionTagVal ())
      nummatch++;
    else if (data.get (converters_[i]->selectionTag(), temp, sizeof (temp))
	     == CDEV_SUCCESS && !strcmp (temp, converters_[i]->selectionTagVal())) 
      nummatch ++;
  }

  if (*size < nummatch) {
#ifdef _CMLOG_DEBUG
    printf ("cmlogXuiConfig error: codeConverter buffer overflow\n");
#endif
    *size = 0;
    return -1;
  }

  *size = nummatch;
  j = 0;
  for (i = 0; i < numConverters_; i++) {
    // if no selection tag we match every code code converter
    if (!converters_[i]->selectionTagVal ())
      cs[j++] = converters_[i];
    else if (data.get (converters_[i]->selectionTag(), temp, sizeof (temp))
	== CDEV_SUCCESS && !strcmp (temp, converters_[i]->selectionTagVal())) 
      cs[j++] = converters_[i];
  }
  return 0;
}

void
cmlogXuiConfig::colorStringsToPixels (void)
{
  int i;
  for (i = 0; i < numConverters_; i++)
    converters_[i]->colorStringsToPixels ();
}

int
cmlogXuiConfig::blinkOn (void) const
{
  int i;
  for (i = 0; i < numConverters_; i++)
    if (converters_[i]->blinkOn ())
      return 1;
  return 0;
}


int
cmlogXuiConfig::parseSelectionTag (char* input, 
				   char seltag[], int seltaglen,
				   char seltagval[], int seltagvallen)
{
  char temp[256];
  char *p, *q;

  // if we do not care about seltag, we put none here
  if (strcasecmp (input, "none") == 0) 
    return 1;

  // get tag value first
  strncpy (temp, input, sizeof (temp) - 1);
  temp [sizeof(temp) - 1] = '\0';

  p = strrchr (temp, '=');
  if (!p || !*p)
    return -1;
  p++;
  while (!isgraph(*p)) // skip white spaces
    p++;
  // here should be first single quote
  if (*p != '\'')
    return -1;
  p++;
  q = strrchr (temp, '\'');
  if (!q || !*q)
    return -1;
  *q = '\0';
  strncpy (seltagval, p, seltagvallen);

  // get tag 
  strncpy (temp, input, sizeof (temp) - 1);
  temp [sizeof(temp) - 1] = '\0';

  p = strchr(temp, '=');
  if (!p || !*p)
    return -1;
  p--;
  // skip white spaces backforward
  while (!isgraph(*p))
    p--;
  p++;
  *p = '\0';

  strncpy (seltag, temp, seltaglen);
  
  return 0;
}

int
cmlogXuiConfig::parseQueryMessageLine (char* line,
				       char  qmsg[], int qlen,
				       char  stime[],int stlen,
				       char  etime[],int etlen)
{
  int i;
  // reset all buffers
  qmsg[0] = '\0';
  stime[0] = '\0';
  etime[0] = '\0';

  // end of line 
  char* end = &line[strlen(line)];

  // find first quote
  char* ptr = strchr (line, '"');
  if (!ptr)
    return -1;

  char* eqstr = strrchr (line, '"');
  if (!eqstr)
    return -1;
  // move to the beginning of qstr
  ptr++;

  // now ptr should be at the beggining of the query string
  // copy query string to qmsg
  i = 0;
  while (ptr != eqstr && i < qlen) {
    qmsg[i++] = *ptr;
    ptr++;
  }
  qmsg[i] = '\0';

  // find out whether qmsg is an empty str
  i = 0;
  int empty = 1;
  while (i < strlen (qmsg)) {
    if (isgraph(qmsg[i])) {
      empty = 0;
      break;
    }
    i++;
  }
  if (empty)
    strcpy (qmsg, "none");
#ifdef _CMLOG_DEBUG
  printf ("Query message is %s\n", qmsg);
#endif

  // find out do we have anything following end of query string
  ptr = eqstr + 1;
  while (!isgraph(*ptr) && ptr != end)
    ptr++;
  // only query string available in the line
  if (ptr == end)
    return 1;
  
  // remember the beginning of starting time
  char* st = ptr;
  while (isgraph(*ptr))
    ptr++;

  // copy this word to stime
  i = 0;
  while (st != ptr && i < stlen) {
    stime[i++] = *st;
    st++;
  }
  stime[i] = '\0';
#ifdef _CMLOG_DEBUG
  printf ("Start time is: %s \n", stime);
#endif

  // find out do we have anything following end of starting time
  while (!isgraph(*ptr) && ptr != end)
    ptr++;
  // only query string available in the line
  if (ptr == end)
    return 2;

  // remember the beginning of end time
  char* set = ptr;
  while (isgraph(*ptr))
    ptr++;

  // copy this word to stime
  i = 0;
  while (set != ptr && i < etlen) {
    etime[i++] = *set;
    set++;
  }
  etime[i] = '\0';
#ifdef _CMLOG_DEBUG
  printf ("End time is: %s \n", etime);
#endif
  return 3;
}

int
cmlogXuiConfig::parseUpdateMessageLine (char* line,
					char  umsg[], int ulen)
{
  int i;
  // reset all buffers
  umsg[0] = '\0';

  // end of line 
  char* end = &line[strlen(line)];

  // find first quote
  char* ptr = strchr (line, '"');
  if (!ptr)
    return -1;

  char* eustr = strrchr (line, '"');
  if (!eustr)
    return -1;
  // move to the beginning of ustr
  ptr++;

  // now ptr should be at the beggining of the update message string
  // copy query string to qmsg
  i = 0;
  while (ptr != eustr && i < ulen) {
    umsg[i++] = *ptr;
    ptr++;
  }
  umsg[i] = '\0';

  // find out whether update is an empty str
  i = 0;
  int empty = 1;
  while (i < strlen (umsg)) {
    if (isgraph(umsg[i])) {
      empty = 0;
      break;
    }
    i++;
  }
  if (empty)
    strcpy (umsg, "none");

#ifdef _CMLOG_DEBUG
  printf ("Update Message is: %s \n", umsg);
#endif
  return 1;
}


int
cmlogXuiConfig::parseNameLine (char* line,
			       char  name[], int nlen)
{
  int i;
  // reset all buffers
  name[0] = '\0';

  // end of line 
  char* end = &line[strlen(line)];
  
  // skip white spaces first
  char* ptr = line;
  while (!isgraph (*ptr))
    ptr++;

  // skip the first word
  while (isgraph (*ptr))
    ptr++;

  // skip the white spaces again
  while (!isgraph (*ptr) && ptr != end)
    ptr++;

  if (ptr == end)
    return -1;

  strncpy (name, ptr, nlen);
#ifdef _CMLOG_DEBUG
  printf ("Config name is : %s\n", name);
#endif
  return 1;
}

int
cmlogXuiConfig::parseCodeConvLine (char* line,
				   char  disc[], int dlen,
				   char  tag[],  int tlen)
{
  int i;
  // reset all buffers
  disc[0] = '\0';
  tag[0] = '\0';

  // end of line 
  char* end = &line[strlen(line)];

  // find first quote
  char* ptr = strchr (line, '"');
  if (!ptr)
    return -1;

  char* eqstr = strrchr (line, '"');
  if (!eqstr)
    return -1;
  // move to the beginning of qstr
  ptr++;

  // now ptr should be at the beggining of the discriminator string
  // copy this string to disc
  i = 0;
  while (ptr != eqstr && i < dlen) {
    disc[i++] = *ptr;
    ptr++;
  }
  disc[i] = '\0';
  
  // find out whether disc is an empty str
  i = 0;
  int empty = 1;
  while (i < strlen (disc)) {
    if (isgraph(disc[i])) {
      empty = 0;
      break;
    }
    i++;
  }
  if (empty)
    strcpy (disc, "none");
#ifdef _CMLOG_DEBUG
  printf ("codeConversion message is %s\n", disc);
#endif

  // find out do we have anything following end of query string
  ptr = eqstr + 1;
  while (!isgraph(*ptr) && ptr != end)
    ptr++;
  // only query string available in the line
  if (ptr == end)
    return -1;
  
  // remember the beginning of starting time
  char* st = ptr;
  while (isgraph(*ptr))
    ptr++;

  // copy this word to tag
  i = 0;
  while (st != ptr && i < tlen) {
    tag[i++] = *st;
    st++;
  }
  tag[i] = '\0';
#ifdef _CMLOG_DEBUG
  printf ("codeConversion Tag is: %s \n", tag);
#endif

  return 2;
}

void
cmlogXuiConfig::printUsage (char *)
{
  fprintf (stderr, "cmlog usage: \n");
  fprintf (stderr, "-(c)onnect: automatic connection to the server\n");
  fprintf (stderr, "-(u)pdate:  automatic put cmlog into updating mode\n");
  fprintf (stderr, "-f filename:use provided config file\n");
  fprintf (stderr, "-cfg file:  use provided config file\n");
  fprintf (stderr, "-q:         do a query with a query message and a time interval provided \n");
  fprintf (stderr, "            in a config file\n");
  fprintf (stderr, "-hu:        do a query with a query message and a time interval specified \n");
  fprintf (stderr, "            in a config file and put cmlog into updating mode\n");
  fprintf (stderr, "-h:         this help\n");
  exit (0);
}


