//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
//-----------------------------------------------------------------------------
//
// Description:
//      Implementation of code conversion from int to string
//
// Author:  
//      Jie Chen
//      Jefferson Lab HPC Group
//
// Revision History:
//   $Log: cmlogXuiCodeConverter.cc,v $
//   Revision 1.5  2000/12/13 16:44:14  chen
//   add action and bell
//
//   Revision 1.4  2000/01/04 14:18:40  chen
//   improve performance for rapid incoming data
//
//   Revision 1.3  1999/11/12 17:32:00  chen
//   beta 2.0
//
//   Revision 1.2  1999/10/29 17:40:06  chen
//   support spaces and quotes in cmlogrc file
//
//   Revision 1.1.1.1  1999/09/07 15:29:13  chen
//   CMLOG version 2.0
//
//
//
#include <XcodaColor.h>
#include <cdevData.h>
#include <cmlogXuiConfig.h>
#include "cmlogXuiCodeConverter.h"

#define CMLOG_FLASH "flash"
#define CMLOG_BLINK "blink"
#define CMLOG_BELL  "bell"

//======================================================================
// Implementation of cmlogXuiCodeItem Class
//======================================================================

cmlogXuiCodeItem::cmlogXuiCodeItem (void)
  :code_ (0), str_ (0), colorStr_ (0), color_ (0), 
  action_ (0), blink_ (0), bell_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogXuiCodeItem Class Object\n");
#endif
}

cmlogXuiCodeItem::cmlogXuiCodeItem (long code, char* str, 
				    char* color, char* desc)
  :code_ (code), color_ (0), action_ (0), blink_ (0), bell_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogXuiCodeItem Class Object\n");
#endif
  if (::strcasecmp (str, "none") == 0)
    str_ = 0;
  else {
    str_ = new char[::strlen (str) + 1];
    ::strcpy (str_, str);
  }

  if (::strcasecmp (color, "none") == 0) 
    colorStr_ = 0;
  else {
    colorStr_ = new char[::strlen (color) + 1];
    ::strcpy (colorStr_, color);
  }

  if (desc)
    parseDescription (desc);
}

cmlogXuiCodeItem::~cmlogXuiCodeItem (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogXuiCodeItem Class Object\n");
#endif
  if (colorStr_)
    delete []colorStr_;

  if (str_)
    delete []str_;

  if (action_)
    delete []action_;
}

cmlogXuiCodeItem::cmlogXuiCodeItem (const cmlogXuiCodeItem& item)
  :code_ (item.code_), color_ (item.color_), str_ (0), colorStr_ (0),
   action_ (0), blink_ (item.blink_), bell_ (item.bell_)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogXuiCodeItem Class Object\n");
#endif
  if (item.str_) {
    str_ = new char[::strlen (item.str_) + 1];
    ::strcpy (str_, item.str_);
  }

  if (item.colorStr_) {
    colorStr_ = new char[::strlen (item.colorStr_) + 1];
    ::strcpy (colorStr_, item.colorStr_);
  }

  if (item.action_) {
    action_ = new char[::strlen (item.action_) + 1];
    ::strcpy (action_, item.action_);
  }
}

cmlogXuiCodeItem &
cmlogXuiCodeItem::operator = (const cmlogXuiCodeItem& item)
{
  if (this != &item) {
    if (str_) {
      delete []str_;
      str_ = 0;
    }
    if (colorStr_) {
      delete []colorStr_;
      colorStr_ = 0;
    }
    if (action_) {
      delete []action_;
      action_ = 0;
    }

    code_ = item.code_;
    color_ = item.color_;
    blink_ = item.blink_;
    bell_ = item.bell_;

    if (item.str_) {
      str_ = new char[::strlen (item.str_) + 1];
      ::strcpy (str_, item.str_);
    }

    if (item.colorStr_) {
      colorStr_ = new char[::strlen (item.colorStr_) + 1];
      ::strcpy (colorStr_, item.colorStr_);
    }

    if (item.action_) {
      action_ = new char[::strlen (item.action_) + 1];
      ::strcpy (action_, item.action_);
    }
  }
  return *this;
}

int
cmlogXuiCodeItem::generateTokens (char* desc, char sep, char* tokens[], unsigned int num)
{
  char  *p, *q;
  char  *t;
  char  item[128];
  int   ntokens = 0;
  int   i = 0;

  p = desc;
  while (p && *p) {
    t = item;

    q = ::strchr (p, sep);
    if (!q) /* end of string */
      q = p + ::strlen (p);


    while (p != q) {
      *t = *p;
      t++;
      p++;
    }
    *t = '\0';

    if (i >= num) 
      return i + 1;
    
    tokens[i] = new char[::strlen (item) + 1];
    strcpy (tokens[i++], item);

    // move p and q forward one position if we are not at the end
    if (*p) {
      p++; q++;
    }
  }
  return i;
}

  

void
cmlogXuiCodeItem::parseDescription (char* desc)
{
  char* token[3];
  int  numtokens = 0;
  int  i;

  if ((numtokens = generateTokens (desc, ':', token, 3)) == 0) {
    fprintf (stderr, "Configuration file error: action description format incorrect\n");
    fprintf (stderr, "Provided action description is: %s\n", desc);
    fprintf (stderr, "Expected action description must be in [blink][:bell:script]\n");
    return;
  }
  
  for (i = 0; i < numtokens; i++) {
    if (::strcasecmp (token[i], CMLOG_FLASH) == 0 ||
	::strcasecmp (token[i], CMLOG_BLINK) == 0) 
      blink_ = 1;
    else if (::strcasecmp (token[i], CMLOG_BELL) == 0) 
      bell_ = 1;
    else {
      action_ = new char[::strlen (token[i]) + 1];
      ::strcpy (action_, token[i]);
    }
    delete []token[i];
  }
}
    

//======================================================================
//  Implementation of cmlogXuiCodeConverter
//======================================================================
cmlogXuiCodeConverter::cmlogXuiCodeConverter (int seltag, int convtag, 
					      char* seltagValue)
  :selTag_ (seltag), convTag_ (convtag), size_ (0), bufsize_ (10)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogXuiCodeConverter Class Object\n");
#endif
  if (seltagValue && *seltagValue) {
    seltagValue_ = new char[::strlen (seltagValue) + 1];
    ::strcpy (seltagValue_, seltagValue);
  }
  else 
    seltagValue_ = 0;

  item_ = new cmlogXuiCodeItem[bufsize_];
}

cmlogXuiCodeConverter::~cmlogXuiCodeConverter (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogXuiCodeConverter Class Object\n");
#endif
  if (seltagValue_)
    delete []seltagValue_;
  delete []item_;
}

int
cmlogXuiCodeConverter::input (FILE* input)
{
  long code;
  char codestr[40];
  char colorstr[40];
  char line[256];
  char desc[80];
  int  foundleftbract = 0;
  int  foundrightbract = 0;
  int  nummatch;

  fgets (line, sizeof (line), input);
  if (line[0] != '#' && !foundleftbract) {
    if (line[0] != '{') {
      fprintf (stderr, "Config file syntax error: Cannot find left bract { \n");
      exit (1);
    }
    foundleftbract = 1;
  }
  while (!foundrightbract && !feof(input)) {
    fgets (line, sizeof (line), input);
    if (line[0] != '#') {
      if (line[0] == '}') 
	foundrightbract = 1;
      else if ((nummatch = sscanf (line, "%d %s %s %s",
				   &code,codestr,colorstr,desc)) == 3) 
	addConversionInfo (code, codestr, colorstr, 0);
      else if (nummatch == 4)
	addConversionInfo (code, codestr, colorstr, desc);	
    }
  }
  if (!foundrightbract) {
    fprintf (stderr, "Config file syntax error: Cannot find right bract }\n");
    exit (1);
  }

  return 0;
}

int
cmlogXuiCodeConverter::output (FILE* out)
{
  int   needsep = 0;
  char* seltag_c = 0;
  char* convtag_c = 0;
  cmlogXuiConfig* config = cmlogXuiConfig::config ();

  if (seltagValue_ && cdevData::tagI2C (selTag_, seltag_c) != CDEV_SUCCESS)
    return -1;
  if (cdevData::tagI2C (convTag_, convtag_c) != CDEV_SUCCESS)
    return -1;

  fprintf (out, "#Code Conversion\n");
  if (seltagValue_)
    fprintf (out, "codeConversion \"%s=='%s'\" %s\n", config->titleOfTag(seltag_c),
	     seltagValue_, config->titleOfTag (convtag_c));
  else
    fprintf (out, "codeConversion \"none\" %s\n", config->titleOfTag (convtag_c));

  fprintf (out, "{\n");
  for (int i = 0; i < size_; i++) {
    needsep = 0;
    fprintf (out, "%d          %s         %s", item_[i].code_,
	     item_[i].str_ ? item_[i].str_ : "none", 
	     item_[i].colorStr_ ? item_[i].colorStr_ : "none");

    if (item_[i].blink_) {
      fprintf (out, "         flash");
      needsep = 1;
    }
    if (item_[i].bell_) {
      if (needsep)
	fprintf (out,":bell");
      else
	fprintf (out,"bell");
      needsep = 1;
    }
    if (item_[i].action_) {
      if (needsep)
	fprintf (out,":%s",item_[i].action_);
      else
	fprintf (out,"%s", item_[i].action_);
    }

    fprintf (out, "\n");

  }
  fprintf (out, "}\n");
}

int
cmlogXuiCodeConverter::blinkOn (void) const
{
  int i = 0;

  for (i = 0; i < size_; i++)
    if (item_[i].blink_)
      return 1;

  return 0;
}


int
cmlogXuiCodeConverter::addConversionInfo (long code, char* str, char* color,
					  char* desc)
{
  int i;
  cmlogXuiCodeItem titem (code, str, color, desc);
	
  if (size_ == bufsize_ - 1) {
    cmlogXuiCodeItem* nitem = new cmlogXuiCodeItem[2*bufsize_];
    for (i = 0; i < bufsize_; i++)
      nitem[i] = item_[i];
    
    delete []item_;
    bufsize_ = 2*bufsize_;
    item_ = nitem;
  }
  item_[size_++] = titem;
  return 0;
}

int
cmlogXuiCodeConverter::find (long code, char str[], int bufsize, 
			     Pixel* color, int* blink)
{
  for (int i = 0; i < size_; i++)
    if (code == item_[i].code_) {
      if (item_[i].str_) 
	strncpy (str, item_[i].str_, bufsize);
      else
	sprintf (str, "%d", code);

      *color = item_[i].color_;
      *blink = item_[i].blink_;
      return 1;
    }
  
  return 0;
}

int
cmlogXuiCodeConverter::find (char* string, Pixel *color, int* blink)
{
  for (int i = 0; i < size_; i++)
    if (item_[i].str_ && ::strcmp (string, item_[i].str_) == 0) {
      *color = item_[i].color_;
      *blink = item_[i].blink_;
      return 1;
    }
  return 0;
}

void
cmlogXuiCodeConverter::colorStringsToPixels (void)
{
  for (int i = 0; i < size_; i++) {
    if (item_[i].colorStr_)
      item_[i].color_ = XcodaColor::getPixel (item_[i].colorStr_);
    else
      item_[i].color_ = 0;
  }
}

int
cmlogXuiCodeConverter::doBell (long code)
{
  for (int i = 0; i < size_; i++)
    if (code == item_[i].code_) 
      return item_[i].bell_;
  
  return 0;  
}

int
cmlogXuiCodeConverter::doBell (char* value)
{
  for (int i = 0; i < size_; i++)
    if (item_[i].str_ && ::strcmp (value, item_[i].str_) == 0) 
      return item_[i].bell_;

  return 0;
}
  

int
cmlogXuiCodeConverter::hasAction (long code, char action[], unsigned int size)
{
  for (int i = 0; i < size_; i++) {
    if (code == item_[i].code_ && item_[i].action_) {
      strncpy (action, item_[i].action_, size);
      return 1;
    }
  }
  return 0;    
}

int
cmlogXuiCodeConverter::hasAction (char* value, char action[], unsigned int size)
{
  for (int i = 0; i < size_; i++) {
    if (item_[i].str_ && ::strcmp (value, item_[i].str_) == 0 && item_[i].action_) {
      strncpy (action, item_[i].action_, size);
      return 1;
    }
  }
  return 0;    
}


