//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Simple pthread synchnization class wrapper
//      Or VxWorks synchronization mechanism
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cpSynch.h,v $
//   Revision 1.1.1.1  1999/09/07 15:29:12  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:30:22  bickley
// Added cmlog to application development system.
//
//
#ifndef _CP_SYNCH_H
#define _CP_SYNCH_H

#include <cpThread.h>

// C++ class wrapper for pthread mutex and VxWorks semaphores
class cpMutex
{
public:
#ifndef __vxworks
  // constructor and destructor
  cpMutex (int type = CP_THREAD_PROCESS_PRIVATE);
#else
  cpMutex (int type = SEM_Q_PRIORITY | SEM_DELETE_SAFE | SEM_INVERSION_SAFE);
#endif
  // implicitly destroy mutex
  virtual ~cpMutex (void);

  // operations

  // destroy mutex
  int remove (void);
  // get lock
  int acquire (void);

  // noblock get lock
  int tryAcquire (void);

  // release lock
  int release (void);

  // class name
  virtual const char* className (void) const {return "cpMutex";}

#ifndef __vxworks
  // not intended for application programmer, for all other classes
  pthread_mutex_t lock_;
#else
  SEM_ID lock_;
#endif

private:
  // deny assignment and copy operations
  cpMutex (const cpMutex& mutex);
  cpMutex& operator = (const cpMutex& mutex);
};

class cpRWMutex
{
public:
#ifndef __vxworks
  // constructor and destructor
  cpRWMutex (int type = CP_THREAD_PROCESS_PRIVATE);
#else
  cpRWMutex (int type = SEM_Q_PRIORITY);
#endif
  virtual ~cpRWMutex (void);

  // operations
  // destroy lock
  int remove (void);
  // acquire read lock
  int acquireRLock (void);
  // get write lock (void);
  int acquireWLock (void);
  // release lock
  int release (void);

  // class name
  virtual const char* className (void) const {return "cpRWMutex";}

#ifndef __vxworks
  // not for public use
  pthread_mutex_t    lock_;
  pthread_cond_t     readers_ok_;
  pthread_cond_t     writers_ok_;
#else
  SEM_ID              lock_;
  // binary semaphores
  SEM_ID              readers_ok_;
  SEM_ID              writers_ok_;
#endif
  int                rwlock_;  // >0 = # readers, <0 writer, 0 none
  unsigned int       waitingWriters_; // number of waiting writers

private:
  // deny access to the copy and assignment operations
  cpRWMutex (const cpRWMutex& );
  cpRWMutex& operator = (const cpRWMutex& );
};

// This class allow recursive call lock to itself
class cpRecursiveMutex
{
public:
  // constructor and destructor
#ifndef __vxworks
  cpRecursiveMutex (int type = CP_THREAD_PROCESS_PRIVATE);
#else
  cpRecursiveMutex (int type = SEM_Q_PRIORITY | SEM_DELETE_SAFE | SEM_INVERSION_SAFE);
#endif

  virtual ~cpRecursiveMutex (void);

  // operations
  int acquire    (void);
  int tryAcquire (void);
  int release    (void);
  // return internal lock
  operator cpMutex &();

  // return thread id that holds the lock
  cp_thread_t getThreadId (void);
  int         getNestingLevel (void);
  
  virtual const char* className (void) const {return "cpRecursiveMutex";}

private:
  void setNestingLevel (int d);
  void setThreadId     (cp_thread_t t);

  // This is the internal lock
  cpMutex lock_;
  // this lock handles the state of the class
  cpMutex mutex_;

  cp_thread_t thrId_;

  int         nestingLevel_;

  // deny access to assign and copy operations
  cpRecursiveMutex (const cpRecursiveMutex& );
  cpRecursiveMutex& operator = (const cpRecursiveMutex& );
};


class cpConditionMutex
{
public:
  // constructor and destructor
#ifndef __vxworks
  cpConditionMutex (cpMutex& m, int type = CP_THREAD_PROCESS_PRIVATE);
#else
  cpConditionMutex (cpMutex& m,
		    int type = SEM_Q_PRIORITY);
#endif

  virtual ~cpConditionMutex (void);

  // Block on a condition for certain time, if time == 0, wait forever
  int wait (const struct timespec* abs = 0);
  
  // signal one waiting thread
  int signal (void);
  // wake up all waiting threads
  int broadcast (void);

  // return reference to the underlying mutex
  cpMutex& mutex (void);

private:
#ifndef __vxworks
  pthread_cond_t cond_;
#else
  SEM_ID          cond_;
#endif
  cpMutex& mutex_;

  // deny access to copy and assignment operations
  cpConditionMutex (const cpConditionMutex& c);
  cpConditionMutex& operator = (const cpConditionMutex &);
};

// Conditional Variable 
// waiting for writer lock has higher priority than waiting
// for reader lock
class cpConditionRWMutex
{
public:
  // constructor and destructor
#ifndef __vxworks
  cpConditionRWMutex (cpRWMutex& m, int type = CP_THREAD_PROCESS_PRIVATE);
#else
  cpConditionRWMutex (cpRWMutex& m,
		      int type = SEM_Q_PRIORITY);
#endif
  virtual ~cpConditionRWMutex (void);

  // Block on a condition for certain time, if time == 0, wait forever
  int wait (const struct timespec* abs = 0);
  
  // signal one waiting thread
  int signal (void);
  // wake up all waiting threads
  int broadcast (void);

  // return reference to the underlying mutex
  cpRWMutex& mutex (void);

private:
#ifndef __vxworks
  pthread_cond_t cond_;
#else
  SEM_ID         cond_;
#endif
  cpRWMutex& mutex_;

  // deny access to copy and assignment operations
  cpConditionRWMutex (const cpConditionRWMutex& c);
  cpConditionRWMutex& operator = (const cpConditionRWMutex &);
};

// This class is to be used whithin a method or function.
// It performs automatic acquisition/release of a cpMutex
class cpMutexGuard
{
public:
  // implicitly acquire lock
  cpMutexGuard (cpMutex& m);
  // implicitly release lock
  ~cpMutexGuard (void);

  virtual const char* className (void) const {return "cpMutexGuard";}

protected:
  cpMutex& m_;

private:
  // deny copy and assignment operations
  cpMutexGuard (const cpMutexGuard& );
  cpMutexGuard& operator = (const cpMutexGuard& );
};

// This class is to be used whithin a method or function.
// It performs automatic acquisition/release of a cpRecursiveMutex
class cpRecursiveMutexGuard
{
public:
  // implicitly acquire lock
  cpRecursiveMutexGuard (cpRecursiveMutex& m);
  // implicitly release lock
  ~cpRecursiveMutexGuard (void);

  virtual const char* className (void) const {return "cpRecursiveMutexGuard";}

protected:
  cpRecursiveMutex& m_;

private:
  // deny copy and assignment operations
  cpRecursiveMutexGuard (const cpRecursiveMutexGuard& );
  cpRecursiveMutexGuard& operator = (const cpRecursiveMutexGuard& );
};

#endif

  
  


  
  
  
