//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Implemenation of signal handler
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogSignalHandler.cc,v $
//   Revision 1.8  2001/07/25 14:30:47  chen
//   64 BIT Initial Port
//
//   Revision 1.7  2000/06/20 19:35:44  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.6  2000/03/17 19:15:31  chen
//   use unique file name to track siginfo
//
//   Revision 1.5  2000/03/01 19:49:29  chen
//   remove error on threaded version
//
//   Revision 1.4  2000/03/01 19:20:10  chen
//   remove SIGEMT from SIGNAL List
//
//   Revision 1.3  2000/03/01 17:09:31  chen
//   Add concurrency control and output signa info to a file
//
//   Revision 1.2  1999/10/29 17:39:11  chen
//   Fix a minor thing for hpux
//
//   Revision 1.1.1.1  1999/09/07 15:29:11  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:30:04  bickley
// Added cmlog to application development system.
//
//
#include "cmlogSignalHandler.h"
#include <cmlogUtil.h>

int cmlogSignalHandler::signals[] = 
{SIGINT, SIGQUIT, SIGILL, SIGIOT, 
 SIGFPE, SIGSYS, SIGPIPE, SIGBUS, SIGSEGV, SIGTERM, 
 SIGTSTP, SIGCONT, SIGIO, SIGPOLL, SIGXCPU, SIGXFSZ};

int cmlogSignalHandler::numSignals = 16;

FILE* cmlogSignalHandler::siginfofile_ = 0;

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
sigset_t cmlogSignalHandler::sig_set;

int
cmlogSignalHandler::registerSignalHandlers (void)
{
  // create a file that store signal information
  char sigfilename[256];
  
  strcpy (sigfilename, "/tmp/cmlog_signal_info.XXXXXX");
  if (!mktemp (sigfilename))
    strcpy (sigfilename, "/tmp/cmlog_signal_info");

  cmlogSignalHandler::siginfofile_ = fopen (sigfilename, "w");
  if (!cmlogSignalHandler::siginfofile_) {
    cmlogUtil::serverPrintf ("Cannot open signal information file %s\n",
			     sigfilename);
    cmlogSignalHandler::siginfofile_ = stderr;
  }
  
  // handles many signals here
  sigemptyset (&(cmlogSignalHandler::sig_set));

  for (int i = 0; i < cmlogSignalHandler::numSignals; i++)
    if (sigaddset (&(cmlogSignalHandler::sig_set), 
		   cmlogSignalHandler::signals[i]) != 0)
      cmlogUtil::serverPrintf ("Cannot add signal %d to signal set\n",
			       cmlogSignalHandler::signals[i]);
  
  // block all these signals in all threads except my signal handler
  return cpThread::sigsetmask (SIG_BLOCK, 
			       &(cmlogSignalHandler::sig_set), NULL);
}

void *
cmlogSignalHandler::signalFunc (void *arg)
{
  int signo, err;
  // signal function send message to this file descriptor
  int writefd = *(int *)arg;

  for (;;) {
    err = sigwait (&(cmlogSignalHandler::sig_set), &signo);
    if (err) 
      ::exit (1);

    cmlogUtil::serverPrintf ("Interrupted by %d signal\n", signo);
    fprintf (cmlogSignalHandler::siginfofile_, 
	     "Interrupted by %d signal\n", signo);
    fflush (cmlogSignalHandler::siginfofile_);
    
    if (signo != SIGPIPE && signo != SIGTSTP &&
	signo != SIGCONT ) {
      cmlogUtil::serverPrintf ("Stop cmlog server\n");
      fprintf (cmlogSignalHandler::siginfofile_, "Stop cmlog server\n"); 
      fflush (cmlogSignalHandler::siginfofile_);

      // since we are using signal waiting mechanism, we are not restricted
      // by usinbg async-safe functions only
      cdevData unused;
      cmlogMsg msg ((long)CMLOG_EXIT, unused, 0);
      int num = writefd << msg;

      if (num <= 0) {
	cmlogUtil::serverPrintf ("Cannot send message back to the tcp thread, quit\n");
	::exit (1);
      }
    }
  }
  return 0;
}
#else
extern void cmlogCleanup (void);

int
cmlogSignalHandler::registerSignalHandlers (void)
{
  // create a file that store signal information
  char sigfilename[256];

  strcpy (sigfilename, "/tmp/cmlog_signal_info.XXXXXX");
  if (!mktemp (sigfilename))
    strcpy (sigfilename, "/tmp/cmlog_signal_info");

  cmlogSignalHandler::siginfofile_ = fopen (sigfilename, "w");
  if (!cmlogSignalHandler::siginfofile_) {
    cmlogUtil::serverPrintf ("Cannot open signal information file %s\n",
			     sigfilename);
    cmlogSignalHandler::siginfofile_ = stderr;
  }

  // handle many signals
  struct sigaction act, oact;

  act.sa_handler = &(cmlogSignalHandler::signalFunc);
  sigemptyset (&act.sa_mask);
  act.sa_flags = 0;
#ifdef SA_RESTART
  act.sa_flags |= SA_RESTART;
#endif

  for (int i = 0; i < cmlogSignalHandler::numSignals; i++) 
    if (sigaction (cmlogSignalHandler::signals[i], &act, &oact) < 0)
      return -1;
  
  return 0;
}

void
cmlogSignalHandler::signalFunc (int signo)
{
#ifdef solaris
  char signame[SIG2STR_MAX];
  if (sig2str (signo, signame) == -1)
    sprintf (signame, "unknown");

  cmlogUtil::serverPrintf (stderr, "Interrupted by %s signal\n", signame);
  fprintf (cmlogSignalHandler::siginfofile_, 
	   "Interrupted by %s signal\n", signame);
  fflush (cmlogSignalHandler::siginfofile_);
    

  if (signo == SIGSEGV || signo == SIGBUS) {
    cmlogUtil::serverPrintf ("Fatal: Caught signal %s, Quit.......\n", signame);
    fprintf (cmlogSignalHandler::siginfofile_, 
	     "Fatal: Caught signal %s, Quit.......\n", signame);
    fflush  (cmlogSignalHandler::siginfofile_);
    manager.sendSignal (signo, 0);
    finished = 1;
    abort ();
  }
  else if (signo != SIGPIPE && signo != SIGTSTP &&
	   signo != SIGCONT ) {
    manager.sendSignal (signo, 0);
    finished = 1; // exit from runcontrol
  }
#else
  cmlogUtil::serverPrintf ("Interrupted by %d signal\n", signo);
  fprintf (cmlogSignalHandler::siginfofile_, 
	   "Interrupted by %d signal\n", signo);
  fflush (cmlogSignalHandler::siginfofile_);

  if (signo == SIGSEGV || signo == SIGBUS) {
    cmlogUtil::serverPrintf ("Fatal: Caught signal %d, Quit.......\n", signo);
    fprintf (cmlogSignalHandler::siginfofile_, 
	     "Fatal: Caught signal %d, Quit.......\n", signo);
    fflush (cmlogSignalHandler::siginfofile_);
    manager.sendSignal (signo, 0);
    finished = 1;
    cmlogCleanup();
    abort ();
  }
  else if (signo != SIGPIPE && signo != SIGTSTP &&
	   signo != SIGCONT ) {
    manager.sendSignal (signo, 0);
    finished = 1; // exit from runcontrol
    cmlogCleanup();
    cmlogUtil::serverPrintf ("Shutting down cmlog server\n");
    exit (0);     
  }
#endif
}

#endif
