//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Implementation of CMLOG Client Daemon User Process (Task) IO Handler
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogUpIO.cc,v $
//   Revision 1.4  1999/12/13 21:24:39  chen
//   Add more error checking using BAD_PTR
//
//   Revision 1.3  1999/12/13 15:17:34  chen
//   Add robust check on all incoming messages
//
//   Revision 1.2  1999/09/20 17:03:16  chen
//   Fix cmlogVxLogDisable(enable)
//
//   Revision 1.1.1.1  1999/09/07 15:29:11  chen
//   CMLOG version 2.0
//
// Revision 1.4  1997/09/30  18:59:55  chen
// fix a potential problem when client daemon is shutdown
//
// Revision 1.3  1997/08/26  16:57:17  chen
// cmlogClientD can be delete/restart on vx
//
// Revision 1.2  1997/08/25  18:24:03  chen
// add umask (0)
//
// Revision 1.1  1997/08/01  15:29:26  bickley
// Added cmlog to application development system.
//
//
//
#include <cmlogUtil.h>
#include "cmlogUpIO.h"

#ifndef __vxworks
cmlogUpIO::cmlogUpIO (ACE_Reactor& reactor, cmlogUxToServer& toServer)
:reactor_ (reactor), toServer_ (toServer), fifo_ ()
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogUpIO Class Object\n");
#endif
  unsigned long ipaddr = cmlogUtil::hostIpAddr ();
  char     pipepath [128];
  sprintf (pipepath, CMLOG_CLNT_PIPE, ipaddr);

  // set umask to 0 to make sure everyone can read/write to this pipe
  umask (0);

  if (fifo_.open (pipepath) == -1) 
    fprintf (stderr, "cannot open pipe at %s\n", pipepath);

  //===============================================================
  //    If I turn a unix pipe into a non-block device,
  //    read from the pipe may result unpredictable reading
  //    when writer (clients) sending large amount of data
  //    especially on Linux (2.0.x)
  //    jie chen 12/11/1999
  //    fifo_.enable (ACE_NONBLOCK);
  //===============================================================
}

cmlogUpIO::~cmlogUpIO (void)
{
#ifdef _TRACE_OBJECTS
  printf ("    Delete cmlogUpIO Class Object\n");
#endif
  fifo_.remove ();  // remove from file system
}

int
cmlogUpIO::get_handle (void) const
{
  return fifo_.get_handle ();
}

int
cmlogUpIO::handle_close (int, ACE_Reactor_Mask)
{
  return fifo_.close ();
}

int
cmlogUpIO::handle_input (int)
{
  cmlog_cdevPacket1 packet;
  int fd = get_handle ();

  int n =  fd >> packet;
#ifdef _CMLOG_DEBUG
  printf ("Received %d bytes from user process\n", n);
#endif

  if (n <= 0) {
#ifdef _CMLOG_DEBUG
    printf ("User process disconnected from client daemon\n");
#endif
    // if this is the last one attached to the daemon, my daemon will
    // stay here forever ::::

    // cannot return -1: since reactor will close this
    return 0;
  }
  toServer_.insertMessageToBuffer (packet, n);

  return 0;
}

#else  /* not vxworks */

/* global symbal for fifo */
char* cmlogFifoFdSymVx_ = 0;

/* global mutex lock      */
extern cpMutex*           cmlog_rlock;

cmlogUpIO::cmlogUpIO (ACE_Reactor& reactor, cmlogUxToServer& toServer)
:reactor_ (reactor), toServer_ (toServer), fifo_ (0), dropped_ (0), dolog_ (1)
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogUpIO Class Object\n");
#endif
  unsigned long ipaddr = cmlogUtil::hostIpAddr ();
  char     pipepath [128];
  sprintf (pipepath, CMLOG_CLNT_PIPE, ipaddr);

  if (pipeDevCreate (pipepath,
		     CMLOG_CLNT_PIPE_NUM_MSGS,
		     CMLOG_CLNT_PIPE_MAX_BYTES) != OK) {
    fprintf (stderr, "Cannot create pipe device %s\n", pipepath);
    exit (1);
  }
  else {
    fifo_ = open (pipepath, O_RDWR, 0666);
    if (fifo_ == ERROR) {
      fprintf (stderr, "Cannot open pipe %s\n", pipepath);
      exit (1);
    }
  }
  cmlogFifoFdSymVx_ = (char *)this;
}

cmlogUpIO::~cmlogUpIO (void)
{
#ifdef _TRACE_OBJECTS
  printf ("    Delete cmlogUpIO Class Object\n");
#endif

  // lock all these symbols
  cpMutexGuard guard (*cmlog_rlock);

  close (fifo_);
  cmlogFifoFdSymVx_ = 0;
  fifo_ = 0;

  // remove device  from device list
  removeDevice ();
}

int
cmlogUpIO::get_handle (void) const
{
  return fifo_;
}

int
cmlogUpIO::handle_close (int, ACE_Reactor_Mask)
{
  // lock all these symbols
  cpMutexGuard guard (*cmlog_rlock);

  close (fifo_);
  cmlogFifoFdSymVx_ = 0;
  fifo_ = 0;

  // remove device  from device list
  removeDevice ();
  return 0;
}

int
cmlogUpIO::handle_input (int)
{
  cmlog_cdevPacket1 packet;
  int fd = get_handle ();

  int n =  fd >> packet;
#ifdef _CMLOG_DEBUG
  // printf ("Received %d bytes from user process\n", n);
#endif

  if (n <= 0) {
#ifdef _CMLOG_DEBUG
    printf ("User process disconnected from client daemon\n");
#endif
    // if this is the last one attached to the daemon, my daemon will
    // stay here forever ::::
    // cannot return -1 since reactor will close this
    return 0;
  }
  if (dolog_) 
    toServer_.insertMessageToBuffer (packet, n);

  // since we are using static buffer, detach packet from static buffer
  packet.detachData ();

  return 0;
}

void
cmlogUpIO::removeDevice (void)
{
  unsigned long ipaddr = cmlogUtil::hostIpAddr ();
  char     pipepath [128];
  sprintf (pipepath, CMLOG_CLNT_PIPE, ipaddr);

  DEV_HDR* dev = iosDevFind (pipepath, 0);
  if (dev) 
    iosDevDelete (dev);
}

// Important note: the following routines are called with lock held
int
cmlogUpIO::incDroppedMsgsCount (void)
{
  dropped_ ++;
  return 0;
}

int
cmlogUpIO::fifoFull ()
{
  int nmessages;

  ioctl (fifo_, FIONMSGS, (int) &nmessages);
  
  if (nmessages == CMLOG_CLNT_PIPE_NUM_MSGS)
    return 1;
  return 0;
}

int
cmlogVxFifoFd (void)
{
  cmlogUpIO* ioh = (cmlogUpIO *)cmlogFifoFdSymVx_;

  if (ioh != 0)
    return ioh->get_handle ();
  return 0;
}

int
cmlogVxIncDroppedMsgsCount (void)
{
  cmlogUpIO* ioh = (cmlogUpIO *)cmlogFifoFdSymVx_;

  if (ioh != 0)
    return ioh->incDroppedMsgsCount ();  
  return 0;
}

int
cmlogVxFifoFull (void)
{
  cmlogUpIO* ioh = (cmlogUpIO *)cmlogFifoFdSymVx_;
  
  if (ioh != 0)
    return ioh->fifoFull ();
  return 1;
}

void
cmlogVxLogDisable (void)
{
  cmlogUpIO* ioh = (cmlogUpIO *)cmlogFifoFdSymVx_;
  if (ioh != 0)
    ioh->disableLog ();
}

void
cmlogVxLogEnable (void)
{
  cmlogUpIO* ioh = (cmlogUpIO *)cmlogFifoFdSymVx_;
  if (ioh != 0)
    ioh->enableLog ();
}
#endif
