//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Implementation of cmlogServer Heart beat monitor at unix side
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogServerHB.cc,v $
//   Revision 1.5  2000/06/20 19:35:16  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.4  2000/03/06 18:37:54  chen
//   Change unsigned short to short for clientid_
//
//   Revision 1.3  2000/02/07 16:13:03  chen
//   check buffer to msg return -1 for bad client
//
//   Revision 1.2  1999/10/19 17:56:36  chen
//   Fix a bug for short interger clientID overflow
//
//   Revision 1.1.1.1  1999/09/07 15:29:11  chen
//   CMLOG version 2.0
//
// Revision 1.3  1998/03/27  14:00:00  chen
// Fix a bug when a client application connected to a non-connected client daemon
//
// Revision 1.2  1998/01/20  20:38:35  chen
// add reconnection message to client daemons
//
// Revision 1.1  1997/08/01  15:29:20  bickley
// Added cmlog to application development system.
//
//
//
#include <cmlogMsg.h>
#include <cmlogUtil.h>
#include <cmlogUxToServer.h>
#include <cmlogUpIO.h>
#include "cmlogServerHB.h"

//=================================================================
//       Implementation of cmlogClientId Class
//=================================================================
cmlogClientId:: cmlogClientId (short id, int proc, cdevData& cxt,
			       cdevData& tagmap )
:id_ (id), pid_ (proc), cxt_ (cxt), tagmap_ (tagmap)
{
  // empty
}

cmlogClientId::~cmlogClientId (void)
{
  // empty
}


short
cmlogClientId::clientId (void) const
{
  return id_;
}

int
cmlogClientId::procId (void) const
{
  return pid_;
}

cdevData&
cmlogClientId::context (void)
{
  return cxt_;
}

cdevData&
cmlogClientId::tagmap (void)
{
  return tagmap_;
}


//=================================================================
//        Implementation of cmlogServerHB Class
//=================================================================

#ifdef __vxworks
/* global symbol for access this object in vxworks             */
char* cmlogServerHBSymVx_ = 0;
#endif

/* if client receives no heart beat 5 consective times, client */
/* assumes the server is dead                                  */
int cmlogServerHB::serverHBTh = 5;  

#ifndef __vxworks
cmlogServerHB::cmlogServerHB (ACE_Reactor& reactor)
:reactor_ (reactor), listener_ (), serverHB_ (0),
 timerOn_ (0), toServer_ (0), clientId_ (1234), clientList_ ()
#else
cmlogServerHB::cmlogServerHB (ACE_Reactor& reactor)
:reactor_ (reactor), listener_ (), serverHB_ (0),
 timerOn_ (0), toServer_ (0), 
 clientId_ (1234), clientList_ ()
#endif
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogServerHB Class Object\n");
#endif

#ifdef __vxworks
  cmlogServerHBSymVx_ = (char *)this;
#endif
}

cmlogServerHB::~cmlogServerHB (void)
{
#ifdef _TRACE_OBEJCTS
  printf ("Delete heartBeat Class Object\n");
#endif
  handle_close (-1, ACE_Event_Handler::READ_MASK);
  if (timerOn_)
    reactor_.cancel_timer (this);
  timerOn_ = 0;

#ifdef __vxworks
  cmlogServerHBSymVx_ = 0;
#endif

  // free memory of client list
  cmlogSlistIterator ite (clientList_);
  cmlogClientId* cid = 0;

  for (ite.init (); !ite; ++ite) {
    cid = (cmlogClientId *) ite ();
    delete cid;
  }
}

int
cmlogServerHB::handle_close (int, ACE_Reactor_Mask)
{
  return listener_.close ();
}

int
cmlogServerHB::open (void)
{
  ACE_INET_Addr addr (CMLOG_CLNT_PORT);

  if (listener_.open (addr) == -1) {
#ifdef _CMLOG_DEBUG
    printf ("Cannot attach to local port %d\n", CMLOG_CLNT_PORT);
#endif
    return -1;
  }
  else
    return 0;
}

int
cmlogServerHB::registerMonTimer (int delay, int period)
{
  ACE_Time_Value dtv (delay);
  ACE_Time_Value itv (period);

  int status = reactor_.schedule_timer (this, (void *)0, dtv, itv);
  if (status == 0)
    timerOn_ = 1;
  else
    timerOn_ = 0;

  return status;
}

int
cmlogServerHB::get_handle (void) const
{
  return listener_.get_handle ();
}

int
cmlogServerHB::handle_input (int)
{
  ACE_INET_Addr sa;
  int           n;
  cmlogMsg      msg;
  char          buf[1024];  // cannot exceed 1024 bytes

  if ((n = listener_.recv (buf, sizeof (buf), sa)) == -1) {
#ifdef _CMLOG_DEBUG
    printf ("Heart beat monitor read error\n");
#endif
    return -1;
  }
  if (cmlogMsg::bufferToMsg (buf, msg) == -1)
    return 0;

  if (msg.type () == CMLOG_SERVERALIVE) { /* never used */
    serverHB_ = 0;
    if (!timerOn_)
      registerMonTimer (CMLOG_HEARTBEAT_INTERVAL/2,CMLOG_HEARTBEAT_INTERVAL);
  }
  else if (msg.type () == CMLOG_USERLKCLNT) {
#ifdef _CMLOG_DEBUG
    printf ("User process is looking for client daemon\n");
#endif
    // check whether client daemon is connected to the server
    if (toServer_->connectedToServer ()) {
#ifdef _CMLOG_DEBUG
      printf ("ClientD is connected to the cmlogServer\n");
#endif
      // send client id back to the client
      sendClientIdToUser (sa);
    }
    else {
      printf ("ClientD is not connected to the cmlogServer, reconnect\n");
      cmlog_cdevMessage& message = msg;
      cdevData* pdata = message.getData ();
      unsigned short svport;

      if (pdata && 
	  pdata->get (cmlogUtil::CMLOG_VALUE_TAG, &svport) == CDEV_SUCCESS) {
	toServer_->serverUdpPort (svport);
	toServer_->reConnToServer ();
	
	if (toServer_->connectedToServer ()) {
	  reactor_.cancel_timer (this);
	  timerOn_ = 0; 
	  printf ("Connection to the server has been reestablished\n");
	  // send all client context and tagmap information again
	  sendConnectionInfo ();
	}
      }
      if (toServer_->connectedToServer ()) 
	sendClientIdToUser (sa);
      else
	sendNoConnToUser   (sa);
    }
  }
  else {
#ifdef _CMLOG_DEBUG
    printf ("Wrong heat beat message :::::\n");
#endif
  }
  return 0;
}

int
cmlogServerHB::handle_timeout (const ACE_Time_Value& tv, const void* arg)
{
/*  serverHB_ ++;
  if (serverHB_ >= cmlogServerHB::serverHBTh) {
#ifdef _CMLOG_DEBUG
    printf ("Server is dead >>>>>>>>>>>>>\n");
#endif
    // keep listening on this port until new server is up
    serverHB_ = 0;
    if (toServer_)
      toServer_->closeDown ();
  }
  return 0;
*/
#ifdef _CMLOG_DEBUG
  printf ("Server is dead, try to reconnect to the server\n");
#endif
  if (toServer_->reConnToServer () == 0) {
    reactor_.cancel_timer (this);
    timerOn_ = 0; 
    printf ("Connection to the server has been reestablished\n");

    // send all client context and tagmap information again
    sendConnectionInfo ();
  }
  return 0;
}

void
cmlogServerHB::tcpToServer (cmlogUxToServer* toServer)
{
  toServer_ = toServer;
}

short
cmlogServerHB::clientId (void)
{
  if (toServer_->connectedToServer ()) {
    short value;
    value = clientId_ ++;
    
    return value;
  }
  else 
    return 1;
}

void
cmlogServerHB::registerClient (int pid, short cid, 
			       cdevData& cxt, cdevData& tagmap)
{
#ifndef __vxworks
  cmlogClientId* newid = new cmlogClientId (cid, pid, cxt, tagmap);

  clientList_.add ((void *)newid);
#else
  cmlogSlistIterator ite (clientList_);
  cmlogClientId* client = 0;
  int found = 0;

  for (ite.init (); !ite; ++ite) {
    client = (cmlogClientId *) ite ();
    if (client->pid_ == pid) {  // vxworks some times assign old task id
      found = 1;
      break;
    }
  }
  if (found) {
    clientList_.remove ((void *)client);
    sendDisconnInfo (client->id_);
    delete client;
  }
  cmlogClientId* newid = new cmlogClientId (cid, pid, cxt, tagmap);

  clientList_.add ((void *)newid);  
#endif  
}

void
cmlogServerHB::removeClient (short id)
{
  cmlogSlistIterator ite (clientList_);
  cmlogClientId* cid = 0;
  int found = 0;

  for (ite.init (); !ite; ++ite) {
    cid = (cmlogClientId *) ite ();
    if (cid->id_ == id) {
      found = 1;
      break;
    }
  }
  if (found) {
    clientList_.remove ((void *)cid);
    delete cid;
#ifdef _CMLOG_DEBUG
    printf ("Remove Client id %d\n", id);
#endif
  }
}

void
cmlogServerHB::checkAllClients (void)
{
  cmlogSlistIterator ite (clientList_);
  cmlogClientId* cid = 0;

  for (ite.init (); !ite; ++ite) {
    cid = (cmlogClientId *) ite ();
    if (cmlogUtil::isProcessDead (cid->pid_)) {
#ifdef _CMLOG_DEBUG
      printf ("Client Process %d is dead\n", cid->pid_);
#endif
      sendDisconnInfo (cid->id_);
    }
  }
  // no need to remove item from the list, this disconnection information
  // will go to cmlogUxToServer and this item will be removed from the
  // removeClient call
}

void
cmlogServerHB::sendDisconnInfo (short clientid)
{
  cmlog_cdevMessage message (clientid, 0, 0, 0, 0,
		       CMLOG_CLOSE_CONN, 0, 0, 0,
		       0, 0, 0);
  cmlogMsg sendmsg (message);

  // open the named pipe to send message as a simple client
  int fd = openNamedPipe ();
  if (fd != -1) {
    int n = fd << sendmsg;
#ifndef __vxworks
    close (fd);
#endif
  }
}

void
cmlogServerHB::sendConnectionInfo (void)
{
  cmlogSlistIterator ite (clientList_);
  cmlogClientId* cid = 0;

  // open the named pipe to send message as a simple client
  int fd = openNamedPipe ();
  int n = 0;
  int ocid = 1234;


  for (ite.init (); !ite; ++ite) {
    cid = (cmlogClientId *) ite ();  
    cmlog_cdevMessage message (cid->clientId(), 0, 0, 0, 0,
			       CMLOG_RECONN_INFO, 0, 0, 0,
			       &(cid->context()), 0, 
			       &(cid->tagmap()) 
			       );
    cmlogMsg sendmsg (message);
    if (fd != -1) 
      n = fd << sendmsg;
    ocid = cid->clientId ();
  }

#ifndef __vxworks
  // get host name
  char host[128];
  int  len = sizeof (host);
  if (::gethostname (host, len) != 0)
    strcpy (host, "unknown");

  cdevData recdata;
  recdata.insert(cmlogUtil::CMLOG_TEXT_TAG, 
		 (char *)("Reconnected to cmlog server"));
  recdata.insert(cmlogUtil::CMLOG_HOST_TAG, host);
  recdata.insert(cmlogUtil::CMLOG_FACILITY_TAG, (char *)("clientD"));

  cmlog_cdevMessage recmsg (ocid, 0, 0, 0, 0,
			    CMLOG_ADD_DATA, 0, 0, 0,
			    &recdata, 0, 0);

  cmlogMsg smsg (recmsg);

  if (fd != -1) {
    n = fd << smsg;
    close (fd);
  }
#else
  logMsg ("Reconnected to cmlog server\n", 0, 0, 0, 0, 0, 0);
#endif
}

int
cmlogServerHB::openNamedPipe (void)
{
  // now try to open named pipe
  unsigned long ipaddr = cmlogUtil::hostIpAddr ();
  char          pipepath[128];
  sprintf       (pipepath, CMLOG_CLNT_PIPE, ipaddr);
  int           fifo;

#ifndef __vxworks
  fifo = ::open (pipepath, O_WRONLY);
  if (fifo < 0) {
    fprintf (stderr, "Cannot open named pipe : %s\n", pipepath);
    return -1;
  }
#else
  // use fifo fd opened by client daemon
  fifo = cmlogVxFifoFd ();
#endif
  return fifo;
}

void
cmlogServerHB::sendClientIdToUser (ACE_INET_Addr& addr)
{
  if (listener_.send (&clientId_, sizeof (clientId_), addr) <=0) {
    fprintf (stderr, "Cannot send back to user client process (task)\n");
  }
  else {
#ifdef _CMLOG_DEBUG
    printf ("Send client id %d to client process\n", clientId_);
#endif
    clientId_ ++;
    if (clientId_ == 32767) /* maximum of short */
      clientId_ = 1234;
  }
}

void
cmlogServerHB::sendNoConnToUser (ACE_INET_Addr& addr)
{
  unsigned short noconn = 1;
  if (listener_.send (&noconn, sizeof (noconn), addr) <=0) {
    fprintf (stderr, "Cannot send back to user client process (task)\n");
  }
}

#ifdef __vxworks
short
cmlogVxClientId (void)
{
  cmlogServerHB *hb = (cmlogServerHB *)cmlogServerHBSymVx_;

  if (hb)
    return hb->clientId ();
  else
    return 1;
}
#endif

