//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Implementation of cmlogNetUtil Class
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogNetUtil.cc,v $
//   Revision 1.3  2001/07/25 14:30:19  chen
//   64 BIT Initial Port
//
//   Revision 1.2  2000/06/20 19:35:16  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.1.1.1  1999/09/07 15:29:11  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:29:08  bickley
// Added cmlog to application development system.
//
//
//
#include "cmlogNetUtil.h"

#define CMLOG_MAX_BRUDP_RETRIES   6

static int cmlogUdpBackOffB[] = {
  1, 2, 4, 8, 16, 32
};

unsigned short
cmlogNetUtil::findServerUdpPort (void)
{
  // find udp server port to broadcast to
  char* tmp = ::getenv ("CMLOG_PORT");
  int   tmpport = 0;
  if (tmp)
    sscanf (tmp, "%d", &tmpport);
  else
    tmpport = CMLOG_PORT;
  
  unsigned short brdPort = (unsigned short)tmpport;
  return brdPort;
}

#ifndef __vxworks
int
cmlogNetUtil::udpOpenLocal (int& udp_fd)
{
  struct sockaddr_in my_addr;

  // open a socket
  if ((udp_fd = socket (AF_INET, SOCK_DGRAM, 0)) < 0) {
    fprintf (stderr, "udp_open: cannot create socket\n");
    udp_fd = -1;
    return -1;
  }
  // set socket option for broadcast
  int one = 1;
  if (setsockopt (udp_fd, SOL_SOCKET, SO_BROADCAST,
		  (char *)&one, sizeof (one)) == -1)
    return -1;

  // bind local address
  ::memset (&my_addr, 0, sizeof (my_addr));
  my_addr.sin_family = AF_INET;
  my_addr.sin_addr.s_addr = htonl (INADDR_ANY);
  my_addr.sin_port = htons (0);
  
  if (bind (udp_fd, (struct sockaddr *)&my_addr, sizeof (my_addr)) < 0) {
    fprintf (stderr, "udp error: bind error\n");
    close (udp_fd);
    udp_fd = -1;
    return -1;
  }
  return 0;
}

int
cmlogNetUtil::udpTalkToGivenServer (int udp_fd, unsigned short port, 
				    char* host, int num,
				    unsigned opcode,
				    unsigned short& tcp_server_port)
{
  unsigned tcode = htonl (opcode);
  return cmlogNetUtil::udpSendRecvWithServer (udp_fd, 
					      (char *)&tcode, sizeof (tcode), 
					      port, host, num,
					      tcp_server_port);
}  

int
cmlogNetUtil::udpSendRecvWithServer (int udp_fd, char* buffer, int len, 
				     unsigned short port, char* host, int num,
				     unsigned short& tcp_server_port)
{
  int i = 0;
  int status;
  int numRetries;
  char* tmp_host = 0;

  numRetries = (num < CMLOG_MAX_BRUDP_RETRIES) ? num : CMLOG_MAX_BRUDP_RETRIES;
  if (numRetries <= 0)
    numRetries = 3;

  while (i < numRetries) {
    // broadcast to server
    if (cmlogNetUtil::directUdpcast (udp_fd, buffer, len, 
				     port, host) < 0) {
      fprintf (stderr, "Send error on udp packet to socket %d to host %s port %d\n",
	       udp_fd, host, port);
      close (udp_fd);
      udp_fd = -1;
      return -1;
    }
    
    // receive message at udp_fd
    if ((status = cmlogNetUtil::recvUdpMessage (udp_fd, cmlogUdpBackOffB[i],
						tcp_server_port, 
						tmp_host)) == 1)
      i++;
    else 
      return status;
  }
  return -1;
}

int
cmlogNetUtil::udpTalkToServer (int udp_fd, unsigned short port, int num,
			       unsigned opcode,
			       unsigned short& server_port,
			       char* &server_host)
{
  unsigned tcode = htonl (opcode);
  return cmlogNetUtil::udpSendRecv (udp_fd, (char *)&tcode, sizeof (tcode), 
				    port, num,
				    server_port, server_host);
}


int
cmlogNetUtil::udpSendRecv (int udp_fd, char* buffer, int len, 
			   unsigned short port, int num,
			   unsigned short& server_port,
			   char* &server_host)
{
  int i = 0;
  int numRetries;
  int status;
  

  numRetries = (num < CMLOG_MAX_BRUDP_RETRIES) ? num : CMLOG_MAX_BRUDP_RETRIES;
  if (numRetries <= 0)
    numRetries = 3;

  while (i < numRetries) {
    // broadcast to server
    if (cmlogNetUtil::broadcast (udp_fd, buffer, len, port) < 0) {
      fprintf (stderr, "Send error on broadcast packet to socket %d\n",
	       udp_fd);
      close (udp_fd);
      udp_fd = -1;
      return -1;
    }
    
    // receive udp message (tcp_port and server)
    if ((status = cmlogNetUtil::recvUdpMessage (udp_fd, cmlogUdpBackOffB[i],
						server_port,
						server_host)) == 1)
      i++;
    else
      return status;
  }
  return -1;
}

int
cmlogNetUtil::broadcast (int udp_fd, char* buffer, int len,
			 unsigned short port)
{
  size_t iterations = 0;
  size_t bytes = 0;

  // first check environment variable CMLOG_HOST
  // if CMLOG_HOST is present, use CMLOG_HOST as the destination of the
  // udp packet
  char* cmlog_host = getenv ("CMLOG_HOST");
  if (cmlog_host) 
    return directUdpcast (udp_fd, buffer, len, port, cmlog_host);
  
  // do regular broadcast
  struct cmlog_ifnode *ptr = cmlogNetUtil::get_ifnode (udp_fd);
  
  if (ptr == 0)
    return -1;

  for (struct cmlog_ifnode* tptr = ptr; tptr != 0; tptr = tptr->next) {
    sockaddr_in to_addr = tptr->brdAddr;
    to_addr.sin_port = htons (port);

    bytes += sendto (udp_fd, buffer, len, 0,
		     (struct sockaddr *)&to_addr, sizeof (to_addr));
    iterations++;
  }
  return iterations == 0 ? 0 : bytes/iterations;
}

int
cmlogNetUtil::directUdpcast (int udp_fd, char* buffer, int len,
			     unsigned short port, char* cmlog_host)
{
  size_t iterations = 0;
  size_t bytes = 0;
  struct hostent* phe;
  struct sockaddr_in to_addr;
  
  ::memset (&to_addr, 0, sizeof (to_addr));
  to_addr.sin_family = AF_INET;
  to_addr.sin_port = htons (port);
  
  // map host ip address
  if (phe = gethostbyname (cmlog_host)) {
    memcpy (&to_addr.sin_addr, phe->h_addr, phe->h_length);
    bytes += sendto (udp_fd, buffer, len, 0,
		     (struct sockaddr *)&to_addr, sizeof (to_addr));
    iterations++;
    return iterations == 0 ? 0 : bytes/iterations; 
  }
  return -1;
}  


int
cmlogNetUtil::recvUdpMessage (int fd, int wait_in_sec,
			      unsigned short& tcp_server_port,
			      char* &server_host)
{
  fd_set readfd;
  int    nfound;
  int    n;
  struct timeval tv;
  struct sockaddr_in sudp;
#ifdef __linux
  socklen_t          sudplen = sizeof (sudp);
#else
  int                sudplen = sizeof (sudp);
#endif


  FD_ZERO (&readfd);
  FD_SET  (fd, &readfd);
  tv.tv_sec = wait_in_sec;
  tv.tv_usec = 0;
    
  do {
#ifndef __hpux
    nfound = select (fd + 1, &readfd, 0, 0, &tv);
#else

#ifndef __hpux10
    nfound = select (fd + 1, (int *)&readfd, (int *)0, (int *)0,
		     &tv);
#else
    nfound = select (fd + 1, &readfd, 0, 0, &tv);
#endif

#endif
  }while (nfound == -1 && errno == EINTR);

  if (nfound == -1) { // something wrong with this fd
    fprintf (stderr, "Bad udp fd %d\n", fd);
    return -1;
  }
  else if (nfound == 0) // timeout
    return 1;
  else {
    // receive from the server
    n = recvfrom (fd, (char *)&tcp_server_port, sizeof (tcp_server_port), 0, 
		  (struct sockaddr *)&sudp, &sudplen);
    if (n < 0) {
      fprintf (stderr, "Error in recvfrom \n");
      close (fd);
      fd = -1;
      return -1;
    }

    // update server host and port information
    tcp_server_port = ntohs (tcp_server_port);
    server_host = cmlogNetUtil::get_host_name (&sudp);
#ifdef _CMLOG_DEBUG
    printf ("Server on host %s at TCP port %d\n", server_host, 
	    tcp_server_port);
#endif

    // close udp fd
    close (fd);
    fd = -1;
    return 0;
  }
}



cmlog_ifnode*
cmlogNetUtil::get_ifnode (int fd)
{
  char   buf[BUFSIZ];
  struct cmlog_ifnode* ifptr = 0;

  struct ifconf ifc;
  struct ifreq *ifr;

  struct ifreq  flags;
  struct ifreq  if_req;

  ifc.ifc_len = sizeof buf;
  ifc.ifc_buf = buf;

  if (ioctl (fd, SIOCGIFCONF, (char *)&ifc) < 0) {
    fprintf (stderr, "ioctl failed on get_ifnode call\n");
    return 0;
  }

  ifr = ifc.ifc_req;

  for (int n = ifc.ifc_len / sizeof (struct ifreq) ; n > 0; n--, ifr++) {
    if (ifr->ifr_addr.sa_family != AF_INET) { // not ip
      continue;
    }

    flags = if_req = *ifr;
    
    if (ioctl(fd, SIOCGIFFLAGS, (char *)&flags) < 0) 
      continue;
      
    if ((flags.ifr_flags & IFF_UP) == 0) 
      continue;

    if (flags.ifr_flags & IFF_LOOPBACK) 
      continue;

    if (flags.ifr_flags & IFF_BROADCAST) {
      if (ioctl(fd, SIOCGIFBRDADDR, (char *)&if_req) < 0) {
	fprintf (stderr, "ioctl error \n");
      }
      else {
	struct cmlog_ifnode *ptr = new cmlog_ifnode;
	ptr->brdAddr       = *(struct sockaddr_in *) &if_req.ifr_broadaddr;
	ptr->next	   = ifptr;
	ifptr = ptr;
      }
    }
    else 
      fprintf (stderr, "Broadcast is not enabled\n");
  }
  return ifptr;
}


char*
cmlogNetUtil::get_host_name (struct sockaddr_in* addr)
{
  char* p;
  int len = sizeof (addr->sin_addr.s_addr);
  hostent *hp = gethostbyaddr ((char *)&addr->sin_addr, 
			       len, AF_INET);
  if (hp == 0)
    return 0;
  else {
    p = new char[::strlen (hp->h_name) + 1];
    ::strcpy (p, hp->h_name);
    return p;
  }
}


int
cmlogNetUtil::tcpConnect (unsigned short server_port,
			  char* server_host, int& tcp_fd)
{
  struct hostent* phe;
  struct sockaddr_in sin;
  
  ::memset (&sin, 0, sizeof (sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons (server_port);
  
  // map host ip address
  if (phe = gethostbyname (server_host))
    memcpy (&sin.sin_addr, phe->h_addr, phe->h_length);
  else {
    fprintf (stderr, "gethostbyname %s error\n", server_host);
    return -1;
  }
  tcp_fd = socket (PF_INET, SOCK_STREAM, 0);
  if (tcp_fd < 0) {
    fprintf (stderr, "Cannot create tcp socket\n");
    return -1;
  }
  
  // connect to the server
  if (::connect (tcp_fd, (struct sockaddr *)&sin, sizeof (sin)) < 0) {
    fprintf (stderr, "tcp_open: cannot connect to server on %s:%d\n",
	     server_host, server_port);
    close (tcp_fd);
    return -1;
  }

  return 0;
}

int
cmlogNetUtil::tcpConnect (unsigned short server_port,
			  char* server_host, int& tcp_fd, int bufsize)
{
  struct hostent* phe;
  struct sockaddr_in sin;
  int    optlen = sizeof (bufsize);
  
  ::memset (&sin, 0, sizeof (sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons (server_port);
  
  // map host ip address
  if (phe = gethostbyname (server_host))
    memcpy (&sin.sin_addr, phe->h_addr, phe->h_length);
  else {
    fprintf (stderr, "gethostbyname %s error\n", server_host);
    return -1;
  }
  tcp_fd = socket (PF_INET, SOCK_STREAM, 0);
  if (tcp_fd < 0) {
    fprintf (stderr, "Cannot create tcp socket\n");
    return -1;
  }

  if (setsockopt(tcp_fd, SOL_SOCKET, SO_SNDBUF, 
		 (char *)&bufsize, optlen) != 0)
    fprintf (stderr, "Cannot set socket send buffer size to %d\n", bufsize);

  if (setsockopt(tcp_fd, SOL_SOCKET, SO_RCVBUF, 
		 (char *)&bufsize, optlen) != 0)
    fprintf (stderr, "Cannot set socket recv buffer size to %d\n", bufsize);
  
  // connect to the server
  if (::connect (tcp_fd, (struct sockaddr *)&sin, sizeof (sin)) < 0) {
    fprintf (stderr, "tcp_open: cannot connect to server %s:%d\n",
	     server_host, server_port);
    close (tcp_fd);
    return -1;
  }

  return 0;
}
#endif
