//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      CMLOG utility routines
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogUtil.cc,v $
//   Revision 1.4  2001/07/25 14:26:39  chen
//   64 BIT Initial Port
//
//   Revision 1.3  2000/06/20 19:32:50  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.2  1999/10/20 20:39:07  chen
//   Fix checkProcess Status Bug
//
//   Revision 1.1.1.1  1999/09/07 15:29:10  chen
//   CMLOG version 2.0
//
// Revision 1.6  1999/02/02  16:04:34  chen
// Add a debug info
//
// Revision 1.5  1999/01/11  20:42:25  chen
// fix a bug for delete dhs under hpux-10.20
//
// Revision 1.4  1998/11/19  17:29:41  chen
// Support multiple directories for database
//
// Revision 1.3  1997/09/15  14:29:34  chen
// change PAGE_SIZE to DBASE_PAGESIZE
//
// Revision 1.2  1997/09/15  13:48:46  chen
// add page size as configuration parameter
//
// Revision 1.1  1997/08/01  15:27:44  bickley
// Added cmlog to application development system.
//
//
//
#include <cmlogConfig.h>
#include <cdevData.h>

#include "cmlogUtil.h"

#ifndef __vxworks

#if defined(SYSV) && !defined (__hpux)

#include <poll.h>
#include <sys/stropts.h>

#else

#ifdef __hpux
#include <poll.h>
#endif

#include <sys/uio.h>
#endif

#endif // !vxworks

int cmlogUtil::tagInserted = 0;

int cmlogUtil::CMLOG_KEY_TAG        = 1234;
int cmlogUtil::CMLOG_STR_TAG        = 1235;
int cmlogUtil::CMLOG_CLNTCXT_TAG    = 1236;
int cmlogUtil::CMLOG_HOST_TAG       = 1237;
int cmlogUtil::CMLOG_PID_TAG        = 1238;
int cmlogUtil::CMLOG_NAME_TAG       = 1239;
int cmlogUtil::CMLOG_CLASS_TAG      = 1240;
int cmlogUtil::CMLOG_DOMAIN_TAG     = 1241;
int cmlogUtil::CMLOG_FACILITY_TAG   = 1242;
int cmlogUtil::CMLOG_CODE_TAG       = 1243;
int cmlogUtil::CMLOG_VERBOSITY_TAG  = 1244;
int cmlogUtil::CMLOG_SEVERITY_TAG   = 1245;
int cmlogUtil::CMLOG_USER_TAG       = 1246;
int cmlogUtil::CMLOG_CLNTID_TAG     = 1247;
int cmlogUtil::CMLOG_DISP_TAG       = 1248;
int cmlogUtil::CMLOG_DROPPED_TAG    = 1249;
int cmlogUtil::CMLOG_START_TAG      = 1250;
int cmlogUtil::CMLOG_END_TAG        = 1251;
int cmlogUtil::CMLOG_TEXT_TAG       = 1252;
int cmlogUtil::CMLOG_STATUS_TAG     = 1253;
int cmlogUtil::CMLOG_INCOMPLETE_TAG = 1254;
int cmlogUtil::CMLOG_RESULT_TAG     = 1255;
int cmlogUtil::CMLOG_VALUE_TAG      = 1256;
int cmlogUtil::CMLOG_PROC_TAG       = 1257;
int cmlogUtil::CMLOG_NUMITEMS_TAG   = 1258;
int cmlogUtil::CMLOG_QUERYMSG_TAG   = 1259;
int cmlogUtil::CMLOG_LINE_TAG       = 1260;

#ifdef _CMLOG_BUILD_SERVER
int cmlogUtil::PORT = 0;
int cmlogUtil::MAXCLNT_CONS = 0;
int cmlogUtil::MAXBRSER_CONS = 0;
int cmlogUtil::DBASE_CHANGE_INTERVAL = 0;
int cmlogUtil::DBASE_PAGESIZE = 0;
char* cmlogUtil::DEFAULT_DIRECTORY = 0;
char* cmlogUtil::LOG_FILE = 0;
char* cmlogUtil::DATABASE_NAME = 0;
char* cmlogUtil::CXTDBASE_NAME = 0;
char* cmlogUtil::TAGTABLE_NAME = 0;
char* cmlogUtil::SECONDARY_DBASE = 0;
char* cmlogUtil::SECONDARY_CXTDBASE = 0;
#endif


#ifdef solaris
extern "C" int gethostname (char* host, int len);
#endif

  

unsigned long
cmlogUtil::hostIpAddr (void)
{
#ifndef __vxworks
  struct hostent *lh = 0;
  unsigned long addr;
  char   hostname[128];
  int    len = sizeof (hostname);
  
  gethostname (hostname, len);

  lh = gethostbyname (hostname);

  if (lh == 0) {
    fprintf (stderr, "Cannot find ip address of local host\n");
    return 0;
  }
  ::memcpy (&addr, lh->h_addr, lh->h_length);
  return addr;
#else
  return 1;
#endif
}

void
cmlogUtil::setTags (void)
{
  // touch tag table first
  cdevGlobalTagTable::tagTable ();

  if (cmlogUtil::tagInserted == 0) {
    int tagval;

    if (cdevData::tagC2I (CMLOG_KEY_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_KEY_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_KEY_TAG, CMLOG_KEY_TAG_STR);

    if (cdevData::tagC2I (CMLOG_STR_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_STR_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_STR_TAG, CMLOG_STR_TAG_STR);    
      
    if (cdevData::tagC2I (CMLOG_CLNTCXT_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_CLNTCXT_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_CLNTCXT_TAG,CMLOG_CLNTCXT_TAG_STR);

    if (cdevData::tagC2I (CMLOG_HOST_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_HOST_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_HOST_TAG, CMLOG_HOST_TAG_STR);

    if (cdevData::tagC2I (CMLOG_PID_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_PID_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_PID_TAG, CMLOG_PID_TAG_STR);

    if (cdevData::tagC2I (CMLOG_NAME_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_NAME_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_NAME_TAG, CMLOG_NAME_TAG_STR);

    if (cdevData::tagC2I (CMLOG_CLASS_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_CLASS_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_CLASS_TAG, CMLOG_CLASS_TAG_STR);

    if (cdevData::tagC2I (CMLOG_DOMAIN_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_DOMAIN_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_DOMAIN_TAG, CMLOG_DOMAIN_TAG_STR);

    if (cdevData::tagC2I (CMLOG_FACILITY_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_FACILITY_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_FACILITY_TAG, 
			   CMLOG_FACILITY_TAG_STR);

    if (cdevData::tagC2I (CMLOG_CODE_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_CODE_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_CODE_TAG, CMLOG_CODE_TAG_STR);

    if (cdevData::tagC2I (CMLOG_VERBOSITY_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_VERBOSITY_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_VERBOSITY_TAG, 
			   CMLOG_VERBOSITY_TAG_STR);

    if (cdevData::tagC2I (CMLOG_SEVERITY_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_SEVERITY_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_SEVERITY_TAG, 
			   CMLOG_SEVERITY_TAG_STR);

    if (cdevData::tagC2I (CMLOG_USER_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_USER_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_USER_TAG, CMLOG_USER_TAG_STR);

    if (cdevData::tagC2I (CMLOG_CLNTID_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_CLNTID_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_CLNTID_TAG, CMLOG_CLNTID_TAG_STR);

    if (cdevData::tagC2I (CMLOG_DISP_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_DISP_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_DISP_TAG, CMLOG_DISP_TAG_STR);

    if (cdevData::tagC2I (CMLOG_DROPPED_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_DROPPED_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_DROPPED_TAG, 
			   CMLOG_DROPPED_TAG_STR);

    if (cdevData::tagC2I (CMLOG_START_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_START_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_START_TAG, 
			   CMLOG_START_TAG_STR);

    if (cdevData::tagC2I (CMLOG_END_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_END_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_END_TAG, 
			   CMLOG_END_TAG_STR);


    if (cdevData::tagC2I (CMLOG_TEXT_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_TEXT_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_TEXT_TAG, 
			   CMLOG_TEXT_TAG_STR);

    if (cdevData::tagC2I (CMLOG_STATUS_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_STATUS_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_STATUS_TAG, 
			   CMLOG_STATUS_TAG_STR);

    if (cdevData::tagC2I (CMLOG_INCOMPLETE_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_INCOMPLETE_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_INCOMPLETE_TAG, 
			   CMLOG_INCOMPLETE_TAG_STR);

    if (cdevData::tagC2I (CMLOG_RESULT_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_RESULT_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_RESULT_TAG, 
			   CMLOG_RESULT_TAG_STR);

    if (cdevData::tagC2I (CMLOG_VALUE_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_VALUE_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_VALUE_TAG, 
			   CMLOG_VALUE_TAG_STR);

    if (cdevData::tagC2I (CMLOG_PROC_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_PROC_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_PROC_TAG, 
			   CMLOG_PROC_TAG_STR);


    if (cdevData::tagC2I (CMLOG_NUMITEMS_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_NUMITEMS_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_NUMITEMS_TAG, 
			   CMLOG_NUMITEMS_TAG_STR);

    if (cdevData::tagC2I (CMLOG_QUERYMSG_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_QUERYMSG_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_QUERYMSG_TAG, 
			   CMLOG_QUERYMSG_TAG_STR);

    if (cdevData::tagC2I (CMLOG_LINE_TAG_STR, &tagval) == CDEV_SUCCESS) 
      cmlogUtil::CMLOG_LINE_TAG = tagval;
    else
      cdevData::insertTag (cmlogUtil::CMLOG_LINE_TAG, 
			   CMLOG_LINE_TAG_STR);

    cmlogUtil::tagInserted = 1;
#if 0
    printf ("cmlog tags %s : %d\n", CMLOG_KEY_TAG_STR, 
		  cmlogUtil::CMLOG_KEY_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_STR_TAG_STR, 
	    cmlogUtil::CMLOG_STR_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_CLNTCXT_TAG_STR, 
	    cmlogUtil::CMLOG_CLNTCXT_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_HOST_TAG_STR, 
	    cmlogUtil::CMLOG_HOST_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_PID_TAG_STR, 
	    cmlogUtil::CMLOG_PID_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_NAME_TAG_STR, 
	    cmlogUtil::CMLOG_NAME_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_CLASS_TAG_STR, 
	    cmlogUtil::CMLOG_CLASS_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_DOMAIN_TAG_STR, 
	    cmlogUtil::CMLOG_DOMAIN_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_FACILITY_TAG_STR, 
	    cmlogUtil::CMLOG_FACILITY_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_CODE_TAG_STR, 
	    cmlogUtil::CMLOG_CODE_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_VERBOSITY_TAG_STR, 
	    cmlogUtil::CMLOG_VERBOSITY_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_SEVERITY_TAG_STR, 
	    cmlogUtil::CMLOG_SEVERITY_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_USER_TAG_STR, 
	    cmlogUtil::CMLOG_USER_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_CLNTID_TAG_STR, 
	    cmlogUtil::CMLOG_CLNTID_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_DISP_TAG_STR, 
	    cmlogUtil::CMLOG_DISP_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_DROPPED_TAG_STR, 
	    cmlogUtil::CMLOG_DROPPED_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_START_TAG_STR, 
	    cmlogUtil::CMLOG_START_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_END_TAG_STR, 
	    cmlogUtil::CMLOG_END_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_TEXT_TAG_STR, 
	    cmlogUtil::CMLOG_TEXT_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_STATUS_TAG_STR, 
	    cmlogUtil::CMLOG_STATUS_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_INCOMPLETE_TAG_STR, 
	    cmlogUtil::CMLOG_INCOMPLETE_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_RESULT_TAG_STR, 
	    cmlogUtil::CMLOG_RESULT_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_VALUE_TAG_STR, 
	    cmlogUtil::CMLOG_VALUE_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_PROC_TAG_STR, 
	    cmlogUtil::CMLOG_PROC_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_NUMITEMS_TAG_STR, 
	    cmlogUtil::CMLOG_NUMITEMS_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_QUERYMSG_TAG_STR, 
	    cmlogUtil::CMLOG_QUERYMSG_TAG);

    printf ("cmlog tags %s : %d\n", CMLOG_LINE_TAG_STR, 
	    cmlogUtil::CMLOG_LINE_TAG);
#endif
  }
}

double
cmlogUtil::currentTime (void)
{
#ifndef __vxworks
  struct timeval tv;
  
  ::gettimeofday (&tv, 0);
  return (double)(tv.tv_sec + tv.tv_usec/1000000.0);
#else
  struct timespec tp;
  
  clock_gettime (CLOCK_REALTIME, &tp);

  return (double)(tp.tv_sec + tp.tv_nsec/1000000000.);
#endif
}

int
cmlogUtil::isProcessDead (long id)
{
#ifndef __vxworks
  if (kill (id, 0) == 0)
    return 0;
  else {
    if (errno == EPERM)
      return 0;
    else if (errno == ESRCH)
      return 1;
    else
      return 0;
  }
#else
  if (taskIdVerify (id) == OK)
    return 0;
  return 1;
#endif
}

#ifndef __vxworks

#ifdef _CMLOG_BUILD_SERVER

int
cmlogUtil::serverPrintf (const char* format,...)
{
  // get text presentation of current time
  char    timebuf[128];
  char*   p;
  time_t  curr = (time_t)cmlogUtil::currentTime();
  strcpy  (timebuf, ctime (&curr));
  
  // add : to the end of string
  p = timebuf;
  while (*p != '\0' && *p != '\n')
    p++;
  *p++ = ':';
  *p = '\0';

  // now print eventhing into a string
  char    buffer[1024];
  va_list argp;
  int     status;

  va_start (argp, format);
  status = vsprintf (buffer, format, argp);
  va_end (argp);

  printf ("%s %s\n", timebuf, buffer);
  return status;
}


int
cmlogUtil::checkSocket (int fd)
{
#ifdef SYSV
  pollfd handle;
  handle.fd = fd;
  handle.events = POLLIN;
  handle.revents = 0;

  if (::poll (&handle, 1, 0) == -1 || (handle.revents && POLLERR) ) {
#ifdef _CMLOG_DEBUG
    serverPrintf ("Error: handle struct is %d 0x%x 0x%x\n", handle.fd, handle.events,
		  handle.revents);
#endif
    return -1;
  }

#ifdef _CMLOG_DEBUG
  serverPrintf ("OK :handle struct is %d 0x%x 0x%x\n", handle.fd, handle.events,
		handle.revents);
#endif
  return 0;
#else
  struct timeval tv;
  fd_set readfd;
  
  tv.tv_sec = 0;
  tv.tv_usec = 0;
  
  FD_ZERO (&readfd);
  FD_SET  (fd, &readfd);

  if (::select (fd + 1, (fd_set *)&readfd, 0, 0, &tv) < 0)
    return -1;
  return 0;
#endif
}

void
cmlogUtil::isort (int *a, int len)
{
  int i = 0, j = 0;
  int key = 0;

  for (j = 1; j < len; j++) {
    key = a[j];
    /* insert a[j] into the sorted sequence a[0 ... j - 1] */
    i = j - 1;
    while (i >= 0 && a[i] > key) {
      /* move up */
      a[i + 1] = a[i];
      i--;
    }
    a[i + 1] = key;
  }
}
  
#include <dirent.h>

int
cmlogUtil::dbaseFilenames (double start, double end,
			   char** &filenames, int& numfiles)
{
  int    tval, tval0;
  int    i, j;
  int    si = 0, ei = 0;
  cmlogFileHandle fhs[10000];
  int    maxsize = 10000;
  cmlogFileHandle *fh = 0;

  
  cmlogDirHandle* cdir = new cmlogDirHandle ();
  if (cdir->parseDir (cmlogUtil::DATABASE_NAME) != 0) {
    serverPrintf ("Parse directory <%s> error\n", cmlogUtil::DATABASE_NAME);
    delete cdir;
    return -1;
  }

  DIR* dbasedir = opendir (cdir->directory());
  if (dbasedir == 0) {
    serverPrintf ("Cannot open directory <%s>\n", cdir->directory());
    delete cdir;
    return -1;
  }
  
  struct dirent *direntp = 0;
  // since we are using different directory stream every time, we do not
  // have to use readdir_r
  i = 0;
  while ( (direntp = readdir (dbasedir)) != NULL) {
    if ((fh = cmlogFileHandle::createFileHandle(direntp->d_name, cdir)) != 0) {
      if (i < maxsize)
	fhs[i++] = *fh;
      delete fh;
    }
  }
  closedir (dbasedir);

  // find secondary directories
  char** dires = 0;
  int    numdires = 0;
  cmlogDirHandle* dhs = 0;
  if (cmlogUtil::SECONDARY_DBASE && 
      cmlogDirHandle::directories (cmlogUtil::SECONDARY_DBASE, dires, &numdires) ==0) {
    dhs = new cmlogDirHandle[numdires];
      
    for (j = 0; j < numdires; j++) {
      if (dhs[j].parseDir (dires[j]) == 0 && 
	  (dbasedir = opendir (dhs[j].directory()) ) != 0) {
	while ((direntp = readdir (dbasedir)) != NULL) 
	  if ((fh=cmlogFileHandle::createFileHandle (direntp->d_name, &dhs[j])) != 0){
	    if (i < maxsize)
	      fhs[i++] = *fh;
	    delete fh;
	  }
	closedir (dbasedir);
      } // end if if parse
    }   // end of for loop
  }     // end of if	  

  // do a quick insertion sort on the index array
  if (i > 0) {
    cmlogFileHandle::isort (fhs, i);

    tval = 0;
    tval0 = 0;
    for (j = 0; j < i; j++) {
      if (fhs[j].timeStamp() < start && fhs[j].timeStamp() > tval) {
	tval = fhs[j].timeStamp();
	si = j;
      }
      if (fhs[j].timeStamp() < end && fhs[j].timeStamp() > tval0) {
	tval0 = fhs[j].timeStamp();
	ei = j;
      }
    }
    
    numfiles = ei - si + 1;
    filenames = new char*[numfiles];

    i = 0;
    for (j = si; j < ei + 1; j++) {
      filenames[i++] = fhs[j].filename ();
#ifdef _CMLOG_DEBUG
      serverPrintf ("Find database filename <%s>\n", filenames[i - 1]);
#endif
    }

    // free all memory
    delete cdir;

    if (numdires) {
      for (i = 0; i < numdires; i++)
	delete []dires[i];
      delete []dires;

      delete []dhs;
    }
    return 0;
  }
  else {
    // free all memory
    delete cdir;

    if (numdires) {
      for (i = 0; i < numdires; i++)
	delete []dires[i];
      delete []dires;

      delete []dhs;
    }

    numfiles = 0;
    filenames = 0;
    return -1;
  }
}

int
cmlogUtil::dbaseCxtFilenames (double start, double end,
			      char** &filenames, int& numfiles)
{
  int    tval, tval0;
  int    i, j;
  int    si = 0;
  int    ei = 0;
  cmlogFileHandle fhs[10000];
  int    maxsize = 10000;
  cmlogFileHandle *fh = 0;

  cmlogDirHandle* cdir = new cmlogDirHandle ();
  if (cdir->parseDir (cmlogUtil::CXTDBASE_NAME) != 0) {
    serverPrintf ("Parse directory <%s> error\n", cmlogUtil::CXTDBASE_NAME);
    delete cdir;
    return -1;
  }

  DIR* dbasedir = opendir (cdir->directory ());
  if (dbasedir == 0) {
    serverPrintf ("Cannot open cxt directory <%s>\n", cdir->directory());
    delete cdir;
    return -1;
  }

  struct dirent *direntp = 0;
  // since we are using different directory stream every time, we do not
  // have to use readdir_r
  i = 0;
  while ( (direntp = readdir (dbasedir)) != NULL) {
    if ((fh = cmlogFileHandle::createFileHandle(direntp->d_name, cdir)) != 0) {
      if (i < maxsize)
	fhs[i++] = *fh;
      delete fh;
    }
  }
  closedir (dbasedir);

  // find secondary directories
  char** dires = 0;
  int    numdires = 0;
  cmlogDirHandle* dhs = 0;
  if (cmlogUtil::SECONDARY_CXTDBASE &&
      cmlogDirHandle::directories (cmlogUtil::SECONDARY_CXTDBASE,dires, &numdires) ==0){
    dhs = new cmlogDirHandle[numdires];
      
    for (j = 0; j < numdires; j++) {
      if (dhs[j].parseDir (dires[j]) == 0 && 
	  (dbasedir = opendir (dhs[j].directory()) ) != 0) {
	while ((direntp = readdir (dbasedir)) != NULL) 
	  if ((fh=cmlogFileHandle::createFileHandle (direntp->d_name, &dhs[j])) != 0){
	    if (i < maxsize)
	      fhs[i++] = *fh;
	    delete fh;
	  }
	closedir (dbasedir);
      } // end if if parse
    }   // end of for loop
  }     // end of if	  
 
  // do a quick insertion sort on the index array
  if (i > 0) {
    cmlogFileHandle::isort (fhs, i);

    // find where start/end time stamp reside in the array of index
    tval = 0;
    tval0 = 0;
    for (j = 0; j < i; j++) {
      if (fhs[j].timeStamp() < start && fhs[j].timeStamp() > tval) {
	tval = fhs[j].timeStamp ();
	si = j;
      }
      if (fhs[j].timeStamp() < end && fhs[j].timeStamp() > tval0) {
	tval0 = fhs[j].timeStamp ();
	ei = j;
      }
    }
    
    numfiles = ei - si + 1;
    filenames = new char*[numfiles];

    i = 0;
    for (j = si; j < ei + 1; j++) {
      filenames[i++] = fhs[j].filename ();
#ifdef _CMLOG_DEBUG
      printf ("Found cxtdatabase <%s>\n", filenames[i - 1]);
#endif
    }

    // free all memory
    delete cdir;

    if (numdires) {
      for (i = 0; i < numdires; i++)
	delete []dires[i];

      delete []dires;
      delete []dhs;
    }
    return 0;
  }
  else {
    // free all memory
    delete cdir;

    if (numdires) {
      for (i = 0; i < numdires; i++)
	delete []dires[i];
      delete []dires;

      delete []dhs;
    }

    numfiles = 0;
    filenames = 0;
    closedir (dbasedir);

    return -1;
  }
}

void
cmlogUtil::freeFileNames (char** fnames, char** cxtfnames, int numfs)
{
  for (int i = 0; i < numfs; i++) {
#ifdef _CMLOG_DEBUG
    printf ("Free database filenames %s %s\n", fnames[i], cxtfnames[i]);
#endif
    delete []fnames[i];
    delete []cxtfnames[i];
  }
  delete []fnames;
  delete []cxtfnames;
}

#define CMLOG_GET_FD "getfd"

#ifdef __linux
struct cmlog_cmsghdr {  
  /* The frist three elements are the same as the cmsghdr */
  unsigned int cmsg_len;
  int cmsg_level;
  int cmsg_type;
  int cmsg_data;
};

#endif

int
cmlogUtil::sendfile (int sendfd, int openedfd)
{
#if defined (SYSV) && !defined (__hpux)
  if (ioctl (sendfd, I_SENDFD, openedfd) < 0) {
    // we cannot afford to block the receiver end
    int fd = CMLOG_BAD_FD;
    write (sendfd, &fd, sizeof (fd));
  }
  return 0;
#else
  struct iovec  iov[1];
  struct msghdr msg;

#ifndef __linux
  iov[0].iov_base = (char *)CMLOG_GET_FD;
  iov[0].iov_len = strlen (CMLOG_GET_FD) + 1;
  msg.msg_iov = iov;
  msg.msg_iovlen = 1;
  msg.msg_name = (caddr_t)0;
  msg.msg_namelen = 0;
  msg.msg_accrights = (caddr_t)&openedfd;
  msg.msg_accrightslen = sizeof (openedfd);

#else
  struct cmlog_cmsghdr acmsg;

  acmsg.cmsg_len = sizeof (struct cmlog_cmsghdr);
  acmsg.cmsg_level = SOL_SOCKET;
  acmsg.cmsg_type = SCM_RIGHTS;
  acmsg.cmsg_data = openedfd;

  iov[0].iov_base = (char *)CMLOG_GET_FD;
  iov[0].iov_len = strlen (CMLOG_GET_FD) + 1;
  msg.msg_iov = iov;
  msg.msg_iovlen = 1; 
  msg.msg_name = (caddr_t)0;
  msg.msg_namelen = 0;
  msg.msg_control = (caddr_t)&(acmsg);
  msg.msg_controllen = sizeof (struct cmlog_cmsghdr);
#endif

  if (sendmsg (sendfd, &msg, 0) < 0) {
    serverPrintf ("Send message failed: errno %d\n", errno);
    // we cannot afford to block the receiver end
    msg.msg_accrights = (caddr_t)0;
    msg.msg_accrightslen = 0;
    return sendmsg (sendfd, &msg, 0);
  }

  return 0;
#endif
}

int
cmlogUtil::recvfile (int recvfd, int* openedfd)
{
#if defined (SYSV) && !defined (__hpux)
  struct strrecvfd recv;

  if (ioctl (recvfd, I_RECVFD, (char *)&recv) < 0) {
    *openedfd = CMLOG_BAD_FD;
    return -1;
  }
  
  *openedfd = recv.fd;
  return 0;
#else
  int fd = CMLOG_BAD_FD;
  struct iovec  iov[1];
  struct msghdr msg;
  char   temp[80];


#ifndef __linux
  iov[0].iov_base = temp;
  iov[0].iov_len = strlen (CMLOG_GET_FD) + 1;
  msg.msg_iov = iov;
  msg.msg_iovlen = 1;
  msg.msg_name = (caddr_t)0;
  msg.msg_namelen = 0;
  msg.msg_accrights = (caddr_t)&fd;
  msg.msg_accrightslen = sizeof (fd);
#else
  struct cmlog_cmsghdr acmsg;

  acmsg.cmsg_level = SOL_SOCKET;
  acmsg.cmsg_type = SCM_RIGHTS;
  acmsg.cmsg_len = sizeof (struct cmlog_cmsghdr);
  acmsg.cmsg_data = fd;

  iov[0].iov_base = temp;
  iov[0].iov_len = strlen (CMLOG_GET_FD) + 1;
  msg.msg_iov = iov;
  msg.msg_iovlen = 1; 
  msg.msg_name = (caddr_t)0;
  msg.msg_namelen = 0;
  msg.msg_control = (caddr_t)&(acmsg);
  msg.msg_controllen = sizeof (struct cmlog_cmsghdr);

#endif

  if (recvmsg (recvfd, &msg, 0) < 0) {
    *openedfd = CMLOG_BAD_FD;
    return -1;
  }

#ifndef __linux
  *openedfd = fd;
#else
  *openedfd = (int)acmsg.cmsg_data;
#endif

  return 0;
#endif
}

int
cmlogUtil::parseConfigFile (char* filename)
{
  // initialize to default parameters
  cmlogUtil::PORT = CMLOG_PORT;
  cmlogUtil::MAXCLNT_CONS = CMLOG_MAXCLNT_CONS;
  cmlogUtil::MAXBRSER_CONS = CMLOG_MAXBRSER_CONS;
  cmlogUtil::DBASE_CHANGE_INTERVAL = CMLOG_DBASE_CHANGE_INTERVAL;
  cmlogUtil::DBASE_PAGESIZE = CMLOG_PAGE_SIZE;
  
#ifdef CMLOG_DEFAULT_DIRECTORY
  cmlogUtil::DEFAULT_DIRECTORY = new char[strlen (CMLOG_DEFAULT_DIRECTORY)+1];
  ::strcpy (cmlogUtil::DEFAULT_DIRECTORY, CMLOG_DEFAULT_DIRECTORY);
#endif

#ifdef CMLOG_LOG_FILE
  cmlogUtil::LOG_FILE = new char[strlen (CMLOG_LOG_FILE)+1];
  ::strcpy (cmlogUtil::LOG_FILE, CMLOG_LOG_FILE);
#endif

  cmlogUtil::DATABASE_NAME = new char[strlen (CMLOG_DATABASE_NAME)+1];
  ::strcpy (cmlogUtil::DATABASE_NAME, CMLOG_DATABASE_NAME);

  cmlogUtil::CXTDBASE_NAME = new char[strlen (CMLOG_CXTDBASE_NAME)+1];
  ::strcpy (cmlogUtil::CXTDBASE_NAME, CMLOG_CXTDBASE_NAME);

#ifdef CMLOG_SECONDARY_DBASE
  cmlogUtil::SECONDARY_DBASE = new char[strlen (CMLOG_SECONDARY_DBASE)+1];
  ::strcpy (cmlogUtil::SECONDARY_DBASE, CMLOG_SECONDARY_DBASE); 
#endif

#ifdef CMLOG_SECONDARY_CXTDBASE
  cmlogUtil::SECONDARY_CXTDBASE =new char[strlen (CMLOG_SECONDARY_CXTDBASE)+1];
  ::strcpy (cmlogUtil::SECONDARY_CXTDBASE, CMLOG_SECONDARY_CXTDBASE); 
#endif


  cmlogUtil::TAGTABLE_NAME = new char[strlen (CMLOG_TAGTABLE_NAME)+1];
  ::strcpy (cmlogUtil::TAGTABLE_NAME, CMLOG_TAGTABLE_NAME);

  if (!filename) {
    serverPrintf ("No configuration provided, use default parameters\n");
    return -1;
  }

  FILE* fd = fopen (filename, "r");
  if (!fd) {
    serverPrintf ("cannot open configuration file %s\n", filename);
    serverPrintf ("use default parameters\n");
    return -1;
  }

  char line[1024];
  char token[80], value[1024];
  int  st = 0;
  while (!feof (fd)) {

    fgets (line, sizeof (line), fd);
    if (line[0] == '#')
      continue;
    
    if ((st = sscanf (line, "%s %s", token, value)) >= 2) {
      if (::strcasecmp (token, "cmlogServerPort") == 0)
	cmlogUtil::PORT = atoi (value);
      else if (::strcasecmp (token, "cmlogDbaseCHangeInterval") == 0)
	cmlogUtil::DBASE_CHANGE_INTERVAL = atoi (value);     
      else if (::strcasecmp (token, "cmlogMaxClntConnections") == 0)
	cmlogUtil::MAXCLNT_CONS = atoi (value);
      else if (::strcasecmp (token, "cmlogMaxBrowserConnections") == 0)
	cmlogUtil::MAXBRSER_CONS = atoi (value);
      else if (::strcasecmp (token, "cmlogDefaultDirectory") == 0) {
	if (cmlogUtil::DEFAULT_DIRECTORY)
	  delete []cmlogUtil::DEFAULT_DIRECTORY;
	cmlogUtil::DEFAULT_DIRECTORY = new char[strlen (value) + 1];
	::strcpy (cmlogUtil::DEFAULT_DIRECTORY, value);
      }
      else if (::strcasecmp (token, "cmlogLogFile") == 0) {
	if (cmlogUtil::LOG_FILE)
	  delete []cmlogUtil::LOG_FILE;
	cmlogUtil::LOG_FILE = new char[strlen (value) + 1];
	::strcpy (cmlogUtil::LOG_FILE, value);
      }
      else if (::strcasecmp (token, "cmlogDatabaseName") == 0) {
	delete []cmlogUtil::DATABASE_NAME;
	cmlogUtil::DATABASE_NAME = new char[strlen (value) + 1];
	::strcpy (cmlogUtil::DATABASE_NAME, value);
      }
      else if (::strcasecmp (token, "cmlogCxtDatabaseName") == 0) {
	delete []cmlogUtil::CXTDBASE_NAME;
	cmlogUtil::CXTDBASE_NAME = new char[strlen (value) + 1];
	::strcpy (cmlogUtil::CXTDBASE_NAME, value);
      }
      else if (::strcasecmp (token, "cmlogSecondaryDatabase") == 0) {
	if (cmlogUtil::SECONDARY_DBASE)
	  delete []cmlogUtil::SECONDARY_DBASE;
	cmlogUtil::SECONDARY_DBASE = new char[strlen (value) + 1];
	::strcpy (cmlogUtil::SECONDARY_DBASE, value);
      }
      else if (::strcasecmp (token, "cmlogSecondaryCxtDatabase") == 0) {
	if (cmlogUtil::SECONDARY_CXTDBASE)
	  delete []cmlogUtil::SECONDARY_CXTDBASE;
	cmlogUtil::SECONDARY_CXTDBASE = new char[strlen (value) + 1];
	::strcpy (cmlogUtil::SECONDARY_CXTDBASE, value);
      }
      else if (::strcasecmp (token, "cmlogTagTableName") == 0) {
	delete []cmlogUtil::TAGTABLE_NAME;
	cmlogUtil::TAGTABLE_NAME = new char[strlen (value) + 1];
	::strcpy (cmlogUtil::TAGTABLE_NAME, value);
      }
      else if (::strcasecmp (token, "cmlogDbasePageSize") == 0) 
	cmlogUtil::DBASE_PAGESIZE = atoi (value);
    }
  }
  fclose (fd);


  serverPrintf ("cmlogServer UDP port: %d\n", cmlogUtil::PORT);
  serverPrintf ("Maximum client connections: %d\n", cmlogUtil::MAXCLNT_CONS);
  serverPrintf ("Maximum broswer connections: %d\n", cmlogUtil::MAXBRSER_CONS);
  if (cmlogUtil::DEFAULT_DIRECTORY)
    serverPrintf ("Default directory: %s\n", cmlogUtil::DEFAULT_DIRECTORY);
  if (cmlogUtil::LOG_FILE)
    serverPrintf ("Server Log file: %s\n", cmlogUtil::LOG_FILE);
  serverPrintf ("Server database name: %s\n", cmlogUtil::DATABASE_NAME);
  serverPrintf ("Server context database name: %s\n", cmlogUtil::CXTDBASE_NAME);
  serverPrintf ("Server tag table name: %s\n", cmlogUtil::TAGTABLE_NAME);
  serverPrintf ("Server database name changes every %d minitues\n", 
	  cmlogUtil::DBASE_CHANGE_INTERVAL);
  serverPrintf ("Server database page size is %d\n", cmlogUtil::DBASE_PAGESIZE);

  return 0;
}


//==========================================================================
//       Implementation of cmlogDirHandle
//==========================================================================
cmlogDirHandle::cmlogDirHandle (void)
  :dir_ (0), prefix_ (0), sprefix_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogDirHandle Class Object\n");
#endif
}

cmlogDirHandle::~cmlogDirHandle (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogDirHandle Class Object\n");
#endif
  if (dir_)
    delete []dir_;
  
  if (prefix_)
    delete []prefix_;

  if (sprefix_)
    delete []sprefix_;
}

int
cmlogDirHandle::parseDir (char* dirname)
{
  char temp[256];
  char prefix[256];
  char sprefix[128];
  
  strncpy (temp, dirname, sizeof (temp) - 1);
  strncpy (prefix, dirname, sizeof (temp) - 1);

  // get directory name
  char* lslash = strrchr (temp, '/');
  if (lslash == 0) {
#ifdef _CMLOG_DEBUG
    printf ("Fatal: cmlog directory name <%s> is wrong\n", dirname);
#endif
    return -1;
  }
  // change lslash into null
  *lslash = '\0';

#ifdef _CMLOG_DEBUG
  printf ("cmlog directory has name <%s> \n", temp);
#endif

  // get prefix
  char* perct = strrchr (prefix, '%');
  if (perct == 0) {
#ifdef _CMLOG_DEBUG
    printf ("Fatal: cmlog directory <%s> has wrong format\n", dirname);
#endif
    return -1;
  }
  // change % to null
  *perct = '\0';

#ifdef _CMLOG_DEBUG
  printf ("cmlog directory has prefix <%s> \n", prefix);
#endif

  // find prefix without full path
  lslash = strrchr (prefix, '/');
  if (lslash == 0) 
    return -1;

  char* sp = lslash + 1;
  strncpy (sprefix, sp, sizeof (sprefix) - 1);

#ifdef _CMLOG_DEBUG
  printf ("cmlog directory has single prefix <%s> \n", sprefix);
#endif

  // copy information into data members
  dir_ = new char[strlen (temp) + 1];
  strcpy (dir_, temp);

  prefix_ = new char[strlen (prefix) + 1];
  strcpy (prefix_, prefix);

  sprefix_ = new char[strlen (sprefix) + 1];
  strcpy (sprefix_, sprefix);
  
  return 0;
}


const char*
cmlogDirHandle::directory (void) const
{
  return dir_;
}

const char*
cmlogDirHandle::prefix (void) const
{
  return prefix_;
}

const char*
cmlogDirHandle::sprefix (void) const
{
  return sprefix_;
}

int
cmlogDirHandle::directories (char* paths, char** &dirs, int* numdirs)
{
  char   temp[256];
  char*  comma = 0;
  char*  nh = 0;
  int    i = 0;
  size_t seg = 0;

  nh = paths;
  while (nh && (comma = strchr (nh, ':')) != 0) {
    nh = comma + 1;
    i++;
  }
  *numdirs = i + 1;

#ifdef _CMLOG_DEBUG
  printf ("cmlog secondary storage contains %d directories\n", i + 1);
#endif
  dirs = new char*[i + 1];

  nh = paths;
  i = 0;
  while (nh && (comma = strchr (nh, ':')) != 0) {
    seg = (size_t) (comma - nh);
    if (seg >= sizeof (temp) - 1) {
#ifdef _CMLOG_DEBUG
      printf ("Individual directory exceeds 256 chars\n");
#endif
      return -1;
    }
    memcpy (temp, nh, seg);
    temp[seg] = '\0';

    // copy to dirs
    dirs[i] = new char[strlen (temp) + 1];
    strcpy (dirs[i], temp);

    nh = comma + 1;
    i++;
  }
  
  // copy last segment to dirs
  if (nh) {
    seg = strlen (nh);
    memcpy (temp, nh, seg);
    temp[seg] = '\0';
    dirs[i] = new char[strlen (temp) + 1];
    strcpy (dirs[i], temp);
  }

#ifdef _CMLOG_DEBUG
  for (i = 0; i < *numdirs; i++) {
    printf ("Directory[%d] is <%s> \n", i, dirs[i]);
  }
#endif

  return 0;
}

//=======================================================================
//     Implementation of cmlogFileHandle
//=======================================================================
cmlogFileHandle::cmlogFileHandle (void)
  :timeStamp_ (0), dh_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogFileHandle Class Object\n");
#endif
}

cmlogFileHandle::cmlogFileHandle (int ts, cmlogDirHandle* dh)
  :timeStamp_ (ts), dh_ (dh)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogFileHandle Class Object\n");
#endif
}


cmlogFileHandle::cmlogFileHandle (const cmlogFileHandle& fh)
  :timeStamp_ (fh.timeStamp_), dh_ (fh.dh_)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogFileHandle Class Object\n");
#endif
}

cmlogFileHandle::~cmlogFileHandle (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogFileHandle Class Object\n");
#endif
}


cmlogFileHandle&
cmlogFileHandle::operator = (const cmlogFileHandle& fh)
{
  if (this != &fh) {
    timeStamp_ = fh.timeStamp_;
    dh_ = fh.dh_;
  }
  return *this;
}

cmlogFileHandle *
cmlogFileHandle::createFileHandle (char* filename, cmlogDirHandle* dh)
{
  char conv[256];
  int  tval;
  int  status = -1;
  cmlogFileHandle* fh = 0;
  
  if (strstr (filename, dh->sprefix()) != NULL) {
    strncpy (conv, filename, sizeof (conv) - 1);
    char *p = conv + strlen (dh->sprefix());

    if (sscanf (p, "%d", &tval) >= 1) 
      fh = new cmlogFileHandle (tval, dh);
  }
  return fh;
}
  

const int
cmlogFileHandle::timeStamp (void) const
{
  return timeStamp_;
}

char* 
cmlogFileHandle::filename (void) const
{
  if (!dh_)
    return 0;

  char fname[1024];
  sprintf (fname, "%s%d", dh_->prefix (), timeStamp_);

  char* rf = new char[strlen (fname) + 1];
  strcpy (rf, fname);

  return rf;
}

int
cmlogFileHandle::compare (const cmlogFileHandle& fh) const
{
  return (timeStamp_ - fh.timeStamp_);
}

int
operator > (const cmlogFileHandle& h1, cmlogFileHandle& h2)
{
  return (h1.compare (h2) > 0);
}

int
operator >= (const cmlogFileHandle& h1, cmlogFileHandle& h2)
{
  return (h1.compare (h2) >= 0);
}

int
operator < (const cmlogFileHandle& h1, cmlogFileHandle& h2)
{
  return (h1.compare (h2) < 0);
}

int
operator <= (const cmlogFileHandle& h1, cmlogFileHandle& h2)
{
  return (h1.compare (h2) <= 0);
}

int
operator == (const cmlogFileHandle& h1, cmlogFileHandle& h2)
{
  return (h1.compare (h2) == 0);
}

int
operator != (const cmlogFileHandle& h1, cmlogFileHandle& h2)
{
  return (h1.compare (h2) != 0);
}

void
cmlogFileHandle::isort (cmlogFileHandle *a, int len)
{
  int i = 0, j = 0;
  cmlogFileHandle key;

  for (j = 1; j < len; j++) {
    key = a[j];
    /* insert a[j] into the sorted sequence a[0 ... j - 1] */
    i = j - 1;
    while (i >= 0 && a[i] > key) {
      /* move up */
      a[i + 1] = a[i];
      i--;
    }
    a[i + 1] = key;
  }
}
#endif  // _CMLOG_BUILD_SERVER


#endif
  
