//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
// Email: coda@cebaf.gov  Tel: (804) 249-7101  Fax: (804) 249-7363
//-----------------------------------------------------------------------------
// 
// Description:
//	 cmlogStrHash: cmlog hash table keyed by a variable length string
//                    Open Hash with buckets implemented by single linked lists
//    
//              Note: void *s are stored inside the table. This class
//                    will not manage these pointers. Callers have to
//                    manage these pointers
// 
//              Note: this is unsafe C++ practice. Use at your own risk
//              
//            Reason: It is so difficult to use a template class inside
//                    a shared library. (Cfront based C++ compiler cannot
//                    instantiate a template class during compilation time
//	
// Author:  Jie Chen
//       CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogStrHash.cc,v $
//   Revision 1.1.1.1  1999/09/07 15:29:10  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:27:34  bickley
// Added cmlog to application development system.
//
//
//
#include <assert.h>
#include "cmlogStrHash.h"

//======================================================================
//      A simple string hash fucntion for cmlogDevice and caChannel
//           Its implementation will be changed for release 1.1
//======================================================================
unsigned int cmlogStrHashFunc (char *src)
{
  unsigned int hash = 0, g;

  for (int i = 0; src[i] != '\0'; i++){
    hash = (hash << 4) + src[i];
    // assume 32 bit integer
    if (g = hash & 0xf0000000){
      hash ^= g >> 24;
      hash ^= g;
    }
  }
  return hash;
}


//======================================================================
//	class cmlogStrHash implementation
//======================================================================

// lock_ has default constructor, so compiler will do it for us

cmlogStrHash::cmlogStrHash (unsigned int max, unsigned int (*f)(cmlogKeyItem))
:tablesize(max), hashCode_ (f)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogStrHash Class Objects\n");
#endif
  // cmlogSlist has a default constructor, so one can create an array of the list
  buckets = new cmlogSlist[tablesize];
  assert (buckets);
}

cmlogStrHash::~cmlogStrHash (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogStrHash Class Objects\n");
#endif
  delete []buckets;
  buckets = 0;
}

int 
cmlogStrHash::isEmpty()
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif
  // if any table is non-empty, return 0
  for (int i = 0; i < tablesize; i++)
    if (!buckets[i].isEmpty_i())
      return 0;

  // all empty
  return 1;
}

void
cmlogStrHash::deleteAllValues()
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif
  // delete all values from a hash table
  // clear the elements from each of teh buckets
  for (int i = 0; i < tablesize; i++)
    buckets[i].deleteAllValues_i();
}

unsigned int 
cmlogStrHash::hash(const cmlogKeyItem& key) const
{
  // return hashed value of key
  return ((*hashCode_)(key) % tablesize);
}

void
cmlogStrHash::add (cmlogKeyItem key, cmlogHashItem newele)
{
  buckets[hash (key)].add (newele);
}

int
cmlogStrHash::remove (cmlogKeyItem key, cmlogHashItem ele)
{
  return (buckets[hash (key)].remove (ele));
}

int
cmlogStrHash::find (cmlogKeyItem key, cmlogHashItem ele) const
{
  return buckets[hash (key)].includes(ele);
}

cmlogSlist&
cmlogStrHash::bucketRef (cmlogKeyItem key)
{
  return (buckets[hash (key)]);
}

//======================================================================
//	class cmlogStrHashIterator implementation
//======================================================================
cmlogStrHashIterator::cmlogStrHashIterator (cmlogStrHash& v)
:base(v), currentIndex(0), itr(0)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogStrHashIterator Class Object \n");
#endif
  // no further initialization
}

cmlogStrHashIterator::~cmlogStrHashIterator (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogStrHashIterator Class Object \n");
#endif
  currentIndex = 0;
  if (itr)
    delete itr;
}

int 
cmlogStrHashIterator::init (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  // initialize iterator, 
  // start search with first bucket
  currentIndex = 0; 
  itr = 0;
  return getNextIterator(); 
}

cmlogHashItem 
cmlogStrHashIterator::operator() (void)
{
  // return current element
  return (*itr)();
}

int 
cmlogStrHashIterator::operator ! (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  // test if there is a current element
  return itr != 0;
}

int 
cmlogStrHashIterator::operator ++ (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  // see if current iterator can be advanced
  if (itr && itr->forward_i ()) 
    return 1;

  // if not, get next iterator
  currentIndex++;
  return getNextIterator();
}

void 
cmlogStrHashIterator::operator = (cmlogHashItem val)
{
  // change the current value
  (*itr) = val;
}

void
cmlogStrHashIterator::removeCurrent (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  if (itr) 
    itr->removeCurrent_i ();
}

int 
cmlogStrHashIterator::getNextIterator (void)
{
  // if there is an old iterator, delete it
  if (itr != 0) 
    delete itr;

  // now search for a new one
  for (; currentIndex < base.tablesize; currentIndex++){
    // generate a new iterator at the current point
    itr = new cmlogSlistIterator (base.buckets[currentIndex]);
    assert(itr != 0);

    // if it has at least one element, we're done
    if (itr->init_i())
      return 1;

    // otherwise delete it, try again
    delete itr;
  }
  // all out of iterators, can quit
  itr = 0;
  return 0;
}
