//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      cmlog server/client message packet
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogMsg.cc,v $
//   Revision 1.6  2001/07/25 14:26:37  chen
//   64 BIT Initial Port
//
//   Revision 1.5  2000/03/17 19:13:32  chen
//   allow multiple protocol support
//
//   Revision 1.4  2000/02/07 16:10:42  chen
//   cmlog protocol 2.0
//
//   Revision 1.3  1999/12/13 21:38:56  chen
//   use CMLOG_BAD_PTR to signal error
//
//   Revision 1.2  1999/12/13 15:19:12  chen
//   Change writev on single cmlogMsg to write a buffer
//
//   Revision 1.1.1.1  1999/09/07 15:29:10  chen
//   CMLOG version 2.0
//
// Revision 1.2  1998/02/12  19:33:21  chen
// remove assert return -1 in reading input
//
// Revision 1.1  1997/08/01  15:27:24  bickley
// Added cmlog to application development system.
//
//
#include <errno.h>
#include "cmlogMsg.h"

#ifdef _CMLOG_BUILD_CLIENT
#include <sys/types.h>
#include <netinet/in.h>
#include <unistd.h>

#ifdef __hpux
#include <sys/uio.h>
#endif

#endif

#ifdef CDEV_64BIT_LONG
#define size_t unsigned
#endif

cmlogMsg::cmlogMsg (void)
:data_ (0, 0, 0, 0, 0, CMLOG_UNKNOWN)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogMsg class object\n");
#endif
}

cmlogMsg::cmlogMsg (unsigned type, cdevData& data, int reqId)
:data_ (0, 0, 0, 0, (unsigned)reqId, type, 0, 0, 0, &data)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogMsg class object\n");
#endif
}

cmlogMsg::cmlogMsg (cmlog_cdevMessage& packet)
:data_ (packet)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogMsg class object\n");
#endif
}

cmlogMsg::cmlogMsg (cmlogMsg& msg)
:data_ (msg.data_)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogMsg class object\n");
#endif
}

cmlogMsg::~cmlogMsg (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogMsg class object\n");
#endif
}

unsigned char
cmlogMsg::getMagicNumber (int iheader)
{
#if (CMLOG_PROTOCOL_MAJOR_VERSION >= 2)
  int header = ntohl (iheader);
  return (unsigned char)(header & 0x000000ff);
#else
  return CMLOG_MAGIC;
#endif
}

unsigned int
cmlogMsg::getMsgSize (int iheader)
{
#if (CMLOG_PROTOCOL_MAJOR_VERSION >= 2)
  int header = ntohl (iheader);
  return (unsigned int)((header >> 8) & 0x00ffffff);
#else
  int header = ntohl (iheader);
  return (unsigned int)(header);
#endif
}

int
cmlogMsg::setMsgHeader (unsigned int size)
{
#if (CMLOG_PROTOCOL_MAJOR_VERSION >= 2)
  if  (size > (unsigned int)(0xffffff00)) {
    fprintf (stderr, "Fatal cmlogMsg: message size is too big (0xffffff00)\n");
    exit (1);
  }
  return htonl ((int) (((size << 8) & 0xffffff00) | CMLOG_MAGIC));
#else
  if  (size > (unsigned int)(0xffffff00)) {
    fprintf (stderr, "Fatal cmlogMsg: message size is too big (0xffffff00)\n");
    exit (1);
  }
  return htonl ((int)(size));
#endif
}

unsigned
cmlogMsg::type (void)
{
  return data_.getOperationCode ();
}

void
cmlogMsg::type (unsigned type)
{
  data_.setOperationCode (type);
}

int
cmlogMsg::size (void)
{
  // total size is message size + size of header
  return (int)(data_.binarySize () + sizeof (int));
}

int
cmlogMsg::reqId (void)
{
  return (int)data_.getForeignDataIndex ();
}

void *
cmlogMsg::channelId (void)
{
  return data_.getArg ();
}

void
cmlogMsg::channelId (void* id)
{
  data_.setArg (id);
}

int
cmlogMsg::sameChannel (void* id)
{
  return (channelId() == id);
}

cmlogMsg::operator cmlog_cdevMessage& (void)
{
  return data_;
}

int
cmlogMsg::msgToBuffer (cmlogMsg& msg, char* buffer)
{
  char* buf;
  size_t bufsize = 0;
  int    n = 0;
  int    header;

  if (msg.type () == CMLOG_EMPTY) {
#ifdef _CMLOG_DEBUG
    printf ("cmlogMsg::msgToBuffer Sending out magic number\n");
#endif
    // create network byte order header
    header = cmlogMsg::setMsgHeader (CMLOG_EMPTY_SIZE);
    memcpy (buffer, &header, sizeof(int));
    n += sizeof (int);

    return 0;
  }

  // convert to binary output
  msg.data_.streamOut (&buf, &bufsize);
#ifdef _CMLOG_DEBUG
  printf ("cmlogMsg::msgToBuffer Sending out magic number\n");
#endif
  // create network byte order header
  header = cmlogMsg::setMsgHeader (bufsize);
  memcpy (buffer, &header, sizeof (int));
  n += sizeof(int);

  // cdevMsg need to implement a new function to shuffle data into
  // an existing buffer
  memcpy (&(buffer[n]), (void *)buf, bufsize);
  n += bufsize;

  // free memory
  delete []buf;
  return 0;
}

int
cmlogMsg::msgToBuffer (cmlogMsg& msg, char* buffer, int* len)
{
  char* buf;
  size_t bufsize = 0;
  int    n = 0;
  int    header;

  if (msg.type () == CMLOG_EMPTY) {
#ifdef _CMLOG_DEBUG
    printf ("cmlogMsg::msgToBuffer 1 Sending out magic number\n");
#endif
    // create header of network byte order
    if (*len < sizeof (int)) {
      fprintf (stderr, "cmlogMsg: Buffer len %d is too small\n", *len);
      return -1;
    }
    header = cmlogMsg::setMsgHeader (CMLOG_EMPTY_SIZE);
    memcpy (buffer, &header, sizeof (int));
    n += sizeof (int);

    *len = sizeof (int);
    return 0;
  }

  // convert to binary output
  msg.data_.streamOut (&buf, &bufsize);
  if (*len < bufsize + sizeof (int)) {
    fprintf (stderr, "cmlogMsg: Buffer len %d is too small\n", *len);
    return -1;
  }

  // magic number first
#ifdef _CMLOG_DEBUG
  printf ("cmlogMsg::msgToBuffer 1 Sending out magic number\n");
#endif
  // create header of network byte order
  header = cmlogMsg::setMsgHeader (bufsize);
  memcpy (buffer, &header, sizeof(int));
  n += sizeof (int);

  // cdevMsg need to implement a new function to shuffle data into
  // an existing buffer
  memcpy (&(buffer[n]), (void *)buf, bufsize);
  n += bufsize;

  // free memory
  delete []buf;

  *len = n;
  return 0;
}

int
cmlogMsg::bufferToMsg (char* buffer, cmlogMsg& msg, int start, int bufsize)
{
  int  n = 0;
  int  header;
  unsigned char magic;
  unsigned int  size = 0;

  // read header information of 4 bytes
  ::memcpy (&header, buffer, sizeof (int));
  n += sizeof (int);

  // get magic number
  magic = cmlogMsg::getMagicNumber (header);
  if (magic != CMLOG_MAGIC) {
    fprintf (stderr, "Magic number mismatch for cmlogMsg\n");
    return -1;
  }
#ifdef _CMLOG_DEBUG
  printf ("cmlogMsg::bufferToMsg get magic number 0x%x\n", magic);
#endif

  // get magic size
  size = cmlogMsg::getMsgSize (header);
  if (size == CMLOG_EMPTY_SIZE) {    // if this is a junk data
#ifdef _CMLOG_DEBUG
    printf ("cmlogMsg OK: return CMLOG_EMPTY_SIZE \n");
#endif
    return CMLOG_EMPTY_SIZE;
  }
  else if (size > bufsize || size + start > bufsize) {
    fprintf (stderr, "cmlogMsg Fatal error: incorrect size <%d> information of a message\n");
    return -1;
  }
#ifdef _CMLOG_DEBUG
  printf ("Get individual message starts at %d with size %u, buffer size %d\n",
	  start, size, bufsize);
#endif

  // convert cmlog_cdevMessage
  if (msg.data_.streamIn (&(buffer[n]), size) == -1)
    return -1;
  return (size + sizeof (int));
}

#if !defined (CMLOG_USE_THREAD) || !defined (_REENTRANT)
int
cmlogMsg::bufferToMsg (char* buffer, cmlogMsg& msg)
{
  int  n = 0;
  int  header;
  unsigned char magic;
  unsigned int  size = 0;

  // read header information of 4 bytes
  ::memcpy (&header, buffer, sizeof (int));
  n += sizeof (int);

  // get magic number
  magic = cmlogMsg::getMagicNumber (header);
  if (magic != CMLOG_MAGIC) {
    fprintf (stderr, "Magic number mismatch for cmlogMsg\n");
    return -1;
  }
#ifdef _CMLOG_DEBUG
  printf ("cmlogMsg::bufferToMsg get magic number 0x%x\n", magic);
#endif

  // get magic size
  size = cmlogMsg::getMsgSize (header);
  if (size == CMLOG_EMPTY_SIZE) {    // if this is a junk data
#ifdef _CMLOG_DEBUG
    printf ("cmlogMsg OK: return CMLOG_EMPTY_SIZE \n");
#endif
    return CMLOG_EMPTY_SIZE;
  }

  // convert cmlog_cdevMessage
  if (msg.data_.streamIn (&(buffer[n]), size) == -1)
    return -1;
  return (size + sizeof (int));
}
#endif

int
operator << (int out, cmlogMsg& msg)
{
  char*   buffer = 0;
  size_t  bufsize = 0;
  int     n = 0;
  int     header;
  unsigned int size = 0;

#ifndef __vxworks
//======================================================================
// Sending out a single cmlogMsg using a single write
//      On Linux writev may be interrupted by a signal
//======================================================================
  // total size of buffer including header
  bufsize = msg.size ();
  if (bufsize > CMLOG_PACKET_MAX_SIZE) {
    // always leave some space for network header
    fprintf (stderr, "Cannot send out message bigger than %d\n",
	     CMLOG_PACKET_MAX_SIZE);
    return -1;
  }
  buffer = new char[bufsize];

  // message size excluding header
  size = bufsize - sizeof (int);
#ifdef _CMLOG_DEBUG
  printf ("Sending out message in size %d bytes\n", size);
#endif

  
  // create header of network byte order
  header = cmlogMsg::setMsgHeader (size);
  ::memcpy (buffer, &header, sizeof (int));
  n += sizeof (int);
#ifdef _CMLOG_DEBUG
  printf ("cmlogMsg::<< cmlogMsg copy magic number\n");
#endif

  // convert message into buffer
  msg.data_.streamOut (&(buffer[n]), size);

  n = ::write (out, buffer, bufsize);
  
  // free memory
  delete []buffer;
#else
  struct iovec iovp[2];
//======================================================================
// On VxWorks writev actually writes multiple times depending
// on how many iovec structures.
//
// VxWorks pipe device is a message based device, a reader of the pipe
// has to read one message a time. If the reader cannot finish reading
// a single message in one time, the remainder of the message will be
// lost.
//======================================================================
  /* lock is held in each call to this routine */
  static char    cmlog_streambuf[CMLOG_CLNT_PIPE_MAX_BYTES];
  static size_t  cmlog_streambufsize = sizeof (cmlog_streambuf);
  
  bufsize = msg.data_.streamOut (cmlog_streambuf, cmlog_streambufsize);

  if (bufsize > CMLOG_CLNT_PIPE_MAX_BYTES - sizeof(int)) {
    fprintf (stderr, "Cannot send out message bigger than %d in size\n",
	     CMLOG_CLNT_PIPE_MAX_BYTES - sizeof(int));
    return -1;
  }
#ifdef _CMLOG_DEBUG
  printf ("Sending out message in size %d bytes\n", bufsize);
#endif


  if (bufsize > 0) {
    // create a header in network byte order
    header = cmlogMsg::setMsgHeader (bufsize);
    iovp[0].iov_base = (char *)&header;
    iovp[0].iov_len = sizeof (int);
#ifdef _CMLOG_DEBUG
    printf ("vxworks cmlogMsg::<< cmlogMsg send out magic number\n");
#endif

    // real binary data
    iovp[1].iov_base = (char *)cmlog_streambuf;
    iovp[1].iov_len = bufsize;

    n = ::writev (out, &(iovp[0]), (size_t)2);
  }
  else
    n = 0;
#endif

  return n;
} 

#ifndef __vxworks
int
operator >> (int in, cmlogMsg& msg)
{
  int  n;
  int  size = 0;
  int  header;
  unsigned char magic;
  unsigned int  msgsize;

  // receiving header first
  n = cmlogMsg::cmlog_recv_n (in, (char *)&header, sizeof (int));
  if (n != sizeof (int)) {
#ifdef _CMLOG_DEBUG
    printf ("cmlogMsg: Receiving error in header in >> operator\n");
#endif
    return -1;
  }
  // get magic number
  magic = cmlogMsg::getMagicNumber (header);
  if (magic != CMLOG_MAGIC) {
    fprintf (stderr, "cmlogMsg: Received wrong magic number in >> operator\n");
    return -1;
  }
#ifdef _CMLOG_DEBUG
  printf ("in >> cmlogMsg received magic number 0x%x\n", magic);
#endif

  // get message size
  msgsize = cmlogMsg::getMsgSize (header);
  size += n;

  if (msgsize <= 0 || msgsize > CMLOG_PACKET_MAX_SIZE - sizeof(int)) {
    fprintf (stderr, "cmlogMsg: error in receiving cmlogMsg msgsize : %d\n", msgsize);
    return -1;
  }
  char* buffer = new char[msgsize];

  n = cmlogMsg::cmlog_recv_n (in, buffer, msgsize);
  if (n != msgsize) {
#ifdef _CMLOG_DEBUG
    printf ("cmlogMsg: Receiving error in retrieve cdevData\n");
#endif
    return -1;
  }
  msg.data_.streamIn (buffer, msgsize);
  delete []buffer;
  size += n;
  return size;
}
#endif

int
operator >> (int in, cmlog_cdevPacket& packet)
{
  int  n;
  int  header;
  int  size = 0;
  unsigned int  msgsize;
  unsigned char magic;
  
  // read input header first
  n = cmlogMsg::cmlog_recv_n (in, (char *)&header, sizeof (int));
  if (n != sizeof (int)) {
#ifdef _CMLOG_DEBUG
    printf ("cmlogMsg: Receiving cmlog_cdevPackage magic number error in >> \n");
#endif
    return -1;
  }
  // get magic number
  magic = cmlogMsg::getMagicNumber (header);
  if (magic != CMLOG_MAGIC) {
    fprintf (stderr, "cmlogMsg: Received a wrong magic number for cmlog_cdevPacket in >> operator \n");
    return -1;
  }
#ifdef _CMLOG_DEBUG
  printf ("in >> cmlog_cdevPacket received magic number 0x%x\n", magic);
#endif
  // get size information
  msgsize = cmlogMsg::getMsgSize (header);
  size += n;
#ifdef _CMLOG_DEBUG
  printf ("ClientD receive message in size %d\n", msgsize);
#endif

#ifndef __vxworks
  if (msgsize <= 0 || msgsize > CMLOG_PACKET_MAX_SIZE - sizeof(int)) {
    fprintf (stderr, "cmlogMsg: error in receiving cmlog_cdevPacket msgsize : %d\n", msgsize);
    return -1;
  }
  char* buffer = new char[msgsize];
#else
  if (msgsize <= 0 || msgsize > CMLOG_CLNT_PIPE_MAX_BYTES - sizeof(int)) {
    fprintf (stderr, "cmlogMsg: error in receiving cmlog_cdevPacket msgsize : %d\n", msgsize);
    return -1;
  }
  // in vxworks buffer size is limited by pipe device
  // this static buffer is accessed by daemon only
  static char buffer[CMLOG_CLNT_PIPE_MAX_BYTES];
#endif

  n = cmlogMsg::cmlog_recv_n (in, buffer, msgsize);
  if (n != msgsize) {
#ifdef _CMLOG_DEBUG
    printf ("Receiving error in retrieve cdevData\n");
#endif
    return -1;
  }
  
  if (packet.attachData (buffer, msgsize) != 0) {
#ifdef _CMLOG_DEBUG
    printf ("packet attach data error\n");
#endif
    return 0;
  }

  size += n;

  return size;
}

int
cmlogMsg::cmlog_recv_n (int in, char* buffer, size_t len)
{
  size_t bytes_read;
  int    m = 0;
  
  for (bytes_read = 0; bytes_read < len; bytes_read += m) {
    if ((m = read (in, (char *)buffer + bytes_read,
		   len - bytes_read)) == -1) 
      return -1;
    else if (m == 0)
      break;
  }
  return bytes_read;
}

//==================================================================
//      implementation of cmlogPacket class
//==================================================================
cmlogPacket::cmlogPacket (void)
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
:number_ (0), size_ (0), full_ (0), lock_ ()
#else
:number_ (0), size_ (0), full_ (0)
#endif
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogPacket Class Object\n");
#endif

  buffer_ = new char[CMLOG_PACKET_MAX_SIZE];
  if (!buffer_) {
    fprintf (stderr, "Cannot allocate char buffer with size %d, Quit\n", 
	     CMLOG_PACKET_MAX_SIZE);
    exit(1);
  }
}


cmlogPacket::~cmlogPacket (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogPacket Class Object\n");
#endif

  // free memory
  delete []buffer_;

  number_ = 0;
  size_ = 0;
  full_ = 0;
}

int
cmlogPacket::insert (cmlogMsg& msg)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (this->lock_);
#endif

  if (full_)
    return -1;

  // calculate total size of buffer
  if (size_ + msg.size () > CMLOG_PACKET_MAX_SIZE) 
    return -1;

  cmlogMsg::msgToBuffer (msg, (char *)&(buffer_[size_]));
  size_ += msg.size ();
  number_ ++;

  if (number_ == CMLOG_PACKET_LIST_SIZE)
    full_ = 1;

  return 0;
}

int
cmlogPacket::insert (cmlog_cdevPacket& packet, int bufsize)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (this->lock_);
#endif

  if (full_)
    return -1;

  if (size_ + bufsize > CMLOG_PACKET_MAX_SIZE) 
    return -1;

  // bufsize is total size of the packet + header, clip requires 
  // size information only
  unsigned int size = (unsigned int) (bufsize - sizeof (int));

  // create a message network header in network byte order
  int header = cmlogMsg::setMsgHeader (size);
#ifdef _CMLOG_DEBUG
  printf ("cmlogPacket::insert cmlog_cdevPacket send out magic\n");
#endif
  
  // copy header to the buffer
  memcpy ((char *)&(buffer_[size_]), (void *)&header, sizeof(int));
  size_ += sizeof (int);

  // get internal buffer of the packet
  char*   pbuffer;
  size_t  plen = 0;
  // stream out packet
  packet.streamOut (&pbuffer, &plen);

  // copy streamed packet into the buffer
  memcpy ((char *)&(buffer_[size_]), (void *)pbuffer, plen);
  size_ += plen;
  number_ ++;

  if (number_ == CMLOG_PACKET_LIST_SIZE)
    full_ = 1;

  return 0;
}

int
cmlogPacket::overflow (cmlogMsg& msg)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (this->lock_);
#endif
  if (full_)
    return 1;
  
  if (size_ + msg.size () > CMLOG_PACKET_MAX_SIZE)
    return 1;

  return 0;
}

int
cmlogPacket::overflow (cmlog_cdevPacket& packet, int bufsize)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (this->lock_);
#endif
  if (full_)
    return 1;
  
  if (size_ + bufsize > CMLOG_PACKET_MAX_SIZE)
    return 1;

  return 0;
}

int
cmlogPacket::dataSize (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (this->lock_);
#endif
  return size_;
}

int
cmlogPacket::packetSize (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (this->lock_);
#endif

#if (CMLOG_PROTOCOL_MAJOR_VERSION >= 2)
  // Total network packet size is size of this total messages +
  // network header information (size + number messages)
  return (size_ + CMLOG_PACKET_PREDATA_SIZE + sizeof (int));
#else
  return (size_ + CMLOG_PACKET_PREDATA_SIZE);
#endif
}

unsigned int
cmlogPacket::numberOfData (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (this->lock_);
#endif
  return number_;
}

cmlogMsg**
cmlogPacket::messages (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (this->lock_);
#endif

  cmlogMsg** msgs = 0;

  if (number_ > 0) {
    msgs = new cmlogMsg*[number_];
    int    j = 0;
    int    msgsize;

    for (int i = 0; i < number_; i++) {
      msgs[i] = new cmlogMsg ();
      if ((msgsize = cmlogMsg::bufferToMsg (&(buffer_[j]), *(msgs[i]), 
					    j, CMLOG_PACKET_MAX_SIZE)) == -1)
	return (cmlogMsg **)CMLOG_BAD_PTR;
      j += msgsize;
    }
    assert (j == size_);
  }
  return msgs;
}

int
cmlogPacket::empty (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(this->lock_);
#endif
  number_ = 0;
  size_ = 0;
  full_ = 0;
  buffer_[0] = '\0';

  return 0;
}

int
cmlogPacket::fullPacket (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (this->lock_);
#endif
  return full_;
}

int
cmlogPacket::emptyPacket (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (this->lock_);
#endif
  if (number_ == 0)
    return 1;
  return 0;
}

void
cmlogPacket::encode (void)
{
  // clip size is 4 bytes bigger than my buffer size
  unsigned int clipsize = size_ + sizeof (size_);

  size_ = htonl (clipsize);
  number_ = htonl (number_);
  full_ = htonl (full_);
}

void
cmlogPacket::decode (void)
{
  // clip size is 4 bytes bigger than my buffer size
  unsigned int clipsize = ntohl (size_);

  // real buffer size
  size_ = clipsize - sizeof (size_);
  number_ = ntohl (number_);
  full_ = ntohl (full_);
}

void
cmlogPacket::reset (void)
{
  size_ = 0;
  number_ = 0;
}


#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
#define CMLOG_BLOCK_SIGNAL
#define CMLOG_UNBLOCK_SIGNAL
#else
#define CMLOG_BLOCK_SIGNAL { \
    sigset_t newset, oldset; \
    ::sigemptyset(&newset); \
    ::sigaddset(&newset, SIGQUIT); \
    ::sigaddset(&newset, SIGINT); \
    ::sigprocmask (SIG_SETMASK, &newset, &oldset);

#define CMLOG_UNBLOCK_SIGNAL \
    ::sigprocmask (SIG_SETMASK, &oldset, 0); \
}
#endif


#ifndef _CMLOG_BUILD_CLIENT
int
operator << (ACE_SOCK_Stream& out, cmlogPacket& p)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(p.lock_);
#endif

  int     n = 0, i = 0, m = 0;
  int     magic;
  int     osize;
  iovec   iovp[3];

#if (CMLOG_PROTOCOL_MAJOR_VERSION >= 2)
  if (p.number_ == 0) {
    magic = htonl (CMLOG_PMAGIC);
    iovp[0].iov_base = (char *)&magic;
    iovp[0].iov_len = sizeof (int);
#ifdef _CMLOG_DEBUG
    printf ("ACE_SOCK_STREAM << cmlogPacket send out magic\n");
#endif

    p.encode ();
    iovp[1].iov_base = (char *)&p;
    iovp[1].iov_len = CMLOG_PACKET_PREDATA_SIZE;
    CMLOG_BLOCK_SIGNAL
    n = out.send (&(iovp[0]), (size_t)2);
    CMLOG_UNBLOCK_SIGNAL
    
    return n;
  }
  else {
    magic = htonl (CMLOG_PMAGIC);
    iovp[0].iov_base = (char *)&magic;
    iovp[0].iov_len = sizeof (int);
#ifdef _CMLOG_DEBUG
    printf ("ACE_SOCK_STREAM << cmlogPacket send out magic\n");
#endif
    // original size information
    osize = p.size_;
    p.encode ();

    // send over the network
    iovp[1].iov_base = (char *)&p;
    iovp[1].iov_len = CMLOG_PACKET_PREDATA_SIZE;
    
    iovp[2].iov_base = p.buffer_;
    iovp[2].iov_len = osize;

    CMLOG_BLOCK_SIGNAL
    n = out.send (&(iovp[0]), (size_t)3);
    CMLOG_UNBLOCK_SIGNAL
    return n;
  }
#else // protocol version 2.x
  if (p.number_ == 0) {
    p.encode ();
    iovp[0].iov_base = (char *)&p;
    iovp[0].iov_len = CMLOG_PACKET_PREDATA_SIZE;
    CMLOG_BLOCK_SIGNAL
    n = out.send (&(iovp[0]), (size_t)1);
    CMLOG_UNBLOCK_SIGNAL
    
    return n;
  }
  else {
    // original size information
    osize = p.size_;
    p.encode ();

    // send over the network
    iovp[0].iov_base = (char *)&p;
    iovp[0].iov_len = CMLOG_PACKET_PREDATA_SIZE;
    
    iovp[1].iov_base = p.buffer_;
    iovp[1].iov_len = osize;

    CMLOG_BLOCK_SIGNAL
    n = out.send (&(iovp[0]), (size_t)2);
    CMLOG_UNBLOCK_SIGNAL
    return n;
  }
#endif // protocol version 1.x
}

int
operator >> (ACE_SOCK_Stream& in, cmlogPacket& p)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(p.lock_);
#endif
  int size = 0;
  int magic;
  int n;

#if (CMLOG_PROTOCOL_MAJOR_VERSION >= 2)
  // receive magic number
  n = in.recv_n (&magic, sizeof (int));
  if (n != sizeof (int)) {
#ifdef _CMLOG_DEBUG
    printf ("cmlogPacket: Receiving magic number error in >> operator\n");
#endif
    return -1;
  }
  magic = ntohl (magic);
  if (magic != CMLOG_PMAGIC) {
    fprintf (stderr, "cmlogPacket: Received a wrong magic number.\n");
    return -1;
  }
  size += n;
#ifdef _CMLOG_DEBUG
  printf ("ACE_SOCKET_STREAM >> cmlogPacket received magic 0x%x\n", magic);
#endif

  n = in.recv_n (&p, CMLOG_PACKET_PREDATA_SIZE);
  if (n != CMLOG_PACKET_PREDATA_SIZE) {
#ifdef _CMLOG_DEBUG
    printf ("Receiving error for cmlogPacket >> operator\n");
#endif
    return -1;
  }
  size += n;

  // decode header
  p.decode ();

  if (p.number_ > 0) {
    if (p.number_ > CMLOG_PACKET_LIST_SIZE || p.size_ > CMLOG_PACKET_MAX_SIZE) {
      fprintf (stderr, "cmlogPacket: error in receiving packet\n");
      return -1;
    }
    
    n = in.recv_n (p.buffer_, p.size_);
    if (n != p.size_) {
#ifdef _CMLOG_DEBUG
      printf ("cmlogPacket: receive whole buffer error\n");
#endif
      return -1;
    }
    size += n;
  }
#else  // protocol version 2.x
  // get message header
  n = in.recv_n (&p, CMLOG_PACKET_PREDATA_SIZE);
  if (n != CMLOG_PACKET_PREDATA_SIZE) {
#ifdef _CMLOG_DEBUG
    printf ("Receiving error for cmlogPacket >> operator\n");
#endif
    return -1;
  }
  size += n;

  // decode header
  p.decode ();

  if (p.number_ > 0) {
    if (p.number_ > CMLOG_PACKET_LIST_SIZE || p.size_ > CMLOG_PACKET_MAX_SIZE) {
      fprintf (stderr, "cmlogPacket: error in receiving packet\n");
      return -1;
    }
    
    n = in.recv_n (p.buffer_, p.size_);
    if (n != p.size_) {
#ifdef _CMLOG_DEBUG
      printf ("cmlogPacket: receive whole buffer error\n");
#endif
      return -1;
    }
    size += n;
  }
#endif  // protocol versin 1.x
  return size;
}
#endif

int
operator << (int out, cmlogPacket& p)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (p.lock_);
#endif
  int     n = 0, i = 0, m = 0;
  int     osize = 0;
  int     magic;

  struct iovec iovp[3];

#if (CMLOG_PROTOCOL_MAJOR_VERSION >= 2)
  if (p.number_ == 0) {
    // magic number 
    magic = htonl (CMLOG_PMAGIC);
    iovp[0].iov_base = (char *)&magic;
    iovp[0].iov_len = sizeof (int);
#ifdef _CMLOG_DEBUG
    printf ("out << cmlogPacket send out magic\n");
#endif

    // header information
    p.encode ();
    iovp[1].iov_base = (char *)&p;
    iovp[1].iov_len = CMLOG_PACKET_PREDATA_SIZE;

    CMLOG_BLOCK_SIGNAL
    n = ::writev (out, &(iovp[0]), (size_t)2);
    CMLOG_UNBLOCK_SIGNAL
    
    return n;
  }
  else {
    // magic number 
    magic = htonl (CMLOG_PMAGIC);
    iovp[0].iov_base = (char *)&magic;
    iovp[0].iov_len = sizeof (int);
#ifdef _CMLOG_DEBUG
    printf ("out << cmlogPacket send out magic\n");
#endif

    // original size information
    osize = p.size_;
    p.encode ();

    // send over the network
    iovp[1].iov_base = (char *)&p;
    iovp[1].iov_len = CMLOG_PACKET_PREDATA_SIZE;
    
    iovp[2].iov_base = p.buffer_;
    iovp[2].iov_len = osize;

    CMLOG_BLOCK_SIGNAL
    n = ::writev (out, &(iovp[0]), (size_t)3);
    CMLOG_UNBLOCK_SIGNAL

    return n;
  }
#else // protocol version 2.x
  if (p.number_ == 0) {
    // header information
    p.encode ();
    iovp[0].iov_base = (char *)&p;
    iovp[0].iov_len = CMLOG_PACKET_PREDATA_SIZE;

    CMLOG_BLOCK_SIGNAL
    n = ::writev (out, &(iovp[0]), (size_t)1);
    CMLOG_UNBLOCK_SIGNAL
    
    return n;
  }
  else {
    // original size information
    osize = p.size_;
    p.encode ();

    // send over the network
    iovp[0].iov_base = (char *)&p;
    iovp[0].iov_len = CMLOG_PACKET_PREDATA_SIZE;
    
    iovp[1].iov_base = p.buffer_;
    iovp[1].iov_len = osize;

    CMLOG_BLOCK_SIGNAL
    n = ::writev (out, &(iovp[0]), (size_t)2);
    CMLOG_UNBLOCK_SIGNAL

    return n;
  }
#endif // protocol version 1.x
}

int
operator >> (int in, cmlogPacket& p)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard(p.lock_);
#endif
  int size = 0;
  int magic;
  int n;

#if (CMLOG_PROTOCOL_MAJOR_VERSION >= 2)
  // receive magic number
  n = cmlogMsg::cmlog_recv_n (in, (char *)&magic, sizeof (int));
  if (n != sizeof (int)) {
#ifdef _CMLOG_DEBUG
    printf ("cmlogPacket: Receiving magic number error in >> operator\n");
#endif
    return -1;
  }
  magic = ntohl (magic);
  if (magic != CMLOG_PMAGIC) {
    fprintf (stderr, "cmlogPacket: Received a wrong magic number in >> \n");
    return -1;
  }
  size += n;
#ifdef _CMLOG_DEBUG
  printf ("in >> cmlogPacket received magic number 0x%x\n", magic);
#endif

  n = cmlogMsg::cmlog_recv_n (in, (char *)&p, CMLOG_PACKET_PREDATA_SIZE);
  if (n != CMLOG_PACKET_PREDATA_SIZE) {
#ifdef _CMLOG_DEBUG
    printf ("Receiving error for cmlogPacket >> operator\n");
#endif
    return -1;
  }
  size += n;

  // decode header
  p.decode ();
#ifdef _CMLOG_DEBUG
  printf ("Receive packet number %d and size %d\n", p.number_, p.size_);
#endif

  if (p.number_ > 0) {
    if (p.number_ > CMLOG_PACKET_LIST_SIZE || p.size_ > CMLOG_PACKET_MAX_SIZE) {
      fprintf (stderr, "cmlogMsg: error in receiving packet\n");
      return -1;
    }

    n = cmlogMsg::cmlog_recv_n (in, p.buffer_, p.size_);
    if (n != p.size_) {
#ifdef _CMLOG_DEBUG
      printf ("fd >> cmlogPacket receive whole buffer error\n");
#endif
      return -1;
    }
    size += n;
  }
#else // protocol version 2.x
  // get header
  n = cmlogMsg::cmlog_recv_n (in, (char *)&p, CMLOG_PACKET_PREDATA_SIZE);
  if (n != CMLOG_PACKET_PREDATA_SIZE) {
#ifdef _CMLOG_DEBUG
    printf ("Receiving error for cmlogPacket >> operator\n");
#endif
    return -1;
  }
  size += n;

  // decode header
  p.decode ();
#ifdef _CMLOG_DEBUG
  printf ("Receive packet number %d and size %d\n", p.number_, p.size_);
#endif

  if (p.number_ > 0) {
    if (p.number_ > CMLOG_PACKET_LIST_SIZE || p.size_ > CMLOG_PACKET_MAX_SIZE) {
      fprintf (stderr, "cmlogMsg: error in receiving packet\n");
      return -1;
    }

    n = cmlogMsg::cmlog_recv_n (in, p.buffer_, p.size_);
    if (n != p.size_) {
#ifdef _CMLOG_DEBUG
      printf ("fd >> cmlogPacket receive whole buffer error\n");
#endif
      return -1;
    }
    size += n;
  }
#endif // protocol version 1.x
  return size;
}

#ifdef CDEV_64BIT_LONG
#undef size_t
#endif

#ifdef __vxworks

cmlogVxPacket1::cmlogVxPacket1 (void)
:clientId_ (0), operationCode_ (0),
 message_ (0), arg0_ (0), arg1_ (0), arg2_ (0),
 arg3_ (0), arg4_ (0), arg5_ (0), arg6_ (0),
 arg7_ (0), arg8_ (0), arg9_ (0)
{
  map_.rawData = 0;
  map_.value.version = 1;
}

cmlogVxPacket1::~cmlogVxPacket1 (void)
{
  // empty
}

void
cmlogVxPacket1::set (short clientId, unsigned opcode,
		     char* message,
		     int arg0, int arg1, int arg2, int arg3,
		     int arg4, int arg5, int arg6, int arg7,
		     int arg8, int arg9)
{
  clientId_ = clientId;
  operationCode_ = opcode;
  message_ = message;
  arg0_ = arg0;
  arg1_ = arg1;
  arg2_ = arg2;
  arg3_ = arg3;
  arg4_ = arg4;
  arg5_ = arg5;
  arg6_ = arg6;
  arg7_ = arg7;
  arg8_ = arg8;
  arg9_ = arg9;

  map_.rawData = 0;
  map_.value.version = 1;
  map_.value.clientIDSet = 1;
  map_.value.operationCodeSet = 1;
  map_.value.messageSet = 1;
}


size_t
cmlogVxPacket1::size (void)
{
  size_t s = 0;

  // xdr converts everything into long
  s += sizeof (int);
  s += sizeof (int);
  s += sizeof (operationCode_);
  
  // plus size of char string len
  s += sizeof (size_t);

  size_t len = strlen (message_);
  len = cmlogRoundup (len, sizeof (int));

  s += len;

  s += (10*sizeof(int));

  return s;
}

int
cmlogVxPacket1::streamOut (char* buffer, size_t& size, size_t maxsize)
{
  int i = 0;
  size_t lsize = this->size ();

  // xdr converts everything into long
  bcopy ((char *)&map_.rawData, &(buffer[i]), sizeof (int));
  i += sizeof (int);
	 
  int temp = (int)clientId_;
  bcopy ((char *)&temp, &(buffer[i]), sizeof (int));
  i += sizeof (int);

  bcopy ((char *)&operationCode_, &(buffer[i]), sizeof (operationCode_));
  i += sizeof (operationCode_);

  // xdr does this without extra 0
  size_t olen = strlen (message_);
  size_t len = cmlogRoundup (olen, sizeof (int));
  bcopy ((char *)&olen, &(buffer[i]), sizeof (olen));
  i += sizeof (olen);

  bcopy (message_, &(buffer[i]), olen);
  i += olen;

  if (len > olen) {
    bcopy (0, &(buffer[i]), len - olen);
    i += (len - olen);
  }

  bcopy ((char *)&arg0_, &(buffer[i]), 10*sizeof(int));
  i += (10*sizeof (int));

  size = i;

  return 0;
}
#endif
