//-----------------------------------------------------------------------------
// Copyright (c) 1991,1992 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//-----------------------------------------------------------------------------
// Description:
//	 cmlogIntHash: cmlog hash table keyed by an integer
//                    Open Hash with buckets implemented by single linked lists
//    
//              Note: void *s are stored inside the table. This class
//                    will not manage these pointers. Callers have to
//                    manage these pointers
//
// Author: Jie Chen
// 
// Revision History:
//   $Log: cmlogIntHash.cc,v $
//   Revision 1.1.1.1  1999/09/07 15:29:10  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:27:22  bickley
// Added cmlog to application development system.
//
//
//
#include <stdio.h>
#include <assert.h>
#include "cmlogIntHash.h"

// Use multiplication method. h(k) = [m(kA mod1)]
// A = (sqrt(5) - 1)/ 2 ~ 0.6180339887 suggested by Dr. Knuth
// m is power of 2

const double cmlogA = 0.6180339887;

//======================================================================
//	class cmlogIntHash implementation
//======================================================================
cmlogIntHash::cmlogIntHash (unsigned int max)
:tablesize(max)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogIntHash Class Objects\n");
#endif
  // cmlogSlist has a default constructor, so one can create an array of the list
  buckets = new cmlogSlist[tablesize];
  assert (buckets);
}

cmlogIntHash::~cmlogIntHash (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogIntHash Class Objects\n");
#endif
  delete []buckets;
  buckets = 0;
}

int 
cmlogIntHash::isEmpty()
{
  // if any table is non-empty, return 0
  for (int i = 0; i < tablesize; i++)
    if (!buckets[i].isEmpty())
      return 0;

  // all empty
  return 1;
}

void
cmlogIntHash::deleteAllValues()
{
  // delete all values from a hash table
  // clear the elements from each of teh buckets
  for (int i = 0; i < tablesize; i++)
    buckets[i].deleteAllValues();
}

unsigned int 
cmlogIntHash::hash(const cmlogIntKeyItem key) const
{
  // return hashed value of key
  double val = (double)key*cmlogA - (int)(key*cmlogA);
  unsigned int slot = (unsigned int)(val*tablesize);
  return slot;
}

void
cmlogIntHash::add (cmlogIntKeyItem key, cmlogHashItem newele)
{
  buckets[hash (key)].add (newele);
}

int
cmlogIntHash::remove (cmlogIntKeyItem key, cmlogHashItem ele)
{
  return (buckets[hash (key)].remove (ele));
}

int
cmlogIntHash::find (cmlogIntKeyItem key, cmlogHashItem ele) const
{
  return buckets[hash (key)].includes(ele);
}

cmlogSlist&
cmlogIntHash::bucketRef (cmlogIntKeyItem key)
{
  return (buckets[hash (key)]);
}

//======================================================================
//	class cmlogIntHashIterator implementation
//======================================================================
cmlogIntHashIterator::cmlogIntHashIterator (cmlogIntHash& v)
:base(v), currentIndex(0), itr(0)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogIntHashIterator Class Object \n");
#endif
  // no further initialization
}

cmlogIntHashIterator::~cmlogIntHashIterator (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogIntHashIterator Class Object \n");
#endif
  currentIndex = 0;
  if (itr)
    delete itr;
}

int 
cmlogIntHashIterator::init (void)
{
  // initialize iterator, 
  // start search with first bucket
  currentIndex = 0; 
  itr = 0;
  return getNextIterator(); 
}

cmlogHashItem 
cmlogIntHashIterator::operator() (void)
{
  // return current element
  return (*itr)();
}

int 
cmlogIntHashIterator::operator ! (void)
{
  // test if there is a current element
  return itr != 0;
}

int 
cmlogIntHashIterator::operator ++ (void)
{
  // see if current iterator can be advanced
  if (itr && ++(*itr)) 
    return 1;

  // if not, get next iterator
  currentIndex++;
  return getNextIterator();
}

void 
cmlogIntHashIterator::operator = (cmlogHashItem val)
{
  // change the current value
  (*itr) = val;
}

int 
cmlogIntHashIterator::getNextIterator (void)
{
  // if there is an old iterator, delete it
  if (itr != 0) 
    delete itr;

  // now search for a new one
  for (; currentIndex < base.tablesize; currentIndex++){
    // generate a new iterator at the current point
    itr = new cmlogSlistIterator (base.buckets[currentIndex]);
    assert(itr != 0);

    // if it has at least one element, we're done
    if (itr->init())
      return 1;

    // otherwise delete it, try again
    delete itr;
  }
  // all out of iterators, can quit
  itr = 0;
  return 0;
}
