//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Implementation of cmlogData which holds information
//      about internal states of the server
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogData.cc,v $
//   Revision 1.4  2001/07/25 14:26:35  chen
//   64 BIT Initial Port
//
//   Revision 1.3  2000/06/20 19:32:49  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.2  2000/02/07 15:47:07  chen
//   handle overflow data when searching
//
//   Revision 1.1.1.1  1999/09/07 15:29:09  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:27:11  bickley
// Added cmlog to application development system.
//
//
//
#include <cmlog_cdevMessage.h>
#include <cmlog_cdevTagMap.h>
#include <cmlogUtil.h>
#include <cmlogMsg.h>
#include <cmlogBrowserIO.h>
#include <cmlogLogicSup.h>
#include "cmlogData.h"

// since lock_ has default constructor, it can be ignored from construction
// list. Compiler will do it for you

cmlogData::cmlogData (char* name)
:data_ (), getCbkList_ (), channels_ (),
 offCbkList_ ()
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogData Class Object\n");
#endif
  name_ = new char[::strlen (name) + 1];
  ::strcpy (name_, name);
}

cmlogData::cmlogData (char* name, const cdevData& data)
:data_ (data), getCbkList_ (), channels_ (),
 offCbkList_ ()
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogData Class Object\n");
#endif
  name_ = new char[::strlen (name) + 1];
  ::strcpy (name_, name);
}

cmlogData::~cmlogData (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogData Class Object\n");
#endif
  delete []name_;

  cmlogSlistIterator ite (channels_);
  cmlog_cdevMessage* msg = 0;

  for (ite.init (); !ite; ++ite) {
    msg = (cmlog_cdevMessage *) ite ();
    delete msg;
  }

  cmlogSlistIterator ite1 (getCbkList_);
  for (ite1.init (); !ite1; ++ite1) {
    msg = (cmlog_cdevMessage *) ite1 ();
    delete msg;
  }

  cmlogSlistIterator ite2 (offCbkList_);
  for (ite2.init (); !ite2; ++ite2) {
    msg = (cmlog_cdevMessage *) ite2 ();
    delete msg;
  }
}

int
cmlogData::connect (cmlogDataManager& manager)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif
  return manager.addData (this);
}

int
cmlogData::disconnect (cmlogDataManager& manager)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  return manager.removeData (this);
}

int
cmlogData::monitorOn  (cmlog_cdevMessage* msg)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  cmlogSlistIterator ite (channels_);
  cmlog_cdevMessage* tmsg = 0;

  for (ite.init (); !ite; ++ite) {
    tmsg = (cmlog_cdevMessage *) ite ();
    if (cmlogData::sameMessage (tmsg, msg))
      return -1;
  }
  channels_.add ((void *)msg);
  return 0;
}

int
cmlogData::monitorOff (cmlog_cdevMessage* msg)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  cmlogSlistIterator ite (channels_);
  cmlog_cdevMessage* tmsg = 0;
  int          found = 0;

  for (ite.init (); !ite; ++ite) {
    tmsg = (cmlog_cdevMessage *) ite ();
    if (cmlogData::sameMessage (tmsg, msg, 0)) {
      found = 1;
      break;
    }
  }
  if (found) {
    channels_.remove ((void *)tmsg);
    // free memory of tmsg which is no longer used
    delete tmsg;
    return 0;
  }
  return -1;
}

int
cmlogData::monitored (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  if (channels_.isEmpty ())
    return 0;
  return 1;
}

cmlogData& 
cmlogData::operator = (const cdevData& data)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  data_ = data;
  // notify all interested channel about changing of value
  notifyChannels ();
  return *this;
}

cmlogData::operator cdevData& (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  return data_;
}

char*
cmlogData::name (void) const
{
  return name_;
}

cmlogSlist&
cmlogData::getCbkList (void)
{
  return getCbkList_;
}

cmlogSlist&
cmlogData::monitorCbkList (void)
{
  return channels_;
}

cmlogSlist&
cmlogData::monitorOffCbkList (void)
{
  return offCbkList_;
}

void
cmlogData::asciiDump (FILE* fd)
{
  data_.asciiDump (fd);
}

int
cmlogData::sameMessage (cmlog_cdevMessage* msg1, cmlog_cdevMessage* msg2, int checkreq)
{
  short clientId1 = msg1->getClientID ();
  short clientId2 = msg2->getClientID ();  

  unsigned reqid1 = msg1->getTransIndex (); 
  unsigned reqid2 = msg2->getTransIndex (); 

  void *sock1 = msg1->getArg (); 
  void *sock2 = msg2->getArg (); 

  unsigned cbkid1 = msg1->getForeignDataIndex ();
  unsigned cbkid2 = msg2->getForeignDataIndex (); 

  unsigned opcode1 = msg1->getOperationCode (); 
  unsigned opcode2 = msg2->getOperationCode (); 

  if (checkreq) {
    if (clientId1 == clientId2 &&
	reqid1    == reqid2    &&
	sock1     == sock2     &&
	cbkid1    == cbkid2    &&
	opcode1   == opcode2   )
      return 1;
  }
  else {
    if (clientId1 == clientId2 &&
	sock1     == sock2     &&
	cbkid1    == cbkid2    &&
	opcode1   == opcode2   )
      return 1;
  }
  return 0;
}

void
cmlogData::update (void)
{
  // empty
}

void
cmlogData::notifyChannels (void)
{
  unsigned       qid;       // request id
  unsigned       cbkid;     // callback id
  char*          reqmsg;    // request message
  cmlogBrowserIO *io;       // iochannel
  cdevData       rdata;     // retuning data
  cmlogPacket    packet;
  cmlog_cdevTagMap     *tagmap = 0;

  cdevData*      qdata = 0;
  char           qstr[1024];    // this is the filtering string 
  int            hasqstr = 0;   // do we have query string 

  cmlogSlistIterator ite (channels_);
  cmlog_cdevMessage *msg = 0;
  

  for (ite.init (); !ite; ++ite) {
    // constructing the returning message
    msg = (cmlog_cdevMessage *) ite ();
    qid = msg->getTransIndex ();
    cbkid = msg->getForeignDataIndex ();
    reqmsg = msg->getMessage ();
    io = (cmlogBrowserIO *)msg->getArg ();
    tagmap = io->tagmap ();

    // check whether we have filtering message
    hasqstr = 0;  
    qdata = msg->getData ();
    if (qdata) {
      if (qdata->get (cmlogUtil::CMLOG_QUERYMSG_TAG, qstr, sizeof (qstr)) 
	  == CDEV_SUCCESS) 
	hasqstr = 1;
    }

    // do logic filtering here
    if (!hasqstr || (hasqstr && dataMatchLogic (data_, qstr) == 1) ) {
      // construct returning message    
      rdata = data_;
      rdata.insert (cmlogUtil::CMLOG_RESULT_TAG, CMLOG_SUCCESS);

      // convert local representation to remote representation
      tagmap->localToRemote (rdata);

      cmlog_cdevMessage rmsg (0, qid, 0, 0, cbkid, 0, 0, 0, reqmsg, &rdata, 0, 0);
      cmlogMsg    cmlogrmsg (rmsg);


      if (packet.overflow (cmlogrmsg)) {
	cdevData tdata;
	tdata.insert (cmlogUtil::CMLOG_FACILITY_TAG, (char *)("cmlogServer"));
	tdata.insert (cmlogUtil::CMLOG_KEY_TAG, cmlogUtil::currentTime());
	tdata.insert (cmlogUtil::CMLOG_RESULT_TAG, CMLOG_SUCCESS);
	tdata.insert (cmlogUtil::CMLOG_TEXT_TAG, (char *)("Incoming message + context overflows packet"));
	cmlog_cdevMessage trmsg (0, qid, 0, 0, cbkid, 0, 0, 0, reqmsg, &tdata, 0, 0);
	cmlogMsg    tcmlogrmsg (trmsg);
	packet.insert (tcmlogrmsg);
      }
      else 
	packet.insert (cmlogrmsg);

#ifdef _CMLOG_DEBUG
      if (io == 0) 
	fprintf (stderr, "IO channel empty: should never happen\n");
#endif
      // send out result
      io->sendResult (packet);
    
      // empty packet
      packet.empty ();
      // empty data object
      rdata.remove ();
    }
  }
}

void
cmlogData::getValue (cmlog_cdevMessage* msg)
{
  unsigned qid;       // request id
  unsigned cbkid;     // callback id
  char*    reqmsg;    // request message
  cmlogIO *io;        // iochannel
  cdevData rdata;     // retuning data
  cmlogPacket packet;


  // constructing the returning message
  qid = msg->getTransIndex ();
  cbkid = msg->getForeignDataIndex ();
  reqmsg = msg->getMessage ();
  io = (cmlogIO *)msg->getArg ();

  // get latest value
  update ();
  // copy data
  rdata = data_;
  // construct returning message 
  rdata.insert (cmlogUtil::CMLOG_RESULT_TAG, CMLOG_SUCCESS);

  cmlog_cdevMessage rmsg (0, qid, 0, 0, cbkid, 0, 0, 0, reqmsg, &rdata, 0, 0);
  cmlogMsg    cmlogrmsg (rmsg);

  
  if (packet.overflow (cmlogrmsg)) {
    cdevData tdata;
    tdata.insert (cmlogUtil::CMLOG_RESULT_TAG, CMLOG_ERROR);
    cmlog_cdevMessage trmsg (0, qid, 0, 0, cbkid, 0, 0, 0, reqmsg, &tdata, 0, 0);
    cmlogMsg    tcmlogrmsg (trmsg);
    packet.insert (tcmlogrmsg);
  }
  else 
    packet.insert (cmlogrmsg);

#ifdef _CMLOG_DEBUG
  if (io == 0) 
    fprintf (stderr, "IO channel empty: should never happen\n");
#endif
  // send out result
  io->sendResult (packet);
    
  // empty packet
  packet.empty ();
  // empty data object
  rdata.remove ();
}

void
cmlogData::removeAllCbksOfCh (void* ch)
{
  void* ich = 0;

  cmlogSlistIterator ite (channels_);
  cmlog_cdevMessage* msg = 0;

  for (ite.init (); !ite; ++ite) {
    msg = (cmlog_cdevMessage *) ite ();
    ich = msg->getArg ();
    if (ich == ch) {
      delete msg;
      ite.removeCurrent ();
    }
  }

  cmlogSlistIterator ite1 (getCbkList_);
  for (ite1.init (); !ite1; ++ite1) {
    msg = (cmlog_cdevMessage *) ite1 ();
    ich = msg->getArg ();
    if (ich == ch) {
      delete msg;
      ite1.removeCurrent ();
    }
  }

  cmlogSlistIterator ite2 (offCbkList_);
  for (ite2.init (); !ite2; ++ite2) {
    msg = (cmlog_cdevMessage *) ite2 ();
    ich = msg->getArg ();
    if (ich == ch) {
      delete msg;
      ite2.removeCurrent ();
    }
  }
}  





			   


  
