//-----------------------------------------------------------------------------
// Copyright (c) 1996 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//-----------------------------------------------------------------------------
//
// Description:
//       Common Message Logger Filter
//
// Author:  Danjin Wu & Jie Chen
//       
//
// Revision History:
//   $Log: cmlogFilter.cc,v $
//   Revision 1.1.1.1  1999/09/07 15:29:09  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:26:38  bickley
// Added cmlog to application development system.
//
//
#include <cmlogUtil.h>
#include "cmlogFilter.h"

//=========================================================================
//      Implementation of cmlogCounter
//=========================================================================
cmlogCounter::cmlogCounter (void) 
:valid_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogCounter Class Object\n");
#endif
}

cmlogCounter::~cmlogCounter (void) 
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogCounter Class Object\n");
#endif  
  if (type_ == CDEV_STRING) 
    delete data_.str;
}
  
int 
cmlogCounter::update (int &dropFlag, int &num)
{
  double t;
  
  if ((t= cmlogUtil::currentTime()) <= (t0_ + deltaTime_)) {
    if(count_ < maxCount_) {
      count_++;
      ts_ = cmlogUtil::currentTime();
      dropFlag = 0;
      return CMLOG_NOFILTERING;
    }
    else {
      count_++;
      dropFlag = 0;
      return CMLOG_FILTERED;
    }
  }
  else { 
    t0_ = t;
    if (count_ < maxCount_) dropFlag = 0;    
    else if (maxCount_ <= count_ < 2*maxCount_) {      
      if (count_ == maxCount_ ) dropFlag = 0; 
      else {
	dropFlag = 1;
	count_ = num = count_- maxCount_;
      }
    }
    else if (count_ >= 2*maxCount_) {
      dropFlag = 1;
      num = count_- maxCount_;
      count_ = maxCount_;
    }
    return CMLOG_NOFILTERING;
  }   
}
  

void 
cmlogCounter::setLimit(int limit, double delta)
{
  maxCount_ = limit;
  deltaTime_ = delta;
}

void 
cmlogCounter::reset(void) 
{
  valid_=0; 
  if(type_==CDEV_STRING) 
    delete data_.str;
}

void 
cmlogCounter::init (void) 
{
  valid_=1; 
  count_=1; 
  t0_ = ts_= cmlogUtil::currentTime ();
}

void 
cmlogCounter::set (BYTE val) 
{
  init(); 
  type_=CDEV_BYTE; 
  data_.cval=val;
}
  
void 
cmlogCounter::set (short val) 
{
  init(); 
  type_=CDEV_INT16;
  data_.sval=val;
}

void 
cmlogCounter::set (unsigned short val) 
{
  init(); 
  type_=CDEV_UINT16; 
  data_.usval=val;
}

void 
cmlogCounter::set (long val) 
{
  init(); 
  type_=CDEV_INT32; 
  data_.lval=val;
}

void 
cmlogCounter::set (unsigned long val) 
{
  init(); 
  type_=CDEV_UINT32; 
  data_.ulval=val;
}

void 
cmlogCounter::set (float val) 
{
  init(); 
  type_=CDEV_FLOAT; 
  data_.fval=val;
}

void 
cmlogCounter::set (double val) 
{
  init(); 
  type_=CDEV_DOUBLE; 
  data_.dval=val;
}

void
cmlogCounter::set (char* val) 
{
  init();
  type_=CDEV_STRING;
  data_.str=new char[strlen(val)+1];
  ::strcpy(data_.str,val);
}

int 
cmlogCounter::match (BYTE val) const 
{
  return (valid_==1 && type_==CDEV_BYTE && val==data_.cval)?1:0;
}
  
int 
cmlogCounter::match (short val) const 
{
  return (valid_==1 && type_==CDEV_INT16 && val==data_.sval)?1:0;
}
  
int 
cmlogCounter::match (unsigned short val) const 
{
  return (valid_==1 && type_==CDEV_UINT16 && val==data_.usval)?1:0;
}
  
int 
cmlogCounter::match (long val) const 
{
  return (valid_==1 && type_==CDEV_INT32 && val==data_.lval)?1:0;
}
 
int
cmlogCounter::match (unsigned long val) const 
{
  return (valid_==1 && type_==CDEV_UINT32 && val==data_.ulval)?1:0;
}
  
int
cmlogCounter::match (float val) const 
{
  return (valid_==1 && type_==CDEV_FLOAT && val==data_.fval)?1:0;
}
  
int 
cmlogCounter::match (double val) const 
{
  return (valid_==1 && type_==CDEV_DOUBLE && val==data_.dval)?1:0;
}
  
int
cmlogCounter::match (char*  val) const 
{
  return (valid_==1 && type_==CDEV_STRING && ::strcmp(val,data_.str)==0)?1:0;
}
  
double
cmlogCounter::lastUpdate (void) const 
{
  return ts_;
}


//=========================================================================
//      Implementation of cmlogFilter
//=========================================================================

// strang tag denotes not initialized tag
int cmlogFilter::tag_no_use = 0xdeadbeaf;


cmlogFilter::cmlogFilter (int maxSeverity, int maxVerbosity, int maxListSize)
:maxSeverity_ (maxSeverity), maxVerbosity_ (maxVerbosity),
 maxListSize_ (maxListSize), tag_ (cmlogFilter::tag_no_use), 
 verbosityThreshold_ (maxVerbosity),
 severityThreshold_ (0), listSize_ (maxListSize), 
 limit_ (CMLOG_DEFAULT_LIMIT), delta_ ((double)CMLOG_DELTA_TIME),
 msgList_ ()
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogFilter Class Object\n");
#endif
}

cmlogFilter::~cmlogFilter (void) 
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogFilter Class Object\n");
#endif
  cmlogSlistIterator ite (msgList_);
  cmlogCounter* counter = 0;

  for (ite.init (); !ite; ++ite) {
    counter = (cmlogCounter *) ite ();
    delete counter;
  }
}

int
cmlogFilter::setVerbosityThreshold (int thresh)
{
  if ( thresh >= 0 && thresh <= maxVerbosity_) {
    verbosityThreshold_ = thresh; 
    return 0; 
  }
  return -1;
}

int 
cmlogFilter::getVerbosityThreshold (void) const 
{
  return verbosityThreshold_;
}

int 
cmlogFilter::setSeverityThreshold (int thresh)
{
  if ( thresh >= 0 && thresh <= maxSeverity_) {
    severityThreshold_ = thresh; 
    return 0; 
  }
  return -1;
}

int
cmlogFilter::getSeverityThreshold (void) const 
{
  return severityThreshold_;
}

int 
cmlogFilter::setThrottle (char *ctag, int listSize, 
			  int limit, double deltaTime)
{
  if ( cdevData::tagC2I(ctag, &tag_) == CDEV_SUCCESS) {
    throttleTag_ = new char[sizeof(::strlen(ctag)+1)];
    ::strcpy (throttleTag_, ctag);
    listSize_ = listSize;
    limit_ = limit;
    delta_ = deltaTime;
    return CMLOG_SUCCESS;
  }
  return CMLOG_ERROR;
}


int
cmlogFilter::getThrottle (char **ctag, int& listSize, 
			  int& limit, double& deltaTime) {
  if (throttleTag_) {
    *ctag =  new char [::strlen(throttleTag_) + 1];
    strcpy(*ctag, throttleTag_);
    listSize = listSize_;
    limit = limit_;
    deltaTime = delta_;
    return CMLOG_SUCCESS;
  } 
  return CMLOG_ERROR;
}

int
cmlogFilter::filter(cdevData& data) 
{
  void *tmp;
  if (data.find(cmlogUtil::CMLOG_SEVERITY_TAG, (void * &)tmp) == CDEV_SUCCESS) 
    if ( *((int *)tmp) < severityThreshold_) 
      return CMLOG_FILTERED; // not send msg

  if (data.find(cmlogUtil::CMLOG_VERBOSITY_TAG, (void * &)tmp) == CDEV_SUCCESS)
    if ( *((int *)tmp) > verbosityThreshold_) 
      return CMLOG_FILTERED;

  if (tag_ != cmlogFilter::tag_no_use
      && data.find(tag_, (void * &)tmp) == CDEV_SUCCESS) { 
    cdevDataTypes type = data.getType(tag_);
    // goto through throttleList to match the unique value  
    if (msgList_.count () == 0) {  // empty list,  append the counterObj  
      cmlogCounter * counterObj = new cmlogCounter;
      counterObj->setLimit (limit_, delta_);
      setCounter (type, tmp, counterObj);  
      msgList_.add((void *)counterObj);
      return CMLOG_NOFILTERING; // send msg
    }
    else { // not an empty list, find match counter object in the list
      cmlogSlistIterator sit(msgList_);
      cmlogCounter* counter = 0;
      int dropped, num, status;

      for (sit.init(); !sit; ++sit) {
	counter = (cmlogCounter *) sit ();
	if (matchCounter(type, tmp, counter)) { 
	  status = counter->update(dropped, num);
	  if (dropped) 
	    data.insert(cmlogUtil::CMLOG_DROPPED_TAG, num);
	  return status;
	}
      }
      if (msgList_.count() < listSize_) {
	cmlogCounter * counterObj = new cmlogCounter;
	counterObj->setLimit(limit_, delta_);
	setCounter (type, tmp, counterObj);
	msgList_.add((void *)counterObj);
	return CMLOG_NOFILTERING;
      }
      else {
	// find out a LRU counter to reset
	double timer = 0;
	cmlogCounter* tmpCount = 0;
	cmlogCounter* counter = 0;

	for (sit.init(); !sit; ++sit) {
	  counter = (cmlogCounter *)sit ();
	  if (timer == 0) {
	    timer = counter->lastUpdate(); 
	    tmpCount = counter;
	  }
	  else {
	    tmpCount = (timer < counter->lastUpdate()) ? tmpCount : counter;
	    timer = (timer < counter->lastUpdate()) ? timer:(counter->lastUpdate());
	  }
	}
	tmpCount->reset();
	setCounter (type, tmp, tmpCount);
	return CMLOG_NOFILTERING;
      }    
    }
  }
  return CMLOG_NOFILTERING;
}

void 
cmlogFilter::setCounter (cdevDataTypes type, void *pdata, 
                         cmlogCounter *counterObj)
{
  switch (type) {
  case CDEV_BYTE: 
    counterObj->set(*(BYTE *)pdata);
    break;
  case CDEV_INT16: 
    counterObj->set(*(short *)pdata);
    break;
  case CDEV_UINT16:
    counterObj->set(*(unsigned short *)pdata);
    break;
  case CDEV_INT32:
    counterObj->set(*(long *)pdata);
    break;
  case CDEV_UINT32:
    counterObj->set(*(unsigned long *)pdata);
    break;
  case CDEV_FLOAT:
    counterObj->set(*(float *)pdata);
    break;
  case CDEV_DOUBLE:
    counterObj->set(*(double *)pdata);
    break;
  case CDEV_STRING:
    counterObj->set((char *)pdata);
    break;
  default:
    break;
  }
}

int 
cmlogFilter::matchCounter(cdevDataTypes type, void *pdata, 
			  cmlogCounter *counterObj)
{
  int result = 0;
  switch (type) {
  case CDEV_BYTE: 
    result = counterObj->match(*(BYTE *)pdata);
    break;
  case CDEV_INT16: 
    result = counterObj->match(*(short *)pdata);
    break;
  case CDEV_UINT16:
    result = counterObj->match(*(unsigned short *)pdata);
    break;
  case CDEV_INT32:
    result = counterObj->match(*(long *)pdata);
    break;
  case CDEV_UINT32:
    result = counterObj->match(*(unsigned long *)pdata);
    break;
  case CDEV_FLOAT:
    result = counterObj->match(*(float *)pdata);
    break;
  case CDEV_DOUBLE:
    result = counterObj->match(*(double *)pdata);
    break;
  case CDEV_STRING:
    result = counterObj->match((char *)pdata);
    break;
  default:
    break;
  }
  return result;
}
