//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
//-----------------------------------------------------------------------------
//
// Description:
//      CMLOG Query Request Object Class
//
// Author:  Jie Chen
//
// Revision History:
//   $Log: cmlogQRequestObject.cc,v $
//   Revision 1.1.1.1  1999/09/07 15:29:09  chen
//   CMLOG version 2.0
//
// Revision 1.4  1998/10/07  17:02:28  chen
// Fix a compiler (egcs) warning about cmlog_cdevMessage& = (cmlog_cdevMessage)
//
// Revision 1.3  1997/09/24  17:40:28  chen
// fix update callback's transaction finished flag
//
// Revision 1.2  1997/09/19  15:21:37  chen
// minor change
//
// Revision 1.1  1997/09/18  14:28:56  chen
// new cmlog query service
//
//
//
#include <cdevErrCode.h>
#include <cdevCommon.h>
#include <cdevDirectory.h>
#include <cmlogBrowser.h>

#include "cmlogQRequestObject.h"
#include "cmlogQService.h"
#include "cmlogQUpdateObj.h"

cmlogQRequestObject::cmlogQRequestObject (char* device, char* msg,
					  cmlogQService* svc,
					  cdevSystem& system)
:cdevRequestObject (device, msg, system), browser_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("             Create cmlogQRequestObject Class Object\n");
#endif
  // get pointer of real client handler
  browser_ = &(svc->browser_);

  // find action
  cmlogQRequestObject::findAction (msg, action_);
}

cmlogQRequestObject::~cmlogQRequestObject (void)
{
#ifdef _TRACE_OBJECTS
  printf ("             Delete cmlogQRequestObject Class Object\n");
#endif
}

int
cmlogQRequestObject::getState (void)
{
  if (browser_->connected ())
    return CDEV_STATE_CONNECTED;
  else
    return CDEV_STATE_NOTCONNECTED;
}

int
cmlogQRequestObject::send (cdevData& out, cdevData& result)
{
  return send (&out, &result);
}

int
cmlogQRequestObject::send (cdevData* out, cdevData& result)
{
  return send (out, &result);
}

int
cmlogQRequestObject::send (cdevData& out, cdevData* result)
{
  return send (&out, result);
}

int
cmlogQRequestObject::send (cdevData* out, cdevData* result)
{
  return CDEV_INVALIDARG;
}

int
cmlogQRequestObject::sendNoBlock (cdevData& out, cdevData& result)
{
  return sendNoBlock (&out, &result);
}

int
cmlogQRequestObject::sendNoBlock (cdevData* out, cdevData& result)
{
  return sendNoBlock (out, &result);
}

int
cmlogQRequestObject::sendNoBlock (cdevData& out, cdevData* result)
{
  return sendNoBlock (&out, result);
}

int
cmlogQRequestObject::sendNoBlock (cdevData* out, cdevData* result)
{
  return CDEV_INVALIDARG;
}

int
cmlogQRequestObject::sendCallback (cdevData& data, cdevCallback& callback)
{
  return sendCallback (&data, callback);
}

int
cmlogQRequestObject::sendCallback (cdevData* out, cdevCallback& userCallback)
{
  if (deferExecution (out, &userCallback))
    return CDEV_SUCCESS;

  int status = CDEV_ERROR;
  cdevTranObj *xobj = 0;
  cmlogQService* qservice = (cmlogQService *)service_;


  // copy userCallback into the following callback, which will free 
  // the user from worrying about cdevCallback object
  // note: cdevCallback's new operator is an overloaded operator
  cdevCallback* callback = 0;

  if (browser_->connected ()) {
    switch (action_) {
    case cmlogQRequestObject::QUERY:
      {
	if (out == 0)
	  return CDEV_INVALIDARG;

	callback = new cdevCallback (userCallback);
	xobj = new cdevTranObj (&system_, this, 0, callback);

	cmlogQUpdateObj* obj = new cmlogQUpdateObj (*xobj);
	xobj->disableDeleteCbk ();

	status = query (*out, &(cmlogQRequestObject::queryCallback),
			(void *)obj);
	if (status != CDEV_SUCCESS) {
	  delete xobj;
	  delete obj;
	}
	else
	  qservice->addUpdateObject (obj);
      }
      break;
    case cmlogQRequestObject::UPDATE:
      {
	if (qservice->hasCallback (userCallback)) {
	  system_.reportError (CDEV_SEVERITY_WARN, deviceName_, this,
			       "%s %s already has this callback\n",
			       deviceName_, message_);
	  return CDEV_INVALIDARG;
	}

	callback = new cdevCallback (userCallback);
	xobj = new cdevTranObj (&system_, this, 0, callback);

	cmlogQUpdateObj* obj = new cmlogQUpdateObj (*xobj);
	xobj->disableDeleteCbk ();
	status = update (out, &(cmlogQRequestObject::updateCallback),
			 (void *)obj);

	if (status != CDEV_SUCCESS) {
	  delete xobj;
	  delete obj;
	}
	else {
	  // we do not want to block group to finish here
	  delete obj->tobj_;
	  obj->tobj_ = 0;

	  qservice->addUpdateObject (obj);
	}
      }
      break;
    case cmlogQRequestObject::UPDATEOFF:
      {
	cmlogQService* qservice = (cmlogQService *)service_;
	if (userCallback.callbackFunction () == 0)  // remove all
	  status = updateOffAll (qservice);
	else {
	  if (!qservice->hasCallback (userCallback))
	    return CDEV_INVALIDARG;
	  status = updateOff    (userCallback);
	}
      }
      break;
    default:
      break;
    }
  }
  return status;
}

int
cmlogQRequestObject::query (cdevData& data, 
			    cmlogBrCallback callback, void* arg)
{
  double from, to;

  if (data.get ("start", &from) != CDEV_SUCCESS)
    return CDEV_ERROR;
  
  if (data.get ("end", &to) != CDEV_SUCCESS)
    return CDEV_ERROR;

  if (browser_->queryCallback ("query", data,
			       callback, arg) != CMLOG_SUCCESS) {
    system_.reportError (CDEV_SEVERITY_ERROR, deviceName_, this,
			 "%s %s cannot send command to cmlogServer\n",
			 deviceName_, message_);
    return CDEV_ERROR;
  }
  return CDEV_SUCCESS;
}

int
cmlogQRequestObject::update (cdevData* data,
			     cmlogBrCallback callback, void* arg)
{
  cdevData tdata;
  if (data)
    tdata = *data;
  
  if (browser_->queryCallback ("monitorOn loggingData", tdata,
			       callback, arg) != CMLOG_SUCCESS) {
    system_.reportError (CDEV_SEVERITY_ERROR, deviceName_, this,
			 "%s %s cannot send command to cmlogServer\n",
			 deviceName_, message_);
    return CDEV_ERROR;
  }
  return CDEV_SUCCESS;    
}

void
cmlogQRequestObject::queryCallback (int status, void* arg,
				    cmlogPacket* data)
{
  cmlogQUpdateObj* xobj = (cmlogQUpdateObj *)arg;

  cdevCallback* cbk = xobj->userCallback_;
  cdevRequestObject* obj = xobj->reqObj_;
  cdevData result;
  int      cbk_finished = 0;

  // remove transaction objects so that groups can finish
  if (xobj->tobj_) {
    delete xobj->tobj_;
    xobj->tobj_ = 0;
  }

  if (status == CMLOG_ERROR || status == CMLOG_NOTFOUND) {
    cbk->fireCallback (CDEV_NOTFOUND, cbk->userarg (),
		       *obj, result, 0);
    cbk_finished = 1;
  }
  else if (status == CMLOG_INCOMPLETE) 
    cmlogQRequestObject::processResult (status, data, xobj);
  else if (status == CMLOG_PAUSED) 
    cbk->fireCallback (CDEV_SUCCESS, cbk->userarg (),
		       *obj, result, 0);
  else if (status == CMLOG_QUERYMSG_ERR) {
    cbk->fireCallback (CDEV_ERROR, cbk->userarg (),
		       *obj, result, 0);
    cbk_finished = 1;
  }
  else { // callback finished
    cbk->fireCallback (CDEV_SUCCESS, cbk->userarg (),
		       *obj, result, 0);
    cbk_finished = 1;
  }
  if (cbk_finished) {
    cmlogQRequestObject* robj = (cmlogQRequestObject *)obj;
    robj->removeQueryCallback (xobj);
  }
}

void
cmlogQRequestObject::updateCallback (int status,
				     void* arg, cmlogPacket* packet)
{
  cmlogQUpdateObj* obj = (cmlogQUpdateObj *)arg;
  cdevData* res = 0;

  if (obj->tobj_) { // release transaction object
    delete obj->tobj_;
    obj->tobj_ = 0;
  }

  // get callback pointer
  cdevCallback* cbk = obj->userCallback_;
  cdevRequestObject* req = obj->reqObj_;

  cdev_TS_STAMP  ts;
  double         ds;

  if (status == CMLOG_SUCCESS) {
    cmlogMsg** msgs = packet->messages ();
    for (int i = 0; i < packet->numberOfData (); i++) {
      cmlog_cdevMessage& idata = (*msgs[i]);
      res = idata.getData ();
      if (res->get ("cmlogTime", &ds) == CDEV_SUCCESS) {
	res->remove ("cmlogTime");
	ts.secPastEpoch = (unsigned long)ds;
	ts.nsec = (ds - (unsigned long)ds) * 1000;  // convert to nano sec
	res->insert ("cmlogTime", ts);
      }
      cbk->fireCallback (CDEV_SUCCESS, cbk->userarg (),
			 *req, *res, 1);
    }
    // free memory
    for (i = 0; i < packet->numberOfData (); i++) 
      delete msgs[i];
    delete []msgs;
  }
  else if (status == CMLOG_CBK_FINISHED) {
    // fire callback to notify user callback that monitor has finished
    cdevData empty;
    cbk->fireCallback (CDEV_SUCCESS, cbk->userarg (),
		       *req, empty, 0);

    cmlogQRequestObject* robj = (cmlogQRequestObject *)req;
    
    robj->removeUpdateCallback ("loggingData",
				&(cmlogQRequestObject::updateCallback),
				(void *)obj);
  }
}

void
cmlogQRequestObject::processResult (int status, 
				    cmlogPacket* packet, cmlogQUpdateObj* xobj)
{
  cdevCallback* cbk = xobj->userCallback_;
  cdevRequestObject* obj = xobj->reqObj_;

  cmlogMsg** msgs = packet->messages ();
  cdevData* result = 0;
  cdev_TS_STAMP  ts;
  double         ds;

  for (int i = 0; i < packet->numberOfData (); i++) {
    cmlog_cdevMessage& idata = (*msgs[i]);
    result = idata.getData ();
    if (result->get ("cmlogTime", &ds) == CDEV_SUCCESS) {
      result->remove ("cmlogTime");
      ts.secPastEpoch = (unsigned long)ds;
      ts.nsec = (ds - (unsigned long)ds) * 1000;  // convert to nano sec
      result->insert ("cmlogTime", ts);
    }
    cbk->fireCallback (CDEV_SUCCESS, cbk->userarg (),
		       *obj, *result, 1);     
  }

  for (i = 0; i < packet->numberOfData (); i++) {
    delete msgs[i];
  }
  delete []msgs; 
}

int
cmlogQRequestObject::updateOffAll (cmlogQService* qsvc)
{
  cmlogSlistIterator ite (qsvc->uobjs_);
  cmlogQUpdateObj*   uobj = 0;

  for (ite.init(); !ite; ++ite) {
    uobj = (cmlogQUpdateObj *) ite ();
    updateOff (*(uobj->userCallback_));
  }
  return CDEV_SUCCESS;
}

int
cmlogQRequestObject::updateOff (cdevCallback& callback)
{
  cmlogQService* qsvc = (cmlogQService *)service_;
  cmlogSlistIterator ite (qsvc->uobjs_);
  cmlogQUpdateObj*   uobj = 0;

  for (ite.init(); !ite; ++ite) {
    uobj = (cmlogQUpdateObj *) ite ();  
    if (*(uobj->userCallback_) == callback)
      break;
  }

  cdevData empty;
  if (browser_->queryCallback ("monitorOff loggingData", empty,
			       &(cmlogQRequestObject::updateCallback),
			       (void *)uobj) != CMLOG_SUCCESS) {
    system_.reportError (CDEV_SEVERITY_ERROR, deviceName_, this,
			 "%s %s cannot send command to cmlogServer\n",
			 deviceName_, message_);
    return CDEV_ERROR;
  }
  return CDEV_SUCCESS;
}

int
cmlogQRequestObject::removeUpdateCallback (char* attr,
					   cmlogBrCallback callback,
					   void* arg)
{
  cmlogQUpdateObj* uobj = (cmlogQUpdateObj *)arg;
  cmlogQService* qservice = (cmlogQService *)service_;

  if (browser_->removeMonitorCallback (attr, callback, arg) == CMLOG_SUCCESS) {
    qservice->removeUpdateObject (uobj);
    delete uobj;
    return CDEV_SUCCESS;
  }
  return CDEV_ERROR;
}

int
cmlogQRequestObject::removeQueryCallback (cmlogQUpdateObj* uobj)
{
  cmlogQService* qservice = (cmlogQService *)service_;

  qservice->removeUpdateObject (uobj);
  delete uobj;
  return CDEV_SUCCESS;
}

      
int
cmlogQRequestObject::findAction (char* msg, int& action)
{
  if (strcasecmp (msg, "query") == 0)
    action = cmlogQRequestObject::QUERY;
  else if (strcasecmp (msg, "update") == 0)
    action = cmlogQRequestObject::UPDATE;
  else if (strcasecmp (msg, "updateOff") == 0)
    action = cmlogQRequestObject::UPDATEOFF;
  else
    action = cmlogQRequestObject::UNKNOWN;

  return CDEV_SUCCESS;
}
