/* -*- C++ -*- */

// ============================================================================
//
// = LIBRARY
//    ace
// 
// = FILENAME
//    Timer_Queue.h
//
// = AUTHOR
//    Doug Schmidt 
// 
// ============================================================================

#if !defined (ACE_TIMER_QUEUE_H)
#define ACE_TIMER_QUEUE_H

#include "ace/Event_Handler.h"
#include "ace/Time_Value.h"

#if defined (ACE_MT_SAFE)
#include "ace/Synch.h"
#endif /* ACE_MT_SAFE */

// This should be nested within the ACE_Timer_Queue class but some C++
// compilers still don't like this...

struct ACE_Timer_Node
  // = TITLE
  //     Maintains the state associated with a Timer entry.
{
friend class ACE_Timer_Queue;
private:
  ACE_Timer_Node (ACE_Event_Handler *h, 
		  const void *a, 
		  const ACE_Time_Value &t, 
		  const ACE_Time_Value &i, 
		  ACE_Timer_Node *n, 
		  int timer_id);
  // Constructor.

  ACE_Event_Handler *handler_;
  // Handler to invoke <handle_timeout> on when a timeout occurs.

  const void *arg_;
  // Argument to pass to <handle_timeout>.

  ACE_Time_Value timer_value_;
  // Time until the timer expires.

  ACE_Time_Value interval_;
  // If this is a periodic timer this holds the time until the next
  // timeout.

  ACE_Timer_Node *next_;
  // Pointer to next timer.
  
  int timer_id_;
  // Id of this timer (used to cancel timers before they expire).

  void dump (void) const;
  // Dump the state of an object.

  ACE_ALLOC_HOOK_DECLARE;
  // Declare the dynamic allocation hooks.
};

class ACE_Timer_Queue
  // = TITLE 
  //      Provides an interface to timers.
  //
  // = DESCRIPTION
  //      This is a simple implementation that uses a linked list of
  //      absolute times.  A more clever implementation would use a
  //      delta-list, a heap, or timing wheels.
{
public: 
  // = Initialization and termination methods.
  ACE_Timer_Queue (void);
  // Default constructor.
  virtual ~ACE_Timer_Queue (void);

  int is_empty (void) const;
  // True if queue is empty, else false.

  const ACE_Time_Value &earliest_time (void) const;
  // Returns the time of the earlier node in the Timer_Queue.

  int schedule (ACE_Event_Handler *event_handler, 
		const void *arg, 
		const ACE_Time_Value &delay,
		const ACE_Time_Value &interval = ACE_Time_Value::zero);
  // Schedule an <event_handler> that will expire after <delay> amount
  // of time.  If it expires then <arg> is passed in as the value to
  // the <event_handler>'s <handle_timeout> callback method.  If
  // <interval> is != to <ACE_Time_Value::zero> then it is used to
  // reschedule the <event_handler> automatically.  This method
  // returns a timer handle that uniquely identifies the
  // <event_handler> in an internal list.  This timer handle can be
  // used to cancel an <event_handler> before it expires.  The
  // cancellation ensures that timer_ids are unique up to values of
  // greater than 2 billion timers.  As long as timers don't stay
  // around longer than this there should be no problems with
  // accidentally deleting the wrong timer.

  int cancel (ACE_Event_Handler *event_handler);
  // Cancel all <event_handlers> that match the address of
  // <event_handler>.

  int cancel (int timer_id);
  // Cancel the single <ACE_Event_Handler> that matches the <timer_id>
  // value (which was returned from the <schedule> method).

  int expire (const ACE_Time_Value &current_time);
  // Run the <handle_timeout> method for all Timers whose values are
  // <= <cur_time>.

private:
  void reschedule (ACE_Timer_Node *);
  // Reschedule a "period" Timer_Node.

  ACE_Timer_Node *head_; 
  // Pointer to linked list of ACE_Timer_Handles. 

  int timer_id_;
  // Keeps track of the timer id that uniquely identifies each timer.
  // This id can be used to cancel a timer via the <cancel (int)>
  // method.

#if defined (ACE_MT_SAFE)
  ACE_Recursive_Mutex lock_; 
  // Synchronization variable for the MT_SAFE ACE_Reactor 
#endif /* ACE_MT_SAFE */

  void dump (void) const;
  // Dump the state of an object.

  ACE_ALLOC_HOOK_DECLARE;
  // Declare the dynamic allocation hooks.
};

#include "ace/Timer_Queue.i"

#endif /* ACE_TIMER_QUEUE_H */
