/* -*- C++ -*- */

// ============================================================================
//
// = LIBRARY
//    ace
// 
// = FILENAME
//    OS.h
//
// = AUTHOR
//    Doug Schmidt <schmidt@cs.wustl.edu> and Jesper S. M|ller <stophph@diku.dk>
// 
// ============================================================================

#if !defined (ACE_OS_H)
#define ACE_OS_H

#include "ace/config.h"

// Define the default constants for ACE.  Many of these are used for
// the ACE tests and applications in ./tests and ./apps.  You may want
// to change some of these to correspond to your environment.

#define ACE_DEFAULT_TIMEOUT 5
#define ACE_DEFAULT_THREADS 1

// The following 3 defines are used in the IP multicast and broadcast tests.
#define ACE_DEFAULT_BROADCAST_PORT 10000
#define ACE_DEFAULT_MULTICAST_PORT 10001
#define ACE_DEFAULT_MULTICAST_ADDR "224.9.9.2"

// Used in many IPC_SAP tests
#define ACE_DEFAULT_SERVER_PORT 10002  

// Used in Acceptor tests
#define ACE_DEFAULT_SERVER_PORT_STR "10002"  

// Used for the Service_Directory test
#define ACE_DEFAULT_SERVICE_PORT 10003 

// Used for the ACE_Thread_Spawn test
#define ACE_DEFAULT_THR_PORT     10004 

// Used for SOCK_Connect::connect() tests
#define ACE_DEFAULT_LOCAL_PORT 10005  

// Used for Connector tests
#define ACE_DEFAULT_LOCAL_PORT_STR "10005" 

// Used for ACE_MMAP_Memory_Pool
#define ACE_DEFAULT_BACKING_STORE "/tmp/ace-malloc-XXXXXX" 

#define ACE_DEFAULT_SERVER_HOST "localhost"

// Default shared memory key
#define ACE_DEFAULT_SHM_KEY 1234 

// Default semaphore key
#define ACE_DEFAULT_SEM_KEY 1234

// Default address for shared memory mapped files
#define ACE_DEFAULT_BASE_ADDR ((char *) (16 * 1024 * 1024)) 

// Used by the FIFO tests.
#define ACE_DEFAULT_RENDEZVOUS "/tmp/fifo.ace"

// The following 3 defines are used by the ACE Name Server...
#define ACE_DEFAULT_NAMESPACE_DIR "/tmp"
#define ACE_DEFAULT_LOCALNAME "/localnames"
#define ACE_DEFAULT_GLOBALNAME "/globalnames"

// Default file permissions.
#define ACE_DEFAULT_PERMS 0666

// Here are all ACE-specific global declarations needed throughout
// ACE.

// This is used to indicate that a platform doesn't support a
// particular feature.
#define ACE_NOTSUP_RETURN(FAILVALUE) do { errno = ENOTSUP ; return FAILVALUE; } while (0)

#if defined (ACE_NDEBUG)
#define ACE_DB(X)
#else
#define ACE_DB(X) X
#endif /* NDEBUG */

// The following is necessary since many C++ compilers don't support
// typedef'd types inside of classes used as formal template
// arguments... ;-(.  Luckily, using the C++ preprocessor I can hide
// most of this nastiness!

#if defined (ACE_HAS_TEMPLATE_TYPEDEFS)
#define ACE_SYNCH_1 class SYNCH
#define ACE_SYNCH_2 SYNCH
#define ACE_SYNCH_MUTEX SYNCH::MUTEX
#define ACE_SYNCH_CONDITION SYNCH::CONDITION
#else /* TEMPLATES are broken (must be a cfront-based compiler...) */
#define ACE_SYNCH_1 class SYNCH_MUTEX, class SYNCH_CONDITION
#define ACE_SYNCH_2 SYNCH_MUTEX, SYNCH_CONDITION
#define ACE_SYNCH_MUTEX SYNCH_MUTEX
#define ACE_SYNCH_CONDITION SYNCH_CONDITION
#endif /* ACE_HAS_TEMPLATE_TYPEDEFS */

// Increase the range of "address families".
#define AF_SPIPE (AF_MAX + 1)
#if !defined (AF_FILE) && !defined (ACE_HAS_AF_FILE)
#define AF_FILE (AF_MAX + 2)
#endif
#define AF_DEV (AF_MAX + 3)
#define AF_UPIPE (AF_SPIPE)

// Turn a number into a string.
#define ACE_ITOA(X) #X

// Create a string of a server address with a "port:host" format.
#define ACE_SERVER_ADDRESS(P,H) P":"H

// A couple useful inline functions for checking whether bits are 
// enabled or disabled. 

#define ACE_BIT_ENABLED(WORD, BIT) ((WORD & BIT) != 0)
#define ACE_BIT_DISABLED(WORD, BIT) ((WORD & BIT) == 0)
#define ACE_SET_BITS(WORD, BITS) (WORD |= BITS)
#define ACE_CLR_BITS(WORD, BITS) (WORD &= ~BITS)

// A useful abstraction for expressions involving operator new since
// we can change memory allocation error handling policies (e.g.,
// depending on whether ANSI/ISO exception handling semantics are
// being used). 

#define ACE_NEW_RETURN(POINTER,CONSTRUCTOR,RET_VAL) \
   { POINTER = new CONSTRUCTOR; \
     if (POINTER == 0) { errno = ENOMEM; return RET_VAL; }}

// These hooks enable ACE to have all dynamic memory management
// automatically handled on a per-object basis.

#if defined (ACE_HAS_ALLOC_HOOKS)
#define ACE_ALLOC_HOOK_DECLARE \
  void *operator new (size_t bytes); \
  void operator delete (void *ptr);

// Note that these are just place holders for now.  They'll
// be replaced by the ACE_Malloc stuff shortly...
#define ACE_ALLOC_HOOK_DEFINE(CLASS) \
  void *CLASS::operator new (size_t bytes) { return ::new char[bytes]; } \
  void CLASS::operator delete (void *ptr) { delete (ptr); }
#else
#define ACE_ALLOC_HOOK_DECLARE struct __Ace {} // Just need a dummy...
#define ACE_ALLOC_HOOK_DEFINE(CLASS) 
#endif /* ACE_HAS_ALLOC_HOOKS */

// A deep understanding of this is what separates the men from the boys ;-)
#define ACE_THREAD_EXIT_HOOK_INIT(TASK_SYNCH, TASK_PTR) \
  ACE_TSS< ACE_Task_Exit<TASK_SYNCH> > exit_hook \
    (new ACE_Task_Exit <TASK_SYNCH> (TASK_PTR))

#define ACE_THREAD_EXIT_HOOK_FINI(STATUS) \
  return exit_hook->status (STATUS)

#undef INLINE
#if defined (__INLINE__)
#define INLINE inline
#else
#define INLINE
#endif /* __INLINE__ */

#include "ace/Time_Value.h"

#if defined (ACE_HAS_H_ERRNO)
void herror (const char *str);
#endif

#if defined (ACE_LACKS_STRRECVFD)
struct strrecvfd {};
#endif /* ACE_LACKS_STRRECVFD */

#if !defined (MAP_FAILED) && !defined (ACE_HAS_MAP_FAILED)
#define MAP_FAILED ((void *) -1)
#endif /* MAP_FAILED */

#if defined (ACE_MT_SAFE)
#define ACE_MT(X) X
#if !defined (_REENTRANT)
#define _REENTRANT
#endif /* _REENTRANT */
#else
#define ACE_MT(X) 
#endif /* ACE_MT_SAFE */

#if !defined (ACE_HAS_THREAD_T) 
typedef int thread_t;
typedef char hthread_t;
typedef int thread_key_t;
#endif /* !ACE_HAS_THREAD_T */

#if defined (ACE_HAS_THREADS)
struct cancel_state
{
  int cancelstate;
  int canceltype;
};
#if defined (ACE_HAS_STHREADS)
#include <synch.h>
#include <thread.h>

// Typedefs to help compatibility with Windows NT and Pthreads.  Note
// that the hthread_t must be different than the thread_t type in
// order to make overloading work correctly in ACE_Thread_Manager.
typedef char hthread_t;

#define THR_CANCEL_DISABLE	0
#define THR_CANCEL_ENABLE	0
#define THR_CANCEL_DEFERRED	0
#define THR_CANCEL_ASYNCHRONOUS	0
#elif defined (ACE_HAS_PTHREADS)
// Definitions for mapping POSIX pthreads onto Solaris threads.

// Typedefs to help compatibility with Windows NT and Pthreads.
typedef tid_t hthread_t;
#define thread_t        pthread_t
#define thread_key_t    pthread_key_t
#define mutex_t         pthread_mutex_t
#define cond_t          pthread_cond_t
#if !defined (timespec)
// Deal with an AIX issue...?
#define timestruc_t 	struct timespec
#endif /* timespec */

// Needed to implement semaphores for pthreads.
struct sema_t
{
  pthread_mutex_t lock_;
  pthread_cond_t count_nonzero_;
  u_long count_;
};

#define USYNC_THREAD 	MUTEX_NONRECURSIVE_NP
#define USYNC_PROCESS 	MUTEX_NONRECURSIVE_NP
#define THR_CANCEL_DISABLE	PTHREAD_CANCEL_DISABLE
#define THR_CANCEL_ENABLE	PTHREAD_CANCEL_ENABLE
#define THR_CANCEL_DEFERRED	PTHREAD_CANCEL_DEFERRED
#define THR_CANCEL_ASYNCHRONOUS	PTHREAD_CANCEL_ASYNCHRONOUS
#define THR_BOUND               0x00000001
#define THR_NEW_LWP             0x00000002
#define THR_DETACHED            0x00000040
#define THR_SUSPENDED           0x00000080
#define THR_DAEMON              0x00000100
#define THR_UNDETACHED          0x00010000
#define THR_SCHED_FIFO          0x00020000
#define THR_SCHED_RR            0x00040000
#define THR_SCHED_DEFAULT       0x00080000
#define THR_SCOPE_SYSTEM        0x00100000
#define THR_SCOPE_PROCESS       0x00200000
#define THR_SCHED_INHERIT       0x00400000
#define THR_SCHED_EXPLICIT      0x00800000

// This is used to implement readers/writer locks for POSIX pthreads. 
struct rwlock_t
{
  mutex_t lock_; 
  // Serialize access to internal state.
 
  cond_t waiting_readers_;
  // Reader threads waiting to acquire the lock.
 
  int num_waiting_readers_;
  // Number of waiting readers.
 
  cond_t waiting_writers_;
  // Writer threads waiting to acquire the lock.
 
  int num_waiting_writers_;
  // Number of waiting writers.
 
  int ref_count_;
  // Value is -1 if writer has the lock, else this keeps track of the
  // number of readers holding the lock.
};
#endif /* ACE_HAS_STHREADS */
#else /* !ACE_HAS_THREADS, i.e., the OS/platform doesn't support threading. */
// Give these things some reasonable value...
#define THR_CANCEL_DISABLE	0
#define THR_CANCEL_ENABLE	0
#define THR_CANCEL_DEFERRED	0
#define THR_CANCEL_ASYNCHRONOUS	0
#define THR_DETACHED    0	// ?? ignore in most places 
#define THR_BOUND    	0	// ?? ignore in most places
#define THR_NEW_LWP	0	// ?? ignore in most places
#define THR_SUSPENDED   0	// ?? ignore in most places
// These are dummies needed for class OS.h

#if defined ( ACE_HAS_RPC_SVC_MT)  // SUNOS 5.6
#include <rpc/rpc.h>
#else
typedef int cond_t;
typedef int mutex_t;
typedef int sema_t;
typedef int rwlock_t;
#endif

#endif /* ACE_HAS_THREADS */

#ifndef __vxworks
#include <sys/types.h>
#include <sys/stat.h>
#include <limits.h>
#include <stdio.h>
#if __GNUC__>=3
#include <new>
#include <iostream>
#else
#include <new.h>
#include <iostream.h>
#endif
#include <ctype.h>
#include <signal.h>
#include <string.h>
#include <assert.h>
#include <stdarg.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#endif

#if defined (ACE_HAS_BROKEN_SENDMSG)
typedef struct msghdr ACE_SENDMSG_TYPE;
#else
typedef const struct msghdr ACE_SENDMSG_TYPE;
#endif /* ACE_HAS_BROKEN_SENDMSG */

#if defined (ACE_HAS_BROKEN_RANDR)
// The SunOS 5.x version of rand_r is inconsistent with the header files...
typedef unsigned int ACE_RANDR_TYPE;
extern "C" int rand_r(ACE_RANDR_TYPE seed);
#else
typedef unsigned int *ACE_RANDR_TYPE;
#endif /* ACE_HAS_BROKEN_RANDR */

#if !defined (ACE_HAS_RTLD_LAZY_V)
#define RTLD_LAZY 1
#endif /* !ACE_HAS_RTLD_LAZY_V */

#if defined (ACE_HAS_CHARPTR_DL)
typedef char *ACE_DL_TYPE;
#else
typedef const char *ACE_DL_TYPE;
#endif /* ACE_HAS_CHARPTR_DL */

#if defined (ACE_HAS_UTIME)
#include <utime.h>
#endif /* ACE_HAS_UTIME */

#if !defined (ACE_HAS_MSG)
struct msghdr {};
#endif /* ACE_HAS_MSG */

#if !defined (ACE_HAS_HI_RES_TIMER)
typedef int hrtime_t;
#endif /* ACE_HAS_HI_RES_TIMER */

#if !defined (ACE_HAS_SIG_ATOMIC_T)
typedef int sig_atomic_t;
#endif /* !ACE_HAS_SIG_ATOMIC_T */

#if !defined (ACE_HAS_SSIZE_T)
typedef int ssize_t;
#endif /* ACE_HAS_SSIZE_T */

#if defined (ACE_HAS_OLD_MALLOC)
typedef char *ACE_MALLOC_T;
#else
typedef void *ACE_MALLOC_T;
#endif /* ACE_HAS_OLD_MALLOC */

#if defined (ACE_HAS_POSIX_NONBLOCK)
#define ACE_NONBLOCK O_NONBLOCK
#else
#define ACE_NONBLOCK O_NDELAY
#endif /* ACE_HAS_POSIX_NONBLOCK */

#if defined (ACE_HAS_CONSISTENT_SIGNAL_PROTOTYPES)
/* Prototypes for both signal() and struct sigaction are consistent. */
typedef void (*ACE_SignalHandler)(int);
typedef void (*ACE_SignalHandlerV)(int);
#elif defined (ACE_HAS_IRIX_53_SIGNALS)
typedef void (*ACE_SignalHandler)(...);
typedef void (*ACE_SignalHandlerV)(...);
#elif defined (ACE_HAS_SPARCWORKS_401_SIGNALS)
typedef void (*ACE_SignalHandler)(int, ...);
typedef void (*ACE_SignalHandlerV)(int,...);
#elif defined (ACE_HAS_SUNOS4_SIGNAL_T)
typedef void (*ACE_SignalHandler)(void);
typedef void (*ACE_SignalHandlerV)(void);
#elif defined (ACE_HAS_SVR4_SIGNAL_T)
/* SVR4 Signals are inconsistent (e.g., see struct sigaction). */
typedef void (*ACE_SignalHandler)(int);
typedef void (*ACE_SignalHandlerV)(void);
#elif defined (ACE_WIN32)
typedef void (__cdecl *ACE_SignalHandler)(int); 
typedef void (__cdecl *ACE_SignalHandlerV)(int);
#else /* This is necessary for some older broken version of cfront */
#define ACE_SignalHandler SIG_PF
typedef void (*ACE_SignalHandlerV)(...);
#endif /* ACE_HAS_CONSISTENT_SIGNAL_PROTOTYPES */

#if !defined (ACE_HAS_SEMUN)
union semun {
	int		val;	
	// value for SETVAL 
	struct semid_ds	*buf;	
	// buffer for IPC_STAT & IPC_SET 
	unsigned short *array;	
	// array for GETALL & SETALL 
};
#endif /* !ACE_HAS_SEMUN */

#if defined (ACE_WIN32)
// The following are #defines and #includes that are specific to WIN32.

#define ACE_STDIN GetStdHandle (STD_INPUT_HANDLE)
#define ACE_STDOUT GetStdHandle (STD_OUTPUT_HANDLE)
#define ACE_STDERR GetStdHandle (STD_ERROR_HANDLE)

// These flags speed up the inclusion of Win32 header files.

// @@ Jesper, these are causing problems for MSVC++ 4.0.  Can you
// please recheck them?
#if 0
#define NOGDICAPMASKS     
#define NOVIRTUALKEYCODES 
#define NOWINMESSAGES     
#define NOWINSTYLES       
#define NOSYSMETRICS      
#define NOMENUS           
#define NOICONS           
#define NOKEYSTATES       
#define NOSYSCOMMANDS     
#define NORASTEROPS       
#define NOSHOWWINDOW      
#define OEMRESOURCE       
#define NOCLIPBOARD       
#define NOCOLOR           
#define NOCTLMGR          
#define NODRAWTEXT        
#define NOGDI             
#define NOUSER            
#define NOMB              
#define NOMETAFILE        
#define NOMSG             
#define NOOPENFILE        
#define NOSCROLL          
#define NOSOUND           
#define NOTEXTMETRIC      
#define NOWH              
#define NOWINOFFSETS      
#define NOCOMM            
#define NOKANJI           
#define NOHELP            
#define NOPROFILER        
#define NODEFERWINDOWPOS  
#endif /* 0 */

// This will help until we figure out everything:
#define NFDBITS 32 /* only used in unused functions... */ 
// These two may be used for internal flags soon:
#define MAP_PRIVATE 1
#define MAP_SHARED  2
#define MAP_FIXED   4

#define RUSAGE_SELF 1

// This one exists only to please Service_Config.h
// and Service_Manager.cpp
#define SIGHUP     1
#define SIGPIPE    13
#define O_NONBLOCK 1

// SYSV IPC FLAGS
#define IPC_CREAT 0
#define IPC_NOWAIT 0
#define IPC_RMID 0
#define IPC_PRIVATE 0
#define GETVAL 0
#define SETVAL 0
#define SEM_UNDO 0
struct sembuf 
{
  unsigned short sem_num; // semaphore # 
  short sem_op; // semaphore operation 
  short sem_flg; // operation flags 
};
struct shmaddr { };
struct shmid_ds { };
struct msgbuf {};
struct msqid_ds {};
#define	SYS_NMLN 257

// Fake the UNIX rusage structure.  Perhaps we can add more to this
// later on?
struct rusage
{
  FILETIME ru_utime;
  FILETIME ru_stime;
};

struct utsname 
{
  char sysname[SYS_NMLN];
  char nodename[SYS_NMLN];
  char release[SYS_NMLN];
  char version[SYS_NMLN];
  char machine[SYS_NMLN];
};

typedef int key_t;

// MMAP flags
#define PROT_READ PAGE_READONLY
#define PROT_WRITE PAGE_READWRITE
#define PROT_RDWR PAGE_READWRITE
/* If we can find suitable use for these flags, here they are:
PAGE_WRITECOPY
PAGE_EXECUTE
PAGE_EXECUTE_READ
PAGE_EXECUTE_READWRITE
PAGE_EXECUTE_WRITECOPY
PAGE_GUARD
PAGE_NOACCESS
PAGE_NOCACHE  */

// This is necessary since MFC users apparently can't #include
// <windows.h> directly.
#if !defined (__AFX_H__)
#include <windows.h>
#endif /* __AFX_H__ */

#include <winsock.h>

#define MAXHOSTNAMELEN  (MAX_COMPUTERNAME_LENGTH+1) // This should be around 16 or so...

// error code mapping
#define ETIME         		ERROR_SEM_TIMEOUT
#define EWOULDBLOCK             WSAEWOULDBLOCK
#define EINPROGRESS             WSAEINPROGRESS
#define EALREADY                WSAEALREADY
#define ENOTSOCK                WSAENOTSOCK
#define EDESTADDRREQ            WSAEDESTADDRREQ
#define EMSGSIZE                WSAEMSGSIZE
#define EPROTOTYPE              WSAEPROTOTYPE
#define ENOPROTOOPT             WSAENOPROTOOPT
#define EPROTONOSUPPORT         WSAEPROTONOSUPPORT
#define ESOCKTNOSUPPORT         WSAESOCKTNOSUPPORT
#define EOPNOTSUPP              WSAEOPNOTSUPP
#define EPFNOSUPPORT            WSAEPFNOSUPPORT
#define EAFNOSUPPORT            WSAEAFNOSUPPORT
#define EADDRINUSE              WSAEADDRINUSE
#define EADDRNOTAVAIL           WSAEADDRNOTAVAIL
#define ENETDOWN                WSAENETDOWN
#define ENETUNREACH             WSAENETUNREACH
#define ENETRESET               WSAENETRESET
#define ECONNABORTED            WSAECONNABORTED
#define ECONNRESET              WSAECONNRESET
#define ENOBUFS                 WSAENOBUFS
#define EISCONN                 WSAEISCONN
#define ENOTCONN                WSAENOTCONN
#define ESHUTDOWN               WSAESHUTDOWN
#define ETOOMANYREFS            WSAETOOMANYREFS
#define ETIMEDOUT               WSAETIMEDOUT
#define ECONNREFUSED            WSAECONNREFUSED
#define ELOOP                   WSAELOOP
#define EHOSTDOWN               WSAEHOSTDOWN
#define EHOSTUNREACH            WSAEHOSTUNREACH
#define EPROCLIM                WSAEPROCLIM
#define EUSERS                  WSAEUSERS
#define EDQUOT                  WSAEDQUOT
#define ESTALE                  WSAESTALE
#define EREMOTE                 WSAEREMOTE
/* Grrr! These two are already defined by the horrible 'standard' library */
/* #define ENAMETOOLONG            WSAENAMETOOLONG */
/* #define ENOTEMPTY               WSAENOTEMPTY */

#include <time.h>
#include <direct.h>
#include <io.h>

typedef DWORD thread_t;
typedef HANDLE hthread_t;
typedef DWORD pid_t;
typedef DWORD thread_key_t;
typedef DWORD nlink_t;

// 64-bit quad-word definitions
typedef	unsigned __int64  ACE_QWORD;
inline ACE_QWORD ACE_MAKE_QWORD (DWORD lo, DWORD hi) { return ACE_QWORD (lo) | (ACE_QWORD (hi) << 32); }
inline DWORD ACE_LOW_DWORD  (ACE_QWORD q) { return (DWORD) q; }
inline DWORD ACE_HIGH_DWORD (ACE_QWORD q) { return (DWORD) (q >> 32); }

// Win32 dummys to help compilation

typedef void *sigset_t;    // Who knows?
typedef int mode_t;
typedef CRITICAL_SECTION tmutex_t;
typedef HANDLE mutex_t;
typedef HANDLE sema_t;
typedef int uid_t;
typedef int gid_t;
typedef int hrtime_t;
typedef char *caddr_t;

// This structure is used to implement condition variables on NT!
struct cond_t
{
  DWORD waiters_;
  // Number of waiting threads/processes.

  sema_t sema_;
  // Queue up threads waiting for the condition to become signaled.

  mutex_t internal_mutex_;
  // Mutex that guards access to waiter count.
};

// This is used to implement readers/writer locks on NT!
struct rwlock_t
{
  mutex_t lock_; 
  // Serialize access to internal state.
 
  cond_t waiting_readers_;
  // Reader threads waiting to acquire the lock.
 
  int num_waiting_readers_;
  // Number of waiting readers.
 
  cond_t waiting_writers_;
  // Writer threads waiting to acquire the lock.
 
  int num_waiting_writers_;
  // Number of waiting writers.
 
  int ref_count_;
  // Value is -1 if writer has the lock, else this keeps track of the
  // number of readers holding the lock.
};

/* This is for file descriptors */
typedef HANDLE ACE_HANDLE;

// For Win32 compatibility. 
typedef SOCKET ACE_SOCKET;

#define ACE_INVALID_HANDLE INVALID_HANDLE_VALUE

#define USYNC_THREAD 0
#define USYNC_PROCESS 0

#define THR_CANCEL_DISABLE	0
#define THR_CANCEL_ENABLE	0
#define THR_CANCEL_DEFERRED	0
#define THR_CANCEL_ASYNCHRONOUS	0
#define THR_DETACHED    0	// ?? ignore in most places 
#define THR_BOUND    	0	// ?? ignore in most places
#define THR_NEW_LWP	0	// ?? ignore in most places
#define THR_SUSPENDED   CREATE_SUSPENDED

// Needed to map calls to NT transparently.
#define MS_ASYNC 0
#define MS_INVALIDATE 0

// Reliance on CRT - I don't really like this.

#define O_NDELAY    0
#define MAXPATHLEN  _MAX_PATH
#define MAXNAMLEN   _MAX_FNAME
#define EADDRINUSE WSAEADDRINUSE

// Undefined structs becomes undeclared overloads with MSVC++ 2.0
// Thus we need to resort to this for unsupported system calls.

struct sigaction 
{
  int sa_flags;
  ACE_SignalHandlerV sa_handler;
  sigset_t sa_mask;
};

#define SIG_BLOCK 0
#define SIG_SETMASK 0

// Why is this defined?  It must be a std C library symbol.
#if !defined (NSIG)
#define NSIG 1
#endif /* NSIG */

struct iovec
{
  char *iov_base; // data to be read/written
  size_t iov_len; // byte count to read/write
};

struct strbuf { };
struct rlimit { };
struct t_call { };
struct t_bind { };
struct t_info { };
struct t_optmgmt { };
struct t_discon { };
struct t_unitdata { };
struct t_uderr { };
struct netbuf { };
struct flock { }; // not used with Win32 locking...

#elif !defined (ACE_WIN32)

// The following are #defines and #includes that are specific to UNIX.

#define ACE_STDIN 0
#define ACE_STDOUT 1
#define ACE_STDERR 2

// Be consistent with Winsock 
#define ACE_INVALID_HANDLE -1

/* Provide compatibility with Windows NT. */
typedef int ACE_HANDLE;
// For Win32 compatibility. 
typedef ACE_HANDLE ACE_SOCKET;

#ifdef __vxworks
#ifdef CPU_FAMILY
#if    CPU_FAMILY==MC680X0 || CPU_FAMILY==TRON
typedef unsigned short INSTR;
#endif
#endif

typedef int key_t;

// SYSV IPC FLAGS
#define IPC_CREAT 0
#define IPC_NOWAIT 0
#define IPC_RMID 0
#define IPC_PRIVATE 0
#define GETVAL 0
#define SETVAL 0
#define SEM_UNDO 0

#define	SYS_NMLN 257


struct sembuf 
{
  unsigned short sem_num; // semaphore # 
  short sem_op; // semaphore operation 
  short sem_flg; // operation flags 
};
struct shmaddr { };
struct shmid_ds { };
struct msgbuf {};
struct msqid_ds {};
struct rlimit { };

struct utsname 
{
  char sysname[SYS_NMLN];
  char nodename[SYS_NMLN];
  char release[SYS_NMLN];
  char version[SYS_NMLN];
  char machine[SYS_NMLN];
};

#include "vxWorks.h"
#ifndef _ARCH_PPC
#include "arch/mc68k/regsMc68k.h"
#endif
#include "stdio.h"
#include "string.h"
#include "stdlib.h"
#include "stdarg.h"
#include "ctype.h"
#include "signal.h"
#include "fcntl.h"
#include "errno.h"
#include "time.h"
#include "sys/types.h"
#include "sys/ioctl.h"
#include "sys/socket.h"
#include "sys/times.h"
#include "sys/wait.h"
#include "sys/stat.h"
#include "in.h"
#include "ifLib.h"
#include "ioLib.h"
#include "logLib.h"
#include "inetLib.h"
#include "hostLib.h"
#include "sysLib.h"
#include "ifLib.h"
#include "taskLib.h"
#include "netLib.h"
#include "sigLib.h"
#include "sockLib.h"
#include "selectLib.h"
#include "kernelLib.h"
#include "net/if.h"

#define   NSIG _NSIGS

#else
#include <time.h>
#include <sys/socket.h>
#include <netdb.h>
#include <net/if.h>
#include <netinet/in.h>
#include <sys/un.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/uio.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <dirent.h>
#include <sys/utsname.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <pwd.h>
#include <unistd.h>
#include <arpa/inet.h>
#endif

#if defined (ACE_HAS_SIGINFO_T)
#include <siginfo.h>
#include <ucontext.h>
#endif /* ACE_HAS_SIGINFO_T */

#if defined (ACE_HAS_POLL)
#include <poll.h>
#endif /* ACE_HAS_POLL */

#if defined (ACE_HAS_STREAMS)
#include <stropts.h>
#endif /* ACE_HAS_STREAMS */

#if defined (ACE_LACKS_T_ERRNO)
extern int t_errno;
#endif /* ACE_LACKS_T_ERRNO */

#if !defined (ACE_HAS_SIGWAIT)
extern "C" int sigwait (sigset_t *set);
#endif /* ACE_HAS_SIGWAIT */

#if defined (ACE_HAS_SELECT_H)
#include <sys/select.h>
#endif /* ACE_HAS_SELECT_H */

#if defined (ACE_HAS_PROC_FS)
#include <sys/procfs.h>
#endif /* ACE_HAS_PROC_FD */

#if defined (ACE_HAS_ALLOCA_H)
#include <alloca.h>
#endif /* ACE_HAS_ALLOCA_H */

#if defined (ACE_HAS_TIUSER_H)
#include <tiuser.h> 
#endif /* ACE_HAS_TIUSER_H */

#if defined (ACE_HAS_SVR4_DYNAMIC_LINKING)
#include <dlfcn.h>
#endif /* ACE_HAS_SVR4_DYNAMIC_LINKING */

#if defined (ACE_HAS_SOCKIO_H)
#include <sys/sockio.h>
#endif 

#if defined (ACE_HAS_TIMOD_H)
#include <sys/timod.h>
#elif defined (ACE_HAS_OSF_TIMOD_H)
#include <tli/timod.h>
#endif /* ACE_HAS_TIMOD_H */

#if !defined (ACE_HAS_TLI_PROTOTYPES)
/* Define ACE_TLI headers for systems that don't prototype them... */
extern "C"
{
  int t_accept(int fildes, int resfd, struct t_call *call);  
  char *t_alloc(int fildes, int struct_type, int fields);
  int t_bind(int fildes, struct t_bind *req, struct t_bind *ret);
  int t_close(int fildes);
  int t_connect(int fildes, struct t_call *sndcall,
		struct t_call *rcvcall);
  void t_error(char *errmsg);
  int t_free(char *ptr, int struct_type);
  int t_getinfo(int fildes, struct t_info *info);
  int t_getname (int fildes, struct netbuf *namep, int type);
  int t_getstate(int fildes);
  int t_listen(int fildes, struct t_call *call);
  int t_look(int fildes);
  int t_open(char *path, int oflag, struct t_info *info);
  int t_optmgmt(int fildes, struct t_optmgmt *req,
		struct t_optmgmt *ret);
  int t_rcv(int fildes, char *buf, unsigned nbytes, int *flags);
  int t_rcvconnect(int fildes, struct t_call *call);
  int t_rcvdis(int fildes, struct t_discon *discon);
  int t_rcvrel(int fildes);
  int t_rcvudata(int fildes, struct t_unitdata *unitdata, int *flags);
  int t_rcvuderr(int fildes, struct t_uderr *uderr);
  int t_snd(int fildes, char *buf, unsigned nbytes, int flags);
  int t_snddis(int fildes, struct t_call *call);
  int t_sndrel(int fildes);
  int t_sndudata(int fildes, struct t_unitdata *unitdata);
  int t_sync(int fildes);
  int t_unbind(int fildes);
}
#endif /* !ACE_HAS_TLI_PROTOTYPES */

#if defined (ACE_LACKS_MODE_MASKS)
/* MODE MASKS */

/* de facto standard definitions */

#define	S_IFMT		0xF000	/* type of file */
#define	S_IAMB		0x1FF	/* access mode bits */
#define	S_IFIFO		0x1000	/* fifo */
#define	S_IFCHR		0x2000	/* character special */
#define	S_IFDIR		0x4000	/* directory */
#define	S_IFNAM		0x5000  /* XENIX special named file */
#define	S_INSEM 	0x1	/* XENIX semaphore subtype of IFNAM */
#define	S_INSHD 	0x2	/* XENIX shared data subtype of IFNAM */
#define	S_IFBLK		0x6000	/* block special */
#define	S_IFREG		0x8000	/* regular */
#define	S_IFLNK		0xA000	/* symbolic link */
#define	S_IFSOCK	0xC000	/* socket */
#define	S_ISUID		0x800	/* set user id on execution */
#define	S_ISGID		0x400	/* set group id on execution */
#define	S_ISVTX		0x200	/* save swapped text even after use */
#define	S_IREAD		00400	/* read permission, owner */
#define	S_IWRITE	00200	/* write permission, owner */
#define	S_IEXEC		00100	/* execute/search permission, owner */
#define	S_ENFMT		S_ISGID	/* record locking enforcement flag */

/* the following macros are for POSIX conformance */

#define	S_IRWXU	00700		/* read, write, execute: owner */
#define	S_IRUSR	00400		/* read permission: owner */
#define	S_IWUSR	00200		/* write permission: owner */
#define	S_IXUSR	00100		/* execute permission: owner */
#define	S_IRWXG	00070		/* read, write, execute: group */
#define	S_IRGRP	00040		/* read permission: group */
#define	S_IWGRP	00020		/* write permission: group */
#define	S_IXGRP	00010		/* execute permission: group */
#define	S_IRWXO	00007		/* read, write, execute: other */
#define	S_IROTH	00004		/* read permission: other */
#define	S_IWOTH	00002		/* write permission: other */
#define	S_IXOTH	00001		/* execute permission: other */

#define	S_ISFIFO(mode)	((mode&0xF000) == 0x1000)
#define	S_ISCHR(mode)	((mode&0xF000) == 0x2000)
#define	S_ISDIR(mode)	((mode&0xF000) == 0x4000)
#define	S_ISBLK(mode)	((mode&0xF000) == 0x6000)
#define	S_ISREG(mode)	((mode&0xF000) == 0x8000)
#define	S_ISLNK(mode)	((mode&0xF000) == 0xa000)
#define	S_ISSOCK(mode)	((mode&0xF000) == 0xc000)

/* POSIX.4 macros */
#define	S_TYPEISMQ(buf)		(0)
#define	S_TYPEISSEM(buf)	(0)
#define	S_TYPEISSHM(buf)	(0)
#endif /* ACE_LACKS_MODE_MASKS */

// IRIX5 defines bzero() in this odd file...
#if defined (ACE_HAS_BSTRING)
#include <bstring.h>
#endif /* ACE_HAS_BSTRING */

// AIX defines bzero() in this odd file...
#if defined (ACE_HAS_STRINGS)
#include <strings.h>
#endif /* ACE_HAS_STRINGS */

#if defined (ACE_HAS_TERM_IOCTLS)
#include <sys/termios.h>
#endif /* ACE_HAS_TERM_IOCTLS */

// Fixes a problem with HP/UX.
#if defined (ACE_HAS_BROKEN_MMAP_H)
extern "C" 
{
#include <sys/mman.h>
}
#else
#include <sys/mman.h>
#endif /* ACE_HAS_BROKEN_MMAP_H */

/* OSF1 has problems with sys/msg.h and C++... */
#if defined (__osf__)
#include "msg_hack.h"
#elif !defined (__vxworks)
#include <sys/msg.h>
#endif /* !defined __osf__ */

#endif /* ACE_WIN32 */

#if defined (ACE_SELECT_USES_INT)
typedef int ACE_FD_SET_TYPE;
#else
typedef fd_set ACE_FD_SET_TYPE;
#endif /* ACE_SELECT_USES_INT */

#if !defined (MAXNAMELEN)
#if defined (FILENAME_MAX)
#define MAXNAMELEN FILENAME_MAX
#else
#define MAXNAMELEN 256
#endif /* FILENAME_MAX */
#endif /* MAXNAMELEN */

#if !defined (ENOTSUP)
#define	ENOTSUP ENOSYS	/* Operation not supported		*/
#endif /* !ENOTSUP */

#if !defined (EDEADLK)
#define	EDEADLK 1000 /* Some large number... */
#endif /* !ENOTSUP */

#if !defined (MS_SYNC)
#define MS_SYNC	0x0
#endif /* !MS_SYNC */

#if !defined (PIPE_BUF)
#define PIPE_BUF 5120
#endif /* PIPE_BUF */

#if !defined (PROT_RDWR)
#define PROT_RDWR (PROT_READ|PROT_WRITE)
#endif /* PROT_RDWR */

#define LOCALNAME 0
#define REMOTENAME 1

#if defined (ACE_HAS_64BIT_LONGS)
// Necessary to support the Alphas, which have 64 bit longs and 32 bit
// ints...
typedef u_int ACE_UINT32;
#else
typedef u_long ACE_UINT32;
#endif /* ACE_HAS_64BIT_LONGS */

#if defined (ACE_HAS_STRUCT_HOSTENT_DATA)
typedef char ACE_HOSTENT_DATA[sizeof(struct hostent_data)];
typedef char ACE_SERVENT_DATA[sizeof(struct servent_data)];
#else
#if !defined ACE_HOSTENT_DATA_SIZE
#define ACE_HOSTENT_DATA_SIZE (4*1024)
#endif /*ACE_HOSTENT_DATA_SIZE */
#if !defined ACE_SERVENT_DATA_SIZE
#define ACE_SERVENT_DATA_SIZE (4*1024)
#endif /*ACE_SERVENT_DATA_SIZE */
typedef char ACE_HOSTENT_DATA[ACE_HOSTENT_DATA_SIZE];
typedef char ACE_SERVENT_DATA[ACE_SERVENT_DATA_SIZE];
#endif /* ACE_HAS_STRUCT_HOSTENT_DATA */

// Max size of an ACE Token.
#define ACE_MAXTOKENNAMELEN 40

// Max size of an ACE Token client ID.
#define ACE_MAXCLIENTIDLEN MAXHOSTNAMELEN + 20

/* Create some useful typedefs */
typedef void *(*ACE_THR_FUNC)(void *);
typedef const char **SYS_SIGLIST;

class ACE_OS
  // = TITLE
  //     This class defines all the OS/platform-related methods that
  //     ACE uses internally.
  //
  // = DESCRIPTION
  //     This class encapsulates all the differences between various
  //     versions of UNIX and WIN32!.  The other components in
  //     ACE are programmed to use only the methods in this class,
  //     which makes it *much* easier to move ACE to a new platform.
  //     The methods in this class also automatically restart when 
  //     interrupts occur during system calls (assuming that the
  //     ACE_Log_Msg::restart() flag is enabled).
{
public:
  struct flock_t
    // = TITLE
    //     OS file locking structure.
  {
#if defined (ACE_WIN32)
    OVERLAPPED overlapped_;
#else
    struct flock lock_;
#endif /* ACE_WIN32 */

    ACE_HANDLE handle_;
    // Handle to the underlying file.
  };

  // = A set of wrappers for miscellaneous operations.
  static int atoi (const char *s);
  static char *getenv (const char *symbol);
  static int getopt (int argc, char *const *argv, const char *optstring); 

  // = A set of wrappers for condition variables.
  static int cond_broadcast (cond_t *cv);
  static int cond_destroy (cond_t *cv);
  static int cond_init (cond_t *cv, int type, const char *name, void *arg);
  static int cond_signal (cond_t *cv);
  static int cond_timedwait (cond_t *cv, mutex_t *m, ACE_Time_Value *);
  static int cond_wait (cond_t *cv, mutex_t *m);

  // = A set of wrappers for determining config info.
  static char *cuserid (char *user, size_t maxlen = 32);
  static int uname (struct utsname *name);
  static long sysinfo (int cmd, char *buf, long count);
  static int hostname (char *name, size_t maxnamelen);

  // = A set of wrappers for explicit dynamic linking.
  static int dlclose (void *handle);
  static char *dlerror (void);
  static void *dlopen (ACE_DL_TYPE filename, int mode);
  static void *dlsym (void *handle, ACE_DL_TYPE symbol);

  // = A set of wrappers for stdio file operations.
  static int fclose (FILE *fp);
  static int fcntl (ACE_HANDLE handle, int cmd, int val = 0);
  static int fdetach (const char *file);
  static FILE *fdopen (ACE_HANDLE handle, const char *mode);
  static char *fgets (char *buf, int size, FILE *fp);
  static int fflush (FILE *fp);
  static FILE *fopen (const char *filename, const char *mode);
  static int fprintf (FILE *fp, const char *format, ...);
  static size_t fread (void *ptr, size_t size, size_t nelems, FILE
		       *fp); 
  static int fstat (ACE_HANDLE, struct stat *);
  static int ftruncate (ACE_HANDLE, off_t);
  static size_t fwrite (const void *ptr, size_t size, size_t nitems,
			FILE *fp); 
  static char *gets (char *str);
  static void perror (const char *s);
  static int printf (const char *format, ...);
  static int puts (const char *s);
  static void rewind (FILE *fp);
  static int sprintf (char *buf, const char *format, ...);

  // = A set of wrappers for file locks.
  static int flock_init (ACE_OS::flock_t *lock, int flags = 0, 
			 const char *name = 0, mode_t perms = 0);
  static int flock_destroy (ACE_OS::flock_t *lock);
  static int flock_rdlock (ACE_OS::flock_t *lock, short whence = 0, off_t start = 0, off_t len = 0);
  static int flock_tryrdlock (ACE_OS::flock_t *lock, short whence = 0, off_t start = 0, off_t len = 0);
  static int flock_trywrlock (ACE_OS::flock_t *lock, short whence = 0, off_t start = 0, off_t len = 0);
  static int flock_unlock (ACE_OS::flock_t *lock, short whence = 0, off_t start = 0, off_t len = 0);
  static int flock_wrlock (ACE_OS::flock_t *lock, short whence = 0, off_t start = 0, off_t len = 0);

  // = A set of wrappers for low-level process operations.
#ifndef _vxworks
  static int execl (const char *path, const char *arg0, ...);
  static int execle (const char *path, const char *arg0, ...);
  static int execlp (const char *file, const char *arg0, ...);
  static int execv (const char *path, char *const argv[]);
  static int execvp (const char *file, char *const argv[]);
  static int execve (const char *path, char *const argv[], char *const envp[]);
  static void _exit (int status);
#endif
  static void exit (int status);
  static pid_t fork (void);
  static uid_t getgid (void);
  static pid_t getpid (void);
  static uid_t getuid (void);
  static pid_t setsid (void);
  static int system (const char *s);
  static pid_t wait (int *);
  static pid_t waitpid (pid_t, int *, int);

  // = A set of wrappers for timers and resource stats.
  static u_int alarm (u_int delay);
  static hrtime_t gethrtime (void);
  static ACE_Time_Value gettimeofday (void);
  static int getrusage (int who, struct rusage *rusage);
  static int getrlimit (int resource, struct rlimit *rl);
  static int setrlimit (int resource, const struct rlimit *rl);
  static u_int sleep (u_int seconds);

  // = A set of wrappers for operations on time.
  static time_t time (time_t *tloc);
  static struct tm *localtime (const time_t *clock);
  static struct tm *localtime_r (const time_t *clock, struct tm *res);
  static char *asctime (const struct tm *tm);
  static char *asctime_r (const struct tm *tm, char *buf, int buflen);
  static char *ctime (const time_t *t);
  static char *ctime_r (const time_t *clock, char *buf, int buflen);

  // = A set of wrappers for memory managment.
  static void *sbrk (int brk);
  static void *malloc (size_t);
  static void *realloc (void *, size_t);
  static void free (void *);

  // = A set of wrappers for memory copying operations.
  static int memcmp (const void *s, const void *t, size_t len);
  static void *memcpy (void *s, const void *t, size_t len);
  static void *memset (void *s, int c, size_t len);

  // = A set of wrappers for System V message queues.
  static int msgctl (int msqid, int cmd, msqid_ds *);
  static int msgget (key_t key, int msgflg);
  static int msgrcv (int int_id, void *buf, size_t len, 
		     long type, int flags); 
  static int msgsnd (int int_id, const void *buf, size_t len, int
		     flags); 

  // = A set of wrappers for memory mapped files.
  static int madvise (caddr_t addr, size_t len, int advice);
  static void *mmap (void *addr, size_t len, int prot, int flags,
		     ACE_HANDLE handle, off_t off = 0,
		     ACE_HANDLE *file_mapping = 0);
  static int mprotect (void *addr, size_t len, int prot);
  static int msync (void *addr, size_t len, int sync);
  static int munmap (void *addr, size_t len);

  // = A set of wrappers for mutex locks.
  static int mutex_init (mutex_t *m, int type, 
			 const char *name, void *arg);
  static int mutex_destroy (mutex_t *m);
  static int mutex_lock (mutex_t *m);
  static int mutex_trylock (mutex_t *m);
  static int mutex_unlock (mutex_t *m);

  // = A set of wrappers for low-level file operations.
  static int close (ACE_HANDLE handle);
  static ACE_HANDLE creat (const char *filename, mode_t mode);
  static ACE_HANDLE dup (ACE_HANDLE handle);
  static int dup2 (ACE_HANDLE oldfd, ACE_HANDLE newfd);
  static int fattach (int handle, const char *path);
  static long filesize (ACE_HANDLE handle);
  static int getmsg (ACE_HANDLE handle, struct strbuf *ctl, struct strbuf
		     *data, int *flags); 
  static int getpmsg (ACE_HANDLE handle, struct strbuf *ctl, struct strbuf
		      *data, int *band, int *flags); 
  static int ioctl (ACE_HANDLE handle, int cmd, void * = 0);
  static int isastream (ACE_HANDLE handle);
  static int isatty (ACE_HANDLE handle);
  static off_t lseek (ACE_HANDLE handle, off_t offset, int whence);
  static ACE_HANDLE open (const char *filename, int mode, int perms = 0);
  static int putmsg (ACE_HANDLE handle, const struct strbuf *ctl, const
		     struct strbuf *data, int flags); 
  static int putpmsg (ACE_HANDLE handle, const struct strbuf *ctl, const
		      struct strbuf *data, int band, int flags); 
  static ssize_t read (ACE_HANDLE handle, void *buf, size_t len); 
  static ssize_t readv (ACE_HANDLE handle, struct iovec *iov, int iovlen); 
  static int recvmsg (ACE_HANDLE handle, struct msghdr *msg, int flags);
  static int sendmsg (ACE_HANDLE handle, ACE_SENDMSG_TYPE *msg, int flags); 
  static ssize_t write (ACE_HANDLE handle, const void *buf, size_t nbyte); 
  static int writev (ACE_HANDLE handle, const struct iovec *iov, int iovcnt);

  // = A set of wrappers for event demultiplexing and IPC.
  static int select (int width, fd_set *rfds, fd_set *wfds, fd_set *efds, ACE_Time_Value *tv);
  static int poll (struct pollfd *pollfds, u_long len, ACE_Time_Value *tv); 
  static int poll (struct pollfd *pollfds, u_long len, const ACE_Time_Value &tv); 
  static int pipe (ACE_HANDLE handles[]);

  // = A set of wrappers for directory operations.
  static mode_t umask (mode_t cmask);
  static int unlink (const char *path);
  static int chdir (const char *path);
  static int mkfifo (const char *file, mode_t mode);
  static char *mktemp (char *t);

  // = A set of wrappers for random number operations.
  static int rand (void);
  static int rand_r (ACE_RANDR_TYPE seed);
  static void srand (u_int seed);

  // = A set of wrappers for readers/writer locks.
  static int rwlock_init (rwlock_t *rw, int type, 
			  const char *name, void *arg);
  static int rwlock_destroy (rwlock_t *rw);
  static int rw_rdlock (rwlock_t *rw);
  static int rw_tryrdlock (rwlock_t *rw);
  static int rw_trywrlock (rwlock_t *rw);
  static int rw_unlock (rwlock_t *rw);
  static int rw_wrlock (rwlock_t *rw);

  // = A set of wrappers for semaphores.
  static int sema_destroy (sema_t *s);
  static int sema_init (sema_t *s, u_int count, int type, 
			const char *name, void *arg); 
  static int sema_post (sema_t *s);
  static int sema_trywait (sema_t *s);
  static int sema_wait (sema_t *s);

  // = A set of wrappers for System V semaphores.
  static int semctl (int int_id, int semnum, int cmd, semun);
  static int semget (key_t key, int nsems, int flags);
  static int semop (int int_id, struct sembuf *sops, size_t nsops); 

  // = A set of wrappers for System V shared memory.
  static void *shmat (int int_id, void *shmaddr, int shmflg);
  static int shmctl (int int_id, int cmd, struct shmid_ds *buf);
  static int shmdt (void *shmaddr);
  static int shmget (key_t key, int size, int flags);

  // = A set of wrappers for Signals.
  static int kill (pid_t pid, int signum);
  static int sigaction (int signum, const struct sigaction *nsa,
			struct sigaction *osa); 
  static int sigaddset (sigset_t *s, int signum);
  static int sigdelset (sigset_t *s, int signum);
  static int sigemptyset (sigset_t *s);
  static int sigfillset (sigset_t *s);
  static int sigismember (sigset_t *s, int signum);
  static ACE_SignalHandler signal (int signum, ACE_SignalHandler);
  static int sigprocmask (int signum, const sigset_t *nsp, sigset_t
			  *osp); 

  // = A set of wrappers for sockets.
  static ACE_HANDLE accept (ACE_HANDLE handle, struct sockaddr *addr, int
		     *addrlen); 
  static int bind (ACE_HANDLE s, struct sockaddr *name, int namelen);
  static int connect (ACE_HANDLE handle, struct sockaddr *addr, int
		      addrlen);
#ifndef __vxworks
  static struct hostent *gethostbyaddr (const char *addr, int length,
					int type); 
  static struct hostent *gethostbyname (const char *name);
  static struct hostent *gethostbyaddr_r (const char *addr, int length,
					  int type, struct hostent *result,
					  ACE_HOSTENT_DATA buffer,
					  int *h_errnop);
  static struct hostent *gethostbyname_r (const char *name, struct
					  hostent *result, ACE_HOSTENT_DATA buffer,
					  int *h_errnop);
#endif
  static int getpeername (ACE_HANDLE handle, struct sockaddr *addr, int
			  *addrlen); 
#ifndef __vxworks
  static struct servent *getservbyname (const char *svc, const char
					*proto); 
  static struct servent *getservbyname_r (const char *svc, const char *proto, 
					  struct servent *result, 
					  ACE_SERVENT_DATA buf);
#endif
  static int getsockname (ACE_HANDLE handle, struct sockaddr *addr, int *addrlen);
  static int getsockopt (ACE_HANDLE handle, int level, int optname, char
			 *optval, int *optlen); 
  static long inet_addr (const char *name);
  static char *inet_ntoa (const struct in_addr addr);
  static int listen (ACE_HANDLE handle, int backlog);
  static int recv (ACE_HANDLE handle, char *buf, int len, int flags = 0);
  static int recvfrom (ACE_HANDLE handle, char *buf, int len, int flags,
		       struct sockaddr *addr, int *addrlen);
  static int send (ACE_HANDLE handle, const char *buf, int len, int
		   flags = 0);
  static int sendto (ACE_HANDLE handle, const char *buf, int len, int
		     flags, const struct sockaddr *addr, int addrlen);
  static int setsockopt (ACE_HANDLE handle, int level, int optname, 
			 const char *optval, int optlen);
  static int shutdown (ACE_HANDLE handle, int how);
  static ACE_HANDLE socket (int domain, int type, int proto);
  static int socketpair (int domain, int type, int protocol,
			 ACE_HANDLE sv[2]); 

  // = A set of wrappers for regular expressions.
  static char *compile (const char *instring, char *expbuf, char
			*endbuf); 
  static int step (const char *str, char *expbuf);

  // = A set of wrappers for string operations.
  static int strcasecmp (const char *s, const char *t);
  static char *strcat (char *s, const char *t);
  static char *strchr (const char *s, int c);
  static int strcmp (const char *s, const char *t);
  static char *strcpy (char *s, const char *t);
  static char *strstr (const char *s, const char *t);
  static char *strdup (const char *s);
  static size_t strlen (const char *s);
  static int strncmp (const char *s, const char *t, size_t len);
  static char *strncpy (char *s, const char *t, size_t len);
  static char *strtok (char *s, const char *tokens);
  static long strtol (const char *s, char **ptr, int base);

  // = A set of wrappers for TLI.
  static int t_accept (ACE_HANDLE fildes, int resfd, struct t_call
		       *call); 
  static char *t_alloc (ACE_HANDLE fildes, int struct_type, int
			fields); 
  static int t_bind (ACE_HANDLE fildes, struct t_bind *req, struct
		     t_bind *ret); 
  static int t_close (ACE_HANDLE fildes);
  static int t_connect(int fildes, struct t_call *sndcall,
		       struct t_call *rcvcall);
  static void t_error (char *errmsg);
  static int t_free (char *ptr, int struct_type);
  static int t_getinfo (ACE_HANDLE fildes, struct t_info *info);
  static int t_getname (ACE_HANDLE fildes, struct netbuf *namep, int
			type); 
  static int t_getstate (ACE_HANDLE fildes);
  static int t_listen (ACE_HANDLE fildes, struct t_call *call);
  static int t_look (ACE_HANDLE fildes);
  static int t_open (char *path, int oflag, struct t_info *info);
  static int t_optmgmt (ACE_HANDLE fildes, struct t_optmgmt *req,
			struct t_optmgmt *ret); 
  static int t_rcv (ACE_HANDLE fildes, char *buf, unsigned nbytes, int
		    *flags); 
  static int t_rcvdis (ACE_HANDLE fildes, struct t_discon *discon);
  static int t_rcvrel (ACE_HANDLE fildes);
  static int t_rcvudata (ACE_HANDLE fildes, struct t_unitdata
			 *unitdata, int *flags); 
  static int t_rcvuderr (ACE_HANDLE fildes, struct t_uderr *uderr); 
  static int t_snd (ACE_HANDLE fildes, char *buf, unsigned nbytes, int
		    flags); 
  static int t_snddis (ACE_HANDLE fildes, struct t_call *call);
  static int t_sndrel (ACE_HANDLE fildes);
  static int t_sync (ACE_HANDLE fildes);
  static int t_unbind (ACE_HANDLE fildes);

  // = A set of wrappers for threads.
  static int thr_continue (hthread_t target_thread);
  static int thr_create (ACE_THR_FUNC,
			 void *args, 
			 long flags, 
			 thread_t *thr_id, 
			 hthread_t *t_handle = 0,
			 u_int priority = 0,
			 void *stack = 0,
			 size_t stacksize = 0);
  static void thr_destroy_tss (void);
  static int thr_equal (thread_t t1, thread_t t2);
  static void thr_exit (void *status = 0);
  static int thr_getconcurrency (void);
  static int thr_getprio (hthread_t thr_id, int *prio);
  static int thr_getspecific (thread_key_t key, void **data);
  static int thr_join (hthread_t waiter_id, void **status); 
  static int thr_join (thread_t waiter_id, thread_t *thr_id, void **status); 
  static int thr_keycreate (thread_key_t *key, void (*dest)(void *));
  static int thr_kill (thread_t thr_id, int signum);
  static size_t thr_min_stack (void);
  static thread_t thr_self (void);
  static void thr_self (hthread_t &);
  static int thr_setconcurrency (int hint);
  static int thr_setprio (hthread_t thr_id, int prio);
  static int thr_setspecific (thread_key_t key, void *data);
  static int thr_sigsetmask (int signum, const sigset_t *nsm, sigset_t *osm);
  static int thr_suspend (hthread_t target_thread);
  static int thr_setcancelstate (int new_state, int *old_state);
  static int thr_setcanceltype (int new_type, int *old_type);
  static int thr_cancel (thread_t t_id);
  static int sigwait (sigset_t *set, int *sig);
  static void thr_testcancel (void);
  static void thr_yield (void);

  static thread_t NULL_thread;
  // This is necessary to deal with POSIX pthreads insanity...

private:
  ACE_OS (void);
  // Ensure we can't define an instance of this class.

  static void mutex_lock_cleanup (void *lock);
};

#include "ace/Trace.h"

#if defined (ACE_HAS_INLINED_OSCALLS)
#if defined (INLINE)
#undef INLINE
#endif /* INLINE */
#define INLINE inline
#include "ace/OS.i"
#endif /* ACE_HAS_INLINED_OSCALLS */

#endif  /* ACE_OS_H */
