/* -*- C++ -*- */

// ============================================================================
//
// = LIBRARY
//    ace
// 
// = FILENAME
//    ACE.h
//
// = AUTHOR
//    Doug Schmidt 
// 
// ============================================================================

#if !defined (ACE_ACE_H)
#define ACE_ACE_H

#include "ace/OS.h"

// Forward declarations.
class ACE_Time_Value;

class ACE
{
  // = TITLE
  //     Contains various static methods used throughout ACE.
  //
  // = DESCRIPTION
  //     This class consolidates all these ACE methods in a single
  //     place in order to manage the namespace better.
public:
  static char *timestamp (char date_and_time[], int time_len);
  // Returns the current timestamp in the form
  // "hour:minute:second:microsecond."  The month, day, and year are
  // also stored in the beginning of the date_and_time array.  Returns
  // 0 if unsuccessful, else returns pointer to beginning of the
  // "time" portion of <day_and_time>.

  static int daemonize (void);
  // Become a daemon process.

  static ssize_t recv_n (ACE_HANDLE handle, 
			 void *buf, 
			 size_t len, 
			 int flags);
  // Receive <len> bytes into <buf> from <handle> (uses the <recv>
  // system call).

  static ssize_t recv_n (ACE_HANDLE handle, 
			 void *buf, 
			 size_t len);
  // Receive <len> bytes into <buf> from <handle> (uses the <read>
  // system call).

  static ssize_t send_n (ACE_HANDLE handle, 
			 const void *buf, 
			 size_t len, 
			 int flags);
  // Receive <len> bytes into <buf> from <handle> (uses the <send>
  // system call).

  static ssize_t send_n (ACE_HANDLE handle, 
			 const void *buf, 
			 size_t len);
  // Receive <len> bytes into <buf> from <handle> (uses the <write>
  // system call).

  static int set_fl (ACE_HANDLE handle, 
		     int flags);
  // Set flags associated with <handle>.

  static int clr_fl (ACE_HANDLE handle, 
		     int flags);
  // Clear flags associated with <handle>.

  static int set_handle_limit (int new_limit = -1);
  // Reset the limit on the number of open handles.

  static int bind_port (ACE_HANDLE handle);
  // Bind a new unused port to <handle>.

  static int handle_timed_accept (ACE_HANDLE listener,
				  ACE_Time_Value *timeout, 
				  int restart);
  // Wait up to <timeout> amount of time to accept a connection.  Note
  // that this method doesn't actually do the accept, it just does the
  // timed waiting...

  static ACE_HANDLE handle_timed_connect (ACE_Time_Value *timeout,
					  const char name[], 
					  int flags, 
					  int perms);
  // Try to "connect" to <name>.

  static ACE_HANDLE handle_timed_complete (ACE_HANDLE, 
					   ACE_Time_Value *);
  // Try to complete a non-blocking connection.

  static size_t	round_to_pagesize (off_t length);
  // Rounds the request to a multiple of the page size.

  static void format_hexdump (char *buffer, char *obuf, int size);
  // Format buffer into printable format.  This is useful for
  // debugging.

private:
  ACE (void);
  // Ensure we can't define an instance of this class...
};

#if defined (__INLINE__)
#include "ace/ACE.i"
#endif /* __INLINE__ */

#endif  /* ACE_ACE_H */
