/* -----------------------------------------------------------------------------
 * Copyright (c) 1994,1995 Southeastern Universities Research Association,
 *                         Continuous Electron Beam Accelerator Facility
 *
 * This software was developed under a United States Government license
 * described in the NOTICE file included as part of this distribution.
 *
 * -----------------------------------------------------------------------------
 *
 * defCallbackCollector : This class is used to collect the callbacks from a
 *                        a single cdevDefCollectionRequest.  This is an 
 *                        implementation class that is only associated with the
 *                        default cdevCollectionObject if the developer's 
 *                        service does not provide specific support for 
 *                        collections.
 *
 *                        The cdevCollectionRequest object will obtain the 
 *                        cdevData object from this object and will return it
 *                        to the caller or to an interim 
 *                        cdevGrpCollectionRequest object.
 *
 * Author:  Walt Akers
 *
 */

#include "defCallbackCollector.h"

// *****************************************************************************
// * defCallbackCollector::defCallbackCollector :
// *	This is the constructor for the defCallbackCollectior.  It initializes
// *	all internal data items.
// *****************************************************************************
defCallbackCollector::defCallbackCollector(int NRequests, cdevData &Format, cdevTranObj &XObj )
	: counter(0),
	  requests (NULL),
	  nRequests(NRequests),
	  result   (Format),
	  format   (Format),
	  xobj     (&XObj)
	{
	requests = new Request[nRequests];
	for(int i=0; i<nRequests; i++)
		{
		requests[i].index     = i;
		requests[i].processed = 0;
		requests[i].parent    = this;
		}
	}

// *****************************************************************************
// * defCallbackCollector::~defCallbackCollector :
// *	This is the destructor for the defCallbackCollector object, it will
// *	delete any memory that was allocated during the constructor.
// *****************************************************************************
defCallbackCollector::~defCallbackCollector ( void ) 
	{ 
	// *********************************************************************
	// * Transfer the data structure to the format cdevData item - this
	// * will improve performance when the cdevRequestObject is reused.
	// *********************************************************************
	cdevDataIterator iter(&result);
	int tag;
	iter.init();
	while((tag=iter.tag())!=0)
		{
		cdevDataTypes fromType = result.getType(tag);
		cdevDataTypes toType   = format.getType(tag);
		
		if(fromType!=CDEV_INVALID && 
		   fromType!=CDEV_STRING  &&
		   fromType!=toType)
			{
			installEmptyArray(format, tag, fromType, nRequests);
			}
		++iter;
		}

	if(requests) delete[] requests; 
	if(xobj)     delete xobj;
	requests = NULL;
	xobj     = NULL;
	}		

// *****************************************************************************
// * defCallbackCollector::finished :
// *	This method allows the caller to determine if all of the requests have
// *	been processed.
// *****************************************************************************
int defCallbackCollector::finished ( void ) 
	{
	return (counter>=nRequests);
	}

// *****************************************************************************
// * defCallbackCollector::processRequest :
// *	This method will copy the contents of the cdevData object associated
// *	with a single request to the result cdevData object that will hold
// *	the output for all of the requests..
// *
// *	This method increments the counter that is used to detect when all of
// *	the collections have been processed.
// *****************************************************************************
void defCallbackCollector::processRequest ( int idx, int status, cdevData *data ) 
	{ 
	if(idx>=0 && idx<nRequests && !requests[idx].processed)
		{
		if(data!=NULL)	
			{
			data->insert(cdevCollectionRequest::resultCodeTag(), status);
			mergeData  (result, *data, nRequests, idx);
			}
		else	{
			cdevData temp;
			temp.insert(cdevCollectionRequest::resultCodeTag(), status);
			mergeData(result, temp, nRequests, idx);
			}
		requests[idx].processed = 1;
		counter++;
		}
	}


// *****************************************************************************
// * defCallbackCollector::mergeData :
// *	This method will copy the data items that are associated with
// *	an inidividual request to the correct location in the arrays of the
// *	result cdevData object that will hold the combined output from all of 
// *	the individual requests.
// *****************************************************************************
void defCallbackCollector::mergeData (cdevData &to, cdevData &from, size_t size, int idx)
	{
	cdevDataIterator iter(&from);
	int tag;
	iter.init();
	while((tag=iter.tag())!=0)
		{
		size_t        arraySize = 0;

		from.getElems(tag, &arraySize);
				
		if(to.getType(tag)==CDEV_INVALID)
			{
			installEmptyArray(to, tag, from.getType(tag), size, arraySize);
			}
		copyItemToArray(to, from, tag, idx);

		++iter;
		}
	}


// *****************************************************************************
// * defCallbackCollector::resolveTargetType :
// *	This method is called when the data type of the inbound object differs
// *	from the data type of the outbound object. It will select the data type
// *	that can best represent both types.
// *****************************************************************************
cdevDataTypes defCallbackCollector::resolveTargetType(cdevDataTypes to, cdevDataTypes from )
	{
	static cdevDataTypes typeMatrix [10][10] = 
		{
/* To Type:     Byte         Short        UShort       Int          UInt         Float        Double       String       TimeStamp        Invalid       */
/* Byte     */  CDEV_BYTE,   CDEV_INT16,  CDEV_UINT16, CDEV_INT32,  CDEV_UINT32, CDEV_FLOAT,  CDEV_DOUBLE, CDEV_STRING, CDEV_BYTE,      CDEV_BYTE,
/* Short    */  CDEV_INT16,  CDEV_INT16,  CDEV_INT32,  CDEV_INT32,  CDEV_INT32,  CDEV_FLOAT,  CDEV_DOUBLE, CDEV_STRING, CDEV_INT16,     CDEV_INT16,   
/* UShort   */  CDEV_UINT16, CDEV_INT32,  CDEV_UINT16, CDEV_INT32,  CDEV_UINT32, CDEV_FLOAT,  CDEV_DOUBLE, CDEV_STRING, CDEV_UINT16,    CDEV_UINT16,  
/* Int      */  CDEV_INT32,  CDEV_INT32,  CDEV_INT32,  CDEV_INT32,  CDEV_INT32,  CDEV_FLOAT,  CDEV_DOUBLE, CDEV_STRING, CDEV_INT32,     CDEV_INT32,   
/* UInt     */  CDEV_UINT32, CDEV_INT32,  CDEV_UINT32, CDEV_INT32,  CDEV_UINT32, CDEV_FLOAT,  CDEV_DOUBLE, CDEV_STRING, CDEV_UINT32,    CDEV_UINT32,  
/* Float    */  CDEV_FLOAT,  CDEV_FLOAT,  CDEV_FLOAT,  CDEV_DOUBLE, CDEV_DOUBLE, CDEV_FLOAT,  CDEV_DOUBLE, CDEV_STRING, CDEV_FLOAT,     CDEV_FLOAT,   
/* Double   */  CDEV_DOUBLE, CDEV_DOUBLE, CDEV_DOUBLE, CDEV_DOUBLE, CDEV_DOUBLE, CDEV_DOUBLE, CDEV_DOUBLE, CDEV_STRING, CDEV_DOUBLE,    CDEV_DOUBLE,  
/* String   */  CDEV_STRING, CDEV_STRING, CDEV_STRING, CDEV_STRING, CDEV_STRING, CDEV_STRING, CDEV_STRING, CDEV_STRING, CDEV_STRING,    CDEV_STRING,  
/* Time     */  CDEV_BYTE,   CDEV_INT16,  CDEV_UINT16, CDEV_INT32,  CDEV_UINT32, CDEV_FLOAT,  CDEV_DOUBLE, CDEV_STRING, CDEV_TIMESTAMP, CDEV_TIMESTAMP,
/* Invalid  */	CDEV_BYTE,   CDEV_INT16,  CDEV_UINT16, CDEV_INT32,  CDEV_UINT32, CDEV_FLOAT,  CDEV_DOUBLE, CDEV_STRING, CDEV_TIMESTAMP, CDEV_INVALID,
		};
	if( to   > 9 ) return CDEV_STRING;
	if( from > 9 ) return CDEV_STRING;
	return typeMatrix[from][to];
	}

// *****************************************************************************
// * defCallbackCollector::copyItemToArray :
// *	This method will copy the element in a single tagged data item to its
// *	correct index in the output array.
// *****************************************************************************
void defCallbackCollector::copyItemToArray ( cdevData &to, cdevData &from, int tag, int idx)
	{
	size_t        size      = 0;
	size_t        fromElems = 1;
	size_t        toElems   = 1;
	size_t        toDim     = 0;
	size_t        toBounds[4];
	
	cdevDataTypes toType     = to.getType(tag);
	cdevDataTypes fromType   = from.getType(tag);
	cdevDataTypes targetType = resolveTargetType(toType, fromType);
	
	if(fromType!=CDEV_INVALID) from.getElems(tag, &fromElems);
	if(toType!=CDEV_INVALID)
		{
		to.getDim(tag, &toDim);
		if(toDim==2)
			{
			to.getBounds(tag, toBounds, 4);
			toElems = toBounds[3];
			size    = toBounds[1];
			}
		else	{			
			to.getElems(tag, &size);
			toElems = 1;
			toBounds[0] = 0 ;
			toBounds[1] = size;
			toBounds[2] = 0;
			toBounds[3] = 1;
			}
		}
		
	if(idx>=0 && idx<size)
		{
		// **************************************************************
		// * At this point, decide whether the array should be resized. 
		// * This will only occur when one of the underlying devices is 
		// * returning an array valued result.
		// **************************************************************
		if(toElems < fromElems)
			{
			resizeArray(to, tag, targetType, size, fromElems);
			toBounds[1] = size;
			toBounds[3] = fromElems;
			}
		
		// **************************************************************
		// * Compute the extended size of the array.
		// **************************************************************
		size_t totalSize = toBounds[1]*toBounds[3];
		
		switch(targetType)
			{
			case CDEV_BYTE:
				{
				unsigned char * ptr;
				
				// **********************************************
				// * If the types do not match, do a simple 
				// * extract and reinsert using the new type.  
				// * This will cause the types to syncronize.
				// **********************************************
				if(targetType!=toType)
					{
					ptr = new unsigned char[totalSize];
					to.get(tag, ptr);
					to.insert(tag, ptr, totalSize, ((toBounds[3]>1)?2:1));
					if(toBounds[3]>1) to.setBounds(tag, toBounds, 4);
					delete[] ptr;
					}

				// **********************************************
				// * Obtain a reference to the array within the
				// * target cdevData object, then simply get
				// * the data into the proper place in the array.
				// **********************************************
				to.find(tag, (void *&)ptr);
				from.get(tag, &ptr[idx*toBounds[3]]);

				break;
				}
		
			case CDEV_INT16:
				{
				short * ptr;
				
				if(targetType!=toType)
					{
					ptr = new short[totalSize];
					to.get(tag, ptr);
					to.insert(tag, ptr, totalSize, ((toBounds[3]>1)?2:1));
					if(toBounds[3]>1) to.setBounds(tag, toBounds, 4);
					delete[] ptr;
					}

				to.find(tag, (void *&)ptr);
				from.get(tag, &ptr[idx*toBounds[3]]);

				break;
				}

			case CDEV_UINT16:
				{
				unsigned short * ptr;
				
				if(targetType!=toType)
					{
					ptr = new unsigned short[totalSize];
					to.get(tag, ptr);
					to.insert(tag, ptr, totalSize, ((toBounds[3]>1)?2:1));
					if(toBounds[3]>1) to.setBounds(tag, toBounds, 4);
					delete[] ptr;
					}

				to.find(tag, (void *&)ptr);
				from.get(tag, &ptr[idx*toBounds[3]]);

				break;
				}

			case CDEV_INT32:
				{
				long * ptr;

				if(targetType!=toType)
					{
					ptr = new long[totalSize];
					to.get(tag, ptr);
					to.insert(tag, ptr, totalSize, ((toBounds[3]>1)?2:1));
					if(toBounds[3]>1) to.setBounds(tag, toBounds, 4);
					delete[] ptr;
					}

				to.find(tag, (void *&)ptr);
				from.get(tag, &ptr[idx*toBounds[3]]);

				break;
				}

			case CDEV_UINT32:
				{
				unsigned long * ptr;
				
				if(targetType!=toType)
					{
					ptr = new unsigned long[totalSize];
					to.get(tag, ptr);
					to.insert(tag, ptr, totalSize, ((toBounds[3]>1)?2:1));
					if(toBounds[3]>1) to.setBounds(tag, toBounds, 4);
					delete[] ptr;
					}

				to.find(tag, (void *&)ptr);
				from.get(tag, &ptr[idx*toBounds[3]]);

				break;
				}

			case CDEV_FLOAT:
				{
				float * ptr;
				
				if(targetType!=toType)
					{
					ptr = new float[totalSize];
					to.get(tag, ptr);
					to.insert(tag, ptr, totalSize, ((toBounds[3]>1)?2:1));
					if(toBounds[3]>1) to.setBounds(tag, toBounds, 4);
					delete[] ptr;
					}

				to.find(tag, (void *&)ptr);
				from.get(tag, &ptr[idx*toBounds[3]]);

				break;
				}

			case CDEV_DOUBLE:
				{
				double * ptr;
				
				if(targetType!=toType)
					{
					ptr = new double[totalSize];
					to.get(tag, ptr);
					to.insert(tag, ptr, totalSize, ((toBounds[3]>1)?2:1));
					if(toBounds[3]>1) to.setBounds(tag, toBounds, 4);
					delete[] ptr;
					}

				to.find(tag, (void *&)ptr);
				from.get(tag, &ptr[idx*toBounds[3]]);

				break;
				}

			case CDEV_STRING:
				{
				char ** ptr;
				int     i;
				
				// **********************************************
				// * You will always need to extract the strings
				// * totally prior to performing a reinsert.  
				// * This is because of the unique structure that
				// * is used to store the strings within the 
				// * cdevData object.
				// **********************************************
				ptr = new char *[totalSize];
				to.get(tag, ptr);

				// **********************************************
				// * If this is a collection of scalar values, 
				// * simply poak the new value into the array and
				// * reinsert it into the cdevData object. 
				// * Remember to delete the data extracted from
				// * these objects.
				// **********************************************
				if(toBounds[3]==1)
					{
					char *   item;
					from.get(tag, &item);
					delete ptr[idx];
					ptr[idx] = item;
					to.insert(tag, ptr, totalSize);
					}
				
				// **********************************************
				// * If this is a collection of array valued 
				// * objects, then we'll need to copy each item 
				// * to the appropriate position in the
				// * array.
				// **********************************************
				else	{
					char ** item = new char *[fromElems];
					from.get(tag, item);
					for(i=0; i<fromElems; i++)
						{
						delete[] ptr[(idx*toBounds[3])+i];
						ptr[(idx*toBounds[3])+i] = item[i];
						}
					// **************************************
					// * Remember to clear the trailing 
					// * string if the inbound array is 
					// * smaller than the existing array --- 
					// * otherwise, it may have '0' valued 
					// * strings following it.
					// **************************************
					while(i<toElems)
						{
						*ptr[(idx*toBounds[3])+i] = 0;
						i++;
						}
						
					to.insert(tag, ptr, totalSize, ((toBounds[3]>1)?2:1));
					if(toBounds[3]>1) to.setBounds(tag, toBounds, 4);
					delete[] item;
					}
				
				for(i=0; i<totalSize; i++) delete[] ptr[i];
				delete[] ptr;
				break;
				}

			case CDEV_TIMESTAMP:
				{
				cdev_TS_STAMP * ptr;
				
				if(targetType!=toType)
					{
					ptr = new cdev_TS_STAMP[totalSize];
					to.get(tag, ptr);
					to.insert(tag, ptr, totalSize, ((toBounds[3]>1)?2:1));
					if(toBounds[3]>1) to.setBounds(tag, toBounds, 4);
					delete[] ptr;
					}

				to.find(tag, (void *&)ptr);
				from.get(tag, &ptr[idx*toBounds[3]]);

				break;
				}
		
			case CDEV_INVALID:
			break;			
			}
		}
	}
	
	
// *****************************************************************************
// * defCallbackCollector::installEmptyArray :
// *	This method will create an empty array that is of the specified size
// *	and type.  By creating empty arrays in the cdevData object, the caller
// *	may insert data directly into the data structure rather than using
// *	the cdevData interface.
// *****************************************************************************
void defCallbackCollector::installEmptyArray ( cdevData &data, int tag, cdevDataTypes type, size_t size, size_t arraySize )
	{
	int elemCnt = size*arraySize;
	
	switch(type)
		{
		case CDEV_BYTE:
			{
			unsigned char * ptr = new unsigned char[elemCnt];
			memset(ptr, 0, sizeof(unsigned char)*elemCnt);
			data.insert(tag, ptr, elemCnt, (arraySize>1)?2:1);
			delete[] ptr;
			break;
			}
			
		case CDEV_INT16:
			{
			short * ptr = new short[elemCnt];
			memset(ptr, 0, sizeof(short)*elemCnt);
			data.insert(tag, ptr, elemCnt, (arraySize>1)?2:1);
			delete[] ptr;
			break;
			}

		case CDEV_UINT16:
			{
			unsigned short * ptr = new unsigned short[elemCnt];
			memset(ptr, 0, sizeof(unsigned short)*elemCnt);
			data.insert(tag, ptr, elemCnt, (arraySize>1)?2:1);
			delete[] ptr;
			break;
			}

		case CDEV_INT32:
			{
			long * ptr = new long[elemCnt];
			memset(ptr, 0, sizeof(long)*elemCnt);
			data.insert(tag, ptr, elemCnt, (arraySize>1)?2:1);
			delete[] ptr;
			break;
			}

		case CDEV_UINT32:
			{
			unsigned long * ptr = new unsigned long[elemCnt];
			memset(ptr, 0, sizeof(unsigned long)*elemCnt);
			data.insert(tag, ptr, elemCnt, (arraySize>1)?2:1);
			delete[] ptr;
			break;
			}

		case CDEV_FLOAT:
			{
			float * ptr = new float[elemCnt];
			memset(ptr, 0, sizeof(float)*elemCnt);
			data.insert(tag, ptr, elemCnt, (arraySize>1)?2:1);
			delete[] ptr;
			break;
			}

		case CDEV_DOUBLE:
			{
			double * ptr = new double[elemCnt];
			memset(ptr, 0, sizeof(double)*elemCnt);
			data.insert(tag, ptr, elemCnt, (arraySize>1)?2:1);
			delete[] ptr;
			break;
			}

		case CDEV_STRING:
			{
			char ** ptr = new char *[elemCnt];
			char    val = '\0';
			for(int i=0; i<elemCnt; i++) ptr[i] = &val;
			data.insert(tag, ptr, elemCnt, (arraySize>1)?2:1);
			delete[] ptr;
			break;
			}

		case CDEV_TIMESTAMP:
			{
			cdev_TS_STAMP * ptr = new cdev_TS_STAMP[elemCnt];
			memset(ptr, 0, sizeof(cdev_TS_STAMP)*elemCnt);
			data.insert(tag, ptr, elemCnt, (arraySize>1)?2:1);
			delete[] ptr;
			break;
			}
		
		case CDEV_INVALID:
		break;			
		}
	
	if(arraySize>1 && type!=CDEV_INVALID)
		{
		size_t bounds[4];

		bounds[0] = 0;
		bounds[1] = size;
		bounds[2] = 0;
		bounds[3] = arraySize;

		data.setBounds(tag, bounds, 4);
		}
	}


// *****************************************************************************
// * defCallbackCollector::resizeArray :
// *	This method is used when the target is an array valued device.  It will 
// *	resize the multiple dimension array to match the new requirements.
// *****************************************************************************
void defCallbackCollector::resizeArray ( cdevData &data, int tag, cdevDataTypes type, size_t size, size_t arraySize )
	{
	size_t elemCnt = size*arraySize;
	size_t oldElemCnt = 0;
	
	// *********************************************************************
	// * If either of the primary size specifiers is 0, then do nothing...
	// * the array has no depth.
	// *********************************************************************
	if(size==0 || arraySize==0) return;
	
	// *********************************************************************
	// * If the specified tag does not exist in the cdevData object, then
	// * call installEmptyArray to generate a new array of the appropriate
	// * size.
	// *********************************************************************
	if(data.getType(tag)==CDEV_INVALID)
		{
		installEmptyArray(data, tag, type, size, arraySize);
		return;
		}

	size_t nDim = 0;
	size_t oldBounds[4];

	// *********************************************************************
	// * The newBounds array represents the new dimensions of the data.
	// *********************************************************************
	size_t newBounds[4];
	newBounds[0] = 0;
	newBounds[1] = size;
	newBounds[2] = 0;
	newBounds[3] = arraySize;
		
	// *********************************************************************
	// * Read the number of elements and the array size from the provided
	// * cdevData object.  This will be used to determine if the size of
	// * the object has been changed.
	// *********************************************************************
	data.getElems(tag, &oldElemCnt);
	if(data.getDim(tag, &nDim)==CDEV_SUCCESS && nDim==2)
		{
		data.getBounds(tag, oldBounds, 4);
		}
	else	{
		oldBounds[0] = 0;
		oldBounds[1] = size;
		oldBounds[2] = 0;
		oldBounds[3] = oldElemCnt/size;
		}

	// *********************************************************************
	// * At this point, if the bound specification does not match the number
	// * of elements stored in the cdevData item --- force an automatic 
	// * correction.  This may cause some of the data to become invalid, 
	// * but it will correct the problem for future data to be introduced 
	// * into the object.
	// *********************************************************************
	if((oldBounds[1]*oldBounds[3])!=oldElemCnt)
		{
		oldBounds[1] = size;
		oldBounds[3] = oldElemCnt/size;
		}
	
	// *********************************************************************
	// * If the oldBounds of the two arrays do not match, then copy the data
	// * from the existing array into a new array and then reinstall it
	// * to the cdevData object.
	// *********************************************************************
	if(oldBounds[3]<arraySize)
		{
		switch(type)
			{
			case CDEV_BYTE:
				{
				unsigned char * oldPtr = new unsigned char[oldElemCnt];
				unsigned char * newPtr = new unsigned char[elemCnt];
				memset(newPtr, 0, sizeof(unsigned char)*elemCnt);
				data.get(tag, oldPtr);
				for(int idx=0; idx<oldBounds[1] && idx<size; idx++)
					{
					for(int jIdx=0; jIdx<oldBounds[3]; jIdx++) 
						{
						newPtr[(idx*newBounds[3])+jIdx] = 
						oldPtr[(idx*oldBounds[3])+jIdx];
						}
					}
				data.insert(tag, newPtr, elemCnt, (arraySize>1)?2:1);
				delete[] newPtr;
				delete[] oldPtr;
				break;
				}
			
			case CDEV_INT16:
				{
				short * oldPtr = new short[oldElemCnt];
				short * newPtr = new short[elemCnt];
				memset(newPtr, 0, sizeof(short)*elemCnt);
				data.get(tag, oldPtr);
				for(int idx=0; idx<oldBounds[1] && idx<size; idx++)
					{
					for(int jIdx=0; jIdx<oldBounds[3]; jIdx++) 
						{
						newPtr[(idx*newBounds[3])+jIdx] = 
						oldPtr[(idx*oldBounds[3])+jIdx];
						}
					}
				data.insert(tag, newPtr, elemCnt, (arraySize>1)?2:1);
				delete newPtr;
				delete oldPtr;
				break;
				}

			case CDEV_UINT16:
				{
				unsigned short * oldPtr = new unsigned short[oldElemCnt];
				unsigned short * newPtr = new unsigned short[elemCnt];
				memset(newPtr, 0, sizeof(unsigned short)*elemCnt);
				data.get(tag, oldPtr);
				for(int idx=0; idx<oldBounds[1] && idx<size; idx++)
					{
					for(int jIdx=0; jIdx<oldBounds[3]; jIdx++) 
						{
						newPtr[(idx*newBounds[3])+jIdx] = 
						oldPtr[(idx*oldBounds[3])+jIdx];
						}
					}
				data.insert(tag, newPtr, elemCnt, (arraySize>1)?2:1);
				delete newPtr;
				delete oldPtr;
				break;
				}

			case CDEV_INT32:
				{
				long * oldPtr = new long[oldElemCnt];
				long * newPtr = new long[elemCnt];
				memset(newPtr, 0, sizeof(long)*elemCnt);
				data.get(tag, oldPtr);
				for(int idx=0; idx<oldBounds[1] && idx<size; idx++)
					{
					for(int jIdx=0; jIdx<oldBounds[3]; jIdx++) 
						{
						newPtr[(idx*newBounds[3])+jIdx] = 
						oldPtr[(idx*oldBounds[3])+jIdx];
						}
					}
				data.insert(tag, newPtr, elemCnt, (arraySize>1)?2:1);
				delete newPtr;
				delete oldPtr;
				break;
				}

			case CDEV_UINT32:
				{
				unsigned long * oldPtr = new unsigned long[oldElemCnt];
				unsigned long * newPtr = new unsigned long[elemCnt];
				memset(newPtr, 0, sizeof(unsigned long)*elemCnt);
				data.get(tag, oldPtr);
				for(int idx=0; idx<oldBounds[1] && idx<size; idx++)
					{
					for(int jIdx=0; jIdx<oldBounds[3]; jIdx++) 
						{
						newPtr[(idx*newBounds[3])+jIdx] = 
						oldPtr[(idx*oldBounds[3])+jIdx];
						}
					}
				data.insert(tag, newPtr, elemCnt, (arraySize>1)?2:1);
				delete newPtr;
				delete oldPtr;
				break;
				}

			case CDEV_FLOAT:
				{
				float * oldPtr = new float[oldElemCnt];
				float * newPtr = new float[elemCnt];
				memset(newPtr, 0, sizeof(float)*elemCnt);
				data.get(tag, oldPtr);
				for(int idx=0; idx<oldBounds[1] && idx<size; idx++)
					{
					for(int jIdx=0; jIdx<oldBounds[3]; jIdx++) 
						{
						newPtr[(idx*newBounds[3])+jIdx] = 
						oldPtr[(idx*oldBounds[3])+jIdx];
						}
					}
				data.insert(tag, newPtr, elemCnt, (arraySize>1)?2:1);
				delete newPtr;
				delete oldPtr;
				break;
				}

			case CDEV_DOUBLE:
				{
				double * oldPtr = new double[oldElemCnt];
				double * newPtr = new double[elemCnt];
				memset(newPtr, 0, sizeof(double)*elemCnt);
				data.get(tag, oldPtr);
				for(int idx=0; idx<oldBounds[1] && idx<size; idx++)
					{
					for(int jIdx=0; jIdx<oldBounds[3]; jIdx++) 
						{
						newPtr[(idx*newBounds[3])+jIdx] = 
						oldPtr[(idx*oldBounds[3])+jIdx];
						}
					}
				data.insert(tag, newPtr, elemCnt, (arraySize>1)?2:1);
				delete newPtr;
				delete oldPtr;
				break;
				}

			// *****************************************************
			// * Note: The string object is special.  Since a NULL
			// * ptr will terminate the list of strings that have
			// * been inserted into the object, the user must set 
			// * all 'unset' entries to point to a NULL terminated
			// * string before performing the insertion.
			// *****************************************************
			case CDEV_STRING:
				{
				char ** oldPtr = new char * [oldElemCnt];
				char ** newPtr = new char * [elemCnt];
				int     idx = 0;
				char    val = '\0';
				for(idx=0; idx<elemCnt; idx++) newPtr[idx] = &val;
				data.get(tag, oldPtr);
				for(idx=0; idx<oldBounds[1] && idx<size; idx++)
					{
					for(int jIdx=0; jIdx<oldBounds[3]; jIdx++) 
						{
						newPtr[(idx*newBounds[3])+jIdx] = 
						oldPtr[(idx*oldBounds[3])+jIdx];
						}
					}
				data.insert(tag, newPtr, elemCnt, (arraySize>1)?2:1);
				delete[] oldPtr;
				for(idx=0; idx<elemCnt; idx++)
					{
					if(newPtr[idx]!=&val) delete newPtr[idx];
					}
				delete[] newPtr;
				break;
				}

			// *****************************************************
			// * The cdev_TS_STAMP object is a structure rather than
			// * a primitive. As such, it does not support the 
			// * direct assignment operations that are used by the 
			// * other objects.  To accomodate this, we'll use the
			// * memmove function to copy data from the old object
			// * to the new one.
			// *****************************************************
			case CDEV_TIMESTAMP:
				{
				cdev_TS_STAMP * oldPtr = new cdev_TS_STAMP[oldElemCnt];
				cdev_TS_STAMP * newPtr = new cdev_TS_STAMP[elemCnt];
				memset(newPtr, 0, sizeof(cdev_TS_STAMP)*elemCnt);
				data.get(tag, oldPtr);
				for(int idx=0; idx<oldBounds[1] && idx<size; idx++)
					{
					memmove(newPtr+(idx*newBounds[3]),
						oldPtr+(idx*oldBounds[3]),
						oldBounds[3]*sizeof(cdev_TS_STAMP));
					}
				data.insert(tag, newPtr, elemCnt, (arraySize>1)?2:1);
				delete newPtr;
				delete oldPtr;
				break;
				}
		
			case CDEV_INVALID:
			break;			
			}

		// *************************************************************
		// * Finally, set the bounds to the new boundaries.
		// *************************************************************
		data.setBounds(tag, newBounds, 4);
		}
	}
