//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
//-----------------------------------------------------------------------------
//
// Description:
//      rsvcDataInputStream that converts a byte stream into a rsvcData
//
// Author:  
//      Jie Chen
//      Jefferson Lab HPC Group
//
// Revision History:
//   $Log: rsvcDataInputStream.java,v $
//   Revision 1.1  1999/10/18 17:12:41  chen
//   *** empty log message ***
//
//
//
import java.io.*;
import java.math.*;

public class rsvcDataInputStream
{
    // for 32 bit machine
    public static final int _RSVC_STREAM_BYTE_UNIT = 4;
    
    /**
     * Return a long word (32 bit) aligned size from original size
     */
    public static final int _RSVC_RNDUP (int x)
    {
	return ((x + _RSVC_STREAM_BYTE_UNIT - 1)/_RSVC_STREAM_BYTE_UNIT)
	    * _RSVC_STREAM_BYTE_UNIT;
    }

    // underlying data output stream
    private DataInputStream input = null;

    /**
     * Construct a data output stream
     */
    public rsvcDataInputStream (InputStream stream)
    {
	try {
	    input = new DataInputStream (stream);
	}catch (Exception e) {
	    e.printStackTrace();
	}
    }

    /**
     * Read a single byte char from stream
     */
    public byte readByte () throws IOException
    {
	int tmp;
	try {
	    tmp = input.readInt ();
	}catch (IOException e) {
	    throw e;
	}
	return (byte)tmp;
    }

    /**
     * Read a value of short from the stream
     */
    public short readShort () throws IOException
    {
	int tmp;
	try {
	    tmp = input.readInt ();
	}catch (IOException e) {
	    throw e;
	}
	return (short)tmp;
    }

    /**
     * Read a value of int from stream
     */
    public int readInt () throws IOException
    {
	int tmp;
	try {
	    tmp = input.readInt ();
	}catch (IOException e) {
	    throw e;
	}
	return tmp;
    }

    /**
     * Read a value of long from stream
     */
    public long readLong () throws IOException
    {
	long tmp;
	try {
	    tmp = input.readLong ();
	}catch (IOException e) {
	    throw e;
	}
	return tmp;
    }


    /**
     * Read a value of float from stream
     */
    public float readFloat () throws IOException
    {
	float tmp;
	try {
	    tmp = input.readFloat ();
	}catch (IOException e) {
	    throw e;
	}
	return tmp;
    }


    /**
     * Read a value of double to stream
     */
    public double readDouble () throws IOException
    {
	double tmp;
	try {
	    tmp = input.readDouble ();
	}catch (IOException e) {
	    throw e;
	}
	return tmp;
    }

    /**
     * Read a string as series of byte using default encoding
     */
    public String readString () throws IOException
    {
	int len, rlen, i;
	
	// first find out str size
	try {
	    len = input.readInt ();
	}catch (IOException e) {
	    throw e;
	}
	
	// readout next bytes of strlen 
	byte[] tmp = new byte[len];
	try {
	    input.readFully (tmp, 0, len);
	}catch (IOException e) {
	    throw e;
	}

	// find out read length of this string by looking for the first 0
	// in this byte array
	rlen = 0;
	for (i = 0; i < len; i++) {
	    if (tmp[i] == 0)
		break;
	}
	rlen = i;

	return new String (tmp, 0, rlen);
    }

    /**
     * Read a series of byte
     */
    public byte[] readBytes () throws IOException
    {
	int len;

	try {
	    len = input.readInt();
	}catch (IOException e){
	    throw e;
	}

	byte[] result = new byte[len];
	try {
	    input.readFully (result, 0, len);
	}catch (IOException e){
	    throw e;
	}

	return result;
    }


    /**
     * Read a number of bytes: blocked until all bytes are read
     */
    public byte[] readBytes (int len) throws IOException
    {
	byte[] result = new byte[len];
	try {
	    input.readFully (result, 0, len);
	}catch (IOException e){
	    throw e;
	}

	return result;
    }
    
    /**
     * Read a time stamp from the stream
     */
    public rsvcTimeStamp readTimeStamp () throws IOException
    {
	rsvcTimeStamp ts = new rsvcTimeStamp ();

	try {
	    ts.secPastEpoch = input.readInt();
	}catch (IOException e){
	    throw e;
	}

	try{
	    ts.nsec = input.readInt();
	}catch (IOException e){
	    throw e;
	}
	return ts;
    }

}
