//-----------------------------------------------------------------------------
// Copyright (c) 1995      Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
//-----------------------------------------------------------------------------
//
// Description:
//	This header file contains the class definitions for the classes
//	associated with a cdevTranNode.  The cdevTranNode is an extension
// 	of the cdevTranObj.  It provides a mechanism for tracking allocated
// 	and unallocated cdevTranObj objects and prevents inadvertent writes
// 	to unallocated memory.
//
// Author:  Walt Akers
//
//
//
#ifndef _CDEV_TRAN_NODE_H_
#define _CDEV_TRAN_NODE_H_

#include <cdevTranObj.h>
	
#ifndef TRUE
	#define TRUE  1
	#define FALSE 0
#endif

class cdevTranNode : public cdevTranObj
{
private:
	static int            count;
	static cdevTranNode * freeList;
	static cdevTranNode * activeList;
	static int            initialized;

	cdevTranNode        * next;
	int                   allocated;
			
protected:
	int     finished_;
	int     permanent_;
	int     transaction_;
	
public:
	cdevTranNode ( void );
	cdevTranNode ( cdevSystem*, cdevRequestObject*, cdevData*, cdevCallback*);
	~cdevTranNode( void );	

	void * operator new    ( size_t size );
	void   operator delete ( void * ptr );

	int    validate        ( int Transaction );
	void   transaction     ( int Transaction );
	int    transaction     ( void );
	void   finished        ( int Finished );
	int    finished        ( void );
	void   permanent       ( int Permanent );
	int    permanent       ( void );
	
	static cdevTranNode * find ( int Transaction );
	static void           init ( void );
};

// *****************************************************************************
// * cdevTranNode :
// *	Allocates a cdevTranNode with no contents.
// *****************************************************************************
inline cdevTranNode::cdevTranNode ( void ) 
	: cdevTranObj(), finished_(FALSE), permanent_(FALSE)
	{
	if(!initialized) init();
	transaction_ = count++;
	disableDeleteCbk();
	}

// *****************************************************************************
// * cdevTranNode :
// *	Allocates a cdevTranNode populated with the required data values.
// *****************************************************************************
inline cdevTranNode::cdevTranNode ( cdevSystem * sys, cdevRequestObject *req,
			     cdevData * data, cdevCallback * callback ) 
	: cdevTranObj(sys, req, data, callback), finished_(FALSE), permanent_(FALSE)
	{
	if(!initialized) init();
	transaction_ = count++;
	disableDeleteCbk();
	}
	
// *****************************************************************************
// * ~cdevTranNode:
// *	Deletes the cdevTranNode object.
// *****************************************************************************
inline cdevTranNode::~cdevTranNode ( void )
	{
	}
	
// *****************************************************************************
// * cdevTranNode::validate:
// *	Allows the user to verify that the cdevTranNode that he has a pointer
// *	to, corresponds to the specified Transaction number.
// *****************************************************************************
inline int cdevTranNode::validate ( int Transaction )
	{
	return (allocated==TRUE && transaction_==Transaction)?CDEV_SUCCESS:CDEV_NOTFOUND; 
	}
	
// *****************************************************************************
// * cdevTranNode::transaction:
// *	Allows the caller to specify a new transaction number.
// *****************************************************************************
inline void cdevTranNode::transaction ( int Transaction )
	{
	transaction_ = Transaction;
	}


// *****************************************************************************
// * cdevTranNode::transaction:
// *	Allows the caller to obtain the transaction number.
// *****************************************************************************
inline int cdevTranNode::transaction ( void )
	{
	return transaction_;
	}
	
// *****************************************************************************
// * cdevTranNode::finished:
// * 	Allows the caller to set the finished flag.
// *****************************************************************************
inline void cdevTranNode::finished ( int Finished )
	{
	finished_ = Finished?TRUE:FALSE;
	}
	
// *****************************************************************************
// * cdevTranNode::finished:
// *	Allows the caller to read the contents of the finished flag.
// *****************************************************************************
inline int cdevTranNode::finished ( void )
	{
	return finished_;
	}
	
// *****************************************************************************
// * cdevTranNode::permanent:
// * 	Allows the caller to set the permanent flag.
// *****************************************************************************
inline void cdevTranNode::permanent ( int Permanent )
	{
	permanent_ = Permanent?TRUE:FALSE;
	}
	
// *****************************************************************************
// * cdevTranNode::permanent:
// *	Allows the caller to read the contents of the permanent flag.
// *****************************************************************************
inline int cdevTranNode::permanent ( void )
	{
	return permanent_;
	}
	
#endif /* _CDEV_TRAN_NODE_H_ */
