/************************************************************************
 *									*
 *	Copyright 1992 by Motorola Mobile Data Division,		*
 *			  Bothell, WA					*
 *									*
 *	Motorola hereby grants permission to use, copy, modify and	*
 *	distribute  this software and its documentation for any		*
 *	purpose and without fee, provided that you retain this		*
 *	copyright notice in all copies.  Motorola makes no		*
 *	representations about the suitability of this software for any	*
 *	purpose.  Motorola provides this software ``as is'' without	*
 *	express or implied warranty.					*
 *									*
 ************************************************************************/
#ifndef lint
static char RCSid[] = "$Header: /usr/local/source/coda_source/Tcl/src7.4/svipc/svipcMsg.c,v 1.3 2003/11/01 00:17:20 gurjyan Exp $";
#endif /* lint */

/*
 * File: SVipc/svipcMsg.c
 * Facility: Tcl C Routines
 * Author: Joe Kelsey
 * Description:
 *	Tcl commands to interact with message queues.
 *
 * Global Functions:
 *	Svipc_MsggetCmd
 *	Svipc_MsgrcvCmd
 *	Svipc_MsgrmidCmd
 *	Svipc_MsgsetCmd
 *	Svipc_MsgsndCmd
 *	Svipc_MsgstatCmd
 *
 * $Log: svipcMsg.c,v $
 * Revision 1.3  2003/11/01 00:17:20  gurjyan
 * *** empty log message ***
 *
 * Revision 1.2  1998/11/06 15:24:51  timmer
 * Linux port
 *
 * Revision 1.1.1.1  1996/08/21 19:30:08  heyes
 * Imported sources
 *
 * Revision 1.5  1993/11/22  23:53:54  kelsey
 * Fix msgctl() calls to remove inappropriate second arg of 0.
 *
 * Revision 1.4  1993/10/01  23:06:12  kelsey
 * Fix all of the AppendElement calls to add last 0 argument.
 *
 * Revision 1.3  1993/08/18  21:34:41  kelsey
 * Use Svipc_ prefix instead of Tcl_.
 * Update for Tcl 7.0.
 *
 * Revision 1.2  1993/04/09  01:30:18  kelsey
 * Make keywords abbreviatable.
 * Make stat return structured status even when variable specified.
 * Fix coding errors.
 *
 * Revision 1.1  1993/04/02  22:14:31  kelsey
 * Initial revision
 *
 *
 * Revision 1.1  2003/10/08  22:14:31  gurjyan
 * add deffinition of msgbuf
 *
*
 */

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include <tcl.h>

#ifdef SHRIKE
struct msgbuf
  {
    long int mtype;             /* type of received/sent message */
    char mtext[1];              /* text of the message */
  };
#endif

/*
 * Function: Svipc_MsggetCmd
 * Description:
 *	Acquire a message queue id.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments (3-6).
 * char **	argv		Vector of command arguments.
 *				[ msgget key mode create excl ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 */
/* ARGSUSED */
int
Svipc_MsggetCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  key_t key;
  int nmsgs;
  int msgflg;
  int msqid;
  char r_msqid[20];

  if (argc < 2 || 5 < argc)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" key ?mode? ?create? ?excl?", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int *)&key) != TCL_OK)
    {
      if (strncmp (argv[1], "private", strlen (argv[1])))
	{
	  return TCL_ERROR;
	}
      key = IPC_PRIVATE;
    }

  if (argc > 2)
    {
      char **pargv = argv + 2;

      if (Tcl_GetInt (interp, *pargv, &msgflg) != TCL_OK)
	{
	  msgflg = 0666;
	}
      else
	{
	  pargv++;
	}

      while (*pargv)
	{
	  int plen = strlen (*pargv);

	  if (strncmp (*pargv, "create", plen) == 0)
	    {
	      msgflg |= IPC_CREAT;
	    }
	  else if (strncmp (*pargv, "excl", plen) == 0)
	    {
	      msgflg |= IPC_EXCL;
	    }
	  else
	    {
	      (void) strcpy (interp->result,
			     "flag argument must be \"create\" or \"excl\".");
	      return TCL_ERROR;
	    }
	  pargv++;
	}
    }
  else
    {
      msgflg = 0666;
    }

  if ((msqid = msgget (key, msgflg)) == -1)
    {
      int i;
      char flgstr[20];

      (void) sprintf (flgstr, "0%o", msgflg);
      Tcl_AppendResult (interp, "cannot get message queue id for \"",
			argv[1], " ", flgstr, (char *)0);
      for (i = 2; i < argc; i++)
	{
	  Tcl_AppendResult (interp, i == 2 ? " (" : " ", argv[i], (char *)0);
	}
      Tcl_AppendResult (interp, argc == 2 ? "\": " : ")\": ",
			Tcl_PosixError (interp), (char *)0);
      return TCL_ERROR;
    }

  (void) sprintf (interp->result, "%d", msqid);

  return TCL_OK;
}

/*
 * Function: Svipc_MsgrcvCmd
 * Description:
 *	Perform a msgrcv system call.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments (3-6).
 * char **	argv		Vector of command arguments.
 *				[ msgrcv msqid msgtyp ?msgsz? ?noerror? 
 *					 ?nowait? ?varName?]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 */
/* ARGSUSED */
int
Svipc_MsgrcvCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
 int msqid;
  struct msgbuf *msgp;
  int msgsz;
  long msgtyp;
  int msgflg = 0;
  char *varName = 0;
  char rstr[20];

  if (argc < 3 || argc > 7)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" msqid msgtyp ?msgsz? ?noerror? ?nowait? ?varName?",
			(char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], &msqid) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[2], (int *)&msgtyp) != TCL_OK)
    {
      return TCL_ERROR;
    }
  
  if (argc > 3)
    {
      char **pargv = argv + 3;

      if (Tcl_GetInt (interp, *pargv, &msgsz) != TCL_OK)
	{
	  msgsz = BUFSIZ;
	}
      else
	{
	  pargv++;
	}

      while (*pargv)
	{
	  int plen = strlen (*pargv);

	  if (plen > 2 && strncmp (*pargv, "noerror", plen) == 0)
	    {
	      msgflg |= MSG_NOERROR;
	    }
	  else if (plen > 2 && strncmp (*pargv, "nowait", plen) == 0)
	    {
	      msgflg |= IPC_NOWAIT;
	    }
	  else
	    {
	      if (varName)
		{
		  Tcl_AppendResult (interp,
				    "cannot return into two variables: \"",
				    varName, "\" and \"", *pargv, "\".",
				    (char *)0);
		  return TCL_ERROR;
		}
	      varName = *pargv;
	    }
	  pargv++;
	}
    }

  msgp = (struct msgbuf *) ckalloc (sizeof(*msgp) + msgsz - 1);
  if ((msgsz = msgrcv (msqid, msgp, msgsz, msgtyp, msgflg)) == -1)
    {
      if (errno == EINTR)
	{
	  (void) strcpy (interp->result, "eintr");
	  ckfree ((char *)msgp);
	  return TCL_OK;
	}
      else if (errno == EAGAIN)
	{
	  (void) strcpy (interp->result, "eagain");
	  ckfree ((char *)msgp);
	  return TCL_OK;
	}
      else
	{
	  Tcl_AppendResult (interp,
			    "cannot receive from message queue id ",
			    argv[1], ": ", Tcl_PosixError (interp), (char *)0);
	  ckfree ((char *)msgp);
	  return TCL_ERROR;
	}
    }

  sprintf (rstr, "%d", msgp->mtype);
  if (varName)
    {
      (void) Tcl_SetVar2 (interp, varName, "mtype", rstr, (int)0);
      (void) Tcl_SetVar2 (interp, varName, "mtext", msgp->mtext, (int)0);
    }

  Tcl_AppendElement (interp, rstr);
  Tcl_AppendElement (interp, msgp->mtext);

  ckfree ((char *)msgp);
  return TCL_OK;	  
}

/*
 * Function: Svipc_MsgrmidCmd
 * Description:
 *	Remove the message queue from the system.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments.
 * char **	argv		Vector of command arguments.
 *				[ msgrmid msqid ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 *	Modifies message queue.
 */
/* ARGSUSED */
int
Svipc_MsgrmidCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  int msqid;

  if (argc != 2)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0], " msqid", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], &msqid) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (msgctl (msqid, IPC_RMID, (struct msqid_ds *)0) == -1)
    {
      Tcl_AppendResult (interp, "cannot remove message queue id ", argv[1],
			": ", Tcl_PosixError (interp), (char *)0);
      return TCL_ERROR;
    }
  return TCL_OK;
}

/*
 * Function: Svipc_MsgsetCmd
 * Description:
 *	Perform the IPC_SET function on the msqid.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments.
 * char **	argv		Vector of command arguments.
 *				[ msgset msqid msg_perm msg_qbytes ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 *	Modifies message queue.
 */
/* ARGSUSED */
int
Svipc_MsgsetCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  int msqid;
  int temp;
  int largc;
  char **largv;
  struct msqid_ds msqid_ds;

  if (argc != 4)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" msqid msg_perm msg_qbytes", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], &msqid) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (Tcl_SplitList (interp, argv[2], &largc, &largv) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (largc < 3)
    {
      (void) strcpy (interp->result,
		     "msg_perm list needs 3 arguments: uid gid mode.");
      ckfree ((char *)largv);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, largv[0], &temp) != TCL_OK)
    {
      ckfree ((char *)largv);
      return TCL_ERROR;
    }
  msqid_ds.msg_perm.uid = temp;

  if (Tcl_GetInt (interp, largv[1], &temp) != TCL_OK)
    {
      ckfree ((char *)largv);
      return TCL_ERROR;
    }
  msqid_ds.msg_perm.gid = temp;

  if (Tcl_GetInt (interp, largv[2], &temp) != TCL_OK)
    {
      ckfree ((char *)largv);
      return TCL_ERROR;
    }
  msqid_ds.msg_perm.mode = temp;

  ckfree ((char *)largv);

  if (Tcl_GetInt (interp, argv[3], &temp) != TCL_OK)
    {
      return TCL_ERROR;
    }
  msqid_ds.msg_qbytes = temp;

  if (msgctl (msqid, IPC_SET, &msqid_ds) == -1)
    {
      Tcl_AppendResult (interp,
			"cannot modify permissions for message queue id ",
			argv[1], " {", argv[2], "} ", argv[3], ": ",
			Tcl_PosixError (interp), (char *)0);
      return TCL_ERROR;
    }
  return TCL_OK;
}

/*
 * Function: Svipc_MsgsndCmd
 * Description:
 *	Perform a msgsnd system call.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments (3-6).
 * char **	argv		Vector of command arguments.
 *				[ msgsnd msqid msgtyp msgtext ?nowait? ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 */
/* ARGSUSED */
int
Svipc_MsgsndCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  int msqid;
  struct msgbuf *msgp;
  int msgsz;
  long msgtyp;
  int msgflg = 0;

  if (argc < 4 || argc > 5)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" msqid msgtyp msgtext ?nowait?",
			(char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], &msqid) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[2], (int *)&msgtyp) != TCL_OK)
    {
      return TCL_ERROR;
    }
  
  if (argc == 5)
    {
      if (strncmp (argv[4], "nowait", strlen (argv[4])) == 0)
	{
	  msgflg |= IPC_NOWAIT;
	}
      else
	{
	  (void) strcpy (interp->result,
			 "msgflg must be \"nowait\".");
	  return TCL_ERROR;
	}
    }

  msgsz = strlen (argv[3]);

  msgp = (struct msgbuf *)ckalloc (sizeof (*msgp) + msgsz);
  msgp->mtype = msgtyp;
  (void) strcpy (msgp->mtext, argv[3]);

  if (msgsnd (msqid, msgp, msgsz+1, msgflg) == -1)
    {
      if (errno == EINTR)
	{
	  (void) strcpy (interp->result, "eintr");
	  ckfree ((char *)msgp);
	  return TCL_OK;
	}
      else if (errno == EAGAIN)
	{
	  (void) strcpy (interp->result, "eagain");
	  ckfree ((char *)msgp);
	  return TCL_OK;
	}
      else
	{
	  Tcl_AppendResult (interp,
			    "cannot send to message queue id ",
			    argv[1], ": ", Tcl_PosixError (interp), (char *)0);
	  ckfree ((char *)msgp);
	  return TCL_ERROR;
	}
    }

  ckfree ((char *)msgp);
  return TCL_OK;	  
}

/*
 * Function: Svipc_MsgstatCmd
 * Description:
 *	Return a vector of information about a message queue.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments.
 * char **	argv		Vector of command arguments.
 *				[ msgstat msqid ?varName? ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 */
/* ARGSUSED */
int
Svipc_MsgstatCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;
/* Detail Design:

 */

{
  int msqid;
  struct msqid_ds msqid_ds;
  int i;
  char rstr[7][20];
  char *margv[7];
  char *msg_list;

  if (argc < 2 || 3 < argc)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0], " msqid ?varName?",
			(char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], &msqid) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (msgctl (msqid, IPC_STAT, &msqid_ds) == -1)
    {
      Tcl_AppendResult (interp, "cannot stat message queue id ", argv[1], ": ",
			Tcl_PosixError (interp), (char *)0);
      return TCL_ERROR;
    }

  (void) sprintf (rstr[0], "%d", msqid_ds.msg_perm.cuid);
  (void) sprintf (rstr[1], "%d", msqid_ds.msg_perm.cgid);
  (void) sprintf (rstr[2], "%d", msqid_ds.msg_perm.uid);
  (void) sprintf (rstr[3], "%d", msqid_ds.msg_perm.gid);
  (void) sprintf (rstr[4], "%07o", msqid_ds.msg_perm.mode);
#ifdef LINUX
  (void) sprintf (rstr[5], "%d", msqid_ds.msg_perm.__seq);
  (void) sprintf (rstr[6], "%d", msqid_ds.msg_perm.__key);
#else
  (void) sprintf (rstr[5], "%d", msqid_ds.msg_perm.seq);
  (void) sprintf (rstr[6], "%d", msqid_ds.msg_perm.key);
#endif
  margv[0] = rstr[0];
  margv[1] = rstr[1];
  margv[2] = rstr[2];
  margv[3] = rstr[3];
  margv[4] = rstr[4];
  margv[5] = rstr[5];
  margv[6] = rstr[6];
  
  msg_list = Tcl_Merge (7, margv);
  Tcl_AppendElement (interp, msg_list);
  ckfree (msg_list);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "cuid", rstr[0], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "cgid", rstr[1], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "uid", rstr[2], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "gid", rstr[3], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "mode", rstr[4], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "seq", rstr[5], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "key", rstr[6], (int)0);
    }

  (void) sprintf (rstr[0], "%d", msqid_ds.msg_cbytes);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "cbytes", rstr[0], (int)0);
    }

  (void) sprintf (rstr[0], "%d", msqid_ds.msg_qnum);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "qnum", rstr[0], (int)0);
    }

  (void) sprintf (rstr[0], "%d", msqid_ds.msg_qbytes);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "qbytes", rstr[0], (int)0);
    }

  (void) sprintf (rstr[0], "%d", msqid_ds.msg_lspid);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "lspid", rstr[0], (int)0);
    }

  (void) sprintf (rstr[0], "%d", msqid_ds.msg_lrpid);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "lrpid", rstr[0], (int)0);
    }

  (void) sprintf (rstr[0], "%d", msqid_ds.msg_stime);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "stime", rstr[0], (int)0);
    }

  (void) sprintf (rstr[0], "%d", msqid_ds.msg_rtime);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "rtime", rstr[0], (int)0);
    }

  (void) sprintf (rstr[0], "%d", msqid_ds.msg_ctime);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "ctime", rstr[0], (int)0);
    }

  return TCL_OK;
}
