/*----------------------------------------------------------------------------*
 *  Copyright (c) 1991, 1992  Southeastern Universities Research Association, *
 *                            Continuous Electron Beam Accelerator Facility   *
 *                                                                            *
 *    This software was developed under a United States Government license    *
 *    described in the NOTICE file included as part of this distribution.     *
 *                                                                            *
 * CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606 *
 *      heyes@cebaf.gov   Tel: (804) 249-7030    Fax: (804) 249-7363          *
 *----------------------------------------------------------------------------*
 * Discription: follows this header.
 *
 * Author:
 *	Graham Heyes
 *	CEBAF Data Acquisition Group
 *
 * Revision History:
 *      $Log: fb_trigLib.h,v $
 *      Revision 1.1.1.1  1996/08/21 19:21:00  heyes
 *      Imported sources
 *
 *	  Revision 1.1  1993/03/16  14:02:06  heyes
 *	  Initial revision
 *
 *
 *----------------------------------------------------------------------------*/

 /* trigLib.c -- Primitive trigger control for the FSCC

 File : trigLib.h

 Author: Graham Heyes
         Data Acquisition Group
         CEBAF Physics Div

 Routines:
	   int wdStart;             watchdog.
	   void fptriglink();       link in trigger isr
	   void fptenable();        enable trigger
	   void fptdisable();       
	   char fpttype();          read trigger type
	   void fpttypeset(tt);     set trigger type
	   void fpttypeset();
	   void fpttoggle();        toggle trigger bit - test...
	   void fpbusy();           raise busy
	   void fpnotbusy();        clear busy
	   void fpttest();          test for trigger (POLL)

-----------------------------------------------------------------------------------------*/
/*
Interrupts input into the 68901 general purpose lines
-----------------------------------------------------

 interrupt enable bits
*/

#define FSCC_TSTRB_ENBIT	7	/* Trigger input strobe */
#define FSCC_TSTRB_EN		(1<<FSCC_TSTRB_ENBIT)	/* Trigger input strobe */

#define PIT_BASE1		0x00640000	/* FCSEL  FASTBUS Parallel Port 1 (MC68230) */
#define APP_BASE                0x004c0000      /* Auxillary Parallel Port (TS Interface) */

#define MFP_BASE		0x00480000	/* MC68901 Multi-Function Peripheral */
#define MFP_GPDR              	0x00		/* GENERAL PURPOSE DATA REGISTER */
#define MFP_IERA              	0x03           	/* INT ENABLE REG A */
#define MFP_IMRA              	0x09           	/* INT MASK REG A */
#define MFP_IPRA             	0x05           	/* INT PENDING REG A  */

/*
------------------------------------------------------------------
 MC68230 Parallel Interface/Timer (PI/T) hardware register offsets
------------------------------------------------------------------
*/
#define PIT_PGCR       0x00                  /* PORT GENERAL CTRL REG */
#define PIT_PSRR       0x01                  /* PORT SERVICE REST REG */
#define PIT_PADDR      0x02                  /* PORT A DATA DIRECTION REG  */
#define PIT_PBDDR      0x03                  /* PORT B DATA DIRECTION REG  */
#define PIT_PCDDR      0x04                  /* PORT C DATA DIRECTION REG  */
#define PIT_PACR       0x06                  /* PORT A CONTROL REG */
#define PIT_PBCR       0x07                  /* PORT B CONTROL REG */
#define PIT_PADR       0x08                  /* PORT A DATA REG  */
#define PIT_PBDR       0x09                  /* PORT B DATA REG  */
#define PIT_PAAR       0x0A                  /* PORT A ALT REG (READ ONLY) */
#define PIT_PBAR       0x0B                  /* PORT B ALT REG (READ ONLY) */
#define PIT_PCDR       0x0C                  /* PORT C DATA REG  */
#define PIT_PSR        0x0D                  /* PORT STATUS REG */
#define PIT_TCR        0x10                  /* TIMER CONTROL REG  */
#define PIT_CPR        0x12                  /* COUNTER PRELOAD REG */
#define PIT_CPRH       0x13                  /* COUNTER PRELOAD REG - HI */
#define PIT_CPRM       0x14                  /* COUNTER PRELOAD REG - MID  */
#define PIT_CPRL       0x15                  /* COUNTER PRELOAD REG - LO */
#define PIT_TSR        0x1A                  /* TIMER STATUS REG */


#define mem8(a) (*(unsigned char *) (a))

static inline void wdStart()
{
  /* PIT 1 timer is watchdog */
  mem8(PIT_BASE1+PIT_TCR)  = 0xf2 ;	     /* Start the clock */           
  mem8(PIT_BASE1+PIT_CPRH)=100;
  mem8(PIT_BASE1+PIT_CPRM)=0x0;
  mem8(PIT_BASE1+PIT_CPRL)=0x0;		     /* set up time constant */
  mem8(PIT_BASE1+PIT_TCR)  = 0xf3 ;	     /* Start the clock */           
}

static inline void fptriglink(isr)
     VOIDFUNCPTR isr;
{
  static FUNCPTR handlerPtr = (FUNCPTR) 0;
  
  if (handlerPtr)
    free(handlerPtr);

  handlerPtr = intHandlerCreate(isr,0);
  intVecSet((VOIDFUNCPTR *) (0x4f<<2),(VOIDFUNCPTR) handlerPtr);

  mem8(MFP_BASE+MFP_IERA) &= ~FSCC_TSTRB_EN; /* disable interrupt */
#ifdef POLLING_MODE
  mem8(MFP_BASE+MFP_IMRA) &= ~FSCC_TSTRB_EN; /* enable interrupt mask */
#else
  mem8(MFP_BASE+MFP_IMRA) |= FSCC_TSTRB_EN;  /* enable interrupt mask */
#endif
}

static inline void fptenable()
{
  mem8(MFP_BASE+MFP_IERA) |= FSCC_TSTRB_EN;  /* interrupt enable */
}

static inline void fptdisable()
{
  mem8(MFP_BASE+MFP_IERA) &= ~FSCC_TSTRB_EN; /* interrupt disable */
}

static inline char fpttype()
{
#ifdef TRIG_SUPV
  return(((mem8(APP_BASE)&0x3c)>>2));
#else
  return(1);
#endif
}

static inline void fpttypeset(char tt)
{
  mem8(PIT_BASE1+PIT_PBDR) = (tt<<5) | (mem8(PIT_BASE1+PIT_PBDR)&0x1f);
}

static inline void fpttoggle()
{
  mem8(MFP_BASE+MFP_IERA) &= ~FSCC_TSTRB_EN; /* ping enable bit to*/
  mem8(MFP_BASE+MFP_IERA) |=  FSCC_TSTRB_EN; /* clear IPR */

  mem8(PIT_BASE1+PIT_PBDR)=0x10; /* Toggle the bit to clear busy logic */
  mem8(PIT_BASE1+PIT_PBDR)=0;	 /*  */
  
}

static inline void fpbusy()
{
  mem8(PIT_BASE1+PIT_PBDR) |=  0x10;
}

static inline void fpnotbusy()
{
  mem8(PIT_BASE1+PIT_PBDR) &= ~0x10;
}

static inline int fpttest()
{
  int ii;
  ii = ((mem8(MFP_BASE+MFP_IPRA) & 0x80)!=0);
  return(ii);
}

